package com.floreantpos.util;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Future;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.floreantpos.PosException;
import com.floreantpos.model.Attribute;
import com.floreantpos.model.ComboGroup;
import com.floreantpos.model.ComboItem;
import com.floreantpos.model.Discount;
import com.floreantpos.model.DosageForms;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.InventoryUnitGroup;
import com.floreantpos.model.Manufacturer;
import com.floreantpos.model.MedGenerics;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemModifierSpec;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.RecepieItem;
import com.floreantpos.model.ReportGroup;
import com.floreantpos.model.Specimen;
import com.floreantpos.model.Store;
import com.floreantpos.model.Tag;
import com.floreantpos.model.Tax;
import com.floreantpos.model.TaxGroup;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemGroup;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.MenuCategoryDAO;
import com.floreantpos.model.dao.MenuGroupDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.RecepieDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TestItemGroupDAO;
import com.floreantpos.model.util.DataProvider;

public class MenuItemJsonManager {
	private Set<String> attributeIdsSet, comboGroupIdsSet, comboItemIdsSet, discountIdsSet, dosageFormsIdsSet, inventoryStockUnitIdsSet, inventoryUnitIdsSet,
			inventoryUnitGroupIdsSet, manufacturerIdsSet, manufacturingItemIdsSet, medGenericsIdsSet, menuCategoryIdsSet, menuGroupIdsSet, menuItemIdsSet,
			menuItemModifierSpecIdsSet, productTypeIdsSet, recepieIdsSet, recepieItemIdsSet, reportGroupIdsSet, specimenIdsSet, storeIdsSet, tagIdsSet,
			taxIdsSet, taxGroupIdsSet, testItemIdsSet, testItemGroupIdsSet;

	public byte[] doExportAndGetFile(ProductType productType, Future<?> currentTask) throws Exception {
		return doExportAndGetFile(productType, currentTask, 0, -1);
	}

	public byte[] doExportAndGetFile(ProductType productType, Future<?> currentTask, int offset, int querySize) throws Exception {
		List<MenuItem> list = MenuItemDAO.getInstance().findByProductType(productType, offset, querySize);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		mapper.addMixIn(MenuItem.class, MenuItemMixIn.class);
		mapper.addMixIn(InventoryUnit.class, InventoryUnitMixIn.class);
		//mapper.addMixIn(Recepie.class, ReceipeMixIn.class);

		JSONArray jsonArray = new JSONArray();
		for (MenuItem menuItem : list) {
			if (currentTask != null && currentTask.isCancelled()) {
				break;
			}

			MenuItemDAO.getInstance().initialize(menuItem);
			JSONObject jsonObject = new JSONObject(mapper.writeValueAsString(menuItem));

			if (jsonObject.has("testItems")) {
				JSONArray testItems = jsonObject.getJSONArray("testItems");
				for (int i = 0; i < testItems.length(); i++) {
					Object object = testItems.get(i);
					if (object instanceof JSONObject) {
						JSONObject testItem = (JSONObject) object;
						if (testItem.has("testItemGroupId") && !testItem.isNull("testItemGroupId")) {
							String testItemGroupId = testItem.getString("testItemGroupId");
							if (currentTask != null && currentTask.isCancelled()) {
								break;
							}
							TestItemGroup testItemGroup = TestItemGroupDAO.getInstance().get(testItemGroupId);
							if (testItemGroup != null) {
								testItem.put("testItemGroup", new JSONObject(mapper.writeValueAsString(testItemGroup)));
							}
						}
						if (testItem.has("unit") && !testItem.isNull("unit")) {
							String unitId = testItem.getString("unit");
							if (currentTask != null && currentTask.isCancelled()) {
								break;
							}
							InventoryUnit unit = InventoryUnitDAO.getInstance().get(unitId);
							if (unit != null) {
								testItem.put("inventoryUnit", new JSONObject(mapper.writeValueAsString(unit)));
							}
						}
					}
				}
			}

			if (StringUtils.isNotBlank(menuItem.getMenuCategoryId())) {
				if (currentTask != null && currentTask.isCancelled()) {
					break;
				}
				MenuCategory menuCategory = MenuCategoryDAO.getInstance().get(menuItem.getMenuCategoryId());
				if (menuCategory != null) {
					jsonObject.put("menuCategory", new JSONObject(mapper.writeValueAsString(menuCategory)));
				}
			}

			if (StringUtils.isNotBlank(menuItem.getMenuGroupId())) {
				if (currentTask != null && currentTask.isCancelled()) {
					break;
				}
				MenuGroup menuGroup = MenuGroupDAO.getInstance().get(menuItem.getMenuGroupId());
				if (menuGroup != null) {
					jsonObject.put("menuGroup", new JSONObject(mapper.writeValueAsString(menuGroup)));
				}
			}

			if (StringUtils.isNotBlank(menuItem.getContainerId())) {
				Tag container = (Tag) DataProvider.get().getObjectOf(Tag.class, menuItem.getContainerId());
				if (container != null) {
					jsonObject.put("container", new JSONObject(mapper.writeValueAsString(container)));
				}
			}

			if (StringUtils.isNotBlank(menuItem.getSpecimenId())) {
				Specimen specimen = (Specimen) DataProvider.get().getObjectOf(Specimen.class, menuItem.getSpecimenId());
				if (specimen != null) {
					jsonObject.put("specimen", new JSONObject(mapper.writeValueAsString(specimen)));
				}
			}

			ReportGroup reportGroup = menuItem.getReportGroup();
			if (reportGroup != null) {
				jsonObject.put("reportGroup", new JSONObject(mapper.writeValueAsString(reportGroup)));
			}

			TaxGroup taxGroup = menuItem.getTaxGroup();
			if (taxGroup != null) {
				jsonObject.put("taxGroup", new JSONObject(mapper.writeValueAsString(taxGroup)));
			}

			if (StringUtils.isNotBlank(menuItem.getDefaultRecipeId())) {
				if (currentTask != null && currentTask.isCancelled()) {
					break;
				}
				Recepie recepie = RecepieDAO.getInstance().get(menuItem.getDefaultRecipeId());
				if (recepie != null && !recepie.isDeleted()) {
					recepie.setMenuItem(null);
					for (RecepieItem recepieItem : recepie.getRecepieItems()) {
						MenuItem inventoryItem = recepieItem.getInventoryItem();
						if (inventoryItem == null) {
							continue;
						}
						if (currentTask != null && currentTask.isCancelled()) {
							break;
						}
						MenuItemDAO.getInstance().initialize(inventoryItem);
					}
					jsonObject.put("recepie", new JSONObject(mapper.writeValueAsString(recepie)));
				}
			}

			jsonArray.put(jsonObject);
		}

		return IOUtils.toByteArray(IOUtils.toInputStream(jsonArray.toString(), "UTF-8"));
	}

	public boolean doImportAndSaveMenuItem(ProductType productType, File file, boolean shouldUpdateIfExists) throws Exception {
		if (!file.exists()) {
			throw new PosException("File not found.");
		}
		return doImportAndSaveMenuItem(productType, file, null, shouldUpdateIfExists);
	}

	public boolean doImportAndSaveMenuItem(ProductType productType, String json, boolean shouldUpdateIfExists) throws Exception {
		if (StringUtils.isBlank(json)) {
			throw new PosException("JSON not found.");
		}
		return doImportAndSaveMenuItem(productType, null, json, shouldUpdateIfExists);
	}

	private boolean doImportAndSaveMenuItem(ProductType productType, File file, String json, boolean shouldUpdateIfExists) throws Exception {
		ObjectMapper mapper = new ObjectMapper();
		mapper.addMixIn(MenuItem.class, MenuItemMixIn.class);
		mapper.addMixIn(RecepieItem.class, RecepieItemMixIn.class);

		JsonNode rootNode = file != null ? mapper.readTree(file) : mapper.readTree(json);

		if (rootNode.isArray()) {

			attributeIdsSet = new HashSet<>();
			comboGroupIdsSet = new HashSet<>();
			comboItemIdsSet = new HashSet<>();
			discountIdsSet = new HashSet<>();
			dosageFormsIdsSet = new HashSet<>();
			inventoryStockUnitIdsSet = new HashSet<>();
			inventoryUnitIdsSet = new HashSet<>();
			inventoryUnitGroupIdsSet = new HashSet<>();
			manufacturerIdsSet = new HashSet<>();
			manufacturingItemIdsSet = new HashSet<>();
			medGenericsIdsSet = new HashSet<>();
			menuCategoryIdsSet = new HashSet<>();
			menuGroupIdsSet = new HashSet<>();
			menuItemIdsSet = new HashSet<>();
			menuItemModifierSpecIdsSet = new HashSet<>();
			productTypeIdsSet = new HashSet<>();
			recepieIdsSet = new HashSet<>();
			recepieItemIdsSet = new HashSet<>();
			reportGroupIdsSet = new HashSet<>();
			specimenIdsSet = new HashSet<>();
			storeIdsSet = new HashSet<>();
			tagIdsSet = new HashSet<>();
			taxIdsSet = new HashSet<>();
			taxGroupIdsSet = new HashSet<>();
			testItemIdsSet = new HashSet<>();
			testItemGroupIdsSet = new HashSet<>();

			Set<String> labTestMethods = new HashSet<>();

			for (JsonNode menuItemJsonNode : rootNode) {
				try (Session session = MenuItemDAO.getInstance().createNewSession()) {
					Transaction beginTransaction = session.beginTransaction();

					convertMenuitemAndSave(mapper, session, menuItemJsonNode, labTestMethods, shouldUpdateIfExists);

					beginTransaction.commit();
				}
			}

			List<String> methods = StoreDAO.getRestaurant().getMethods();
			labTestMethods.removeIf(method -> methods.contains(method) && StringUtils.isBlank(method));

			if (!labTestMethods.isEmpty()) {
				Store restaurant = StoreDAO.getRestaurant();
				boolean isStoreSaveRequired = false;
				for (String method : labTestMethods) {
					if (restaurant.addToMethod(method) && !isStoreSaveRequired) {
						isStoreSaveRequired = true;
					}
				}
				if (isStoreSaveRequired) {
					StoreDAO.getInstance().saveOrUpdate(restaurant);
				}
			}
		}

		return true;
	}

	private <T> void saveOrUpdateIfAllowed(Session session, T entity, Object existingItem, boolean shouldUpdateIfExists) {
		if (existingItem == null) {
			session.save(entity);
		}
		else {
			if (shouldUpdateIfExists) {
				session.merge(entity);
			}
		}
	}

	private void convertMenuitemAndSave(ObjectMapper mapper, Session session, JsonNode menuItemJsonNode, Set<String> labTestMethod,
			boolean shouldUpdateIfExists) {
		MenuItem menuItem = mapper.convertValue(menuItemJsonNode, MenuItem.class);

		labTestMethod.add(menuItem.getLabTestMethod());

		List<InventoryStockUnit> stockUnits = menuItem.getStockUnits();
		menuItem.setStockUnits(new ArrayList<>());

		List<TestItem> testItems = menuItem.getTestItems();
		menuItem.setTestItems(new ArrayList<>());

		List<MenuItem> variants = menuItem.getVariants();
		menuItem.setVariants(new ArrayList<>());

		List<ComboItem> comboItems = menuItem.getComboItems();
		menuItem.setComboItems(new ArrayList<>());

		List<ComboGroup> comboGroups = menuItem.getComboGroups();
		menuItem.setComboGroups(new ArrayList<>());

		List<MenuItemModifierSpec> menuItemModiferSpecs = menuItem.getMenuItemModiferSpecs();
		menuItem.setMenuItemModiferSpecs(new ArrayList<>());

		List<Discount> discounts = menuItem.getDiscounts();
		menuItem.setDiscounts(new ArrayList<>());

		List<Attribute> attributes = menuItem.getAttributes();
		menuItem.setAttributes(new ArrayList<>());

		saveOrUpdateMenuItem(session, menuItem, shouldUpdateIfExists);

		if (stockUnits != null && !stockUnits.isEmpty()) {
			for (InventoryStockUnit inventoryStockUnit : stockUnits) {
				inventoryStockUnit.setMenuItemId(menuItem.getId());
				if (!inventoryStockUnitIdsSet.contains(inventoryStockUnit.getId())) {
					InventoryStockUnit existingItem = session.get(InventoryStockUnit.class, inventoryStockUnit.getId());
					if (existingItem != null) {
						inventoryStockUnit.setVersion(existingItem.getVersion());
					}
					saveOrUpdateIfAllowed(session, inventoryStockUnit, existingItem, shouldUpdateIfExists);
					inventoryStockUnitIdsSet.add(inventoryStockUnit.getId());
				}
			}
		}

		if (testItems != null && !testItems.isEmpty()) {
			for (TestItem testItem : testItems) {
				if (testItem == null || StringUtils.isBlank(testItem.getId())) {
					continue;
				}
				if (!testItemIdsSet.contains(testItem.getId())) {
					TestItem existingItem = session.get(TestItem.class, testItem.getId());
					saveOrUpdateIfAllowed(session, testItem, existingItem, shouldUpdateIfExists);
					testItemIdsSet.add(testItem.getId());
				}
			}
			menuItem.setTestItems(testItems);
		}

		if (variants != null && !variants.isEmpty()) {
			for (MenuItem variant : variants) {
				if (variant == null || StringUtils.isBlank(variant.getId())) {
					continue;
				}
				if (!menuItemIdsSet.contains(variant.getId())) {
					MenuItem existingItem = session.get(MenuItem.class, variant.getId());
					saveOrUpdateIfAllowed(session, variant, existingItem, shouldUpdateIfExists);
					menuItemIdsSet.add(variant.getId());
				}
			}
		}

		if (comboGroups != null && !comboGroups.isEmpty()) {
			for (ComboGroup comboGroup : comboGroups) {
				if (comboGroup == null || StringUtils.isBlank(comboGroup.getId())) {
					continue;
				}
				if (!comboGroupIdsSet.contains(comboGroup.getId())) {
					ComboGroup existingItem = session.get(ComboGroup.class, comboGroup.getId());
					saveOrUpdateIfAllowed(session, comboGroup, existingItem, shouldUpdateIfExists);
					comboGroupIdsSet.add(comboGroup.getId());
				}
			}
		}

		if (comboItems != null && !comboItems.isEmpty()) {
			for (ComboItem comboItem : comboItems) {
				if (comboItem == null || StringUtils.isBlank(comboItem.getId())) {
					continue;
				}
				if (!comboItemIdsSet.contains(comboItem.getId())) {
					ComboItem existingItem = session.get(ComboItem.class, comboItem.getId());
					saveOrUpdateIfAllowed(session, comboItem, existingItem, shouldUpdateIfExists);
					comboItemIdsSet.add(comboItem.getId());
				}
			}
		}

		if (menuItemModiferSpecs != null && !menuItemModiferSpecs.isEmpty()) {
			for (MenuItemModifierSpec menuItemModifierSpec : menuItemModiferSpecs) {
				if (menuItemModifierSpec == null || StringUtils.isBlank(menuItemModifierSpec.getId())) {
					continue;
				}
				if (!menuItemModifierSpecIdsSet.contains(menuItemModifierSpec.getId())) {
					MenuItemModifierSpec existingItem = session.get(MenuItemModifierSpec.class, menuItemModifierSpec.getId());
					saveOrUpdateIfAllowed(session, menuItemModifierSpec, existingItem, shouldUpdateIfExists);
					menuItemModifierSpecIdsSet.add(menuItemModifierSpec.getId());
				}
			}
		}

		if (discounts != null && !discounts.isEmpty()) {
			for (Discount discount : discounts) {
				if (discount == null || StringUtils.isBlank(discount.getId())) {
					continue;
				}
				if (!discountIdsSet.contains(discount.getId())) {
					Discount existingItem = session.get(Discount.class, discount.getId());
					saveOrUpdateIfAllowed(session, discount, existingItem, shouldUpdateIfExists);
					discountIdsSet.add(discount.getId());
				}
			}
		}

		if (attributes != null && !attributes.isEmpty()) {
			for (Attribute attribute : attributes) {
				if (attribute == null || StringUtils.isBlank(attribute.getId())) {
					continue;
				}
				if (!attributeIdsSet.contains(attribute.getId())) {
					Attribute existingItem = session.get(Attribute.class, attribute.getId());
					saveOrUpdateIfAllowed(session, attribute, existingItem, shouldUpdateIfExists);
					attributeIdsSet.add(attribute.getId());
				}
			}
		}

		if (menuItemJsonNode.has("testItems")) {
			JsonNode testItemsJsonNode = menuItemJsonNode.get("testItems");
			for (JsonNode testItemJson : testItemsJsonNode) {
				if (testItemJson.has("testItemGroup")) {
					JsonNode testItemGroupJson = testItemJson.get("testItemGroup");
					if (testItemGroupJson != null && testItemGroupJson.isObject() && !testItemGroupJson.isNull()) {
						TestItemGroup testItemGroup = mapper.convertValue(testItemGroupJson, TestItemGroup.class);
						if (!testItemGroupIdsSet.contains(testItemGroup.getId())) {
							TestItemGroup existingItem = StringUtils.isBlank(testItemGroup.getId()) ? null
									: session.get(TestItemGroup.class, testItemGroup.getId());
							saveOrUpdateIfAllowed(session, testItemGroup, existingItem, shouldUpdateIfExists);
							testItemGroupIdsSet.add(testItemGroup.getId());
						}
					}
				}
				if (testItemJson.has("inventoryUnit")) {
					JsonNode inventoryUnitJson = testItemJson.get("inventoryUnit");
					convertInventoryUnitAndSave(mapper, session, inventoryUnitJson, shouldUpdateIfExists);
				}
			}
		}

		JsonNode menuCategoryJsonNode = menuItemJsonNode.get("menuCategory");
		if (menuCategoryJsonNode != null && menuCategoryJsonNode.isObject() && !menuCategoryJsonNode.isNull()) {
			MenuCategory menuCategory = mapper.convertValue(menuCategoryJsonNode, MenuCategory.class);
			if (!menuCategoryIdsSet.contains(menuCategory.getId())) {
				MenuCategory existingItem = StringUtils.isBlank(menuCategory.getId()) ? null : session.get(MenuCategory.class, menuCategory.getId());
				if (existingItem != null) {
					menuCategory.setVersion(existingItem.getVersion());
				}
				saveOrUpdateIfAllowed(session, menuCategory, existingItem, shouldUpdateIfExists);
				menuCategoryIdsSet.add(menuCategory.getId());
			}
		}

		JsonNode menuGroupJsonNode = menuItemJsonNode.get("menuGroup");
		if (menuGroupJsonNode != null && menuGroupJsonNode.isObject() && !menuGroupJsonNode.isNull()) {
			MenuGroup menuGroup = mapper.convertValue(menuGroupJsonNode, MenuGroup.class);
			if (!menuGroupIdsSet.contains(menuGroup.getId())) {
				MenuGroup existingItem = StringUtils.isBlank(menuGroup.getId()) ? null : session.get(MenuGroup.class, menuGroup.getId());
				if (existingItem != null) {
					menuGroup.setVersion(existingItem.getVersion());
				}
				saveOrUpdateIfAllowed(session, menuGroup, existingItem, shouldUpdateIfExists);
				menuGroupIdsSet.add(menuGroup.getId());
			}
		}

		JsonNode containerJsonNode = menuItemJsonNode.get("container");
		if (containerJsonNode != null && containerJsonNode.isObject() && !containerJsonNode.isNull()) {
			Tag container = mapper.convertValue(containerJsonNode, Tag.class);
			if (!tagIdsSet.contains(container.getId())) {
				Tag existingItem = StringUtils.isBlank(container.getId()) ? null : session.get(Tag.class, container.getId());
				if (existingItem != null) {
					container.setVersion(existingItem.getVersion());
				}
				saveOrUpdateIfAllowed(session, container, existingItem, shouldUpdateIfExists);
				tagIdsSet.add(container.getId());
			}
		}

		JsonNode specimenJsonNode = menuItemJsonNode.get("specimen");
		if (specimenJsonNode != null && specimenJsonNode.isObject() && !specimenJsonNode.isNull()) {
			Specimen specimen = mapper.convertValue(specimenJsonNode, Specimen.class);
			if (!specimenIdsSet.contains(specimen.getId())) {
				Specimen existingItem = StringUtils.isBlank(specimen.getId()) ? null : session.get(Specimen.class, specimen.getId());
				saveOrUpdateIfAllowed(session, specimen, existingItem, shouldUpdateIfExists);
				specimenIdsSet.add(specimen.getId());
			}
		}

		JsonNode manufactureJsonNode = menuItemJsonNode.get("manufacture");
		if (manufactureJsonNode != null && manufactureJsonNode.isObject() && !manufactureJsonNode.isNull()) {
			Manufacturer manufacturer = mapper.convertValue(manufactureJsonNode, Manufacturer.class);
			if (!manufacturerIdsSet.contains(manufacturer.getId())) {
				Manufacturer existingItem = StringUtils.isBlank(manufacturer.getId()) ? null : session.get(Manufacturer.class, manufacturer.getId());
				saveOrUpdateIfAllowed(session, manufacturer, existingItem, shouldUpdateIfExists);
				manufacturerIdsSet.add(manufacturer.getId());
			}
		}

		JsonNode medGenericsJsonNode = menuItemJsonNode.get("medGenerics");
		if (medGenericsJsonNode != null && medGenericsJsonNode.isObject() && !medGenericsJsonNode.isNull()) {
			MedGenerics medGenerics = mapper.convertValue(medGenericsJsonNode, MedGenerics.class);
			if (!medGenericsIdsSet.contains(medGenerics.getId())) {
				MedGenerics existingItem = StringUtils.isBlank(medGenerics.getId()) ? null : session.get(MedGenerics.class, medGenerics.getId());
				saveOrUpdateIfAllowed(session, medGenerics, existingItem, shouldUpdateIfExists);
				medGenericsIdsSet.add(medGenerics.getId());
			}
		}

		JsonNode dosageFormsJsonNode = menuItemJsonNode.get("dosageForms");
		if (dosageFormsJsonNode != null && dosageFormsJsonNode.isObject() && !dosageFormsJsonNode.isNull()) {
			DosageForms dosageForms = mapper.convertValue(dosageFormsJsonNode, DosageForms.class);
			if (!dosageFormsIdsSet.contains(dosageForms.getId())) {
				DosageForms existingItem = StringUtils.isBlank(dosageForms.getId()) ? null : session.get(DosageForms.class, dosageForms.getId());
				saveOrUpdateIfAllowed(session, dosageForms, existingItem, shouldUpdateIfExists);
				dosageFormsIdsSet.add(dosageForms.getId());
			}
		}

		JsonNode reportGroupJsonNode = menuItemJsonNode.get("reportGroup");
		if (reportGroupJsonNode != null && reportGroupJsonNode.isObject() && !reportGroupJsonNode.isNull()) {
			ReportGroup reportGroup = mapper.convertValue(reportGroupJsonNode, ReportGroup.class);
			if (!reportGroupIdsSet.contains(reportGroup.getId())) {
				ReportGroup existingItem = StringUtils.isBlank(reportGroup.getId()) ? null : session.get(ReportGroup.class, reportGroup.getId());
				if (existingItem != null) {
					reportGroup.setVersion(existingItem.getVersion());
				}
				saveOrUpdateIfAllowed(session, reportGroup, existingItem, shouldUpdateIfExists);
				reportGroupIdsSet.add(reportGroup.getId());
			}
		}

		JsonNode taxGroupJsonNode = menuItemJsonNode.get("taxGroup");
		if (taxGroupJsonNode != null && taxGroupJsonNode.isObject() && !taxGroupJsonNode.isNull()) {
			TaxGroup taxGroup = mapper.convertValue(taxGroupJsonNode, TaxGroup.class);

			String currentOutletId = DataProvider.get().getCurrentOutletId();

			taxGroup.setOutletId(currentOutletId);

			List<Tax> taxes = taxGroup.getTaxes();
			if (taxes != null && !taxes.isEmpty()) {
				for (Tax tax : taxes) {
					tax.setOutletId(currentOutletId);
					Tax existingItem = StringUtils.isBlank(tax.getId()) ? null : session.get(Tax.class, new Tax(tax.getId(), currentOutletId));
					if (!taxIdsSet.contains(tax.getId())) {
						if (existingItem != null) {
							tax.setVersion(existingItem.getVersion());
						}
						saveOrUpdateIfAllowed(session, tax, existingItem, shouldUpdateIfExists);
						tagIdsSet.add(tax.getId());
					}
				}
			}

			if (!taxGroupIdsSet.contains(taxGroup.getId())) {
				TaxGroup existingItem = StringUtils.isBlank(taxGroup.getId()) ? null
						: session.get(TaxGroup.class, new TaxGroup(taxGroup.getId(), currentOutletId));
				if (existingItem != null) {
					taxGroup.setVersion(existingItem.getVersion());
				}
				saveOrUpdateIfAllowed(session, taxGroup, existingItem, shouldUpdateIfExists);
				taxGroupIdsSet.add(taxGroup.getId());
			}
		}

		JsonNode unitNode = menuItemJsonNode.get("inventoryUnit");
		convertInventoryUnitAndSave(mapper, session, unitNode, shouldUpdateIfExists);

		JsonNode recepieNode = menuItemJsonNode.get("recepie");
		if (recepieNode != null && recepieNode.isObject() && !recepieNode.isNull()) {
			Recepie recepie = convertRecepieAndSave(mapper, session, recepieNode, labTestMethod, shouldUpdateIfExists);
			menuItem.setDefaultRecipeId(recepie.getId());
		}

		JsonNode additionalItemsNode = menuItemJsonNode.get("additionalItems");
		if (additionalItemsNode != null && additionalItemsNode.isArray() && !additionalItemsNode.isNull()) {
			for (JsonNode additionalItemJson : additionalItemsNode) {
				JsonNode additionalItemJsonNode = additionalItemJson.get("inventoryItem");
				convertMenuitemAndSave(mapper, session, additionalItemJsonNode, labTestMethod, shouldUpdateIfExists);
			}
		}

		saveOrUpdateMenuItem(session, menuItem, shouldUpdateIfExists);
	}

	private void convertInventoryUnitAndSave(ObjectMapper mapper, Session session, JsonNode unitNode, boolean shouldUpdateIfExists) {
		if (unitNode != null && unitNode.isObject() && !unitNode.isNull()) {
			JsonNode unitGroupNode = unitNode.get("inventoryUnitGroup");
			if (unitGroupNode != null && unitGroupNode.isObject() && !unitGroupNode.isNull()) {
				InventoryUnitGroup unitGroup = mapper.convertValue(unitGroupNode, InventoryUnitGroup.class);
				InventoryUnitGroup existingItem = StringUtils.isBlank(unitGroup.getId()) ? null : session.get(InventoryUnitGroup.class, unitGroup.getId());
				if (existingItem != null) {
					unitGroup.setUnits(existingItem.getUnits());
				}
				else {
					unitGroup.setUnits(new ArrayList<>());
				}

				if (!inventoryUnitGroupIdsSet.contains(unitGroup.getId())) {
					if (existingItem != null) {
						unitGroup.setVersion(existingItem.getVersion());
					}
					saveOrUpdateIfAllowed(session, unitGroup, existingItem, shouldUpdateIfExists);
					inventoryUnitGroupIdsSet.add(unitGroup.getId());
				}
			}

			InventoryUnit unit = mapper.convertValue(unitNode, InventoryUnit.class);
			if (StringUtils.isBlank(unit.getId())) {
				return;
			}
			if (!inventoryUnitIdsSet.contains(unit.getId())) {
				InventoryUnit existingItem = StringUtils.isBlank(unit.getId()) ? null : session.get(InventoryUnit.class, unit.getId());
				if (existingItem != null) {
					unit.setVersion(existingItem.getVersion());
				}
				saveOrUpdateIfAllowed(session, unit, existingItem, shouldUpdateIfExists);
				inventoryUnitIdsSet.add(unit.getId());
			}
		}
	}

	private Recepie convertRecepieAndSave(ObjectMapper mapper, Session session, JsonNode recepieNode, Set<String> labTestMethod, boolean shouldUpdateIfExists) {
		Recepie recepie = mapper.convertValue(recepieNode, Recepie.class);
		if (StringUtils.isNotBlank(recepie.getId())) {
			recepie.setRecepieItems(null);
			if (!recepieIdsSet.contains(recepie.getId())) {
				Recepie existingItem = session.get(Recepie.class, recepie.getId());
				if (existingItem != null) {
					recepie.setVersion(existingItem.getVersion());
				}
				saveOrUpdateIfAllowed(session, recepie, existingItem, shouldUpdateIfExists);
				recepieIdsSet.add(recepie.getId());
			}
		}

		JsonNode recepieItemsJsonNode = recepieNode.get("recepieItems");
		if (recepieItemsJsonNode != null && recepieItemsJsonNode.isArray() && !recepieItemsJsonNode.isNull()) {
			for (JsonNode recepieItemJson : recepieItemsJsonNode) {
				JsonNode menuItemJson = recepieItemJson.get("inventoryItem");
				convertMenuitemAndSave(mapper, session, menuItemJson, labTestMethod, shouldUpdateIfExists);

				RecepieItem recepieItem = mapper.convertValue(recepieItemJson, RecepieItem.class);
				recepieItem.setRecepie(recepie);
				recepie.addRecepieItem(recepieItem);
				if (!recepieItemIdsSet.contains(recepieItem.getId())) {
					RecepieItem existingItem = StringUtils.isBlank(recepieItem.getId()) ? null : session.get(RecepieItem.class, recepieItem.getId());
					if (existingItem != null) {
						recepieItem.setVersion(existingItem.getVersion());
					}
					saveOrUpdateIfAllowed(session, recepieItem, existingItem, shouldUpdateIfExists);
					recepieItemIdsSet.add(recepieItem.getId());
				}
			}
		}
		return recepie;
	}

	private void saveOrUpdateMenuItem(Session session, MenuItem menuItem, boolean shouldUpdateIfExists) {
		if (menuItem == null) {
			return;
		}

		if (!menuItemIdsSet.contains(menuItem.getId())) {
			MenuItem existingItem = StringUtils.isBlank(menuItem.getId()) ? null : session.get(MenuItem.class, menuItem.getId());
			if (existingItem != null) {
				menuItem.setVersion(existingItem.getVersion());
			}
			saveOrUpdateIfAllowed(session, menuItem, existingItem, shouldUpdateIfExists);
			menuItemIdsSet.add(menuItem.getId());
		}
	}

	public abstract class MenuItemMixIn {
		@JsonDeserialize(using = MenuItemBooleanDeserializer.class)
		public abstract Boolean getInventoryItem();

		@JsonIgnore(false)
		@JsonProperty("additionalItems")
		public abstract List<RecepieItem> getAdditionalItems();

		@JsonIgnore(false)
		@JsonProperty("inventoryUnit")
		public abstract InventoryUnit getUnit();

		@JsonIgnore(false)
		@JsonProperty("inventoryStockUnit")
		public abstract InventoryStockUnit getStockUnits();

		@JsonIgnore(false)
		@JsonProperty("manufacture")
		public abstract Manufacturer getManufacture();

		@JsonIgnore(false)
		@JsonProperty("medGenerics")
		public abstract MedGenerics getMedGenerics();

		@JsonIgnore(false)
		@JsonProperty("dosageForms")
		public abstract DosageForms getDosageForms();

	}

	public abstract class InventoryUnitMixIn {
		@JsonProperty("inventoryUnitGroup")
		public abstract InventoryUnitGroup getUnitGroup();
	}

	//public abstract class ReceipeMixIn {
	//	@JsonIgnore()
	//	public abstract List<ManufacturingItem> getManufacturingItems();
	//}

	public static class MenuItemBooleanDeserializer extends JsonDeserializer<Boolean> {

		public MenuItemBooleanDeserializer() {
		}

		@Override
		public Boolean deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
			JsonToken token = p.getCurrentToken();

			if (token == JsonToken.VALUE_TRUE)
				return true;
			if (token == JsonToken.VALUE_FALSE)
				return false;

			if (token == JsonToken.START_OBJECT) {
				p.skipChildren();
				return false;
			}

			return null;
		}
	}

	public abstract class RecepieItemMixIn {
		@JsonDeserialize(using = MenuItemDeserializer.class)
		public abstract MenuItem getInventoryItem();
	}

	public static class MenuItemDeserializer extends JsonDeserializer<MenuItem> {

		public MenuItemDeserializer() {
		}

		private final ObjectMapper mapper = new ObjectMapper();

		@Override
		public MenuItem deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
			JsonNode node = p.getCodec().readTree(p);
			return mapper.treeToValue(node, MenuItem.class);
		}
	}
}
