package io.github.escposjava;

import static io.github.escposjava.print.Commands.BARCODE_CODE39;
import static io.github.escposjava.print.Commands.BARCODE_EAN13;
import static io.github.escposjava.print.Commands.BARCODE_ITF;
import static io.github.escposjava.print.Commands.BARCODE_NW7;
import static io.github.escposjava.print.Commands.BARCODE_UPC_A;
import static io.github.escposjava.print.Commands.BARCODE_UPC_E;
import static io.github.escposjava.print.Commands.BEEPER;
import static io.github.escposjava.print.Commands.CD_KICK_2;
import static io.github.escposjava.print.Commands.CD_KICK_5;
import static io.github.escposjava.print.Commands.CHARCODE_GREEK;
import static io.github.escposjava.print.Commands.CHARCODE_HEBREW;
import static io.github.escposjava.print.Commands.CHARCODE_JIS;
import static io.github.escposjava.print.Commands.CHARCODE_PC1252;
import static io.github.escposjava.print.Commands.CHARCODE_PC437;
import static io.github.escposjava.print.Commands.CHARCODE_PC850;
import static io.github.escposjava.print.Commands.CHARCODE_PC852;
import static io.github.escposjava.print.Commands.CHARCODE_PC858;
import static io.github.escposjava.print.Commands.CHARCODE_PC860;
import static io.github.escposjava.print.Commands.CHARCODE_PC863;
import static io.github.escposjava.print.Commands.CHARCODE_PC865;
import static io.github.escposjava.print.Commands.CHARCODE_PC866;
import static io.github.escposjava.print.Commands.CHARCODE_THAI11;
import static io.github.escposjava.print.Commands.CHARCODE_THAI13;
import static io.github.escposjava.print.Commands.CHARCODE_THAI14;
import static io.github.escposjava.print.Commands.CHARCODE_THAI16;
import static io.github.escposjava.print.Commands.CHARCODE_THAI17;
import static io.github.escposjava.print.Commands.CHARCODE_THAI18;
import static io.github.escposjava.print.Commands.CHARCODE_THAI42;
import static io.github.escposjava.print.Commands.CHARCODE_WEU;
import static io.github.escposjava.print.Commands.CTL_LF;
import static io.github.escposjava.print.Commands.HW_INIT;
import static io.github.escposjava.print.Commands.PAPER_FULL_CUT;
import static io.github.escposjava.print.Commands.PAPER_PART_CUT;
import static io.github.escposjava.print.Commands.PD_0;
import static io.github.escposjava.print.Commands.PD_N12;
import static io.github.escposjava.print.Commands.PD_N25;
import static io.github.escposjava.print.Commands.PD_N37;
import static io.github.escposjava.print.Commands.PD_N50;
import static io.github.escposjava.print.Commands.PD_P12;
import static io.github.escposjava.print.Commands.PD_P25;
import static io.github.escposjava.print.Commands.PD_P37;
import static io.github.escposjava.print.Commands.PD_P50;
import static io.github.escposjava.print.Commands.SELECT_BIT_IMAGE_MODE;
import static io.github.escposjava.print.Commands.TXT_2HEIGHT;
import static io.github.escposjava.print.Commands.TXT_2WIDTH;
import static io.github.escposjava.print.Commands.TXT_4SQUARE;
import static io.github.escposjava.print.Commands.TXT_ALIGN_CT;
import static io.github.escposjava.print.Commands.TXT_ALIGN_LT;
import static io.github.escposjava.print.Commands.TXT_ALIGN_RT;
import static io.github.escposjava.print.Commands.TXT_BOLD_OFF;
import static io.github.escposjava.print.Commands.TXT_BOLD_ON;
import static io.github.escposjava.print.Commands.TXT_FONT_A;
import static io.github.escposjava.print.Commands.TXT_FONT_B;
import static io.github.escposjava.print.Commands.TXT_NORMAL;
import static io.github.escposjava.print.Commands.TXT_UNDERL2_ON;
import static io.github.escposjava.print.Commands.TXT_UNDERL_OFF;
import static io.github.escposjava.print.Commands.TXT_UNDERL_ON;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;

import com.floreantpos.Messages;
import com.floreantpos.PosException;

import io.github.escposjava.print.Commands;
import io.github.escposjava.print.Printer;
import io.github.escposjava.print.exceptions.BarcodeSizeError;
import io.github.escposjava.print.exceptions.QRCodeException;
import io.github.escposjava.print.image.Image;
import io.github.escposjava.print.qrcode.QRCodeGenerator;

public class PrinterService {
	private static final String CARRIAGE_RETURN = System.getProperty("line.separator");

	private final Printer printer;

	public PrinterService(Printer printer) {
		this.printer = printer;
	}

	public void print(String text) {
		if (text == null) {
			return;
		}
		write(text.getBytes());
	}

	public void printLeft(String text) {
		write(TXT_ALIGN_LT);
		print(text);
	}

	public void printCentered(String text) {
		write(TXT_ALIGN_CT);
		print(text);
	}

	public void printRight(String text) {
		write(TXT_ALIGN_RT);
		print(text);
	}

	public void print(byte[] align, String text) {
		write(align);
		print(text);
	}

	public void printLn(byte[] align, String text) {
		write(align);
		printLn(text);
	}

	public void printLn(String text) {
		print(text + CARRIAGE_RETURN);
	}

	public void lineBreak() {
		lineBreak(1);
	}

	public void lineBreak(int nbLine) {
		for (int i = 0; i < nbLine; i++) {
			write(CTL_LF);
		}
	}

	public void printQRCode(String value) throws QRCodeException {
		printQRCode(value, 150);
	}

	public void printQRCode(String value, int size) throws QRCodeException {
		QRCodeGenerator q = new QRCodeGenerator();
		printImage(q.generate(value, size));
	}

	public void setTextSizeNormal() {
		setTextSize(1, 1);
	}

	public void setTextSize2H() {
		setTextSize(1, 2);
	}

	public void setTextSize2W() {
		setTextSize(2, 1);
	}

	public void setText4Square() {
		setTextSize(2, 2);
	}

	private void setTextSize(int width, int height) {
		if (height == 2 && width == 2) {
			write(TXT_NORMAL);
			write(TXT_4SQUARE);
		}
		else if (height == 2) {
			write(TXT_NORMAL);
			write(TXT_2HEIGHT);
		}
		else if (width == 2) {
			write(TXT_NORMAL);
			write(TXT_2WIDTH);
		}
		else {
			write(TXT_NORMAL);
		}
	}

	public void setTextTypeBold() {
		setTextType("B");
	}

	public void setTextTypeUnderline() {
		setTextType("U");
	}

	public void setTextType2Underline() {
		setTextType("U2");
	}

	public void setTextTypeBoldUnderline() {
		setTextType("BU");
	}

	public void setTextTypeBold2Underline() {
		setTextType("BU2");
	}

	public void setTextTypeNormal() {
		setTextType("NORMAL");
	}

	private void setTextType(String type) {
		if (type.equalsIgnoreCase("B")) {
			write(TXT_BOLD_ON);
			write(TXT_UNDERL_OFF);
		}
		else if (type.equalsIgnoreCase("U")) {
			write(TXT_BOLD_OFF);
			write(TXT_UNDERL_ON);
		}
		else if (type.equalsIgnoreCase("U2")) {
			write(TXT_BOLD_OFF);
			write(TXT_UNDERL2_ON);
		}
		else if (type.equalsIgnoreCase("BU")) {
			write(TXT_BOLD_ON);
			write(TXT_UNDERL_ON);
		}
		else if (type.equalsIgnoreCase("BU2")) {
			write(TXT_BOLD_ON);
			write(TXT_UNDERL2_ON);
		}
		else if (type.equalsIgnoreCase("NORMAL")) {
			write(TXT_BOLD_OFF);
			write(TXT_UNDERL_OFF);
		}
	}

	public void cutPart() {
		cut("PART");
	}

	public void cutFull() {
		cut("FULL");
	}

	private void cut(String mode) {
		for (int i = 0; i < 5; i++) {
			write(CTL_LF);
		}
		if (mode.toUpperCase().equals("PART")) {
			write(PAPER_PART_CUT);
		}
		else {
			write(PAPER_FULL_CUT);
		}
	}

	public void printBarcode(String code, String bc, int width, int height, String pos, String font) throws BarcodeSizeError {
		// Align Bar Code()
		write(CTL_LF);
		//write(TXT_ALIGN_CT);
		// Height
		/*if (height >= 2 || height <= 6) {
			write(BARCODE_HEIGHT);
		}
		else {
			throw new BarcodeSizeError("Incorrect Height");
		}
		//Width
		if (width >= 1 || width <= 255) {
			write(BARCODE_WIDTH);
		}
		else {
			throw new BarcodeSizeError("Incorrect Width");
		}*/
		//Font
		//if (font.equalsIgnoreCase("B")) {
		//	write(BARCODE_FONT_B);
		//}
		//else {
		///	write(BARCODE_FONT_A);
		//}
		//Position
		//if (pos.equalsIgnoreCase("OFF")) {
		//	write(BARCODE_TXT_OFF);
		//}
		//else if (pos.equalsIgnoreCase("BOTH")) {
		//	write(BARCODE_TXT_BTH);
		//}
		//else if (pos.equalsIgnoreCase("ABOVE")) {
		//	write(BARCODE_TXT_ABV);
		//}
		//else {
		//	write(BARCODE_TXT_BLW);
		//}
		//Type
		switch (bc.toUpperCase()) {
			case "UPC-A":
				write(BARCODE_UPC_A);
				break;
			case "UPC-E":
				write(BARCODE_UPC_E);
				break;
			default:
			case "EAN13":
				write(BARCODE_EAN13);
				break;
			case "EAN8":
				write(Commands.BARCODE_EAN8);
				break;
			case "CODE39":
				write(BARCODE_CODE39);
				break;
			case "ITF":
				write(BARCODE_ITF);
				break;
			case "NW7":
				write(BARCODE_NW7);
				break;
			case "CODE128":
				write(new byte[] { 0x1d, 0x68, 0x40 });
				write(new byte[] { 0x1d, 0x77, 0x02 });
				setTextFont(font);
				setTextAlignCenter();
				write(new byte[] { 0x1d, 0x6b, 0x49, 0x0f });
				write(new byte[] { 0x7b, 0x42 });
				write(code.getBytes());
				write(CTL_LF);
				return;
		}
		//Print Code
		if (!code.equals("")) {
			write(code.getBytes());
			write(CTL_LF);
		}
		else {
			throw new BarcodeSizeError("Incorrect Value");
		}
	}

	public void printBarcode(String code, int type, int h, int w, int font, int pos) throws IOException {
		// need to test for errors in length of code
		// also control for input type=0-6
		// GS H = HRI position
		write(0x1D);
		write("H");
		write(pos); // 0=no print, 1=above, 2=below, 3=above & below

		// GS f = set barcode characters
		write(0x1D);
		write("f");
		write(font);

		// GS h = sets barcode height
		write(0x1D);
		write("h");
		write(h);

		// GS w = sets barcode width
		write(0x1D);
		write("w");
		write(w);// module = 1-6

		// GS k
		write(0x1D); // GS
		write("k"); // k
		write(type);// m = barcode type 0-6
		write(code.length()); // length of encoded string
		write(code);// d1-dk
		write(0);// print barcode
	}

	private void write(String string) {
		write(string.getBytes());
	}

	private void write(int i) {
		write(new byte[] { (byte) i });
	}

	public void setTextFontA() {
		setTextFont("A");
	}

	public void setTextFontB() {
		setTextFont("B");
	}

	private void setTextFont(String font) {
		if (font.equalsIgnoreCase("B")) {
			write(TXT_FONT_B);
		}
		else {
			write(TXT_FONT_A);
		}
	}

	public void setTextAlignCenter() {
		setTextAlign("CENTER");
	}

	public void setTextAlignRight() {
		setTextAlign("RIGHT");
	}

	public void setTextAlignLeft() {
		setTextAlign("LEFT");
	}

	private void setTextAlign(String align) {
		if (align.equalsIgnoreCase("CENTER")) {
			write(TXT_ALIGN_CT);
		}
		else if (align.equalsIgnoreCase("RIGHT")) {
			write(TXT_ALIGN_RT);
		}
		else {
			write(TXT_ALIGN_LT);
		}
	}

	public void setTextDensity(int density) {
		switch (density) {
			case 0:
				write(PD_N50);
				break;
			case 1:
				write(PD_N37);
				break;
			case 2:
				write(PD_N25);
				break;
			case 3:
				write(PD_N12);
				break;
			case 4:
				write(PD_0);
				break;
			case 5:
				write(PD_P12);
				break;
			case 6:
				write(PD_P25);
				break;
			case 7:
				write(PD_P37);
				break;
			case 8:
				write(PD_P50);
				break;
		}
	}

	public void setTextNormal() {
		setTextProperties("LEFT", "A", "NORMAL", 1, 1, 9);
	}

	public void setTextProperties(String align, String font, String type, int width, int height, int density) {
		setTextAlign(align);
		setTextFont(font);
		setTextType(type);
		setTextSize(width, height);
		setTextDensity(density);
	}

	public void printImage(String filePath) throws IOException {
		File img = new File(filePath);
		printImage(ImageIO.read(img));
	}

	public void printImage(BufferedImage image) {
		Image img = new Image();
		int[][] pixels = img.getPixelsSlow(image);
		for (int y = 0; y < pixels.length; y += 24) {
			write(TXT_ALIGN_CT);
			write(Commands.LINE_SPACE_24);
			write(SELECT_BIT_IMAGE_MODE);
			write(new byte[] { (byte) (0x00ff & pixels[y].length), (byte) ((0xff00 & pixels[y].length) >> 8) });
			for (int x = 0; x < pixels[y].length; x++) {
				write(img.recollectSlice(y, x, pixels));
			}
			write(CTL_LF);
		}
		write(Commands.LINE_SPACE_60);
		//        bus.write(CTL_LF);
		//        bus.write(LINE_SPACE_30);
	}

	public void setCharCode(String code) {
		switch (code) {
			case "USA":
				write(CHARCODE_PC437);
				break;
			case "JIS":
				write(CHARCODE_JIS);
				break;
			case "MULTILINGUAL":
				write(CHARCODE_PC850);
				break;
			case "PORTUGUESE":
				write(CHARCODE_PC860);
				break;
			case "CA_FRENCH":
				write(CHARCODE_PC863);
				break;
			default:
			case "NORDIC":
				write(CHARCODE_PC865);
				break;
			case "WEST_EUROPE":
				write(CHARCODE_WEU);
				break;
			case "GREEK":
				write(CHARCODE_GREEK);
				break;
			case "HEBREW":
				write(CHARCODE_HEBREW);
				break;
			case "WPC1252":
				write(CHARCODE_PC1252);
				break;
			case "CIRILLIC2":
				write(CHARCODE_PC866);
				break;
			case "LATIN2":
				write(CHARCODE_PC852);
				break;
			case "EURO":
				write(CHARCODE_PC858);
				break;
			case "THAI42":
				write(CHARCODE_THAI42);
				break;
			case "THAI11":
				write(CHARCODE_THAI11);
				break;
			case "THAI13":
				write(CHARCODE_THAI13);
				break;
			case "THAI14":
				write(CHARCODE_THAI14);
				break;
			case "THAI16":
				write(CHARCODE_THAI16);
				break;
			case "THAI17":
				write(CHARCODE_THAI17);
				break;
			case "THAI18":
				write(CHARCODE_THAI18);
				break;
		}
	}

	public void init() {
		if (!printer.isConnected()) {
			throw new PosException(Messages.getString("COULD_NOT_CONNECT_TO_PRINTER")); //$NON-NLS-1$
		}
		write(HW_INIT);
	}

	public void openCashDrawerPin2() {
		write(CD_KICK_2);
	}

	public void openCashDrawerPin5() {
		write(CD_KICK_5);
	}

	public void open() {
		printer.open();
	}

	public void close() throws Exception {
		printer.close();
	}

	public void beep() {
		write(BEEPER);
	}

	public void write(byte[] command) {
		if (command == null) {
			return;
		}
		printer.write(command);
	}

	public void setTextColor(byte[] colorCode) {
		write(colorCode);
	}

}
