package com.floreantpos.print;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;

import org.apache.commons.lang.StringUtils;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.PDFRenderer;

import com.floreantpos.Messages;
import com.floreantpos.model.ITicketItem;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.ReceiptParam;
import com.floreantpos.model.Store;
import com.floreantpos.model.ext.ReciptPaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.report.AbstractReportDataSource;
import com.floreantpos.report.KitchenTicketDataSource;
import com.floreantpos.report.ReceiptPrintService;
import com.floreantpos.report.TicketDataSource;
import com.floreantpos.util.POSUtil;
import com.lowagie.text.BadElementException;
import com.lowagie.text.Chunk;
import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Phrase;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.Barcode128;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfWriter;
import com.lowagie.text.pdf.draw.DottedLineSeparator;

import net.miginfocom.swing.MigLayout;

public class ITextPrinter {
	private static final int FONT_SIZE = 9;
	private static final int KITCHEN_FONT_SIZE_16 = 16;
	private static final int KITCHEN_FONT_SIZE_14 = 14;

	private Font headerLine1Font = new Font(Font.HELVETICA, 11, Font.BOLD);
	private Font font_normal = new Font(Font.HELVETICA, 9, Font.NORMAL);
	private Font font_bold = new Font(Font.HELVETICA, 9, Font.BOLD);
	private Font table_font = new Font(Font.HELVETICA, 9, Font.NORMAL);
	private Font seperator_font = new Font(Font.HELVETICA, 7, Font.NORMAL);

	private Document document = null;
	private PdfWriter pdfWriter;
	private ByteArrayOutputStream os = new ByteArrayOutputStream();

	public void createReport(ReciptPaperSize paperSize, Map<String, Object> params, AbstractReportDataSource dataSource) throws Exception {
		try {
			renderFonts(dataSource);
			Document document = createDocument();
			int marginHorizontal = 19;
			int pageWidth = 228;
			int pageHeight = 600;

			int[] pointColumnWidths = { 138, 50 };
			int tableWidth = pointColumnWidths[0] + pointColumnWidths[1];

			Rectangle pagesize = new Rectangle(pageWidth, pageHeight);
			document.setPageSize(pagesize);
			document.setMargins(marginHorizontal, marginHorizontal, 0, 0);
			document.newPage();
			Paragraph br = new Paragraph(" "); //$NON-NLS-1$

			document.add(br);
			addText(params, ReceiptPrintService.HEADER_LINE1, Paragraph.ALIGN_CENTER, headerLine1Font);
			addText(params, ReceiptPrintService.TICKET_HEADER, Paragraph.ALIGN_CENTER);
			addText(params, ReceiptPrintService.ADDITIONAL_ORDER_INFO, Paragraph.ALIGN_LEFT);
			document.add(getSep());

			PdfPTable table = createTable(2, tableWidth, pointColumnWidths);
			populateRows(dataSource, table);
			document.add(table);
			document.add(br);

			document.add(getSep());
			if (dataSource instanceof TicketDataSource) {
				int[] summaryColumnWidths = { 126, 66 };

				PdfPTable summarySection = createTable(2, tableWidth, summaryColumnWidths);
				addSummaryCell(summarySection, params, ReceiptPrintService.TOTAL_TEXT, ReceiptPrintService.GRAND_SUBTOTAL, false);
				addSummaryCell(summarySection, params, ReceiptPrintService.DISCOUNT_TEXT, ReceiptPrintService.DISCOUNT_AMOUNT);
				boolean isShowTaxBreakdown = Boolean.parseBoolean(String.valueOf(params.get(ReceiptPrintService.IS_SHOW_TAX_BREAKDOWN))); //$NON-NLS-1$
				if (!isShowTaxBreakdown) {
					addSummaryCell(summarySection, params, ReceiptPrintService.TAX_TEXT, ReceiptPrintService.TAX_AMOUNT);
				}
				else {
					document.add(summarySection);
					addText(params, ReceiptPrintService.TAX_BREAKDOWN_TEXT, Paragraph.ALIGN_RIGHT, null, false);
					summarySection = createTable(2, tableWidth, summaryColumnWidths);
				}
				addSummaryCell(summarySection, params, ReceiptPrintService.SERVICE_CHARGE_TEXT, ReceiptPrintService.SERVICE_CHARGE);
				addSummaryCell(summarySection, params, ReceiptPrintService.DELIVERY_CHARGE_TEXT, ReceiptPrintService.DELIVERY_CHARGE);

				summarySection.setSpacingAfter(5);
				document.add(summarySection);
				document.add(getSep());

				summarySection = createTable(2, tableWidth, summaryColumnWidths);
				addSummaryCell(summarySection, params, ReceiptPrintService.NET_AMOUNT_TEXT, ReceiptPrintService.NET_AMOUNT, false);
				addSummaryCell(summarySection, params, ReceiptPrintService.PAID_AMOUNT_TEXT, ReceiptPrintService.PAID_AMOUNT, false);
				addSummaryCell(summarySection, params, ReceiptPrintService.REFUND_AMOUNT_TEXT, ReceiptPrintService.REFUND_AMOUNT);
				addSummaryCell(summarySection, params, ReceiptPrintService.TENDER_AMOUNT_TEXT, ReceiptPrintService.TENDER_AMOUNT);
				addSummaryCell(summarySection, params, ReceiptPrintService.DUE_AMOUNT_TEXT, ReceiptPrintService.DUE_AMOUNT, false);
				if (params.get(ReceiptPrintService.TENDER_AMOUNT) != null) {
					addSummaryCell(summarySection, params, ReceiptPrintService.CHANGE_AMOUNT_TEXT, ReceiptPrintService.CHANGED_AMOUNT, false);
				}
				addSummaryCell(summarySection, params, ReceiptPrintService.TIPS_TEXT, ReceiptPrintService.TIP_AMOUNT);
				document.add(summarySection);
				addText(params, ReceiptPrintService.SHOW_TIPS, Paragraph.ALIGN_CENTER);
				summarySection = createTable(2, tableWidth, summaryColumnWidths);
				addSummaryCell(summarySection, params, ReceiptPrintService.FEE_AMOUNT_TEXT, ReceiptPrintService.FEE_AMOUNT);
				document.add(summarySection);

				addText(params, ReceiptPrintService.ADDITIONAL_PAYMENT_PROPERTIES, Paragraph.ALIGN_RIGHT);
				addText(params, ReceiptPrintService.ADDITIONAL_PROPERTIES, Paragraph.ALIGN_RIGHT);

				boolean cardPayment = Boolean.parseBoolean(String.valueOf(params.get(ReceiptPrintService.CARD_PAYMENT)));
				boolean showTipsSection = cardPayment && Boolean.parseBoolean(String.valueOf(params.get(ReceiptPrintService.SHOW_TIPS_BLOCK)));
				if (showTipsSection) {
					summarySection = createTable(2, tableWidth, summaryColumnWidths);
					summarySection.setWidthPercentage(80);
					summarySection.setHorizontalAlignment(PdfPTable.ALIGN_RIGHT);
					addTextWithUnderline(summarySection, Messages.getString("EscPosPrintService.4"));//$NON-NLS-1$
					addTextWithUnderline(summarySection, Messages.getString("EscPosPrintService.5"));//$NON-NLS-1$
					addTextWithUnderline(summarySection, Messages.getString("EscPosPrintService.6"));//$NON-NLS-1$
					document.add(summarySection);
				}
				if (cardPayment) {
					addText(params, ReceiptPrintService.APPROVAL_CODE, Paragraph.ALIGN_LEFT);
				}
				addText(params, ReceiptPrintService.BOTTOM_MESSAGE, Paragraph.ALIGN_LEFT);
				document.add(br);

				addText(params, ReceiptPrintService.COPY_TYPE, Paragraph.ALIGN_CENTER);
				addText(params, ReceiptPrintService.FOOTER_MESSAGE, Paragraph.ALIGN_CENTER);
			}
			else {
				addText(params, ReceiptPrintService.PROP_PRINTER_NAME, Paragraph.ALIGN_LEFT);
				addText(params, ReceiptPrintService.BOTTOM_MESSAGE, Paragraph.ALIGN_LEFT);
				document.add(br);
				addText(params, ReceiptPrintService.FOOTER_MESSAGE, Paragraph.ALIGN_CENTER);
			}
		} finally {
			closeDocument();
		}
	}

	private void renderFonts(AbstractReportDataSource dataSource) {
		if (dataSource instanceof KitchenTicketDataSource) {
			headerLine1Font.setSize(KITCHEN_FONT_SIZE_14);
			font_bold.setSize(KITCHEN_FONT_SIZE_14);
			font_normal.setSize(KITCHEN_FONT_SIZE_14);
			table_font.setSize(KITCHEN_FONT_SIZE_16);
		}
		else {
			headerLine1Font.setSize(FONT_SIZE);
			font_bold.setSize(FONT_SIZE);
			font_normal.setSize(FONT_SIZE);
			table_font.setSize(FONT_SIZE);
		}
	}

	private void populateRows(AbstractReportDataSource dataSource, PdfPTable table) {
		Store store = DataProvider.get().getStore();
		List rows = dataSource.getRows();
		if (rows != null) {
			if (dataSource instanceof TicketDataSource) {
				TicketDataSource ticketDataSource = (TicketDataSource) dataSource;
				for (Iterator iterator = rows.iterator(); iterator.hasNext();) {
					ITicketItem item = (ITicketItem) iterator.next();
					String color = ticketDataSource.getColorCode(store, item);
					table_font.setColor(decodeHtmlColorString(color));
					PdfPCell nameCell = getCell(item.getNameDisplay(), PdfPCell.ALIGN_LEFT);
					PdfPCell subtotalCell = getCell(item.getSubTotalAmountDisplay(), PdfPCell.ALIGN_RIGHT);
					table.addCell(nameCell);
					table.addCell(subtotalCell);
				}
			}
			else if (dataSource instanceof KitchenTicketDataSource) {
				KitchenTicketDataSource kitchenTicketDataSource = (KitchenTicketDataSource) dataSource;
				int countLineSeperator = 0;
				for (Iterator iterator = rows.iterator(); iterator.hasNext();) {
					KitchenTicketItem item = (KitchenTicketItem) iterator.next();
					String color = kitchenTicketDataSource.getColorCode(item, item.getKitchenTicket().getOrderTypeId());
					table_font.setColor(decodeHtmlColorString(color));

					PdfPCell cell = getSeperatorCell("-------------------------------------------------------------------------------", PdfPCell.ALIGN_LEFT);//$NON-NLS-1$
					if (!item.isModifierItem() && !item.isCookingInstruction() && countLineSeperator > 0) {
						cell.setColspan(2);
						table.addCell(cell);
					}
					cell = getCell(item.getMenuItemNameDisplay(), PdfPCell.ALIGN_LEFT);
					cell.setColspan(2);
					table.addCell(cell);
					countLineSeperator += 1;
				}
			}
		}
		table_font.setColor(Color.black);
	}

	public static Color decodeHtmlColorString(String colorString) {
		if (StringUtils.isEmpty(colorString)) {
			return Color.BLACK;
		}
		Color color;
		if (colorString.startsWith("#")) { //$NON-NLS-1$
			colorString = colorString.substring(1);
		}
		if (colorString.endsWith(";")) { //$NON-NLS-1$
			colorString = colorString.substring(0, colorString.length() - 1);
		}

		int red, green, blue;
		switch (colorString.length()) {
			case 6:
				red = Integer.parseInt(colorString.substring(0, 2), 16);
				green = Integer.parseInt(colorString.substring(2, 4), 16);
				blue = Integer.parseInt(colorString.substring(4, 6), 16);
				color = new Color(red, green, blue);
				break;
			case 3:
				red = Integer.parseInt(colorString.substring(0, 1), 16);
				green = Integer.parseInt(colorString.substring(1, 2), 16);
				blue = Integer.parseInt(colorString.substring(2, 3), 16);
				color = new Color(red, green, blue);
				break;
			case 1:
				red = green = blue = Integer.parseInt(colorString.substring(0, 1), 16);
				color = new Color(red, green, blue);
				break;
			default:
				return Color.BLACK;
		}
		return color;
	}

	private PdfPTable createTable(int col, int tableWidth, int[] summaryColumnWidths) throws DocumentException {
		PdfPTable summarySection = new PdfPTable(col);
		summarySection.setHorizontalAlignment(PdfPTable.ALIGN_CENTER);
		summarySection.setWidths(summaryColumnWidths);
		summarySection.setKeepTogether(true);
		summarySection.setTotalWidth(tableWidth);
		summarySection.setWidthPercentage(100);
		return summarySection;
	}

	private Image getBarcode(Map<String, Object> parameters) throws DocumentException {
		Object barcode = parameters.get(ReceiptParam.BARCODE.getParamName());
		if (barcode != null) {
			PdfContentByte cb = getPdfWriter().getDirectContent();
			Barcode128 code128 = new Barcode128();
			code128.setBaseline(-1);
			code128.setSize(12);
			//code128.setAltText(String.valueOf(barcode));
			code128.setFont(font_normal.getBaseFont());
			code128.setCode(String.valueOf(barcode));
			code128.setCodeType(Barcode128.CODE128);
			code128.setBarHeight(20);
			code128.setX(1f);
			code128.setTextAlignment(Image.ALIGN_CENTER);

			Image code128Image = code128.createImageWithBarcode(cb, null, null);
			code128Image.setAlignment(Image.ALIGN_CENTER);
			code128Image.setSpacingBefore(1);
			code128Image.setSpacingAfter(1);
			code128Image.setWidthPercentage(90);
			return code128Image;
		}
		return null;
	}

	private Image getImage(Map<String, Object> parameters, int maxWidth) throws BadElementException, MalformedURLException, IOException, DocumentException {
		java.awt.Image orinImage = (java.awt.Image) parameters.get(ReceiptPrintService.STORE_LOGO_ICON);
		if (orinImage != null) {
			Image image = Image.getInstance(POSUtil.convertBlackAndWhiteImage(orinImage), null);
			image.setAlignment(Image.ALIGN_CENTER);
			float width = image.getWidth();
			float height = image.getHeight();
			if (width > maxWidth) {
				height = (height * maxWidth) / width;
				width = maxWidth;
			}
			image.scaleToFit(width, height);
			return image;
		}
		return null;
	}

	public PdfWriter getPdfWriter() {
		return pdfWriter;
	}

	private void addSummaryCell(PdfPTable table, Map<String, Object> parameters, String labelKey, String valueKey) {
		addSummaryCell(table, parameters, labelKey, valueKey, true);
	}

	private void addSummaryCell(PdfPTable table, Map<String, Object> parameters, String labelKey, String valueKey, boolean removeNullOrBlank) {
		Object value = parameters.get(valueKey);
		if (removeNullOrBlank && (value == null || value.equals("null") || value.equals("0.00"))) { //$NON-NLS-1$ //$NON-NLS-2$
			return;
		}
		table.addCell(getCell(String.valueOf(parameters.get(labelKey)), Element.ALIGN_RIGHT));
		table.addCell(getCell(String.valueOf(value), Element.ALIGN_RIGHT));
	}

	private PdfPCell getCell(String text, int align) {
		Font font = new Font(table_font);
		PdfPCell c1 = new PdfPCell(new Phrase(text, font));
		c1.setHorizontalAlignment(align);
		c1.setBorder(PdfPCell.NO_BORDER);
		return c1;
	}

	private PdfPCell getSeperatorCell(String text, int align) {
		Font font = new Font(seperator_font);
		PdfPCell c1 = new PdfPCell(new Phrase(text, font));
		c1.setHorizontalAlignment(align);
		c1.setBorder(PdfPCell.NO_BORDER);
		return c1;
	}

	private DottedLineSeparator getSep() {
		return getSep(2, 2);
	}

	private DottedLineSeparator getSep(float offset, float gap) {
		DottedLineSeparator sep = new DottedLineSeparator();
		sep.setAlignment(DottedLineSeparator.ALIGN_CENTER);
		sep.setOffset(offset);
		sep.setGap(gap);
		return sep;
	}

	public Document createDocument() throws Exception {
		document = new Document();
		pdfWriter = PdfWriter.getInstance(document, os);
		document.open();
		return document;
	}

	public void closeDocument() {
		if (document != null) {
			document.close();
			document = null;
		}
	}

	public void rendererReport(JPanel reportPanel) throws Exception {
		PDDocument document = PDDocument.load(new ByteArrayInputStream(getPrintableByteArray()));
		MigLayout layout = new MigLayout("fill,hidemode 3,wrap 1,ins 0", "", "[]0px[]"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		JPanel pagePanel = new JPanel(layout);
		pagePanel.setBackground(Color.WHITE);
		float scale = Toolkit.getDefaultToolkit().getScreenResolution() / 72f;
		for (int i = 0; i < document.getNumberOfPages(); i++) {
			BufferedImage croppedImage = crop(new PDFRenderer(document).renderImage(i, scale), (i + 1 == document.getNumberOfPages()) ? 20 : 0);
			if (croppedImage == null) {
				continue;
			}
			pagePanel.add(new JLabel(new ImageIcon(croppedImage)), "gap 0"); //$NON-NLS-1$
		}
		reportPanel.add(pagePanel);
	}

	public BufferedImage crop(BufferedImage bi, int bottomMergin) {
		int w = bi.getWidth();

		int trimHeight = getTrimmedHeight(bi);
		/*if (trimHeight == 0) {
			return null;
		}*/
		BufferedImage newImg = new BufferedImage(w, trimHeight + bottomMergin, BufferedImage.TYPE_INT_RGB);
		Graphics g = newImg.createGraphics();
		g.drawImage(bi, 0, 0, null);
		return newImg;
	}

	private int getTrimmedHeight(BufferedImage img) {
		int width = img.getWidth();
		int height = img.getHeight();
		int trimmedHeight = 0;

		for (int i = 0; i < width; i++) {
			for (int j = height - 1; j >= 0; j--) {
				if (img.getRGB(i, j) != Color.WHITE.getRGB() && j > trimmedHeight) {
					trimmedHeight = j;
					break;
				}
			}
		}
		return trimmedHeight;
	}

	public boolean allWhiteRow(int row, BufferedImage bi, int w) {
		for (int i = 0; i < w; i++) {
			if (bi.getRGB(i, row) == 255) {
				return true;
			}
		}
		return false;
	}

	public boolean allWhiteColumn(int col, BufferedImage bi, int h) {
		for (int i = 0; i < h; ++i) {
			if (bi.getRGB(i, col) == 255) {
				return true;
			}
		}
		return false;
	}

	public byte[] getPrintableByteArray() {
		return os.toByteArray();
	}

	public void addText(Map<String, Object> parameters, String key, int align) throws Exception {
		addText(parameters, key, align, null);
	}

	public Document getDocument() {
		return document;
	}

	public void addText(Map<String, Object> parameters, String key, int align, Font font) throws Exception {
		addText(parameters, key, align, font, true);
	}

	public void addText(Map<String, Object> parameters, String key, int align, Font font, boolean addNewLine) throws Exception {
		Document doc = getDocument();
		String text = String.valueOf(parameters.get(key));
		if (StringUtils.isEmpty(text) || text.equals("null")) { //$NON-NLS-1$
			return;
		}
		boolean html = false;
		if (text.contains("<div>")) { //$NON-NLS-1$
			text = text.replaceAll("<div>", "").replaceAll("</div>", "<br>").replaceAll("&nbsp;", " "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
			html = true;
		}
		PdfPTable table = new PdfPTable(1);
		table.setHorizontalAlignment(PdfPTable.ALIGN_CENTER);
		table.setWidthPercentage(100);
		String[] splitTexts = text.split("<br>"); //$NON-NLS-1$
		if (splitTexts.length > 0) {
			for (int i = 0; i < splitTexts.length; i++) {
				String p = splitTexts[i];
				if (p.contains("<b>")) { //$NON-NLS-1$
					font = font_bold;
				}
				if (font == null) {
					font = font_normal;
				}
				String displayText = p.replaceAll("<.*?>", "").replaceAll("</", "").replaceAll("&nbsp;", " "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
				if (StringUtils.isEmpty(displayText)) {
					continue;
				}
				if (html) {
					displayText = displayText.replaceAll("\n", ""); //$NON-NLS-1$ //$NON-NLS-2$
				}
				if (p.contains("<strike>")) { //$NON-NLS-1$
					displayText = Messages.getString("EscPosPrintService.25") + " " + displayText; //$NON-NLS-1$
				}
				PdfPCell c1 = new PdfPCell(new Phrase(displayText, font));
				c1.setHorizontalAlignment(align);
				c1.setBorder(PdfPCell.NO_BORDER);
				table.addCell(c1);
				if (p.contains("<barcode>")) { //$NON-NLS-1$
					Image barcodeImage = getBarcode(parameters);
					if (barcodeImage != null) {
						c1 = new PdfPCell(barcodeImage);
						c1.setHorizontalAlignment(align);
						c1.setBorder(PdfPCell.NO_BORDER);
						table.addCell(c1);
					}
				}
				else if (p.contains("<storeLogo>")) { //$NON-NLS-1$
					Image logo = getImage(parameters, 200);
					if (logo != null) {
						c1 = new PdfPCell(logo);
						c1.setHorizontalAlignment(align);
						c1.setBorder(PdfPCell.NO_BORDER);
						table.addCell(c1);
					}
				}
				if (p.contains("</b>")) { //$NON-NLS-1$
					font = font_normal;
				}
			}
		}
		if (addNewLine) {
			PdfPCell c1 = new PdfPCell(new Phrase(Chunk.NEWLINE));
			c1.setBorder(PdfPCell.NO_BORDER);
			table.addCell(c1);
		}
		doc.add(table);
	}

	private Chunk getStrikeThroughText(String text, Font font) {
		Chunk chunk = new Chunk(text, font);
		chunk.setUnderline(.5f, 3f);
		return chunk;
	}

	public void addTextWithUnderline(PdfPTable summarySection, String text) throws DocumentException {
		PdfPCell cell2 = getCell("          ", Element.ALIGN_LEFT); //$NON-NLS-1$
		PdfPCell cell = getCell(text, Element.ALIGN_LEFT);
		cell.setColspan(2);
		cell.setBorder(PdfPCell.BOTTOM);
		summarySection.addCell(cell);
		summarySection.addCell(cell2);
		summarySection.addCell(cell2);
	}

	public void addLine(Document document, int number) throws DocumentException {
		for (int i = 0; i < number; i++) {
			document.add(new Paragraph(" ")); //$NON-NLS-1$
		}
	}

	public static ITextPrinter create() {
		return new ITextPrinter();
	}

}
