package com.floreantpos.model.dao;

import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.Customer;
import com.floreantpos.model.RentalDurationType;
import com.floreantpos.model.RentalLease;
import com.floreantpos.model.RentalStatus;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.util.DateUtil;

public class RentalLeaseDAO extends BaseRentalLeaseDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public RentalLeaseDAO() {
	}

	public void loadFullLease(RentalLease ticket) {
		if (ticket.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(ticket.getInvoices())) {
			return;
		}
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(ticket);

			initialize(ticket, session);
		} finally {
			closeSession(session);
		}
	}

	public void initialize(RentalLease ticket, Session session) {
		Hibernate.initialize(ticket.getInvoices());
		cleanupLeaseItems(ticket);
	}

	private void cleanupLeaseItems(RentalLease ticket) {
		List<Ticket> invoices = ticket.getInvoices();
		if (invoices != null) {
			for (Iterator iterator = invoices.iterator(); iterator.hasNext();) {
				Ticket invoice = (Ticket) iterator.next();
				if (invoice == null) {
					iterator.remove();
				}
			}
		}
	}

	public RentalLease loadFullLease(String id, String outletId) {
		if (StringUtils.isBlank(id)) {
			return null;
		}
		try (Session session = createNewSession()) {
			RentalLease ticket = (RentalLease) get(getReferenceClass(), id, session);
			if (ticket == null)
				return null;

			initialize(ticket, session);
			return ticket;
		}
	}

	public void createOrUpdateLease(RentalLease lease, Ticket ticket) {
		try (Session session = this.createNewSession()) {
			Transaction transaction = session.beginTransaction();
			if (StringUtils.isBlank(lease.getId())) {
				Date nextBillingDate = lease.getNextBillingDate();
				if (nextBillingDate == null) {
					lease.setNextBillingDate(
							nextBillingDate = getNextBillingDate(lease.getFromDate(), lease.getBillingCycleDuration(), lease.getBillingCycleDurationType()));
				}
				List<TicketItem> ticketItems = ticket.getTicketItems();
				for (TicketItem ticketItem : ticketItems) {
					ticketItem.setRentalStartDate(lease.getFromDate());
					ticketItem.setRentalEndDate(nextBillingDate);
				}
				RentalLeaseDAO.getInstance().save(lease, session);
				ticket.setRentalLeaseId(lease.getId());
				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}
			else {
				Date billingStartDate = DateUtil.startOfDay(StringUtils.isBlank(ticket.getId()) ? lease.getNextBillingDate() : lease.getFromDate());
				Date billingEndDate = getNextBillingDate(billingStartDate, lease.getBillingCycleDuration(), lease.getBillingCycleDurationType());
				lease.setNextBillingDate(billingEndDate);
				List<TicketItem> ticketItems = ticket.getTicketItems();
				for (TicketItem ticketItem : ticketItems) {
					if (!ticketItem.isRentalItem()) {
						continue;
					}
					ticketItem.setRentalStartDate(billingStartDate);
					ticketItem.setRentalEndDate(billingEndDate);
				}
				ticket.setRentalLeaseId(lease.getId());
				RentalLeaseDAO.getInstance().update(lease, session);
				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}
			transaction.commit();
		}
	}

	public Date getNextBillingDate(Date date, int duration, String durationType) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		switch (RentalDurationType.valueOf(durationType)) {
			case HOUR:
				cal.add(Calendar.HOUR_OF_DAY, duration);
				break;
			case DAY:
				cal.add(Calendar.DAY_OF_MONTH, duration);
				break;
			case WEEK:
				cal.add(Calendar.WEEK_OF_YEAR, duration);
				break;
			case MONTH:
				cal.add(Calendar.MONTH, duration);
				break;
			default:
				throw new IllegalArgumentException("Invalid rental duration type: " + durationType);
		}
		return cal.getTime();
	}

	public RentalLease getRecurringLease(String id) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(RentalLease.PROP_ID, id));
			criteria.add(Restrictions.eq(RentalLease.PROP_RECURRING, Boolean.TRUE));
			criteria.add(Restrictions.eq(RentalLease.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.isNotNull(RentalLease.PROP_NEXT_BILLING_DATE));
			addDeletedFilter(criteria);
			return (RentalLease) criteria.uniqueResult();
		}
	}

	public RentalLease findActiveLeaseByLeaseId(String leaseId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(RentalLease.PROP_ID, leaseId));
			criteria.add(Restrictions.ne(RentalLease.PROP_STATUS, RentalStatus.CLOSED.name()));
			criteria.setMaxResults(1);
			addDeletedFilter(criteria);
			criteria.addOrder(Order.desc(RentalLease.PROP_CREATED_DATE));
			return (RentalLease) criteria.uniqueResult();
		}
	}

	public RentalLease findActiveLease(String rentalItemId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(RentalLease.PROP_RENTAL_ITEM_ID, rentalItemId));
			criteria.add(Restrictions.ne(RentalLease.PROP_STATUS, RentalStatus.CLOSED.name()));
			criteria.setMaxResults(1);
			addDeletedFilter(criteria);
			criteria.addOrder(Order.desc(RentalLease.PROP_CREATED_DATE));
			return (RentalLease) criteria.uniqueResult();
		}
	}

	public List<String> findRentalItemIds(Customer customer) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(RentalLease.PROP_RENTAL_ITEM_ID));
			criteria.add(Restrictions.eq(RentalLease.PROP_CUSTOMER_ID, customer.getId()));
			addDeletedFilter(criteria);
			criteria.addOrder(Order.desc(RentalLease.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

}