package com.floreantpos.astm;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MachineResult;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemResult;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.MachineResultDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.floreantpos.model.ext.LabWorkStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.InventoryUnitConversionRule;
import com.google.gson.Gson;

public class AstmUtil {
	private static AstmResultListener astmResultListener;

	public static Test saveResult(String postData) {
		Test test = parseAstmMessage(postData);

		String sampleId = test.sampleId;
		if (StringUtils.isBlank(sampleId)) {
			//throw new PosException("Sample id is empty");
			PosLog.error(AstmUtil.class, "Sample id not found for machine data");
			sampleId = "00000";
		}
		PosLog.debug(AstmUtil.class, String.format("Sample id: %s", sampleId));

		MachineResult machineResult = new MachineResult();
		machineResult.setResultDate(StoreDAO.getServerTimestamp());
		machineResult.setSampleId(sampleId);
		machineResult.setOutletId(DataProvider.get().getCurrentOutletId());
		machineResult.setResultJson(new Gson().toJson(test));
		machineResult.putMachineRawData(postData);
		MachineResultDAO.getInstance().saveOrUpdate(machineResult);
		PosLog.debug(AstmUtil.class, String.format("Ticket item for sample %s not found. Result has been stored.", sampleId));

		if (astmResultListener != null) {
			astmResultListener.lastResult(machineResult);
		}

		TicketItem ticketItem = TicketItemDAO.getInstance().findByLabTest(sampleId);
		if (ticketItem == null) {
			return test;
		}

		addResultToTicketItem(test, ticketItem);
		TicketItemDAO.getInstance().saveOrUpdate(ticketItem);
		return test;
	}

	public static void addResultToTicketItem(String testId, TicketItem ticketItem) {
		MachineResult machineResult = MachineResultDAO.getInstance().getOrderHistoryBySampleId(testId);
		addResultToTicketItem(ticketItem, machineResult);
	}

	public static void addResultToTicketItem(TicketItem ticketItem, MachineResult machineResult) {
		ticketItem.setTestItemResults(Collections.emptyList());
		if (machineResult != null && ticketItem != null) {
			String resultJson = machineResult.getResultJson();
			Test test = new Gson().fromJson(resultJson, Test.class);
			addResultToTicketItem(test, ticketItem);
			ticketItem.putMachineResultTime(machineResult.getResultDate());
		}
		else {
			ticketItem.putMachineResultReceived(false);
			ticketItem.putMachineResultTime(null);
			ticketItem.setLabWorkStatusValue(LabWorkStatus.RUN_TEST);
		}
	}

	private static void addResultToTicketItem(Test test, TicketItem ticketItem) {
		Map<String, Result> machineResultmap = test.resultmap;
		List<TestItem> menuItemTestItems = getMenuItemTestItems(ticketItem.getMenuItemId());
		for (TestItem testItem : menuItemTestItems) {
			String machineCode = testItem.getMachineCode();
			if (StringUtils.isBlank(machineCode)) {
				continue;
			}

			Result machineResult = machineResultmap.get(machineCode);
			if (machineResult == null) {
				continue;
			}
			testItem.putOriginalMachineResult(machineResult.result);
			if (StringUtils.isBlank(testItem.getUnit())) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);
			testItem.putMachineName(test.machine);
			machineResultmap.remove(machineCode);
		}

		int sortOrder = menuItemTestItems.size();
		List<Result> remainResults = new ArrayList<Result>(machineResultmap.values());
		for (int i = 0; i < remainResults.size(); i++) {
			Result machineResult = remainResults.get(i);

			String machineCode = machineResult.testName;
			TestItem testItem = TestItemDAO.getInstance().findByMachineCode(machineCode);
			if (testItem == null) {
				testItem = new TestItem();
				testItem.setName(machineCode);
				testItem.setMachineCode(machineCode);
			}
			else {
				testItem.setLastSyncTime(null);
				testItem.setLastUpdateTime(null);
			}

			testItem.putMachineName(test.machine);
			testItem.setId(machineResult.resultNo);
			testItem.putOriginalMachineResult(machineResult.result);
			String testItemUnit = testItem.getUnit();
			if (StringUtils.isBlank(testItemUnit)) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);

			testItem.putShowTestItem(false);
			testItem.setSortOrder(++sortOrder);
			menuItemTestItems.add(testItem);
			//parsedResult += String.format("Result no: %s, name=%s, value: %s, unit: %s", result.resultNo, result.testName, result.result, result.unit);
			//parsedResult += "\n";
		}

		List<TestItemResult> testItemResults = ticketItem.getTestItemResults();
		testItemResults.clear();
		TestItemResult testItemResult = new TestItemResult();
		testItemResult.setSortOrder(testItemResults.size());
		testItemResult.setTestItems(menuItemTestItems);
		testItemResults.add(testItemResult);
		ticketItem.setTestItemResults(testItemResults);

		ticketItem.setTestItems(menuItemTestItems);
		ticketItem.putMachineResultReceived(true);
		ticketItem.putMachineResultTime(StoreDAO.getServerTimestamp());
		ticketItem.setLabWorkStatusValue(LabWorkStatus.RESULT_RECORDED);
	}

	private static void convertMachineResultToItemUnit(TestItem testItem, Result result) {
		try {
			String unitid = testItem.getUnit();
			if (org.apache.commons.lang.StringUtils.isBlank(unitid)) {
				return;
			}
			InventoryUnit inventoryUnit = InventoryUnitDAO.getInstance().get(unitid);
			if (inventoryUnit == null) {
				return;
			}
			if (inventoryUnit.getConversionRate() > 0) {
				double newRsult = Double.valueOf(result.result);

				String inventoryUnitConversionRuleName = inventoryUnit.getProperty(InventoryUnit.PROP_CONVERSION_RULE,
						InventoryUnitConversionRule.DIVISION.getName());
				InventoryUnitConversionRule inventoryUnitConversionRule = InventoryUnitConversionRule.fromName(inventoryUnitConversionRuleName);
				if (inventoryUnitConversionRule == InventoryUnitConversionRule.MULTIPLICATION) {
					newRsult = newRsult * 1D / inventoryUnit.getConversionRate();
				}
				else {
					newRsult = newRsult / inventoryUnit.getConversionRate();
				}
				result.result = String.valueOf(newRsult);
			} 
		} catch (Exception e) {
			//ignore
		}

	}

	public static Test parseAstmMessage(String astmMessage) {
		String regex = "(?:^(R)\\|(\\d+)\\|\\^*([^|]*)\\|([^|]*)\\|(?:([^|]*)\\|)?.*$)|(?:^(P)\\|(\\d+)\\|?.*$)|(?:^(O)\\|(\\d+)\\|\\|\\^*\\s*(\\d+)\\^\\w.*$)|(?:^(H)\\|\\\\\\^&\\|*([\\w-*\\d]*)\\^*.*$)";

		final Pattern pattern = Pattern.compile(regex, Pattern.CASE_INSENSITIVE | Pattern.MULTILINE);
		final Matcher matcher = pattern.matcher(astmMessage);

		Test test = new Test();

		while (matcher.find()) {

			if (matcher.group(11) != null && matcher.group(11).equals("H")) {
				test.machine = matcher.group(12);
			}
			else if (matcher.group(6) != null && matcher.group(6).equalsIgnoreCase("P")) {
				test.patiendId = matcher.group(7);
				
				PosLog.debug(AstmUtil.class, "Patient ID: " + test.patiendId);
			}
			else if (matcher.group(8) != null && matcher.group(8).equalsIgnoreCase("O")) {
				test.sampleId = matcher.group(10);
				
				PosLog.debug(AstmUtil.class, "Sample ID: " + test.sampleId);
			}
			else if (matcher.group(1) != null && matcher.group(1).equalsIgnoreCase("R")) {
				Result result = new Result();
				result.resultNo = matcher.group(2);
				result.testName = cleanTestName(matcher.group(3));
				result.result = matcher.group(4);
				result.unit = matcher.group(5);

				InventoryUnitDAO instance = InventoryUnitDAO.getInstance();
				InventoryUnit inventoryUnit = instance.findByName(result.unit);
				if (inventoryUnit == null) {
					inventoryUnit = new InventoryUnit();
					inventoryUnit.setConversionRate(1.0);
					inventoryUnit.setName(result.unit);
					if (result.unit.length() > 10) {
						inventoryUnit.setCode(String.format("%.10s", result.unit)); //$NON-NLS-1$
					}
					else {
						inventoryUnit.setCode(result.unit);
					}
					instance.save(inventoryUnit);
				}

				result.unit = inventoryUnit.getId();
				result.inventoryUnit = inventoryUnit;

				test.resultmap.put(result.testName, result);

				PosLog.debug(AstmUtil.class,
						String.format("Result no: %s, name=%s, value: %s, unit: %s", result.resultNo, result.testName, result.result, result.unit));
			}
		}

		return test;
	}

	public static List<TestItem> getMenuItemTestItems(String menuItemId) {
		MenuItem menuItem = MenuItemDAO.getInstance().loadInitialized(menuItemId);

		List<TestItem> testItems = menuItem.getTestItems();
		if (testItems != null) {
			int sortorder = 1;
			for (TestItem testItem : testItems) {
				if (testItem == null) {
					continue;
				}
				testItem.setSortOrder(sortorder);
				sortorder++;
			}
		}
		return testItems;
	}

	static String cleanTestName(String testName) {
		//String[] chars = { "%", "#", "^" };
		String[] chars = { "^" }; //$NON-NLS-1$
		//System.out.println("cleaning: " + testName);
		for (int i = 0; i < chars.length; i++) {
			int index = testName.indexOf(chars[i]);
			if (index > 0) {
				return testName.substring(0, index);
			}
		}

		return testName;
	}

	public static class Test {
		public String machine = null;
		public String patiendId = null;
		public String sampleId = null;
		//List<Result> results = new ArrayList<AstmService.Result>();
		public Map<String, Result> resultmap = new HashMap<String, Result>();
	}

	public static class Result {
		public String resultNo;
		public String result;
		public String testName;
		public String unit;
		public InventoryUnit inventoryUnit;
		public InventoryUnit menuItemInventoryUnit;
	}

	public static void setAstmResultListener(AstmResultListener astmResultListener) {
		AstmUtil.astmResultListener = astmResultListener;
	}

	public static void main(String[] args) {
		Test astmMessage = parseAstmMessage(getSampleMachineData());
		System.out.println(astmMessage);
	}

	public static String getSampleMachineData() {
		return "H|\\^&|||XP-100^00-13^^^^A7355^BS649542||||||||E1394-97\r\n" + 
				"P|1\r\n" + 
				"O|1||^^          05146^A|^^^^WBC\\^^^^RBC\\^^^^HGB\\^^^^HCT\\^^^^MCV\\^^^^MCH\\^^^^MCHC\\^^^^PLT\\^^^^LYM%\\^^^^MXD%\\^^^^NEUT%\\^^^^LYM#\\^^^^MXD#\\^^^^NEUT#\\^^^^RDW-SD\\^^^^RDW-CV\\^^^^PDW\\^^^^MPV\\^^^^P-LCR\\^^^^PCT|||||||N||||||||||||||F\r\n" + 
				"R|1|^^^^WBC^1| 16.7|10*3/uL||H||||               ||20240205132426\r\n" + 
				"R|2|^^^^RBC^1| 4.27|10*6/uL||N||||               ||20240205132426\r\n" + 
				"R|3|^^^^HGB^1| 12.1|g/dL||N||||               ||20240205132426\r\n" + 
				"R|4|^^^^HCT^1| 33.6|%||N||||               ||20240205132426\r\n" + 
				"R|5|^^^^MCV^1| 78.7|fL||L||||               ||20240205132426\r\n" + 
				"R|6|^^^^MCH^1| 28.3|pg||N||||               ||20240205132426\r\n" + 
				"R|7|^^^^MCHC^1| 36.0|g/dL||N||||               ||20240205132426\r\n" + 
				"R|8|^^^^PLT^1|  281|10*3/uL||W||||               ||20240205132426\r\n" + 
				"R|9|^^^^LYM%^1| 12.7|%||N||||               ||20240205132426\r\n" + 
				"R|10|^^^^MXD%^1| 16.2|%||N||||               ||20240205132426\r\n" + 
				"R|11|^^^^NEUT%^1| 71.1|%||N||||               ||20240205132426\r\n" + 
				"R|12|^^^^LYM#^1|  2.1|10*3/uL||N||||               ||20240205132426\r\n" + 
				"R|13|^^^^MXD#^1|  2.7|10*3/uL||N||||               ||20240205132426\r\n" + 
				"R|14|^^^^NEUT#^1| 11.9|10*3/uL||N||||               ||20240205132426\r\n" + 
				"R|15|^^^^RDW-SD^1| 43.6|fL||N||||               ||20240205132426\r\n" + 
				"R|16|^^^^RDW-CV^1| 15.3|%||N||||               ||20240205132426\r\n" + 
				"R|17|^^^^PDW^1| 10.7|fL||N||||               ||20240205132426\r\n" + 
				"R|18|^^^^MPV^1| 10.1|fL||N||||               ||20240205132426\r\n" + 
				"R|19|^^^^P-LCR^1| 25.0|%||N||||               ||20240205132426\r\n" + 
				"R|20|^^^^PCT^1| 0.28|%||W||||               ||20240205132426\r\n" + 
				"L|1|N\r\n" + 
				"";

	}
}
