/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseBookingInfo;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import liquibase.util.StringUtils;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "userName", "customer", "user" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class BookingInfo extends BaseBookingInfo implements TimedModel {
	private static final long serialVersionUID = 1L;

	public static final String STATUS_CANCEL = "cancel"; //$NON-NLS-1$
	public static final String STATUS_CLOSE = "close"; //$NON-NLS-1$
	public static final String STATUS_NO_APR = "no appear"; //$NON-NLS-1$
	public static final String STATUS_SEAT = "seat"; //$NON-NLS-1$
	public static final String STATUS_DELAY = "delay"; //$NON-NLS-1$
	public static final String STATUS_OPEN = "open"; //$NON-NLS-1$
	public static final String STATUS_CONFIRM = "confirm"; //$NON-NLS-1$
	public static final String STATUS_EXPIRED = "expired"; //$NON-NLS-1$

	private transient JsonObject propertiesContainer;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public BookingInfo() {
	}

	/**
	 * Constructor for primary key
	 */
	public BookingInfo(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/
	public String toString() {
		return getId() == null ? "" : getId().toString();
	}

	@XmlTransient
	private String customerName;
	@XmlTransient
	private String customerPhone;
	@XmlTransient
	private String bookedTableNumbers;

	@XmlTransient
	private Customer customer;
	@XmlTransient
	private User user;

	public void setUser(User user) {
		setUserId(user == null ? null : user.getId());
		this.user = user;
	}

	public User getUser() {
		String userId = getUserId();
		if (userId == null) {
			return null;
		}
		if (user != null && userId.equals(user.getId())) {
			return user;
		}
		return DataProvider.get().getUserById(userId, getOutletId());
	}

	public void setCustomer(Customer customer) {
		if (customer == null) {
			setCustomerId(null);
		}
		else {
			setCustomerId(customer.getId());
		}
		this.customer = customer;
	}

	@XmlTransient
	public Customer getCustomer() {
		String customerId = getCustomerId();
		if (customerId == null) {
			return null;
		}
		if (customer != null && customerId.equals(customer.getId())) {
			return customer;
		}
		return DataProvider.get().getCustomer(customerId);
	}

	/**
	 * @return the customerInfo
	 */
	public String getCustomerName() {
		Customer customer = getCustomer();

		if (customer == null) {
			return customerName;
		}
		if (StringUtils.isNotEmpty(customer.getFirstName())) { //$NON-NLS-1$
			customerName = customer.getFirstName() + " "; //$NON-NLS-1$
		}

		if (StringUtils.isNotEmpty(customer.getLastName())) { //$NON-NLS-1$
			customerName += customer.getLastName();
		}

		return customerName;
	}

	/**
	 * @param customerInfo the customerInfo to set
	 */
	public void setCustomerName(String customerInfo) {
		this.customerName = customerInfo;
	}

	public String getCustomerPhone() {
		Customer customer = getCustomer();

		if (customer == null) {
			return customerPhone;
		}
		if (StringUtils.isNotEmpty(customer.getMobileNo())) { //$NON-NLS-1$
			customerPhone = customer.getMobileNo() + " "; //$NON-NLS-1$
		}
		return customerPhone;
	}

	public void setCustomerPhone(String customerPhone) {
		this.customerPhone = customerPhone;
	}

	public void setBookedTableNumbers(String bookTableNumbers) {
		this.bookedTableNumbers = bookTableNumbers;
	}

	public BookingInfo clone(BookingInfo source) {
		return (BookingInfo) SerializationUtils.clone(source);
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		propertiesContainer.addProperty(key, value);
	}

	public String getProperty(String key) {
		if (propertiesContainer == null) {
			return null;
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public String getPatientName() {
		Customer patient = getCustomer();
		if (patient == null) {
			return "";
		}
		return patient.getName();
	}

	public String getPatientPhoneNo() {
		Customer patient = getCustomer();
		if (patient == null) {
			return "";
		}
		return patient.getMobileNo();
	}

	public void calculateBookingPrice() {
		Ticket ticket = getTicket();
		if (ticket == null) {
			return;
		}
		double bookingDays = 1.0;
		if (getToDate() != null) {
			bookingDays = ChronoUnit.DAYS.between(getFromDate().toInstant(), getToDate().toInstant());
		}
		Map<String, TicketItem> ticketItemMap = new HashMap<String, TicketItem>();
		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			String bed = ticketItem.getProperty("bedId");
			if (org.apache.commons.lang3.StringUtils.isBlank(bed)) {
				continue;
			}
			ticketItem.setQuantity(bookingDays);
			ticketItemMap.put(bed, ticketItem);
		}
		List<String> addedBedIds = new ArrayList<>();
		for (Bed bed : getBeds()) {
			addedBedIds.add(bed.getId());
			TicketItem ticketItem = ticketItemMap.get(bed.getId());
			if (ticketItem == null) {
				MenuItem menuItem = new MenuItem();
				menuItem.setName(bed.getNumber().trim());
				menuItem.setPrice(bed.getRoom().getRentPerBed());
				menuItem.setProductType(ProductType.GOODS.name());
				TicketItem tItem = menuItem.convertToTicketItem(ticket, bookingDays);
				tItem.addProperty("bedId", bed.getId());
				ticket.addToticketItems(tItem);
			}
		}
		for (Iterator iterator = ticket.getTicketItems().iterator(); iterator.hasNext();) {
			TicketItem ticketItem = (TicketItem) iterator.next();
			if (!addedBedIds.contains(ticketItem.getProperty("bedId"))) {
				iterator.remove();
			}
		}
		ticket.calculatePrice();
	}

	@XmlTransient
	@JsonIgnore
	public List<Symptom> getSymptoms() {
		List<Symptom> symptoms = new ArrayList<>();
		if (propertiesContainer == null || !propertiesContainer.has("symptoms")) { //$NON-NLS-1$
			return symptoms;
		}
		com.google.gson.JsonArray jsonArray = propertiesContainer.getAsJsonArray("symptoms"); //$NON-NLS-1$
		for (int i = 0; i < jsonArray.size(); i++) {
			JsonObject jsonObject = jsonArray.get(i).getAsJsonObject();
			Symptom symptom = new Symptom();
			symptom.setName(getString(jsonObject, Symptom.PROP_NAME));
			symptom.setId(getString(jsonObject, Symptom.PROP_ID));
			symptoms.add(symptom);
		}
		return symptoms;
	}

	private String getString(JsonObject jsonObject, String key) {
		if (!jsonObject.has(key)) {
			return null;
		}

		return jsonObject.get(key).getAsString();
	}

	public void setSymptoms(List<Symptom> symptoms) {
		if (symptoms == null) {
			symptoms = new ArrayList<>();
		}
		com.google.gson.JsonArray symptomsArray = new com.google.gson.JsonArray();
		for (Symptom symptom : symptoms) {
			JsonObject jsonObject = new JsonObject();
			jsonObject.addProperty(Symptom.PROP_ID, symptom.getId());
			jsonObject.addProperty(Symptom.PROP_NAME, symptom.getName());
			symptomsArray.add(jsonObject);
		}
		if (propertiesContainer == null) {
			propertiesContainer = new Gson().fromJson(super.getProperties(), JsonObject.class);
		}
		propertiesContainer.add("symptoms", symptomsArray); //$NON-NLS-1$
		setProperties(propertiesContainer.toString());
	}

}