/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosLog;
import com.floreantpos.model.AppointmentStatus;
import com.floreantpos.model.Bed;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.PatientBookingStatus;
import com.floreantpos.model.SequenceNumber;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.ShopTableStatus;
import com.floreantpos.model.util.BookingUtil;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class BookingInfoDAO extends BaseBookingInfoDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public BookingInfoDAO() {
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof BookingInfo) {
			BookingInfo bookingInfo = (BookingInfo) obj;
			bookingInfo.setDeleted(Boolean.TRUE);
			super.update(bookingInfo, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<BookingInfo> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getBookedTables(Date startDate, Date endDate) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.or(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate), Restrictions.le(BookingInfo.PROP_TO_DATE, endDate)));
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getAllOpenBooking() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.ne(BookingInfo.PROP_STATUS, BookingInfo.STATUS_CLOSE));
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;

	}

	@SuppressWarnings("unchecked")
	public void setUrgentBookingInfoInTables(Set<ShopTable> tables, Session session) {
		Transaction tx = null;
		try {
			tx = session.beginTransaction();
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(BookingInfo.PROP_CLOSED, false));
			criteria.add(Restrictions.eq(BookingInfo.PROP_STATUS, BookingInfo.STATUS_OPEN));

			Calendar cal = Calendar.getInstance();
			Date start = cal.getTime();
			cal.add(Calendar.MINUTE, 30);
			Date end = cal.getTime();

			criteria.add(Restrictions.between(BookingInfo.PROP_FROM_DATE, start, end));

			List<Integer> tableIds = new ArrayList<Integer>();
			for (ShopTable shopTable : tables) {
				tableIds.add(shopTable.getId());
			}
			criteria.createAlias("tables", "tableIds");
			criteria.add(Restrictions.in("tableIds." + ShopTable.PROP_ID, tableIds));

			List<BookingInfo> updatedBookingInfoList = new ArrayList<BookingInfo>();
			if (!updatedBookingInfoList.isEmpty()) {
				for (BookingInfo bookingInfo : updatedBookingInfoList) {
					update(bookingInfo, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			PosLog.error(getClass(), e);
		}
	}

	public void setBookingStatus(BookingInfo bookingInfo, String bookingStatus, List<ShopTableStatus> tableStatusList) {

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			bookingInfo.setStatus(bookingStatus);
			saveOrUpdate(bookingInfo);

			if (bookingStatus.equals(BookingInfo.STATUS_SEAT) || bookingStatus.equals(BookingInfo.STATUS_DELAY)) {
				ShopTableDAO.getInstance().bookedTables(tableStatusList);
			}

			if (bookingStatus.equals(BookingInfo.STATUS_CANCEL) || bookingStatus.equals(BookingInfo.STATUS_NO_APR)
					|| bookingStatus.equals(BookingInfo.STATUS_CLOSE)) {
				ShopTableDAO.getInstance().freeTables(tableStatusList);
			}

			tx.commit();

		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(BookingInfo.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}

	}

	public List<BookingInfo> getTodaysBooking() {
		return getTodaysBooking(null, null);
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getTodaysBooking(String bookingStatus, String searchTxt) {
		try (Session session = this.createNewSession()) {
			Calendar startDate = Calendar.getInstance();
			startDate.setLenient(false);
			startDate.setTime(new Date());
			startDate.set(Calendar.HOUR_OF_DAY, 0);
			startDate.set(Calendar.MINUTE, 0);
			startDate.set(Calendar.SECOND, 0);
			startDate.set(Calendar.MILLISECOND, 0);

			Calendar endDate = Calendar.getInstance();
			endDate.setLenient(false);
			endDate.setTime(new Date());
			endDate.set(Calendar.HOUR_OF_DAY, 23);
			endDate.set(Calendar.MINUTE, 59);
			endDate.set(Calendar.SECOND, 59);

			Criteria criteria = null;
			List<String> customerIds = null;

			if (StringUtils.isNotEmpty(searchTxt)) {
				criteria = session.createCriteria(Customer.class);
				this.addDeletedFilter(criteria);
				// @formatter:off
				criteria.add(
						Restrictions.or(
								Restrictions.ilike(Customer.PROP_NAME, searchTxt, MatchMode.ANYWHERE),
								Restrictions.ilike(Customer.PROP_MOBILE_NO,  searchTxt, MatchMode.START)
								)
						);
				// @formatter:on
				criteria.setProjection(Projections.property(Customer.PROP_ID));
				customerIds = criteria.list();
			}
			criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			if (bookingStatus == null) {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate.getTime()))
						.add(Restrictions.le(BookingInfo.PROP_FROM_DATE, endDate.getTime()));
			}
			else {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate.getTime()))
						.add(Restrictions.le(BookingInfo.PROP_FROM_DATE, endDate.getTime())).add(Restrictions.eq(BookingInfo.PROP_STATUS, bookingStatus));
			}

			if (customerIds != null && !customerIds.isEmpty()) {
				criteria.add(Restrictions.in(BookingInfo.PROP_CUSTOMER_ID, customerIds));
			}
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	//	@SuppressWarnings("unchecked")
	//	public List<ShopTable> getAllBookedTablesByDate(Date startDate, Date endDate) {
	//		try (Session session = this.createNewSession()) {
	//			Criteria criteria = session.createCriteria(BookingInfo.class);
	//			this.addDeletedFilter(criteria);
	//			criteria.add(Restrictions.and(Restrictions.gt(BookingInfo.PROP_TO_DATE, startDate), Restrictions.lt(BookingInfo.PROP_FROM_DATE, endDate)))
	//					.add(Restrictions.eq(BookingInfo.PROP_STATUS, BookingInfo.STATUS_OPEN));
	//
	//			List<BookingInfo> list = criteria.list();
	//			List<ShopTable> bookedTableList = new ArrayList<ShopTable>();
	//			for (Iterator<BookingInfo> iterator = list.iterator(); iterator.hasNext();) {
	//				BookingInfo bookingInfo = iterator.next();
	//				for (ShopTable shopTable : bookingInfo.getTables()) {
	//					bookedTableList.add(shopTable);
	//				}
	//			}
	//			return bookedTableList;
	//		} catch (Exception e) {
	//			PosLog.error(getClass(), e);
	//		}
	//		return null;
	//	}

	public List<BookingInfo> getAllBookingInfoByDate(Date startDate, Date endDate) {
		return getAllBookingInfoByDate(startDate, endDate, null, null);
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getAllBookingInfoByDate(Date startDate, Date endDate, String status, String searchTxt) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = null;
			List<String> customerIds = null;
			if (StringUtils.isNotEmpty(searchTxt)) {
				criteria = session.createCriteria(Customer.class);
				this.addDeletedFilter(criteria);
				// @formatter:off
				criteria.add(
						Restrictions.or(
								Restrictions.ilike(Customer.PROP_NAME, searchTxt, MatchMode.ANYWHERE),
								Restrictions.ilike(Customer.PROP_MOBILE_NO,  searchTxt, MatchMode.START)
								)
						);
				// @formatter:on
				criteria.setProjection(Projections.property(Customer.PROP_ID));
				customerIds = criteria.list();
			}
			criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			if (status == null) {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate)).add(Restrictions.lt(BookingInfo.PROP_FROM_DATE, endDate));
			}
			else {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate)).add(Restrictions.lt(BookingInfo.PROP_FROM_DATE, endDate))
						.add(Restrictions.eq(BookingInfo.PROP_STATUS, status));
			}
			if (customerIds != null && !customerIds.isEmpty()) {
				criteria.add(Restrictions.in(BookingInfo.PROP_CUSTOMER_ID, customerIds));
			}
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public BookingInfo findById(String currentBookingId) {
		if (StringUtils.isEmpty(currentBookingId)) {
			return null;
		}
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(BookingInfo.PROP_ID, currentBookingId));
			return (BookingInfo) criteria.uniqueResult();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public BookingInfo findByBookingId(String bookingId) {
		if (StringUtils.isEmpty(bookingId)) {
			return null;
		}
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(BookingInfo.PROP_BOOKING_ID, bookingId));
			return (BookingInfo) criteria.uniqueResult();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> findByShopTable(ShopTable table, Session session) {
		try {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			criteria.createAlias("tables", "tables");
			criteria.add(Restrictions.eq("tables." + ShopTable.PROP_ID, table.getId()));
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public void saveOrUpdateBookingInfos(List<BookingInfo> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<BookingInfo> iterator = dataList.iterator(); iterator.hasNext();) {
				BookingInfo item = (BookingInfo) iterator.next();
				BookingInfo existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public int rowCount(String appointmentId, Doctor doctor, Date date) {
		return rowCount(appointmentId, doctor, date, true);
	}

	public int rowCount(String appointmentId, Doctor doctor, Date date, boolean filterCanceled) {
		return rowCount(appointmentId, null, doctor, date, filterCanceled);
	}

	public int rowCount(String appointmentId, String shiftId, Doctor doctor, Date date, boolean filterCanceled) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			updateCriteria(appointmentId, doctor, date, filterCanceled, criteria);
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult == null ? 0 : uniqueResult.intValue();
		}
	}

	private void updateCriteria(String appointmentId, Doctor doctor, Date date, boolean filterCanceled, Criteria criteria) {
		updateCriteria(appointmentId, doctor, date, date, filterCanceled, criteria);
	}

	private void updateCriteria(String appointmentId, Doctor doctor, Date fromDate, Date toDate, boolean filterCanceled, Criteria criteria) {
		criteria.add(Restrictions.eq(BookingInfo.PROP_DOCTOR_ID, doctor.getId()));
		if (filterCanceled) {
			addDeletedFilter(criteria);
			criteria.add(Restrictions.ne(BookingInfo.PROP_STATUS, AppointmentStatus.CANCELLED.name()));
		}
		if (StringUtils.isNotBlank(appointmentId)) {
			criteria.add(Restrictions.ne(BookingInfo.PROP_ID, appointmentId));
		}
		criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, DateUtil.startOfDay(fromDate)));
		if (toDate != null) {
			criteria.add(Restrictions.le(BookingInfo.PROP_FROM_DATE, DateUtil.endOfDay(toDate)));
		}
	}

	public List<BookingInfo> getBookingInfos(Date startTime, Date endTime, Doctor doctor) {
		return getBookingInfos(startTime, endTime, doctor, true);
	}

	public List<BookingInfo> getBookingInfos(Date startTime, Date endTime, Doctor doctor, boolean completed) {
		return getBookingInfos(startTime, endTime, doctor, completed, true);
	}

	public List<BookingInfo> getBookingInfos(Date startTime, Date endTime, Doctor doctor, boolean completed, boolean excludeCancelled) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (doctor != null) {
				criteria.add(Restrictions.eq(BookingInfo.PROP_DOCTOR_ID, doctor.getId()));
			}
			if (completed) {
				criteria.add(Restrictions.eq(BookingInfo.PROP_STATUS, PatientBookingStatus.DISCHARGE.name()));
			}
			if (excludeCancelled) {
				criteria.add(Restrictions.ne(BookingInfo.PROP_STATUS, PatientBookingStatus.CANCELLED.name()));
			}
			if (startTime != null && endTime != null) {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, DateUtil.startOfDay(startTime)));
				criteria.add(Restrictions.le(BookingInfo.PROP_FROM_DATE, DateUtil.endOfDay(endTime)));
			}
			return criteria.list();
		}
	}

	public static String generateBookingId(String outletId) {
		try (Session session = getInstance().createNewSession()) {
			return generateBookingId(session, outletId);
		}
	}

	public static String generateBookingId(Session session, String outletId) {
		if (StringUtils.isBlank(outletId)) {
			outletId = DataProvider.get().getOutletId();
		}
		String prefix = SequenceNumber.yearMonthDayFormat.format(new Date());
		String bookingInfo = prefix + SequenceNumber.threeDigitDecimalFormat
				.format(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.BOOKING_SEQUENCE_NUMBER, prefix));
		while (getInstance().get(bookingInfo, session) != null) {
			bookingInfo = prefix + SequenceNumber.threeDigitDecimalFormat
					.format(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.BOOKING_SEQUENCE_NUMBER, prefix));
		}
		return bookingInfo;
	}

	public boolean isOccupied(Bed bed, Date selectedFromDate, Date selectedToDate) {
		List list = getBookings(bed);
		if (list == null || list.isEmpty()) {
			return false;
		}
		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			BookingInfo bookingInfo = (BookingInfo) iterator.next();
			Date startDate = DateUtil.startOfDay(bookingInfo.getFromDate());
			Date endDate = bookingInfo.getToDate();
			if (endDate == null) {
				endDate = DateUtil.endOfDay(startDate);
			}
			else {
				endDate = BookingUtil.buildBookingEndTime(endDate);
			}
			if (DateUtil.between(startDate, endDate, selectedFromDate)) {
				return true;
			}
			if (selectedToDate != null && DateUtil.between(startDate, endDate, selectedToDate)) {
				return true;
			}
			if (selectedToDate != null) {
				Calendar c = Calendar.getInstance();
				c.setTime(selectedFromDate);
				do {
					if (DateUtil.between(startDate, endDate, c.getTime())) {
						return true;
					}
					c.add(Calendar.DAY_OF_MONTH, 1);
				} while (!c.getTime().after(selectedToDate));
			}
		}
		return false;
	}

	public List<BookingInfo> getBookings(Bed bed) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.createAlias("beds", "beds");
			criteria.add(Restrictions.eq("beds." + Bed.PROP_ID, bed.getId()));
			criteria.add(Restrictions.eq(BookingInfo.PROP_STATUS, PatientBookingStatus.BOOKED.name()));
			criteria.add(Restrictions.eq(BookingInfo.PROP_CLOSED, Boolean.FALSE));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(BookingInfo.PROP_FROM_DATE), BookingInfo.PROP_FROM_DATE);
			projectionList.add(Projections.property(BookingInfo.PROP_TO_DATE), BookingInfo.PROP_TO_DATE);
			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(BookingInfo.class));
			return criteria.list();
		}
	}

	public List<String> getBookingIds(Bed bed) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.createAlias("beds", "beds");
			criteria.add(Restrictions.eq("beds." + Bed.PROP_ID, bed.getId()));
			criteria.add(Restrictions.eq(BookingInfo.PROP_STATUS, PatientBookingStatus.BOOKED.name()));
			criteria.add(Restrictions.eq(BookingInfo.PROP_CLOSED, Boolean.FALSE));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(BookingInfo.PROP_BOOKING_ID));
			criteria.setProjection(projectionList);
			return criteria.list();
		}
	}

	public void saveOrUpdateBooking(BookingInfo booking) {
		//		Transaction tx = null;
		//		try (Session session = BookingInfoDAO.getInstance().createNewSession()) {
		//			tx = session.beginTransaction();
		//			TicketDAO.getInstance().saveOrUpdate(booking.getTicket(), session);
		//			BookingInfoDAO.getInstance().saveOrUpdate(booking, session);
		//			tx.commit();
		//		} catch (Exception e) {
		//			if (tx != null) {
		//				tx.rollback();
		//			}
		//		}
		TicketDAO.getInstance().saveOrUpdate(booking.getTicket());
		BookingInfoDAO.getInstance().saveOrUpdate(booking);
	}

	public List<BookingInfo> getAllBookings(Bed bed) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.createAlias("beds", "beds");
			criteria.add(Restrictions.eq("beds." + Bed.PROP_ID, bed.getId()));
			criteria.add(Restrictions.eq(BookingInfo.PROP_STATUS, PatientBookingStatus.BOOKED.name()));
			criteria.add(Restrictions.eq(BookingInfo.PROP_CLOSED, Boolean.FALSE));
			criteria.addOrder(Order.asc(BookingInfo.PROP_FROM_DATE));
			return criteria.list();
		}
	}
}