package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.GlobalConfig;
import com.floreantpos.util.GsonUtil;
import com.floreantpos.util.POSUtil;
import com.google.common.reflect.TypeToken;
import com.orocube.medlogics.Module;

public class GlobalConfigDAO extends BaseGlobalConfigDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public GlobalConfigDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void addProperty(String key, String value) {
		GlobalConfig globalConfig = findByKey(key);
		if (globalConfig == null) {
			globalConfig = new GlobalConfig();
			globalConfig.setKey(key);
		}
		globalConfig.setValue(value);
		saveOrUpdate(globalConfig);
	}

	public String getProperty(String key) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(GlobalConfig.PROP_VALUE));
			criteria.add(Restrictions.eq(GlobalConfig.PROP_KEY, key));
			return (String) criteria.uniqueResult();
		}
	}

	public String getProperty(String key, String defaultValue) {
		String propValue = getProperty(key);
		if (StringUtils.isBlank(propValue)) {
			return defaultValue;
		}
		return propValue;
	}

	public GlobalConfig findByKey(String key) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(GlobalConfig.PROP_KEY, key));
			criteria.setMaxResults(1);
			return (GlobalConfig) criteria.uniqueResult();
		}
	}

	public static boolean getBooleanProperty(String key, boolean defaultValue) {
		return POSUtil.getBoolean(getInstance().getProperty(key), defaultValue);
	}

	public static Boolean isShouldSentAppointmentSMS() {
		return getBooleanProperty("appointment.sent.sms", false); //$NON-NLS-1$
	}

	public static void setSentAppointmentSMS(boolean sentSms) {
		getInstance().addProperty("appointment.sent.sms", String.valueOf(sentSms)); //$NON-NLS-1$
	}

	public void deleteByKey(String key) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(GlobalConfig.class);
			criteria.add(Restrictions.eq(GlobalConfig.PROP_KEY, key));
			criteria.setMaxResults(1);

			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null) {
				Transaction transaction = session.beginTransaction();
				session.delete(uniqueResult);
				transaction.commit();
			}
		}
	}

	public boolean isDoctorsCanChangePrescriptionPad() {
		return getBooleanProperty("doctors_can_change_prescription_pad", false); //$NON-NLS-1$
	}

	public void putDoctorsCanChangePrescriptionPad(boolean doctorsCanChangePrescriptionPad) {
		addProperty("doctors_can_change_prescription_pad", String.valueOf(doctorsCanChangePrescriptionPad)); //$NON-NLS-1$
	}

	public void putEnabledModules(List<Module> enableKeyList) {
		addProperty("module.enabled.list", GsonUtil.createGson().toJson(enableKeyList));
	}

	public List<Module> getEnabledModules() {
		String enableModulesNames = getProperty("module.enabled.list", "[" + Module.DIAGNOSTICS_LAB.name() + "," + Module.FRONT_DESK.name() + "]");
		return GsonUtil.createGson().fromJson(enableModulesNames, new TypeToken<List<Module>>() {
		}.getType());
	}

	public static List<Module> getEnabledStandAloneModules() {
		List<Module> standAloneModules = new ArrayList<Module>();
		for (Module module : getInstance().getEnabledModules()) {
			if (module == null) {
				continue;
			}
			if (module.isStandalone()) {
				standAloneModules.add(module);
			}
		}
		Comparator<Module> comparator = Comparator.comparing(Module::toString, Comparator.nullsLast(String.CASE_INSENSITIVE_ORDER));
		standAloneModules.sort(comparator);
		return standAloneModules;
	}

	public static boolean isEnabledInventoryPlugin() {
		return getInstance().isEnabledPlugin(Module.INVENTORY);
	}

	public static Boolean isEnabledRefSystemPlugin() {
		return getInstance().isEnabledPlugin(Module.REFERRAL);
	}

	public static boolean isEnabledFrontDeskPlugin() {
		return getInstance().isEnabledPlugin(Module.FRONT_DESK);
	}

	public static boolean isEnabledIPDPlugin() {
		return getInstance().isEnabledPlugin(Module.IPD);
	}

	public static boolean isEnabledDiagnosticsLabPlugin() {
		return getInstance().isEnabledPlugin(Module.DIAGNOSTICS_LAB);
	}

	public static boolean isEnabledPharmacyPlugin() {
		return getInstance().isEnabledPlugin(Module.PHARMACY);
	}

	public static boolean isEnabledAppointmentPlugin() {
		return getInstance().isEnabledPlugin(Module.OPD);
	}

	public boolean isEnabledPlugin(Module module) {
		return getInstance().getEnabledModules().contains(module);
	}

	public void setLastSelectedModule(String terminalKey, Module selectedDepartment) {
		addProperty(terminalKey + ".selected_department", selectedDepartment.name());
	}

	public Module getLastSelectedModule(String terminalKey) {
		String selectedDepartment = getProperty(terminalKey + ".selected_department");
		if (StringUtils.isNotBlank(selectedDepartment)) {
			return Module.valueOf(selectedDepartment);
		}
		return null;
	}

}