package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.StockCount;
import com.floreantpos.model.StockCountItem;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class StockCountDAO extends BaseStockCountDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public StockCountDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdate(StockCount stockCount, boolean updateStock) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(stockCount, updateStock, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdate(StockCount stockCount, boolean updateStock, Session session) {
		if (stockCount.getId() == null) {
			stockCount.setCreatedDate(new Date());
		}
		stockCount.setLastModifiedDate(new Date());
		if (updateStock) {
			adjustInventoryItems(session, stockCount);
		}
		session.saveOrUpdate(stockCount);
	}

	private void adjustInventoryItems(Session session, StockCount stockCount) {
		List<StockCountItem> orderItems = stockCount.getCountItems();
		if (orderItems == null) {
			return;
		}
		for (StockCountItem orderItem : orderItems) {
			if (orderItem.isAdjusted())
				continue;
			orderItem.setAdjusted(Boolean.TRUE);

			Double stockDifference = orderItem.getActualUnit() - orderItem.getUnitOnHand();

			if (stockDifference == 0)
				continue;

			MenuItem menuItem = MenuItemDAO.getInstance().get(orderItem.getItemId(), session);
			if (menuItem == null) {
				menuItem = MenuItemDAO.getInstance().getMenuItemBySKU(orderItem.getSku(), session, true);
				if (menuItem != null) {
					orderItem.setItemId(menuItem.getId());
				}
			}
			Hibernate.initialize(menuItem.getStockUnits());

			if (ProductType.match(menuItem.getProductType(), ProductType.MEDICINE)) {
				List<InventoryStock> inventoryStocks = InventoryStockDAO.getInstance().getInventoryStocks(menuItem.getId(), orderItem.getUnitId(),
						orderItem.getUnitType(), orderItem.getInventoryLocation().getId(), session);
				for (InventoryStock inventoryStock : inventoryStocks) {
					InventoryTransaction stockTrans = createInventoryTransaction(menuItem, (inventoryStock.getQuantityInHand() * -1), orderItem, stockCount);
					stockTrans.setBatchNumber(inventoryStock.getBatchNumber());
					stockTrans.putExpiryDate(inventoryStock.getExpireDate());
					InventoryTransactionDAO.getInstance().adjustInventoryStock(stockTrans, session);
				}
				Date expireDate = InventoryStockDAO.getInstance().getNearestExpireDate(menuItem, orderItem, session);
				InventoryTransaction stockTrans = createInventoryTransaction(menuItem, orderItem.getActualUnit(), orderItem, stockCount);
				stockTrans.putExpiryDate(expireDate);
				InventoryTransactionDAO.getInstance().adjustInventoryStock(stockTrans, session);
			}
			else {
				InventoryTransaction stockTrans = createInventoryTransaction(menuItem, stockDifference, orderItem, stockCount);
				InventoryTransactionDAO.getInstance().adjustInventoryStock(stockTrans, session);
			}
		}
	}

	private InventoryTransaction createInventoryTransaction(MenuItem menuItem, Double stockDifference, StockCountItem orderItem, StockCount stockCount) {
		InventoryTransaction stockTrans = new InventoryTransaction();
		stockTrans.setMenuItem(menuItem);
		stockTrans.setQuantity(Math.abs(stockDifference));
		stockTrans.setUnit(orderItem.getUnitId());
		stockTrans.setUnitType(orderItem.getUnitType());
		double baseUnitQuantity = menuItem.getBaseUnitQuantity(orderItem.getUnitId());
		stockTrans.setUnitCost(baseUnitQuantity * menuItem.getCost());
		stockTrans.setTotal(baseUnitQuantity * stockTrans.getQuantity() * menuItem.getCost());
		stockTrans.setTransactionDate(new Date());
		stockTrans.setOutletId(stockCount.getOutletId());
		IUnit unit = DataProvider.get().getUnitById(orderItem.getUnitId(), orderItem.getUnitType());
		if (unit instanceof InventoryStockUnit) {
			InventoryStockUnit inventoryStockUnit = (InventoryStockUnit) unit;
			stockTrans.setUnitPrice(inventoryStockUnit.getPrice());
			stockTrans.setUnitCost(inventoryStockUnit.getCost());
		}
		String reason = orderItem.getReason();

		if (StringUtils.isEmpty(reason)) {
			String strReason = (stockDifference > 0) ? InventoryTransaction.REASON_ADJUST_IN : InventoryTransaction.REASON_ADJUST_OUT;
			stockTrans.setReason(strReason);
		}

		else {
			stockTrans.setReason(reason);
		}

		if (stockDifference > 0) {
			stockTrans.setType(InventoryTransactionType.IN.getType());
			stockTrans.setToInventoryLocation(orderItem.getInventoryLocation());
		}
		else {
			stockTrans.setType(InventoryTransactionType.OUT.getType());
			stockTrans.setFromInventoryLocation(orderItem.getInventoryLocation());
		}
		return stockTrans;
	}

	public String getNextStockCountSequenceNumber() {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.max(StockCount.PROP_REF_NUMBER));
			Object maxNumber = criteria.uniqueResult();
			if (maxNumber == null)
				return "10001";
			try {
				return String.valueOf(Integer.parseInt((String) maxNumber) + 1);
			} catch (Exception e) {
				return "";
			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void findBy(PaginationSupport dataModel, String ref, Date from, Date to, Boolean showVerified) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (StringUtils.isNotEmpty(ref)) {
				criteria.add(Restrictions.ilike(StockCount.PROP_REF_NUMBER, ref));
			}
			if (from != null && to != null) {
				criteria.add(Restrictions.ge(StockCount.PROP_CREATED_DATE, from));
				criteria.add(Restrictions.le(StockCount.PROP_CREATED_DATE, to));
			}
			if (!showVerified) {
				criteria.add(Restrictions.isNull(StockCount.PROP_VERIFIED_BY));
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());

			dataModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	//download and upload
	public void saveOrUpdateStockCounts(List<StockCount> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<StockCount> iterator = dataList.iterator(); iterator.hasNext();) {
				StockCount item = (StockCount) iterator.next();
				List<StockCountItem> items = item.getCountItems();
				item.setCountItems(null);
				item.setUpdateLastUpdateTime(updateLastUpdateTime);
				item.setUpdateSyncTime(updateSyncTime);

				StockCountDAO dao = StockCountDAO.getInstance();
				StockCount existingItem = dao.get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					//					long version = existingItem.getVersion();
					//					item.setVersion(version);
				}
				else {
					save(item, session);
				}
				List<StockCountItem> existingItems = null;
				if (existingItem != null && existingItem.getCountItems() != null) {
					existingItems = existingItem.getCountItems();
				}
				if (existingItems == null) {
					existingItems = new ArrayList<>();
				}
				if (items != null && items.size() > 0) {
					for (StockCountItem stockCountItem : items) {
						stockCountItem.setStockCount(item);
						int idx = existingItems.indexOf(stockCountItem);
						StockCountItem existingStockCountItem = null;
						if (idx != -1) {
							existingStockCountItem = existingItems.get(idx);
							if (existingStockCountItem == null) {
								//stockCountItem.setVersion(0);
								StockCountItemDAO.getInstance().save(stockCountItem, session);
							}
							else {
								//stockCountItem.setVersion(existingStockCountItem.getVersion());
							}
						}
						else {
							//stockCountItem.setVersion(0);
							StockCountItemDAO.getInstance().save(stockCountItem, session);
						}
					}
				}
				item.setCountItems(items);
				saveOrUpdate(item, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

}