/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Agent;
import com.floreantpos.model.Customer;
import com.floreantpos.model.CustomerGroup;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.MenuModifier;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.Patient;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.BeanTableModel;
import com.floreantpos.swing.PaginatedListModel;
import com.floreantpos.swing.PaginatedTableModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.AESencrp;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class CustomerDAO extends BaseCustomerDAO {

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public CustomerDAO() {
	}

	@Override
	public List<Customer> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		Customer customer = (Customer) obj;
		if (customer == null) {
			throw new PosException(Messages.getString("CustomerNotFound")); //$NON-NLS-1$
		}

		customer.setDeleted(Boolean.TRUE);
		saveOrUpdate(customer, session);
		removeCustomerFromCustomerGroup(customer, session);
	}

	public static void doCheckValidation(Object obj) {
		Customer customer = (Customer) obj;

		String firstName = customer.getFirstName();
		if (StringUtils.isNotBlank(firstName) && firstName.length() > 128) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "first name", 128)); //$NON-NLS-1$
		}

		String lastName = customer.getLastName();
		if (StringUtils.isNotBlank(lastName) && lastName.length() > 128) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "last name", 128)); //$NON-NLS-1$
		}

		String name = customer.getName();
		if (StringUtils.isNotBlank(name) && name.length() > 256) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "name", 256)); //$NON-NLS-1$
		}

		String email = customer.getEmail();
		if (StringUtils.isNotBlank(email) && email.length() > 128) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "email", 128)); //$NON-NLS-1$
		}

		String mobileNo = customer.getMobileNo();
		if (StringUtils.isNotBlank(mobileNo) && mobileNo.length() > 30) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "phone", 30)); //$NON-NLS-1$
		}

		String address = customer.getAddress();
		if (StringUtils.isNotBlank(address) && address.length() > 255) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "address", 255)); //$NON-NLS-1$
		}

		String zipCode = customer.getZipCode();
		if (StringUtils.isNotBlank(zipCode) && zipCode.length() > 10) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "zip code", 10)); //$NON-NLS-1$
		}

		String note = customer.getNote();
		if (StringUtils.isNotBlank(note) && note.length() > 255) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "note", 255)); //$NON-NLS-1$
		}

	}

	public static void doSaveCreateDate(Object obj) {
		Customer customer = (Customer) obj;
		if (customer.getCreateDate() == null) {
			Calendar currentTime = DateUtil.getServerTimeCalendar();
			customer.setCreateDate(currentTime.getTime());
		}
	}

	public void removeCustomerFromCustomerGroup(Customer customer, Session session) {
		Criteria criteria = session.createCriteria(CustomerGroup.class);
		addDeletedFilter(criteria);
		criteria.createAlias("customers", "customer"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("customer." + Customer.PROP_ID, customer.getId())); //$NON-NLS-1$
		List<CustomerGroup> customerGroups = criteria.list();
		if (customerGroups == null || customerGroups.isEmpty()) {
			return;
		}
		for (CustomerGroup group : customerGroups) {
			group.getCustomers().remove(customer);
			CustomerGroupDAO.getInstance().update(group, session);
		}
	}

	@Override
	public Order getDefaultOrder() {
		return Order.asc(Customer.PROP_ID);
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void findByMobileLoyalityName(String mobileEmailName, PaginatedTableModel tableModel) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Disjunction disjunction = Restrictions.disjunction();

			if (StringUtils.isNotEmpty(mobileEmailName)) {
				disjunction.add(Restrictions.like(Customer.PROP_MOBILE_NO, mobileEmailName, MatchMode.ANYWHERE));
				disjunction.add(Restrictions.like(Customer.PROP_NAME, mobileEmailName, MatchMode.ANYWHERE));
				disjunction.add(Restrictions.like(Customer.PROP_EMAIL, mobileEmailName, MatchMode.ANYWHERE));
				disjunction.add(Restrictions.like(Customer.PROP_LOYALTY_NO, mobileEmailName, MatchMode.ANYWHERE));
			}
			addDeletedFilter(criteria);
			criteria.add(disjunction);
			criteria.setFirstResult(0);
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setNumRows(tableModel.getPageSize());
			tableModel.setRows(criteria.list());

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	@Deprecated
	public int getNumberOfCustomers(String searchString) {
		Session session = null;
		Criteria criteria = null;
		try {
			if (StringUtils.isEmpty(searchString)) {
				return 0;
			}
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, "%" + searchString + "%")); //$NON-NLS-1$ //$NON-NLS-2$
			disjunction.add(Restrictions.ilike(Customer.PROP_NAME, "%" + searchString + "%")); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);
			criteria.add(disjunction);
			List list = criteria.list();
			if (list != null) {
				return list.size();
			}
			return 0;
		} finally {
			closeSession(session);
		}

	}

	@Deprecated
	public int getNumberOfCustomers(String mobile, String loyalty, String name) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			Disjunction disjunction = Restrictions.disjunction();

			if (StringUtils.isNotEmpty(mobile))
				disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, "%" + mobile + "%")); //$NON-NLS-1$ //$NON-NLS-2$

			if (StringUtils.isNotEmpty(loyalty))
				disjunction.add(Restrictions.ilike(Customer.PROP_LOYALTY_NO, "%" + loyalty + "%")); //$NON-NLS-1$ //$NON-NLS-2$

			if (StringUtils.isNotEmpty(name))
				disjunction.add(Restrictions.ilike(Customer.PROP_NAME, "%" + name + "%")); //$NON-NLS-1$ //$NON-NLS-2$

			addDeletedFilter(criteria);
			criteria.add(disjunction);

			List list = criteria.list();
			if (list != null) {
				return list.size();
			}
			return 0;
		} finally {
			closeSession(session);
		}

	}

	public List<Customer> findBy(String mobile, String loyalty, String name) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			Disjunction disjunction = Restrictions.disjunction();
			if (StringUtils.isNotEmpty(mobile))
				disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, "%" + mobile + "%")); //$NON-NLS-1$ //$NON-NLS-2$
			if (StringUtils.isNotEmpty(loyalty))
				disjunction.add(Restrictions.ilike(Customer.PROP_LOYALTY_NO, "%" + loyalty + "%")); //$NON-NLS-1$ //$NON-NLS-2$
			if (StringUtils.isNotEmpty(name))
				disjunction.add(Restrictions.ilike(Customer.PROP_NAME, "%" + name + "%")); //$NON-NLS-1$ //$NON-NLS-2$

			criteria.add(disjunction);
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * @deprecated use findByPhoneOrName instead
	 * @param searchString
	 * @param tableModel
	 */
	public void findBy(String searchString, PaginationSupport tableModel) {
		findByPhoneOrName(searchString, tableModel);
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_LOYALTY_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}

	}

	public void findByNameOrPhone(String searchString, String status, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			if (StringUtils.isNotEmpty(status)) {
				if (status.equals(POSConstants.ACTIVE)) {
					criteria.add(Restrictions.eq(Patient.PROP_ACTIVE, true));
				}
				else if (status.equals(POSConstants.DEACTIVE)) {
					criteria.add(Restrictions.eq(Patient.PROP_ACTIVE, false));
				}
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}

	}

	public void findReferrer(String searchString, PaginationSupport tableModel) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Property.forName("class").eq(Agent.class), Property.forName("class").eq(Doctor.class))); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}

	}

	public void findCustomers(String searchString, PaginationSupport tableModel) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void findReferrerByNameOrPhoneNoOrType(String nameString, String phoneNo, String agentAgentType, PaginationSupport tableModel) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(Agent.class);
			//criteria.add(Restrictions.or(Property.forName("class").eq(Agent.class), Property.forName("class").eq(Doctor.class))); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(nameString)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, nameString, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotEmpty(phoneNo)) {
				criteria.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, phoneNo, MatchMode.ANYWHERE));
			}
			if (StringUtils.isNotEmpty(agentAgentType)) {
				criteria.add(Restrictions.ilike(Customer.PROP_PROPERTIES, "\"agent.type\":\"%" + agentAgentType + "%\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}

	}

	public List<String> findReferrerIds() {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			criteria.setProjection(Projections.property(Customer.PROP_ID));
			criteria.add(Restrictions.or(Property.forName("class").eq(Agent.class), Property.forName("class").eq(Doctor.class))); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			return criteria.list();
		}
	}

	public List<Customer> findReferrer() {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			criteria.add(Restrictions.or(Property.forName("class").eq(Agent.class), Property.forName("class").eq(Doctor.class))); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			return criteria.list();
		}
	}

	public void findLabDoctorFee(PaginatedListModel tableModel, Customer customer) {
		findLabDoctorFee(tableModel, customer, false);
	}

	public void findReferralCommissionReport(PaginatedListModel tableModel, Customer customer) {
		findLabDoctorFee(tableModel, customer, true);
	}

	public void findLabDoctorFee(PaginatedListModel tableModel, Customer customer, boolean isRefefrralReport) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			addDeletedFilter(criteria);

			if (isRefefrralReport) {
				criteria.add(Restrictions.gt(Customer.PROP_COMMISSION, 0D));
			}
			else {
				criteria.add(Restrictions.gt(Customer.PROP_LAB_DOCTOR_FEE, 0D));
			}
			if (customer != null) {
				criteria.add(Restrictions.eq(Customer.PROP_ID, customer.getId()));
			}

			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_NAME).ignoreCase());
			tableModel.setData(criteria.list());
		}
	}

	public List<Customer> findDueReportOrders(Date beginDate, Date endDate) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Property.forName("class").eq(Agent.class), Property.forName("class").eq(Doctor.class))); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			criteria.add(Restrictions.gt(Customer.PROP_COMMISSION, 0d));

			return criteria.list();
		}
	}

	public List<Customer> findByMobileNumber(String mobileNo) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(mobileNo))
				criteria.add(Restrictions.eq(Customer.PROP_MOBILE_NO, mobileNo));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<Customer> findByName(String name) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(name))
				criteria.add(Restrictions.ilike(Customer.PROP_FIRST_NAME, name + "%".trim(), MatchMode.ANYWHERE)); //$NON-NLS-1$

			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}

	}

	public Customer findByEmailOrPhone(String searchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE));
			disjunction.add(Restrictions.ilike(Customer.PROP_EMAIL, searchString, MatchMode.ANYWHERE));
			criteria.add(disjunction);

			criteria.setMaxResults(1);
			Customer customer = (Customer) criteria.uniqueResult();
			return customer;
		}
	}

	public void findByNameOrEmail(Pagination tableModel, String searchString) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Disjunction disjunction = Restrictions.disjunction();

			if (StringUtils.isNotEmpty(searchString)) {
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_EMAIL, searchString, MatchMode.ANYWHERE)));

			}
			addDeletedFilter(criteria);
			criteria.add(disjunction);

			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			tableModel.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}

	}

	public Customer findById(String customerId) {
		return findById(customerId, false);
	}

	/**
	 * This method is used to get deleted customer if needed.
	 * 
	 * @param customerId 
	 * @param filterDeleted if true get deleted customer.
	 * 
	  */
	public Customer findById(String customerId, boolean filterDeleted) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (filterDeleted) {
				addDeletedFilter(criteria);
			}
			criteria.add(Restrictions.eq(Customer.PROP_ID, customerId));
			return (Customer) criteria.uniqueResult();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	@Deprecated
	public Customer findByPhoneOrEmail(String mobileNo, String email) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isEmpty(mobileNo)) {
				criteria.add(Restrictions.eq(Customer.PROP_EMAIL, email));
			}
			else {
				criteria.add(Restrictions.or(Restrictions.eq(Customer.PROP_MOBILE_NO, mobileNo), Restrictions.eq(Customer.PROP_EMAIL, email)));
			}
			List list = criteria.list();
			if (list.size() > 0) {
				return (Customer) list.get(0);
			}
			return null;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	@Deprecated
	public boolean deleteAll(List<Customer> customerList) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			String queryDeliveryAddress = "update DELIVERY_ADDRESS set CUSTOMER_ID = null"; //$NON-NLS-1$
			String queryTableBookingInfo = "update TABLE_BOOKING_INFO set customer_id = null"; //$NON-NLS-1$
			String queryTicketItemSeat = "update TICKET_ITEM_SEAT set MEMBER_ID = null"; //$NON-NLS-1$

			Query deliveryAddressUpdateQuery = session.createSQLQuery(queryDeliveryAddress);
			Query tableBookingInfoUpdateQuery = session.createSQLQuery(queryTableBookingInfo);
			Query ticketItemSeatUpdateQuery = session.createSQLQuery(queryTicketItemSeat);

			deliveryAddressUpdateQuery.executeUpdate();
			tableBookingInfoUpdateQuery.executeUpdate();
			ticketItemSeatUpdateQuery.executeUpdate();

			for (Customer customer : customerList) {
				delete(customer, session);
			}

			tx.commit();
			return true;
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(CustomerDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}

	}

	@Deprecated
	public int getNumberOfCustomers() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			criteria.list();
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public int getNumberOfCustomers(CustomerGroup customerGroup) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass(), "customer"); //$NON-NLS-1$
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			if (customerGroup != null) {
				criteria.add(Restrictions.eq(Customer.PROP_CUSTOMER_GROUP_ID, customerGroup.getId()));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public void loadCustomers(PaginatedTableModel tableModel) {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	public void loadCustomers(CustomerGroup customerGroup, PaginatedListModel tableModel) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (customerGroup != null) {
				criteria.add(Restrictions.eq(Customer.PROP_CUSTOMER_GROUP_ID, customerGroup.getId()));
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_NAME).ignoreCase());
			tableModel.setData(criteria.list());
			return;
		} finally {
			closeSession(session);
		}
	}

	public int getRowCount(String searchString) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.ilike(MenuModifier.PROP_NAME, searchString, MatchMode.START));
			}

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
		} finally {
			closeSession(session);
		}
		return 0;
	}

	public void loadCustomers(String searchString, BeanTableModel<Customer> listModel) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Customer.PROP_NAME).ignoreCase());
			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.START));
			}

			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			listModel.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	public Customer initialize(Customer customer) {
		if (customer == null || customer.getId() == null)
			return customer;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(customer);

			Hibernate.initialize(customer.getDeliveryAddresses());
			return customer;
		} finally {
			closeSession(session);
		}
	}

	//	public List<Customer> findAllUnSyncCustomer() {
	//		Session session = null;
	//		try {
	//			session = createNewSession();
	//			Criteria criteria = session.createCriteria(getReferenceClass());
	//			Criterion falseCriteria = Restrictions.eq(Customer.PROP_CLOUD_SYNCED, Boolean.FALSE);
	//			Criterion nullCriteria = Restrictions.isNull(Customer.PROP_CLOUD_SYNCED);
	//			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
	//			return criteria.list();
	//		} finally {
	//			if (session != null) {
	//				closeSession(session);
	//			}
	//		}
	//	}
	//
	//	public void updateCustomerSync(List<String> ids, boolean synced) {
	//
	//		if (ids == null || ids.isEmpty())
	//			return;
	//
	//		String whereCondition = "(";
	//		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
	//			String id = (String) iterator.next();
	//			whereCondition += "'" + id + "'";
	//			if (iterator.hasNext())
	//				whereCondition += ",";
	//		}
	//		whereCondition += ")";
//		//@formatter:off
//		Transaction tx=null;
//		Session session = null;
//		try {
//			session = getSession();
//			tx = session.beginTransaction();
//		String hqlString = "update Customer set %s=:synced where %s in"+whereCondition;
//		hqlString = String.format(hqlString, Customer.PROP_CLOUD_SYNCED, Customer.PROP_ID);
//		//@formatter:on
	//			Query query = session.createQuery(hqlString);
	//			query.setParameter("synced", synced);
	//			query.executeUpdate();
	//			tx.commit();
	//		} catch (Exception e) {
	//			tx.rollback();
	//			throw e;
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public void saveOrUpdateCustomers(List<Customer> customers, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (customers == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Customer customer : customers) {
				//				saveOrUpdate(customer, session);

				Customer existingCustomer = get(customer.getId());
				if (existingCustomer == null) {
					customer.setUpdateLastUpdateTime(updateLastUpdateTime);
					customer.setUpdateSyncTime(updateSyncTime);
					//customer.setBalance(0D);
					save(customer, session);
				}
				else {
					if (!BaseDataServiceDao.get().shouldSave(customer.getLastUpdateTime(), existingCustomer.getLastUpdateTime())) {
						PosLog.info(getClass(), customer.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingCustomer.getId();
					long version = existingCustomer.getVersion();
					double existingBalance = existingCustomer.getBalance();
					Integer existingLoyaltyPoint = existingCustomer.getLoyaltyPoint();
					PropertyUtils.copyProperties(existingCustomer, customer);
					existingCustomer.setId(id);
					existingCustomer.setVersion(version);
					existingCustomer.setBalance(existingBalance);
					existingCustomer.setLoyaltyPoint(existingLoyaltyPoint);
					existingCustomer.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingCustomer.setUpdateSyncTime(updateSyncTime);
					update(existingCustomer, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public Customer findByEmail(String email) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Customer.PROP_EMAIL, email));
			List list = criteria.list();
			if (list != null && list.size() > 0) {
				return (Customer) list.get(0);
			}
			return null;
		}
	}

	public boolean isEmailExist(String email) {
		if (StringUtils.isEmpty(email)) {
			return false;
		}
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Customer.PROP_EMAIL, email));
			criteria.setProjection(Projections.rowCount());
			Number number = (Number) criteria.uniqueResult();
			return number != null && number.intValue() > 0;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public boolean isEmailExistExceptCustomer(String email, Customer customer) {
		if (StringUtils.isEmpty(email)) {
			return false;
		}
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Customer.PROP_EMAIL, email));
			criteria.add(Restrictions.ne(Customer.PROP_ID, customer.getId()));
			criteria.setProjection(Projections.rowCount());
			Number number = (Number) criteria.uniqueResult();
			return number != null && number.intValue() > 0;

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<Customer> getCustomerByCustomerGroup(CustomerGroup customerGroup, Session session) {
		String sql = "SELECT DISTINCT CUSTOMER_ID FROM CUSTOMER_CUST_GROUP WHERE CUST_GROUP_ID = :tt_id"; //$NON-NLS-1$
		Query query = session.createSQLQuery(sql);
		query.setParameter("tt_id", customerGroup.getId()); //$NON-NLS-1$
		List customerIds = query.list();
		if (customerIds != null && !customerIds.isEmpty()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			criteria.add(Restrictions.in(Customer.PROP_ID, customerIds));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(Customer.PROP_FIRST_NAME), Customer.PROP_FIRST_NAME);
			projectionList.add(Projections.property(Customer.PROP_LAST_NAME), Customer.PROP_LAST_NAME);
			criteria.setProjection(projectionList);
			criteria.setMaxResults(100);
			return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
		}
		return null;
	}

	public Customer findCustomerByEmail(String email) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_EMAIL, email));
			criteria.setMaxResults(1);

			Customer uniqueResult = (Customer) criteria.uniqueResult();
			return uniqueResult;
		}
	}

	public Customer findCustomerBySecretKey(String secretKey) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_SECRET_CODE, AESencrp.getEncrypedString(secretKey)));
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			criteria.setMaxResults(1);

			Customer uniqueResult = (Customer) criteria.uniqueResult();
			return uniqueResult;
		}
	}

	public void findReferrerByName(String searchString, PaginationSupport tableModel) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Property.forName("class").eq(Agent.class), Property.forName("class").eq(Doctor.class))); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}

	}

	public void setCustomerLastUpdateTime(String customerId) {
		try (Session session = createNewSession()) {
			Transaction beginTransaction = session.beginTransaction();
			String customerLastUpdateTime = "update customer set last_update_time = :last_update_time where id =:id"; //$NON-NLS-1$
			SQLQuery createSQLQuery = session.createSQLQuery(customerLastUpdateTime);
			createSQLQuery.setParameter("last_update_time", StoreDAO.getServerTimestamp()); //$NON-NLS-1$
			createSQLQuery.setParameter("id", customerId); //$NON-NLS-1$
			createSQLQuery.executeUpdate();
			beginTransaction.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public String findCustomerDisplayName(String customerId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			criteria.add(Restrictions.eq(Customer.PROP_ID, customerId));
			criteria.setProjection(Projections.property(Customer.PROP_NAME));
			return (String) criteria.uniqueResult();
		}
	}

	public void findConcernedAgentsByDoctorId(String doctorId, PaginationSupport tableModel) {
		findConcernedAgentsByDoctorId(doctorId, tableModel, null);
	}

	public void findConcernedAgentsByDoctorId(String doctorId, PaginationSupport tableModel, String searchKeyword) {
		if (StringUtils.isBlank(doctorId)) {
			return;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Customer.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotBlank(searchKeyword)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.ilike(Customer.PROP_NAME, searchKeyword, MatchMode.ANYWHERE));
				disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, searchKeyword, MatchMode.ANYWHERE));
				criteria.add(disjunction);
			}

			DetachedCriteria dCriteria = DetachedCriteria.forClass(Ticket.class);
			dCriteria.add(Restrictions.eq(Ticket.PROP_DOCTOR_ID, doctorId));
			dCriteria.setProjection(Projections.distinct(Property.forName(Ticket.PROP_REFERRER_ID)));

			criteria.add(Property.forName(Customer.PROP_ID).in(dCriteria));

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void findCustomerByNameOrInvoiceOrPhoneOrEmail(String nameString, String phoneOrEmailString, String invoiceNo, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Customer.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(nameString)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, nameString, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotEmpty(phoneOrEmailString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_MOBILE_NO, phoneOrEmailString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_EMAIL, phoneOrEmailString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			if (StringUtils.isNotEmpty(invoiceNo)) {
				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Ticket.class);
				detachedCriteria.add(Restrictions.ilike(Ticket.PROP_ID, invoiceNo, MatchMode.END));
				detachedCriteria.setProjection(Projections.property(Ticket.PROP_CUSTOMER_ID));
				criteria.add(Property.forName(Patient.PROP_ID).in(detachedCriteria));
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

}