package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Department;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.Ticket;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.AESencrp;

public class DoctorDAO extends BaseDoctorDAO {

	public DoctorDAO() {
	}

	@Override
	public List<Doctor> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			return criteria.list();
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Doctor doctor = (Doctor) obj;
		if (doctor == null) {
			throw new PosException("Doctor not found"); //$NON-NLS-1$
		}

		doctor.setDeleted(Boolean.TRUE);
		saveOrUpdate(doctor, session);
		ActionHistoryDAO.saveHistory(ActionHistory.DOCTOR_DELETED,
				"Doctor name: " + doctor.getFirstName() + ", id: " + doctor.getId() + ", phone: " + doctor.getWorkPhoneNo() + ", email: " + doctor.getEmail());
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel) {
		findByPhoneOrName(searchString, tableModel, null);
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel, Department department) {
		findByAgentPhoneOrName(searchString, null, tableModel, department);
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel, Department department, Boolean isShowLabDoctor, Boolean isActive) {
		findByAgentPhoneOrName(searchString, null, tableModel, department, isShowLabDoctor, isActive);
	}

	public void findByAgentPhoneOrName(String searchString, String referrerId, PaginationSupport tableModel, Department department) {
		findByAgentPhoneOrName(searchString, referrerId, tableModel, department, false, true);
	}

	public void findByAgentPhoneOrName(String searchString, String referrerId, PaginationSupport tableModel, Department department, Boolean isShowLabDoctor,
			Boolean isActive) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);

			addDeletedFilter(criteria);

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			if (StringUtils.isNotBlank(referrerId)) {
				criteria.add(Restrictions.ilike(Customer.PROP_PROPERTIES, "\"doctor.agent.id\":\"%" + referrerId + "%\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			}
			if (department != null) {
				criteria.add(Restrictions.ilike(Customer.PROP_PROPERTIES, "\"doctor.department.id\":\"%" + department.getId() + "%\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			}
			if (isShowLabDoctor) {
				criteria.add(Restrictions.ilike(Customer.PROP_PROPERTIES, "\"doctor.lab\":\"" + isShowLabDoctor + "\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			}

			if (isActive != null) {
				criteria.add(Restrictions.eq(Doctor.PROP_ACTIVE, isActive));
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void findByAgentPhoneOrNameOrDepartment(String nameString, String mobileNoString, Department department, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(nameString)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, nameString, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotEmpty(mobileNoString)) {
				criteria.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, mobileNoString, MatchMode.ANYWHERE));
			}

			if (department != null) {
				criteria.add(Restrictions.ilike(Customer.PROP_PROPERTIES, "\"doctor.department.id\":\"%" + department.getId() + "%\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public List<Doctor> findByPhoneOrName(String searchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MEMBER_ID, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());

			List<Doctor> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return result;
		}
	}

	public List<Doctor> findLabDoctor() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			criteria.add(Restrictions.ilike(Customer.PROP_PROPERTIES, "\"doctor.lab\":\"" + true + "\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());

			List<Doctor> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return result;
		}
	}

	public List<Doctor> findDoctorsByAgent(String agentId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.ilike(Customer.PROP_PROPERTIES, "\"doctor.agent.id\":\"%" + agentId + "%\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			return criteria.list();
		}
	}

	public void findDoctorByName(String searchString, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public Doctor findDoctorByPIN(String PIN) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_PIN, AESencrp.getEncrypedString(PIN)));
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			criteria.setMaxResults(1);

			return (Doctor) criteria.uniqueResult();
		}
	}

	public Doctor findDoctorBySignatureCode(String code) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_SIGNATURE_APPROVAL_CODE, AESencrp.getEncrypedString(code)));
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			criteria.setMaxResults(1);

			return (Doctor) criteria.uniqueResult();
		}
	}

	public Doctor findByEmailOrPhone(String emailOrPhone) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);
			addDeletedFilter(criteria);

			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, emailOrPhone));
			disjunction.add(Restrictions.ilike(Customer.PROP_EMAIL, emailOrPhone));
			criteria.add(disjunction);

			criteria.setMaxResults(1);
			Doctor doctor = (Doctor) criteria.uniqueResult();
			return doctor;
		}
	}

	public void findConcernedDoctorsByAgentId(String referrerId, PaginationSupport tableModel) {
		findConcernedDoctorsByAgentId(referrerId, tableModel, null);
	}

	public void findConcernedDoctorsByAgentId(String referrerId, PaginationSupport tableModel, String searchKeyword) {
		if (StringUtils.isBlank(referrerId)) {
			return;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Doctor.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotBlank(searchKeyword)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.ilike(Customer.PROP_NAME, searchKeyword, MatchMode.ANYWHERE));
				disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, searchKeyword, MatchMode.ANYWHERE));
				criteria.add(disjunction);
			}

			DetachedCriteria dCriteria = DetachedCriteria.forClass(Ticket.class);
			dCriteria.add(Restrictions.eq(Ticket.PROP_REFERRER_ID, referrerId));
			dCriteria.setProjection(Projections.distinct(Property.forName(Ticket.PROP_DOCTOR_ID)));

			criteria.add(Property.forName(Customer.PROP_ID).in(dCriteria));

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}
}