/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.WordUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Address;
import com.floreantpos.model.Attribute;
import com.floreantpos.model.AttributeGroup;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.COAAccountType;
import com.floreantpos.model.COAAccountTypeGroup;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.CookingInstruction;
import com.floreantpos.model.Course;
import com.floreantpos.model.Currency;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.Customer;
import com.floreantpos.model.CustomerGroup;
import com.floreantpos.model.DeclaredTips;
import com.floreantpos.model.Department;
import com.floreantpos.model.Discount;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.InventoryClosingBalance;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.InventoryUnitGroup;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.InventoryVendorItems;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemSize;
import com.floreantpos.model.MenuModifier;
import com.floreantpos.model.ModifierGroup;
import com.floreantpos.model.Multiplier;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PayoutRecepient;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.PriceTable;
import com.floreantpos.model.PrinterGroup;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.ReportGroup;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.Shift;
import com.floreantpos.model.ShopFloor;
import com.floreantpos.model.ShopFloorTemplate;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.ShopTableStatus;
import com.floreantpos.model.ShopTableType;
import com.floreantpos.model.Specimen;
import com.floreantpos.model.StockCount;
import com.floreantpos.model.Store;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.StoreSessionControl;
import com.floreantpos.model.Tax;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TerminalType;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemGroup;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.User;
import com.floreantpos.model.UserType;
import com.floreantpos.model.VirtualPrinter;
import com.floreantpos.model.VoidItem;
import com.floreantpos.model.VoidReason;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;

public class GenericDAO extends _RootDAO {

	private final static GenericDAO instance = new GenericDAO();

	public GenericDAO() {
		super();
	}

	public static GenericDAO getInstance() {
		return instance;
	}

	@Override
	protected Class getReferenceClass() {
		return null;
	}

	@Override
	public Serializable save(Object obj) {
		updateTime(obj);
		return super.save(obj);
	}

	@Override
	public void saveOrUpdate(Object obj) {
		updateTime(obj);
		super.saveOrUpdate(obj);
	}

	@Override
	public Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	public void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void update(Object obj) {
		updateTime(obj);
		super.update(obj);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	public void delete(Object obj, Session s) {
		super.delete(obj, s);
	}

	@Override
	public Session getSession(String configFile, boolean createNew) {
		return super.getSession(configFile, createNew);
	}

	public List findAll(Class clazz, Session session) {
		Criteria crit = session.createCriteria(clazz);
		return crit.list();
	}

	public void saveAll(List list, Session session) {
		Transaction tx = session.beginTransaction();
		for (Object object : list) {
			session.saveOrUpdate(object);
		}
		tx.commit();
	}

	@Override
	public void closeSession(Session session) {
		try {
			super.closeSession(session);
		} catch (Exception x) {
		}
	}

	@SuppressWarnings("rawtypes")
	public List findAllUnSyncItem(Class refClass) {
		return findAllUnSyncItem(refClass, false, null);
	}

	@SuppressWarnings("rawtypes")
	public List findAllUnSyncItem(Class<?> refClass, boolean findAll, Date syncTime) {
		return getAllUnSyncItems(refClass, findAll, syncTime, 0);
	}

	public List findAllUnSyncItem(Class<?> refClass, boolean findAll, Date syncTime, int batchSize) {
		return getAllUnSyncItems(refClass, findAll, syncTime, batchSize);
	}

	private List getAllUnSyncItems(Class<?> refClass, boolean findAll, Date syncTime, int batchSize) {
		try (Session session = createNewSession()) {
			return getAllUnSyncItems(refClass, findAll, syncTime, session, batchSize);
		}
	}

	private List getAllUnSyncItems(Class<?> refClass, boolean findAll, Date syncTime, Session session, int batchSize) {
		Criteria criteria = session.createCriteria(refClass);

		createCriteriaToSearchUnSyncedItems(findAll, syncTime, criteria);
		if (batchSize > 0) {
			criteria.setMaxResults(batchSize);
		}

		criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
		return criteria.list();
	}

	private void createCriteriaToSearchUnSyncedItems(boolean findAll, Date syncTime, Criteria criteria) {
		if (!findAll) {
			Criterion nullUpdateTime = Restrictions.isNull(InventoryStock.PROP_LAST_UPDATE_TIME);
			Criterion nullSyncTime = Restrictions.isNull(InventoryStock.PROP_LAST_SYNC_TIME);
			Criterion gtQuery = Restrictions.gtProperty(InventoryStock.PROP_LAST_UPDATE_TIME, InventoryStock.PROP_LAST_SYNC_TIME);
			criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));
		}
		else if (findAll && syncTime != null) {
			Criterion nullUpdateTime = Restrictions.isNull(InventoryStock.PROP_LAST_SYNC_TIME);
			Criterion ltQuery = Restrictions.lt(InventoryStock.PROP_LAST_SYNC_TIME, syncTime);
			criteria.add(Restrictions.or(nullUpdateTime, ltQuery));
		}
	}

	public int getUnSyncItemCount(Class<?> refClass, boolean findAll, Date syncTime) {
		try (Session session = createNewSession()) {
			return getUnSyncItemCount(refClass, findAll, syncTime, session);
		}
	}

	private int getUnSyncItemCount(Class<?> refClass, boolean findAll, Date syncTime, Session session) {
		Criteria criteria = session.createCriteria(refClass);
		createCriteriaToSearchUnSyncedItems(findAll, syncTime, criteria);

		int rowCount = rowCount(criteria);
		return rowCount;
	}

	public void updateItemsLastSyncTime(List<String> ids, String tableName) {
		updateItemsLastSyncTime(null, ids, tableName);
	}

	public void updateItemsLastSyncTime(Date syncTime, List<String> ids, String tableName) {
		if (ids == null || ids.isEmpty())
			return;

		if (syncTime == null) {
			syncTime = new Date();
		}
		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			String id = (String) iterator.next();
			whereCondition += "'" + id + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + tableName + " set %s=:lastSyncTime where %s in %s"; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, InventoryStock.PROP_LAST_SYNC_TIME, InventoryStock.PROP_ID, whereCondition);
			Query query = session.createQuery(hqlString);
			query.setParameter("lastSyncTime", syncTime); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();

		} catch (Exception e) {
			tx.rollback();
			throw e;
		}

		//update last update time only if last update time is null
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(syncTime);
		calendar.add(Calendar.SECOND, -1);

		updateLastUpdateTimeIfNull(tableName, calendar.getTime());
	}

	public void updateItemsLastSyncTimeByInt(Date syncTime, List<Integer> ids, String tableName) {
		if (ids == null || ids.isEmpty())
			return;
		if (syncTime == null) {
			syncTime = new Date();
		}
		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			int id = (Integer) iterator.next();
			whereCondition += id;
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + tableName + " set %s=:lastSyncTime where %s in" + whereCondition; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, InventoryStock.PROP_LAST_SYNC_TIME, InventoryStock.PROP_ID);
			Query query = session.createQuery(hqlString);
			query.setParameter("lastSyncTime", syncTime); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();

		} catch (Exception e) {
			tx.rollback();
			throw e;
		}

		//update last update time only if last update time is null
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(syncTime);
		calendar.add(Calendar.SECOND, -1);

		updateLastUpdateTimeIfNull(tableName, calendar.getTime());
	}

	public void updateLastUpdateTimeIfNull(String tableName) {
		Calendar realTime = Calendar.getInstance();
		realTime.set(Calendar.MILLISECOND, realTime.get(Calendar.MILLISECOND));
		updateLastUpdateTimeIfNull(tableName, realTime.getTime());
	}

	public void updateLastUpdateTimeIfNull(String tableName, Date realTime) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + tableName + " set %s=:lastUpdateTime where %s is null"; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, Terminal.PROP_LAST_UPDATE_TIME, Terminal.PROP_LAST_UPDATE_TIME);
			Query query = session.createQuery(hqlString);
			query.setParameter("lastUpdateTime", realTime); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			PosLog.error(getClass(), Messages.getString("GenericDAO.19")); //$NON-NLS-1$
		}
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass) {
		try (Session session = createNewSession()) {
			checkIdOrNameExists(existingId, name, modelClass, session);
		}
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass, String outletId) {
		try (Session session = createNewSession()) {
			checkIdOrNameExists(existingId, name, modelClass, outletId, session);
		}
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass, Session session) {
		checkIdOrNameExists(existingId, name, modelClass, null, session); //$NON-NLS-1$
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass, String outletId, Session session) {
		if (StringUtils.isBlank(name)) {
			throw new PosException(POSConstants.NAME_IS_EMPTY);
		}
		checkDifferentObjectExists(existingId, name, modelClass, "name", outletId, session); //$NON-NLS-1$
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName) {
		try (Session session = createNewSession()) {
			checkDifferentObjectExists(existingId, fieldValue, modelClass, colName, session);
		}
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName, String outletId) {
		try (Session session = createNewSession()) {
			checkDifferentObjectExists(existingId, fieldValue, modelClass, colName, outletId, session);
		}
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName, Session session) {
		checkDifferentObjectExists(existingId, fieldValue, modelClass, colName, null, session);
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName, String outletId, Session session) {
		Criteria criteria = session.createCriteria(modelClass);
		criteria.setProjection(Projections.rowCount());
		criteria.add(Restrictions.eq(colName, fieldValue).ignoreCase());
		if (StringUtils.isNotEmpty(existingId)) {
			criteria.add(Restrictions.ne("id", existingId)); //$NON-NLS-1$
		}
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$
		}
		addDeletedFilter(criteria);
		Number rowCount = (Number) criteria.uniqueResult();
		if (rowCount != null && rowCount.intValue() > 0) {
			throw new DuplicateDataException(
					String.format("%s" + " (%s) " + Messages.getString("GenericDAO.22"), WordUtils.capitalizeFully(colName), fieldValue), //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
					colName);
		}
	}

	public Object findObjectByFieldValue(String value, Class modelClass, String id, String... colNames) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(modelClass);
			Disjunction disjunction = Restrictions.disjunction();
			for (String field : colNames) {
				disjunction.add(Restrictions.eq(field, value).ignoreCase());
			}
			criteria.add(disjunction);
			if (StringUtils.isNotEmpty(id)) {
				criteria.add(Restrictions.ne("id", id)); //$NON-NLS-1$
			}
			criteria.setMaxResults(1);
			addDeletedFilter(criteria);
			return criteria.uniqueResult();
		}
	}

	public Object findObjectByCompositeKey(Class modelClass, String id, String outletId) {
		if (StringUtils.isBlank(id) || StringUtils.isBlank(outletId)) {
			return null;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(modelClass);

			criteria.add(Restrictions.eq("id", id)); //$NON-NLS-1$
			criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$

			criteria.setMaxResults(1);
			return criteria.uniqueResult();
		}
	}

	public void updateLastUpdateTimeProperty(Class class1, Date time) {
		Store store = StoreDAO.getRestaurant();
		updateLastUpdateTimeProperty(store, class1, time);
		StoreDAO.getInstance().saveOrUpdate(store);
	}

	public void updateLastUpdateTimeProperty(Store store, Class class1, Date time) {
		updateLastUpdateTimeProperty(store, class1.getSimpleName(), time);
	}

	public void updateLastUpdateTimeProperty(Store store, String simpleName, Date time) {
		store.addProperty(simpleName + "." + Store.PROP_LAST_UPDATE_TIME, DateUtil.formatSyncTime(time)); //$NON-NLS-1$
	}

	@SuppressWarnings("unchecked")
	public List<String> getForeignDataListNames(Session session, Class<?> clazz, String column, String id) {
		return getForeignDataList(session, clazz, column, id, "name");//$NON-NLS-1$
	}

	public List getForeignDataList(Session session, Class<?> clazz, String column, String id, String fetchColumn) {
		Criteria criteria = session.createCriteria(clazz);
		criteria.setProjection(Projections.property(fetchColumn));
		criteria.add(Restrictions.eq(column, id));
		addDeletedFilter(criteria, clazz);
		return criteria.list();
	}

	@SuppressWarnings("unchecked")
	public List<String> getForeignDataListNames(Session session, Class<?> clazz, String collectionName, String column, String id) {
		return getForeignDataList(session, clazz, collectionName, column, id, "name"); //$NON-NLS-1$
	}

	@SuppressWarnings("rawtypes")
	public List getForeignDataList(Session session, Class<?> clazz, String collectionName, String column, String id, String fetchColumn) {
		Criteria criteria = session.createCriteria(clazz);
		criteria.createAlias(collectionName, "c"); //$NON-NLS-1$
		criteria.setProjection(Projections.property(fetchColumn));
		criteria.add(Restrictions.eq("c." + column, id)); //$NON-NLS-1$
		addDeletedFilter(criteria, clazz);
		return criteria.list();
	}

	public int getUnSyncPosTransactionItemCount(boolean findAll, Date syncTime) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			createCriteriaToSearchUnSyncedItems(findAll, syncTime, criteria);
			criteria.add(Restrictions.isNull(PosTransaction.PROP_TICKET));
			int rowCount = rowCount(criteria);
			return rowCount;
		}
	}

	public List findAllUnSyncPosTransactions(boolean findAll, Date syncTime, int batchSize) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			createCriteriaToSearchUnSyncedItems(findAll, syncTime, criteria);
			criteria.add(Restrictions.isNull(PosTransaction.PROP_TICKET));
			if (batchSize > 0) {
				criteria.setMaxResults(batchSize);
			}
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			return criteria.list();
		}
	}

	public List getUnSyncMenuItems(boolean findAll, Date syncTime, int batchSize) {
		try (Session session = createNewSession()) {
			List unSyncItems = getSyncableMenuItems(findAll, syncTime, session, batchSize);

			Set<MenuItem> variantParents = new LinkedHashSet<>();
			Set<MenuItem> itemSet = new HashSet<>();

			for (Iterator iterator = unSyncItems.iterator(); iterator.hasNext();) {
				MenuItem menuItem = (MenuItem) iterator.next();
				MenuItemDAO.getInstance().initialize(menuItem, session);

				if (menuItem.getVariants() != null) {
					variantParents.add(menuItem);

					for (MenuItem variantMenuItem : menuItem.getVariants()) {
						MenuItemDAO.getInstance().initialize(variantMenuItem);
						itemSet.add(variantMenuItem);
					}
				}
				else if (menuItem.isVariant()) {
					variantParents.add(menuItem.getParentMenuItem());
				}
				else {
					itemSet.add(menuItem);
				}
			}

			List<MenuItem> items = new ArrayList<>();
			for (MenuItem menuItem : variantParents) {
				items.add(menuItem);
			}
			for (MenuItem menuItem : itemSet) {
				items.add(menuItem);
			}

			return items;
		}
	}

	public List getSyncableMenuItems(boolean findAll, Date syncTime, Session session, int batchSize) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		createCriteriaToSearchUnSyncedItems(findAll, syncTime, criteria);

		if (batchSize > 0) {
			criteria.setMaxResults(batchSize);
		}

		criteria.addOrder(Order.asc(MenuItem.PROP_COMBO_ITEM));
		criteria.addOrder(Order.asc(MenuItem.PROP_VARIANT));
		criteria.addOrder(Order.asc(MenuItem.PROP_HAS_VARIANT));

		criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
		List list = criteria.list();

		return list;
	}

	public void updateTicketsLastSyncTime(Date syncTime, List<String> ids) {
		if (ids == null || ids.isEmpty())
			return;

		if (syncTime == null) {
			syncTime = new Date();
		}
		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			String id = (String) iterator.next();
			whereCondition += "'" + id + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + Ticket.REF + " set %s=:lastSyncTime,%s=:cloudSynced where %s in %s"; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, Ticket.PROP_LAST_SYNC_TIME, Ticket.PROP_CLOUD_SYNCED, InventoryStock.PROP_ID, whereCondition);
			Query query = session.createQuery(hqlString);
			query.setParameter("lastSyncTime", syncTime); //$NON-NLS-1$
			query.setParameter("cloudSynced", Boolean.TRUE); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
		//update last update time only if last update time is null
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(syncTime);
		calendar.add(Calendar.SECOND, -1);
		updateLastUpdateTimeIfNull(Ticket.REF, calendar.getTime());
	}

	public void changeCurrentOutletAndRelatedModels(List<String> outletRelatedModels, Outlet selectedOutlet) throws Exception {
		if (selectedOutlet == null) {
			return;
		}
		String selectedOutletId = selectedOutlet.getId();
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			OutletDAO.getInstance().saveOrUpdateOutlet(selectedOutlet, selectedOutletId, true, false, true, session);

			Outlet currentOutlet = OutletDAO.getInstance().get(DataProvider.get().getCurrentOutletId());
			currentOutlet.setDeleted(true);
			OutletDAO.getInstance().update(currentOutlet, session);

			String hqlStoreUpdate = "update " + Store.REF + " set %s=:outletId"; //$NON-NLS-1$ //$NON-NLS-2$
			hqlStoreUpdate = String.format(hqlStoreUpdate, Store.PROP_DEFAULT_OUTLET_ID);
			Query queryStoreUpdate = session.createQuery(hqlStoreUpdate);
			queryStoreUpdate.setParameter("outletId", selectedOutletId); //$NON-NLS-1$
			queryStoreUpdate.executeUpdate();

			for (String outletModel : outletRelatedModels) {
				String hql = "update " + outletModel + " set %s=:outletId"; //$NON-NLS-1$ //$NON-NLS-2$
				hql = String.format(hql, "outletId"); //$NON-NLS-1$
				Query query = session.createQuery(hql);
				query.setParameter("outletId", selectedOutletId); //$NON-NLS-1$
				query.executeUpdate();
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void updateOutletRelatedModelsIfNullExists(List<String> outletRelatedModels) throws Exception {
		String currentOutletId = DataProvider.get().getCurrentOutletId();
		if (StringUtils.isEmpty(currentOutletId)) {
			return;
		}
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (String outletModel : outletRelatedModels) {
				String hqlString = "update " + outletModel + " set %s=:outletId where %s is null"; //$NON-NLS-1$ //$NON-NLS-2$
				hqlString = String.format(hqlString, "outletId", "outletId"); //$NON-NLS-1$ //$NON-NLS-2$
				Query query = session.createQuery(hqlString);
				query.setParameter("outletId", currentOutletId); //$NON-NLS-1$
				query.executeUpdate();
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public void clearLastSyncTime() {
		List<String> tableNames = new ArrayList<String>();
		tableNames.add(SalesArea.REF);
		tableNames.add(ImageResource.REF);
		tableNames.add(Currency.REF);
		tableNames.add(Shift.REF);
		tableNames.add(StoreSession.REF);
		tableNames.add(StoreSessionControl.REF);
		tableNames.add(CashDrawer.REF);
		tableNames.add(TerminalType.REF);
		tableNames.add(Terminal.REF);
		tableNames.add(Tax.REF);
		tableNames.add(OrderType.REF);
		tableNames.add(Department.REF);
		tableNames.add(UserType.REF);
		tableNames.add(User.REF);
		tableNames.add(BalanceUpdateTransaction.REF);
		tableNames.add(Customer.REF);
		tableNames.add(Multiplier.REF);
		tableNames.add(MenuItemSize.REF);

		tableNames.add(MenuModifier.REF);
		tableNames.add(ModifierGroup.REF);
		tableNames.add(MenuCategory.REF);
		tableNames.add(MenuGroup.REF);
		tableNames.add(AttributeGroup.REF);
		tableNames.add(Attribute.REF);
		tableNames.add(InventoryUnitGroup.REF);
		tableNames.add(InventoryUnit.REF);
		tableNames.add(PackagingUnit.REF);
		tableNames.add(TestItemGroup.REF);
		tableNames.add(TestItem.REF);
		tableNames.add(MenuItem.REF);
		tableNames.add(InventoryVendor.REF);
		tableNames.add(InventoryVendorItems.REF);
		tableNames.add(InventoryLocation.REF);
		tableNames.add(InventoryTransaction.REF);
		tableNames.add(InventoryClosingBalance.REF);
		tableNames.add(ShopFloor.REF);
		tableNames.add(ShopTableStatus.REF);
		tableNames.add(ShopTable.REF);
		tableNames.add(BookingInfo.REF);
		tableNames.add(Ticket.REF);
		tableNames.add(PosTransaction.REF);
		tableNames.add(Address.REF);
		tableNames.add(CookingInstruction.REF);
		tableNames.add(Course.REF);
		tableNames.add(CustomerGroup.REF);
		tableNames.add(CustomPayment.REF);
		tableNames.add(DeclaredTips.REF);

		tableNames.add(Discount.REF);
		tableNames.add(GiftCard.REF);
		tableNames.add(PrinterGroup.REF);
		tableNames.add(ReportGroup.REF);
		tableNames.add(VoidReason.REF);
		tableNames.add(VirtualPrinter.REF);
		tableNames.add(ShopTableType.REF);
		tableNames.add(VoidItem.REF);
		tableNames.add(Recepie.REF);
		tableNames.add(PurchaseOrder.REF);
		tableNames.add(StockCount.REF);
		tableNames.add(ShopFloorTemplate.REF);
		tableNames.add(PriceTable.REF);
		tableNames.add(PriceRule.REF);
		tableNames.add(Specimen.REF);
		tableNames.add(PayoutReason.REF);
		tableNames.add(PayoutRecepient.REF);
		tableNames.add(COAAccountTypeGroup.REF);
		tableNames.add(COAAccountType.REF);
		tableNames.add(ChartOfAccounts.REF);
		tableNames.add(LedgerEntry.REF);

		clearLastSyncTime(tableNames);
	}

	public void clearLastSyncTime(List<String> tableNames) {
		if (tableNames.isEmpty()) {
			return;
		}

		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (String tableName : tableNames) {
				String hql = "update " + tableName + " set last_sync_time = null";
				Query query = session.createQuery(hql);
				int rows = query.executeUpdate();
				PosLog.debug(GenericDAO.class, "updated " + rows + " rows from " + tableName);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}
}
