package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.NullPrecedence;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.Appointment;
import com.floreantpos.model.AppointmentStatus;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Patient;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;

public class PatientDAO extends BasePatientDAO {

	public PatientDAO() {
	}

	@Override
	public List<Patient> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			return criteria.list();
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Patient patient = (Patient) obj;
		if (patient == null) {
			throw new PosException("Patient not found");
		}

		Criteria criteria = session.createCriteria(Appointment.class);
		addDeletedFilter(criteria);
		criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.CANCELLED.name()));
		criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp())));
		criteria.setMaxResults(1);
		Appointment appointment = (Appointment) criteria.uniqueResult();
		if (appointment != null) {
			throw new PosException("The patient cannot be deleted because an appointment is booked on " + DateUtil.formatDateAsString(appointment.getFromDate()));
		}

		patient.setDeleted(Boolean.TRUE);
		saveOrUpdate(patient, session);
		ActionHistoryDAO.saveHistory(ActionHistory.PATIENT_DELETED, "Agent name: " + patient.getFirstName() + ", id: " + patient.getId() + ", phone: "
				+ patient.getWorkPhoneNo() + ", email: " + patient.getEmail());
	}

	public void findBy(String patientNameOrPhoneOrId, Date patientDOB, PaginationSupport tableModel, String sortValue, String status) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			//			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotBlank(patientNameOrPhoneOrId)) {
				Disjunction or = Restrictions.disjunction();
				or.add(Restrictions.ilike(Customer.PROP_NAME, patientNameOrPhoneOrId, MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, patientNameOrPhoneOrId, MatchMode.END));
				or.add(Restrictions.ilike(Customer.PROP_ID, patientNameOrPhoneOrId, MatchMode.END));
				criteria.add(or);
			}
			if (patientDOB != null) {
				criteria.add(Restrictions.between(Customer.PROP_DATE_OF_BIRTH, DateUtil.startOfDay(patientDOB), DateUtil.endOfDay(patientDOB)));
			}

			patientStatus(status, criteria);

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			if (sortValue != null) {
				if (sortValue.equals("Name")) { //$NON-NLS-1$
					criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
				}
				if (sortValue.equals("Last update")) { //$NON-NLS-1$
					criteria.addOrder(Order.desc(Customer.PROP_LAST_UPDATE_TIME).nulls(NullPrecedence.LAST));
				}
			}
			tableModel.setRows(criteria.list());
		}
	}

	private void patientStatus(String status, Criteria criteria) {
		if (StringUtils.isNotEmpty(status)) {
			if (status.equals(POSConstants.ACTIVE)) {
				criteria.add(Restrictions.eq(Patient.PROP_ACTIVE, true));
			}
			else if (status.equals(POSConstants.DEACTIVE)) {
				criteria.add(Restrictions.eq(Patient.PROP_ACTIVE, false));
			}
		}
	}

	public void findByNameOrInvoiceOrPhoneOrEmail(String nameString, String phoneOrEmailString, String invoiceNo, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(nameString)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, nameString, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotEmpty(phoneOrEmailString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_MOBILE_NO, phoneOrEmailString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_EMAIL, phoneOrEmailString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			if (StringUtils.isNotEmpty(invoiceNo)) {
				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Ticket.class);
				detachedCriteria.add(Restrictions.ilike(Ticket.PROP_ID, invoiceNo, MatchMode.END));
				detachedCriteria.setProjection(Projections.property(Ticket.PROP_CUSTOMER_ID));
				criteria.add(Property.forName(Patient.PROP_ID).in(detachedCriteria));
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel) {
		findByPhoneOrName(searchString, null, tableModel);
	}

	public void findByPhoneOrName(String searchString, String status, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();

				Criterion dobCriteria = Restrictions.ilike(Customer.PROP_PROPERTIES, "\"paitent.dob.date\":\"%" + searchString + "%\"", MatchMode.ANYWHERE); //$NON-NLS-1$ //$NON-NLS-2$
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_LOYALTY_NO, searchString, MatchMode.ANYWHERE), dobCriteria));
				criteria.add(disjunction);
			}

			patientStatus(status, criteria);

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public List<Patient> findByPhoneOrName(String searchString) {
		try (Session session = createNewSession()) {
			return findByPhoneOrName(searchString, session);
		}
	}

	public List<Patient> findByPhoneOrName(String searchString, Session session) {
		Criteria criteria = session.createCriteria(Patient.class);

		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

		if (StringUtils.isNotEmpty(searchString)) {
			Disjunction disjunction = Restrictions.disjunction();
			Criterion dobCriteria = Restrictions.ilike(Customer.PROP_PROPERTIES, "\"paitent.dob.date\":\"%" + searchString + "%\"", MatchMode.ANYWHERE); //$NON-NLS-1$ //$NON-NLS-2$
			disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
					Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE),
					Restrictions.ilike(Customer.PROP_MEMBER_ID, searchString, MatchMode.ANYWHERE), dobCriteria));
			criteria.add(disjunction);
		}
		criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());

		List<Patient> result = criteria.list();
		if (result == null || result.isEmpty()) {
			return null;
		}
		return result;
	}

	public Patient findByEmailOrPhone(String searchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);
			addDeletedFilter(criteria);

			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString));
			disjunction.add(Restrictions.ilike(Customer.PROP_EMAIL, searchString));
			criteria.add(disjunction);

			criteria.setMaxResults(1);
			Patient patient = (Patient) criteria.uniqueResult();
			return patient;
		}
	}

	public boolean isMatchedMobileNoAndId(String phoneNumber, String patientId) {
		if (StringUtils.isBlank(phoneNumber) || StringUtils.isBlank(patientId)) {
			return false;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);
			criteria.add(Restrictions.eq(Patient.PROP_ID, patientId));
			criteria.add(Restrictions.eq(Patient.PROP_MOBILE_NO, phoneNumber));
			criteria.setReadOnly(true);
			criteria.setProjection(Projections.id());
			criteria.setMaxResults(1);
			return criteria.uniqueResult() != null;
		}
	}

	public void findByName(String searchString, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

}