package com.floreantpos.model.dao;

import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemGroup;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class TestItemGroupDAO extends BaseTestItemGroupDAO {

	@Override
	public List<TestItemGroup> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TestItemGroup.class);
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(TestItemGroup.PROP_NAME).ignoreCase());
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		TestItemGroup testItemGroup = (TestItemGroup) obj;
		if (testItemGroup == null) {
			throw new PosException("Test unit group not found!");
		}
		checkIfGroupCanbeDeleted(s, testItemGroup);
		testItemGroup.setDeleted(true);
		update(testItemGroup, s);
	}

	private void checkIfGroupCanbeDeleted(Session s, TestItemGroup testItemGroup) {
		String details = ""; //$NON-NLS-1$
		List<TestItem> list = TestItemDAO.getInstance().findByGroupId(testItemGroup.getId());
		if (list != null && !list.isEmpty()) {
			if (StringUtils.isNotBlank(details)) {
				details += "\n\n"; //$NON-NLS-1$
			}
			details += "Test unit group " + testItemGroup.getName() + " " + Messages.getString("MenuGroupDAO.4"); //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			int count = 1;
			for (TestItem testItem : list) {
				details += "\n" + count + ". " + testItem.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}
		if (StringUtils.isNotBlank(details)) {
			throw new PosException("Test unit group " + " <b>" + testItemGroup.getName() + "</b> " + Messages.getString("MenuItemDAO.15"), //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$
					details);
		}
	}

	public void findByName(PaginationSupport tableModel, String searchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TestItemGroup.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.ilike(TestItemGroup.PROP_NAME, searchString, MatchMode.ANYWHERE));

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(TestItemGroup.PROP_NAME).ignoreCase());
			criteria.addOrder(Order.asc(TestItemGroup.PROP_SORT_ORDER));
			tableModel.setRows(criteria.list());
		}
	}

	public TestItemGroup findByName(String groupName) {
		try (Session session = createNewSession()) {
			return findByName(groupName, session);
		}
	}

	public TestItemGroup findByName(String groupName, Session session) {
		Criteria criteria = session.createCriteria(TestItemGroup.class);

		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(TestItemGroup.PROP_NAME, groupName).ignoreCase());
		criteria.setMaxResults(1);
		return (TestItemGroup) criteria.uniqueResult();
	}

	public void saveOrUpdateTestItemGroupList(List<TestItemGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateTestItemGroupList(dataList, updateLastUpdateTime, updateSyncTime, false, false);
	}

	public void saveOrUpdateTestItemGroupList(List<TestItemGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean saveNewDataOnly,
			boolean forceUpdate) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<TestItemGroup> iterator = dataList.iterator(); iterator.hasNext();) {
				TestItemGroup item = (TestItemGroup) iterator.next();
				TestItemGroup existingItem = get(item.getId());
				if (existingItem != null) {
					if (!forceUpdate && (saveNewDataOnly || !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime()))) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

}