/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.Date;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseInventoryTransaction;
import com.floreantpos.model.dao.InventoryVendorDAO;
import com.floreantpos.model.dao.OutletDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.InventoryUnitConversionRule;
import com.floreantpos.model.util.InventoryUnitConvertionUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "unitCostDisplay", "vendor", "toInventoryLocation", "fromInventoryLocation", "transactionDateAsString" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class InventoryTransaction extends BaseInventoryTransaction implements TimedModel, PropertyContainer {
	private static final long serialVersionUID = 1L;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private transient com.google.gson.JsonObject propertiesContainer;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public InventoryTransaction() {
	}

	/**
	 * Constructor for primary key
	 */
	public InventoryTransaction(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public final static String PROPERTY_VENDOR = "vendor"; //$NON-NLS-1$
	//public final static String PROPERTY_MENU_ITEM = "menuItem";
	public final static String PROPERTY_FROM_LOC = "fromInventoryLocation"; //$NON-NLS-1$
	public final static String PROPERTY_TO_LOC = "toInventoryLocation"; //$NON-NLS-1$

	public static final String REASON_NEW_STOCK = "NEW STOCK"; //$NON-NLS-1$
	public static final String REASON_RETURN = "RETURN"; //$NON-NLS-1$
	public static final String REASON_PURCHASE = "PURCHASE"; //$NON-NLS-1$
	public static final String REASON_DAMAGED = "DAMAGED"; //$NON-NLS-1$
	public static final String REASON_TICKET_SALES = "SALES"; //$NON-NLS-1$
	public static final String REASON_TRANSFER = "TRANSFER"; //$NON-NLS-1$
	public static final String REASON_UNIT_CONVERSION = "UNIT. CONV."; //$NON-NLS-1$
	public static final String REASON_ADJUST = "ADJUST"; //$NON-NLS-1$
	public static final String REASON_ADJUST_IN = "ADJUST_IN"; //$NON-NLS-1$
	public static final String REASON_ADJUST_OUT = "ADJUST_OUT"; //$NON-NLS-1$
	public static final String REASON_PREPARE_IN = "PREPARE IN"; //$NON-NLS-1$
	public static final String REASON_PREPARE_OUT = "PREPARE OUT"; //$NON-NLS-1$
	public static final String REASON_VOID = "VOID"; //$NON-NLS-1$

	public static final String[] REASON_IN = { REASON_NEW_STOCK, REASON_RETURN, REASON_PURCHASE, REASON_ADJUST_IN, REASON_PREPARE_IN };
	public static final String[] REASON_OUT = { REASON_TICKET_SALES, REASON_DAMAGED, REASON_ADJUST_OUT, REASON_PREPARE_OUT };
	public static final String[] REASON_TRANS = { REASON_TRANSFER };

	public static InventoryTransactionType transactionType;

	private double openingQty;
	private double openingCost;
	private double openingTotalCost;
	private String unitCodeDisplay;

	private transient InventoryLocation inventoryToLocation;
	private transient InventoryLocation inventoryFromLocation;

	public InventoryTransactionType getTransactionType() {
		return transactionType = InventoryTransactionType.fromInt(super.getType());
	}

	public void setTransactionType(InventoryTransactionType type) {
		super.setType(type.getType());
	}

	public String getTransactionDateAsString() {
		String dateAsString = ""; //$NON-NLS-1$
		if (super.getTransactionDate() == null) {
			return dateAsString;
		}
		return DateUtil.formatReportDateWithBrowserTimeOffset(super.getTransactionDate());
	}

	public void setTransactionDateAsString(String transactionDateAsString) {
	}

	public double getOpeningQty() {
		return openingQty;
	}

	public void setOpeningQty(double openingQty) {
		this.openingQty = openingQty;
	}

	public double getOpeningCost() {
		return openingCost;
	}

	public void setOpeningCost(double openingCost) {
		this.openingCost = openingCost;
	}

	public double getOpeningTotalCost() {
		return openingTotalCost;
	}

	public void setOpeningTotalCost(double openingTotalCost) {
		this.openingTotalCost = openingTotalCost;
	}

	public void setUnitCostDisplay(String s) {
		//do nothing
		//unitCostDisplay
	}

	@XmlTransient
	public String getUnitCostDisplay() {
		return NumberUtil.format3DigitNumber(getUnitCost());
	}

	public void setUnitPriceDisplay(String s) {
		//do nothing
		//unitCostDisplay
	}

	public String getUnitPriceDisplay() {
		return NumberUtil.format3DigitNumber(getUnitPrice());
	}

	public String getSku() {
		return getMenuItem().getSku();
	}

	public void setSku(String sku) {
	}

	public String getItemName() {
		return getMenuItem().getName();
	}

	public void setItemName(String itemName) {
	}

	public void setToInventoryLocation(InventoryLocation location) {
		this.inventoryToLocation = location;
		setToLocationId(location == null ? null : location.getId());
		setToOultetId(location == null ? null : location.getOutletId());
	}

	@XmlTransient
	public InventoryLocation getToInventoryLocation() {
		if (this.inventoryToLocation != null) {
			return this.inventoryToLocation;
		}
		return DataProvider.get().getInventoryLocationById(getToLocationId(), getToOultetId());
	}

	public void setFromInventoryLocation(InventoryLocation location) {
		this.inventoryFromLocation = location;
		setFromLocationId(location == null ? null : location.getId());
		setOutletId(location == null ? null : location.getOutletId());
	}

	@XmlTransient
	public InventoryLocation getFromInventoryLocation() {
		if (this.inventoryFromLocation != null) {
			return this.inventoryFromLocation;
		}
		return DataProvider.get().getInventoryLocationById(getFromLocationId(), getOutletId());
	}

	public void setUser(User user) {
		setUserId(user == null ? null : user.getId());
	}

	public void setVendor(InventoryVendor vendor) {
		setVendorId(vendor == null ? null : vendor.getId());
	}

	@XmlTransient
	public InventoryVendor getVendor() {
		if (getVendorId() == null) {
			return null;
		}

		return InventoryVendorDAO.getInstance().get(getVendorId());
	}

	public void calculateTotal() {
		Double unitCost = super.getUnitCost();
		IUnit transactionUnit = null;
		if (StringUtils.isNotBlank(getUnitType()) && UnitType.match(getUnitType(), UnitType.PACKAGING_UNIT)) {
			transactionUnit = (IUnit) DataProvider.get().getObjectOf(InventoryStockUnit.class, getUnit());
		}
		else {
			transactionUnit = (IUnit) DataProvider.get().getInventoryUnitById(getUnit());
		}
		if (transactionUnit == null) {
			setTotal(getQuantity() * unitCost);
			return;
		}
		MenuItem menuItem = getMenuItem();
		if (StringUtils.isEmpty(menuItem.getUnitId())) {
			setTotal(getQuantity() * unitCost);
			return;
		}
		double outUnitCost = InventoryUnitConvertionUtil.calculateCost(unitCost, menuItem.getUnit(), transactionUnit, menuItem);
		setUnitCost(outUnitCost);
		setTotal(getQuantity() * outUnitCost);
	}

	@Override
	public void addProperty(String key, String value) {
		initPropertyContainer();
		propertiesContainer.addProperty(key, value);
		super.setProperties(propertiesContainer.toString());
	}

	@Override
	public String getProperty(String key) {
		initPropertyContainer();

		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	@Override
	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	@Override
	public void removeProperty(String propertyName) {
		initPropertyContainer();
		propertiesContainer.remove(propertyName);
		super.setProperties(propertiesContainer.toString());
	}

	@Override
	public com.google.gson.JsonObject getPropertyStore() {
		initPropertyContainer();
		return propertiesContainer;
	}

	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}

	public void setDisplayUnit(IUnit displayUnit, Double displayUnitQuantity, MenuItem menuItem) {
		try {
			if (displayUnit != null) {
				addProperty("display_unit_id", displayUnit.getId()); //$NON-NLS-1$
				addProperty("display_unit", displayUnit.getUniqueCode()); //$NON-NLS-1$
				addProperty("display_unit_quantity", String.valueOf(displayUnitQuantity)); //$NON-NLS-1$
				addProperty("display_unit_conversion_rate", String.valueOf(displayUnit.getConversionRate())); //$NON-NLS-1$
				addProperty("display_unit_stock_countable", String.valueOf(displayUnit.isStockCountable())); //$NON-NLS-1$
				if (displayUnit instanceof InventoryUnit) {
					InventoryUnit inventoryUnit = (InventoryUnit) displayUnit;
					InventoryUnitConversionRule conversionRule = inventoryUnit.getConversionRule();
					if (conversionRule != null) {
						addProperty("display_unit_conversion_rule", conversionRule.getName()); //$NON-NLS-1$
					}
				}
			}
			if (menuItem == null) {
				return;
			}
			InventoryUnit menuItemUnit = menuItem.getUnit();
			if (menuItemUnit != null) {
				String menuItemUnitId = menuItemUnit.getId();
				addProperty("menu_item_unit", menuItemUnitId); //$NON-NLS-1$
				addProperty("menu_item_unit_cost", String.valueOf(menuItem.getItemCost())); //$NON-NLS-1$

				if (menuItemUnit.isBaseUnit()) {
					addProperty("base_unit", menuItemUnit.getId()); //$NON-NLS-1$
					addProperty("base_unit_quantity", //$NON-NLS-1$
							String.valueOf(InventoryUnitConvertionUtil.getUnitQuantity(displayUnit, menuItemUnit, menuItem) * displayUnitQuantity));
				}
				else {
					InventoryUnitGroup unitGroup = menuItemUnit.getUnitGroup();
					if (unitGroup != null) {
						List<InventoryUnit> units = unitGroup.getUnits();
						if (units != null && units.size() > 0) {
							for (InventoryUnit inventoryUnit : units) {
								if (inventoryUnit.isBaseUnit()) {
									addProperty("base_unit", inventoryUnit.getId()); //$NON-NLS-1$
									addProperty("base_unit_quantity", String.valueOf( //$NON-NLS-1$
											InventoryUnitConvertionUtil.getUnitQuantity(displayUnit, inventoryUnit, menuItem) * displayUnitQuantity));
									break;
								}
							}
						}
					}
				}
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public String getUnitCodeDisplay() {
		return unitCodeDisplay;
	}

	public void setUnitCodeDisplay(String unitCodeDisplay) {
		this.unitCodeDisplay = unitCodeDisplay;
	}

	public void putExpiryDate(Date date) {
		if (date == null) {
			return;
		}
		addProperty("expiry.date", DateUtil.formatDateAsUTCString(date));
	}

	public Date getExpiryDate() {
		String dateString = getProperty("expiry.date");
		if (StringUtils.isNotBlank(dateString)) {
			try {
				return DateUtil.parseUTCString(dateString);
			} catch (Exception ignored) {
				return null;
			}
		}
		return null;
	}
	
	public void setFromOutletDisplay(String outletId) {
	}

	@JsonIgnore
	public String getFromInventoryLocationDisplayString() {
		//if (getTransactionType() == InventoryTransactionType.IN) {
			InventoryLocation fromInventoryLocation = getFromInventoryLocation();
			Outlet outlet = null;
			if (StringUtils.isNotBlank(getOutletId())) {
				outlet = OutletDAO.getInstance().get(getOutletId());
			}
			if (outlet != null && fromInventoryLocation != null) {
				return outlet.getName() + "-> " + fromInventoryLocation.getName();
			}
			else if (fromInventoryLocation != null) {
				return fromInventoryLocation.getName();
			}
			else if (outlet != null) {
				return outlet.getName();
			}
			return null;
//		}
//		else {
//			InventoryLocation fromInventoryLocation = getToInventoryLocation();
//			Outlet outlet = null;
//			if (StringUtils.isNotBlank(getToOultetId())) {
//				outlet = OutletDAO.getInstance().get(getToOultetId());
//			}
//			if (outlet != null && fromInventoryLocation != null) {
//				return outlet.getName() + "-> " + fromInventoryLocation.getName();
//			}
//			else if (fromInventoryLocation != null) {
//				return fromInventoryLocation.getName();
//			}
//			else if (outlet != null) {
//				return outlet.getName();
//			}
//			return null;
//		}
	}
	
	public void setFromInventoryLocationString(String location) {
	}
	
	@JsonIgnore
	public String getToInventoryLocationDisplayString() {
		//if (getTransactionType() == InventoryTransactionType.IN) {
			InventoryLocation toInventoryLocation = getToInventoryLocation();
			Outlet outlet = null;
			if (StringUtils.isNotBlank(getToOultetId())) {
				outlet = OutletDAO.getInstance().get(getToOultetId());
			}
			if (outlet != null && toInventoryLocation != null) {
				return outlet.getName() + "-> " + toInventoryLocation.getName();
			}
			else if (toInventoryLocation != null) {
				return toInventoryLocation.getName();
			}
			else if (outlet != null) {
				return outlet.getName();
			}
			return null;
//		}
//		else {
//			InventoryLocation toInventoryLocation = getFromInventoryLocation();
//			Outlet outlet = null;
//			if (StringUtils.isNotBlank(getOutletId())) {
//				outlet = OutletDAO.getInstance().get(getOutletId());
//			}
//			if (outlet != null && toInventoryLocation != null) {
//				return outlet.getName() + "-> " + toInventoryLocation.getName();
//			}
//			else if (toInventoryLocation != null) {
//				return toInventoryLocation.getName();
//			}
//			else if (outlet != null) {
//				return outlet.getName();
//			}
//			return null;
//		}
	}
	
	public void setToInventoryLocationString(String location) {
	}
}