/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseInventoryUnit;
import com.floreantpos.model.dao.InventoryUnitGroupDAO;
import com.floreantpos.model.util.InventoryUnitConversionRule;
import com.floreantpos.util.NumberUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "uniqueId", "unitGroup" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class InventoryUnit extends BaseInventoryUnit implements IUnit, TimedModel, PropertyContainer {
	private static final long serialVersionUID = 1L;

	public static final String PROP_CONVERSION_RULE = "conversion.rule"; //$NON-NLS-1$

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private transient JsonObject propertiesContainer;
	private InventoryUnitGroup unitGroup;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public InventoryUnit() {
	}

	/**
	 * Constructor for primary key
	 */
	public InventoryUnit(java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public InventoryUnit(java.lang.String id, java.lang.String code) {

		super(id, code);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@Override
	public String toString() {
		return getName();
	}

	@JsonIgnore
	@Override
	public String getUniqueCode() {
		return super.getCode();
	}

	@XmlTransient
	public InventoryUnitGroup getUnitGroup() {
		String unitGroupId = getUnitGroupId();
		if (StringUtils.isEmpty(unitGroupId))
			return null;
		if (unitGroup != null && unitGroupId.equals(unitGroup.getId()))
			return unitGroup;
		return unitGroup = InventoryUnitGroupDAO.getInstance().get(unitGroupId);
	}

	public void setUnitGroup(InventoryUnitGroup unitGroup) {
		this.unitGroup = unitGroup;
	}

	public String getUniqueId() {
		return ("unit_" + getName() + "_" + getId()).replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}

	public double getBaseUnitConversionValue() {
		double baseConversionValue = 0;
		InventoryUnitGroup group = getUnitGroup();
		if (group != null) {
			List<InventoryUnit> units = group.getUnits();
			if (units != null && units.size() > 0) {
				for (InventoryUnit groupUnit : units) {
					if (groupUnit.isBaseUnit()) {
						baseConversionValue = groupUnit.getConversionRate();
						break;
					}
				}
			}
		}
		return baseConversionValue * getConversionRate();
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		return propertiesContainer;
	}

	public InventoryUnitConversionRule getConversionRule() {
		String inventoryUnitConversionRuleName = getProperty(InventoryUnit.PROP_CONVERSION_RULE, InventoryUnitConversionRule.DIVISION.getName());
		return InventoryUnitConversionRule.fromName(inventoryUnitConversionRuleName);
	}

	@JsonIgnore
	@XmlTransient
	public String getDisplayName() {
		return getProperty("unit_display_name", getName()); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void setDisplayName(String displayName) {
		addProperty("unit_display_name", displayName); //$NON-NLS-1$
	}

	public boolean hideFromSalesScreen() {
		return getBooleanProperty("hideFromSalesScreen", Boolean.FALSE); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void setHideFromSalesScreen(boolean hide) {
		addProperty("hideFromSalesScreen", String.valueOf(hide)); //$NON-NLS-1$
	}

	public void setStockCountable(boolean stockCountable) {
		addProperty("stockCountable", String.valueOf(stockCountable)); //$NON-NLS-1$
	}

	public boolean isStockCountable() {
		return getBooleanProperty("stockCountable", Boolean.TRUE); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public String getConversionRateDisplay() {
		String conversionRule = getProperty(InventoryUnit.PROP_CONVERSION_RULE, InventoryUnitConversionRule.DIVISION.getName());
		InventoryUnitConversionRule selectedRule = InventoryUnitConversionRule.fromName(conversionRule);
		if (selectedRule == InventoryUnitConversionRule.MULTIPLICATION) {
			return NumberUtil.format6DigitNumber(1D / getConversionRate());
		}
		return NumberUtil.format6DigitNumber(getConversionRate());
	}

	@Override
	public String getUnitType() {
		return UnitType.UNIT.name();
	}

	public java.lang.Double getConversionRate() {
		return super.getConversionRate() == 0 ? 1 : super.getConversionRate();
	}

}