package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BalanceForwardTransaction;
import com.floreantpos.model.BalanceSubType;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.Customer;
import com.floreantpos.model.EndBalanceTransaction;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PayoutRecepient;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.DefaultDataInserter;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class BalanceUpdateTransactionDAO extends BaseBalanceUpdateTransactionDAO {

	public BalanceUpdateTransactionDAO() {
	}

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private void adjustBalanceUpdateTransaction(BalanceUpdateTransaction balanceTransaction, Session session) {
		save(balanceTransaction, session);

		BalanceType balanceType = balanceTransaction.getBalanceType();
		if (balanceType == BalanceType.GIFT_CARD) {
			GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, balanceTransaction.getAccountNumber());
			if (giftCard == null) {
				//throw new PosException("No gift card found with account number " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
				return;
			}
			giftCard.setBalance(giftCard.getBalance() + balanceTransaction.getAmount());
			GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
			return;
		}

		Customer customer = CustomerDAO.getInstance().get(balanceTransaction.getAccountNumber(), session);
		if (customer == null) {
			//throw new PosException("No customer found with id " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
			return;
		}

		if (balanceType == BalanceType.CUSTOMER) {
			customer.setBalance(customer.getBalance() + balanceTransaction.getAmount());
		}
		else if (balanceType == BalanceType.LOYALTY) {
			customer.setLoyaltyPoint((int) (customer.getLoyaltyPoint() + balanceTransaction.getAmount()));
		}
		customer.setUpdateLastUpdateTime(false);
		customer.setUpdateSyncTime(false);
		CustomerDAO.getInstance().saveOrUpdate(customer, session);
	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, Session session) {
		saveBalanceUpdateTrans(balanceType, ticket, transaction, transactionType, accountNumber, amount, balanceBefore, subType, null, null, session);

	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, CustomPayment customPayment, String customPaymnetRef,
			Session session) {

		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(transaction == null ? PaymentType.CASH : transaction.getPaymentType(),
				transactionType, amount);
		if (customPayment != null) {
			balanceUpdateTransaction.setPaymentType(PaymentType.CUSTOM_PAYMENT);
			balanceUpdateTransaction.setCustomPaymnetId(customPayment.getId());
			balanceUpdateTransaction.setCustomPaymnetRef(customPaymnetRef);
		}
		balanceUpdateTransaction.setTransactionId(transaction == null ? null : transaction.getId());
		balanceUpdateTransaction.setTicketId(ticket == null ? null : ticket.getId());
		balanceUpdateTransaction.setAccountNumber(accountNumber);

		if (subType != null) {
			balanceUpdateTransaction.setTransactionSubType(subType.name());
			balanceUpdateTransaction.setDescription(subType.name());
		}
		if (!NumberUtil.isZero(balanceBefore)) {
			balanceBefore = NumberUtil.round(balanceBefore);
			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		}

		BalanceUpdateTransactionDAO.getInstance().save(balanceUpdateTransaction, session);
	}

	public void saveOrUpdateBalanceUpdateTransactions(List<BalanceUpdateTransaction> customerBalanceTransactions, boolean updateLastUpdateTime,
			boolean updateSyncTime) throws Exception {
		if (customerBalanceTransactions == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (BalanceUpdateTransaction transaction : customerBalanceTransactions) {
				BalanceUpdateTransaction existingTransaction = get(transaction.getId());
				if (existingTransaction != null) {
					continue;
				}
				transaction.setUpdateLastUpdateTime(updateLastUpdateTime);
				transaction.setUpdateSyncTime(updateSyncTime);
				adjustBalanceUpdateTransaction(transaction, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public double getTicketCommissionAmount(Session session, String ticketId, BalanceType balanceType) {
		return getTicketCommissionAmount(session, ticketId, "", balanceType); //$NON-NLS-1$
	}

	public double getTicketCommissionAmount(Session session, String ticketId, String accountNo, BalanceType balanceType) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
		criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TICKET_ID, ticketId));
		if (StringUtils.isNotBlank(accountNo)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
		}
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, balanceType.name()));
		double scFromItems = POSUtil.getDoubleAmount(criteria.uniqueResult());

		return scFromItems;
	}

	//	public void findBy(String accountNo, Date beginDate, Date currentDate, PaginationSupport tableModel) {
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
	//			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));
	//			tableModel.setNumRows(rowCount(criteria));
	//
	//			criteria.setFirstResult(tableModel.getCurrentRowIndex());
	//			criteria.setMaxResults(tableModel.getPageSize());
	//			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
	//			tableModel.setRows(criteria.list());
	//		}
	//	}

	public Map<String, Double> findSummaryBy(String accountNo, Date beginDate, Date currentDate) {
		Map<String, Double> map = new HashMap<String, Double>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
			//criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER), "accountNo"); //$NON-NLS-1$
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING), "balanceType"); //$NON-NLS-1$
			projectionList.add(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT), "total"); //$NON-NLS-1$
			criteria.setProjection(projectionList);

			List<Object[]> list = criteria.list();
			for (Object[] obj : list) {
				String balanceTypeString = (String) obj[1];
				double amount = (double) obj[2];
				map.put(balanceTypeString, amount);
			}
		}
		return map;
	}

	public List<BalanceUpdateTransaction> loadAccountsManagerPayment(Date fromDate, Date toDate, User accountsManager, String recipientId, String reasonId) {
		Pagination pagination = new Pagination<>(0, -1);
		loadAccountsManagerPayment(pagination, fromDate, toDate, accountsManager, TransactionType.DEBIT, BalanceType.ACCOUNTS_MANAGER_BALANCE, true,
				recipientId, reasonId);
		return pagination.getDataList();
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, User accountsManager) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountsManager, null, BalanceType.ACCOUNTS_MANAGER_BALANCE);
	}

	public void loadStoreBalancePayment(PaginationSupport model, Date fromDate, Date toDate, User accountsManager) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountsManager, null, BalanceType.STORE_BALANCE);
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, User accountsManager, TransactionType transactionType,
			BalanceType balanceType) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountsManager, transactionType, balanceType, false, null, null);
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, User accountsManager, TransactionType transactionType,
			BalanceType balanceType, boolean isExpenseOnly, String recipientId, String reasonId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			initAccountsManagerPaymentCriteria(fromDate, toDate, accountsManager, transactionType, balanceType, isExpenseOnly, recipientId, reasonId, criteria);

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());

			if (!(balanceType == BalanceType.STORE_BALANCE)) {
				criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER));
			}

			if (isExpenseOnly) {
				criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
			}
			else {
				criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
			}

			criteria.setProjection(null);
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());

			List<BalanceUpdateTransaction> transactions = criteria.list();
			if (isExpenseOnly) {
				model.setRows(transactions);
				return;
			}

			List<BalanceUpdateTransaction> allTransactionList = new ArrayList<BalanceUpdateTransaction>();
			if (transactions != null) {
				String accountsManagerNumber = null;
				double totalAmount = 0;
				for (BalanceUpdateTransaction posTransaction : transactions) {
					if (balanceType == BalanceType.STORE_BALANCE) {
						if (StringUtils.isNotBlank(accountsManagerNumber)) {
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
						else {
							accountsManagerNumber = "STORE_PAYMENT";
							allTransactionList.add(createForwardBalanceByTransction(accountsManagerNumber, posTransaction));
							totalAmount += posTransaction.getBalanceBefore();
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
					}
					else {
						if (posTransaction.getAccountNumber().equals(accountsManagerNumber)) {
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
						else {
							if (StringUtils.isNotBlank(accountsManagerNumber)) {
								allTransactionList.add(createEndBalanceByTransction(accountsManagerNumber, totalAmount));
								totalAmount = 0;
							}
							accountsManagerNumber = posTransaction.getAccountNumber();
							allTransactionList.add(createForwardBalanceByTransction(accountsManagerNumber, posTransaction));
							totalAmount += posTransaction.getBalanceBefore();
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
					}
				}
				allTransactionList.add(createEndBalanceByTransction(accountsManagerNumber, totalAmount));
			}
			model.setRows(allTransactionList);
		}
	}

	private void initAccountsManagerPaymentCriteria(Date fromDate, Date toDate, User accountsManager, TransactionType transactionType, BalanceType balanceType,
			boolean isExpenseOnly, String recipientId, String reasonId, Criteria criteria) {
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, balanceType.name()));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));

		criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
		if (accountsManager != null) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountsManager.getId()));
		}

		if (transactionType != null) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_TYPE, transactionType.name()));
		}

		if (isExpenseOnly) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.EXPENSE.name()));
		}

		if (StringUtils.isNotBlank(recipientId)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_RECEPIENT_ID, recipientId));
		}

		if (StringUtils.isNotBlank(reasonId)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_REASON_ID, reasonId));
		}
	}

	public double findTotalExpanses(Date fromDate, Date toDate, User accountsManager) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			initAccountsManagerPaymentCriteria(fromDate, toDate, accountsManager, null, BalanceType.ACCOUNTS_MANAGER_BALANCE, true, null, null, criteria);

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				return ((Number) uniqueResult).doubleValue();
			}
		}
		return 0;
	}

	private BalanceUpdateTransaction createForwardBalanceByTransction(String accountsManagerNumber, BalanceUpdateTransaction balanceUpdateTransaction) {
		BalanceForwardTransaction forwardTransaction = new BalanceForwardTransaction();
		if (balanceUpdateTransaction != null) {
			forwardTransaction.setAmount(balanceUpdateTransaction.getBalanceBefore());
			forwardTransaction.setPaymentTypeString(balanceUpdateTransaction.getPaymentTypeString());
			forwardTransaction.setTransactionType(balanceUpdateTransaction.getTransactionType());
			forwardTransaction.setAccountNumber(accountsManagerNumber);
			forwardTransaction.setOutletId(balanceUpdateTransaction.getOutletId());
			//forwardTransaction.setFromCashdrawerId(balanceUpdateTransaction.getFromCashdrawerId());
		}
		forwardTransaction.setTicketId("Balance forward");
		return forwardTransaction;
	}

	private BalanceUpdateTransaction createEndBalanceByTransction(String accountsManagerNumber, double totalAmount) {
		EndBalanceTransaction endBalanceTransaction = new EndBalanceTransaction();
		endBalanceTransaction.setAccountNumber(accountsManagerNumber);
		endBalanceTransaction.setAmount(totalAmount);
		endBalanceTransaction.setTicketId("End Balance");
		return endBalanceTransaction;
	}

	//	public List<BalanceUpdateTransaction> findPaymentsBy(Date startTime, Date endTime, Customer customer, boolean showAgentPayment) {
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
	//
	//			if (customer != null) {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, customer.getId()));
	//			}
	//
	//			if (showAgentPayment) {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
	//			}
	//			else {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.LAB_DOCTOR_FEE_PAYMENT.name()));
	//			}
	//			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(startTime), DateUtil.endOfDay(endTime)));
	//			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
	//			return criteria.list();
	//		}
	//	}

	public BalanceUpdateTransaction saveAccountsManagerAddBalanceTransaction(User accountsManager, Double balanceToAdd, String note) {
		Transaction tx = null;
		BalanceUpdateTransaction transaction;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			transaction = populateAccountsManagerTransaction(null, accountsManager, balanceToAdd, note, session);
			tx.commit();

		}

		LedgerEntryDAO.getInstance().saveCashToAccountsManagerLedgerEntry(transaction, false);

		return transaction;
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Session session) {
		return populateAccountsManagerTransaction(cashDrawer, accountsManager, cashDrawer.getDrawerAccountable(), null, session);
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Double balanceToAdd, String note,
			Session session) {
		if (accountsManager == null) {
			return null;
		}
		UserDAO userDao = UserDAO.getInstance();
		userDao.refresh(accountsManager, session);

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT,
				balanceToAdd);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());

		balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
		if (cashDrawer != null) {
			balanceUpdateTransaction.setFromCashdrawerId(cashDrawer.getId());
		}
		if (StringUtils.isNotBlank(note)) {
			balanceUpdateTransaction.setDescription(note);
		}

		Double balanceBefore = accountsManager.getAccountsManagerAmount();
		accountsManager.putAccountsManagerAmount(NumberUtil.round(balanceBefore + balanceToAdd));

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);

		userDao.saveOrUpdate(accountsManager, session);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public void saveAccountsManagerExpensesTransaction(Date eventTime, User accountsManager, String batchNo, PayoutRecepient recepient, PayoutReason reason,
			double expensesAmount, String note) throws Exception {

		if (accountsManager == null) {
			throw new PosException("Please select a accounts manager");
		}

		BalanceUpdateTransaction balanceUpdateTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			UserDAO userInstance = UserDAO.getInstance();
			userInstance.refresh(accountsManager, session);

			BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, expensesAmount);
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.EXPENSE.name());
			if (eventTime != null) {
				balanceUpdateTransaction.setEventTime(eventTime);
			}

			balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
			balanceUpdateTransaction.setBatchNo(batchNo);
			balanceUpdateTransaction.setRecepientId(recepient == null ? "" : recepient.getId());
			balanceUpdateTransaction.setReasonId(reason == null ? "" : reason.getId());
			balanceUpdateTransaction.setDescription(note);

			Double balanceBefore = accountsManager.getAccountsManagerAmount();
			accountsManager.putAccountsManagerAmount(balanceBefore - expensesAmount);
			userInstance.saveOrUpdate(accountsManager, session);

			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
			save(balanceUpdateTransaction, session);

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction);
	}

	public void saveAccManagerToStoreTransaction(User accountsManager, Double balanceToTransfer, String note) throws Exception {

		if (accountsManager == null) {
			throw new PosException("Please select a accounts manager");
		}

		BalanceUpdateTransaction balanceUpdateTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			GenericDAO genericDAO = GenericDAO.getInstance();
			genericDAO.refresh(accountsManager, session);
			Double balanceBefore = accountsManager.getAccountsManagerAmount();

			Store store = DataProvider.get().getStore();
			Double storeCashBeforeBalance = store.getCashBalance();

			BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToTransfer);
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());

			balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
			balanceUpdateTransaction.setDescription(note);

			accountsManager.putAccountsManagerAmount(NumberUtil.round(balanceBefore - balanceToTransfer));
			genericDAO.saveOrUpdate(accountsManager, session);

			double storeCashBalance = NumberUtil.round(storeCashBeforeBalance + balanceToTransfer);
			Query query = session.createQuery("update " + Store.REF + " set " + Store.PROP_CASH_BALANCE + " = " + storeCashBalance); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			query.executeUpdate();

			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
			save(balanceUpdateTransaction, session);

			BalanceUpdateTransaction depositToStore = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToTransfer);
			depositToStore.setTransactionSubType(BalanceSubType.TRANSFER_IN.name());
			depositToStore.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
			depositToStore.setReasonId(DefaultDataInserter.addOrGetStoreReason(true).getId());
			depositToStore.setBalanceBefore(storeCashBeforeBalance);
			depositToStore.setAccountNumber(accountsManager.getId());
			depositToStore.setDescription("Add balance to store");
			depositToStore.setAccountProcessed(true);
			save(depositToStore, session);

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveAccManagerToStoreLedgerEntry(balanceUpdateTransaction);
	}

	public void saveWithdrawBalanceFromStoreTransaction(Double balanceToWithdraw, String note) throws Exception {

		BalanceUpdateTransaction balanceUpdateTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			Store store = DataProvider.get().getStore();
			Double storeCashBeforeBalance = store.getCashBalance();

			balanceUpdateTransaction = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToWithdraw);
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.WITHDRAW.name());
			balanceUpdateTransaction.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
			balanceUpdateTransaction.setReasonId(DefaultDataInserter.addOrGetStoreReason(false).getId());
			balanceUpdateTransaction.setBalanceBefore(storeCashBeforeBalance);
			balanceUpdateTransaction.setDescription("Withdraw balance from store");
			save(balanceUpdateTransaction, session);

			double storeCashBalance = NumberUtil.round(storeCashBeforeBalance - balanceToWithdraw);
			Query query = session.createQuery("update " + Store.REF + " set " + Store.PROP_CASH_BALANCE + " = " + storeCashBalance); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			query.executeUpdate();

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveWithdrawBalanceFromStoreLedgerEntry(balanceUpdateTransaction);
	}

	public void saveBeginCashTransaction(CashDrawer cashDrawer) throws Exception {
		Double beginCashAmount = cashDrawer.getBeginCash();
		if (beginCashAmount <= 0) {
			return;
		}

		BalanceUpdateTransaction balanceUpdateTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			balanceUpdateTransaction = BalanceType.CASH_DRAWER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, beginCashAmount);
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.BEGIN_CASH_DEPOSIT.name());
			balanceUpdateTransaction.setAccountNumber(cashDrawer.getId());
			balanceUpdateTransaction.setDescription("Cash drawer begin cash");
			save(balanceUpdateTransaction, session);

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveBeginCashLedgerEntry(balanceUpdateTransaction);
	}

	public boolean isPresent(String batchNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BATCH_NO, batchNo));
			criteria.setMaxResults(1);
			List list = criteria.list();
			return list != null && !list.isEmpty();
		}
	}

}