/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.text.DecimalFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.SerializationUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.EntityType;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.model.PurchaseOrderType;
import com.floreantpos.model.PurchaseTransaction;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.UnitType;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.services.InventoryService;
import com.floreantpos.swing.PaginatedListModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PurchaseOrderDAO extends BasePurchaseOrderDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PurchaseOrderDAO() {
	}

	@Override
	public Order getDefaultOrder() {
		return Order.desc(PurchaseOrder.PROP_CREATED_DATE);
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PurchaseOrder) {
			PurchaseOrder purchaseOrder = (PurchaseOrder) obj;
			purchaseOrder.setDeleted(Boolean.TRUE);
			super.update(purchaseOrder, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	public synchronized void saveOrUpdate(PurchaseOrder order, boolean updateStock) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(order, updateStock, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public synchronized void saveOrUpdateWithItemAvailBalance(PurchaseOrder order) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			updateAvailableUnit(order, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private void updateAvailableUnit(PurchaseOrder purchaseOrder, Session session) {
		List<PurchaseOrderItem> orderItems = purchaseOrder.getOrderItems();
		if (orderItems == null) {
			return;
		}
		for (PurchaseOrderItem purchaseOrderItem : orderItems) {

			MenuItem menuItem = purchaseOrderItem.getMenuItem();
			Double quantityToReceive = purchaseOrderItem.getItemQuantity();

			if (UnitType.match(purchaseOrderItem.getUnitType(), UnitType.PACKAGING_UNIT)) {
				quantityToReceive *= menuItem.getUnits().stream().filter(t -> t.getId().equals(purchaseOrderItem.getItemUnitName()))
						.map(t -> t.getConversionRate()).findFirst().orElse(1D);
			}

			MenuItemDAO.getInstance().updateStockQuantity(menuItem.getId(), purchaseOrder.getOutletId(), quantityToReceive, true, false, session);
		}
		session.saveOrUpdate(purchaseOrder);
	}

	public void saveOrUpdate(PurchaseOrder order, boolean updateStock, Session session) {
		if (order.getId() == null) {
			order.setCreatedDate(new Date());
			order.setStatus(PurchaseOrder.ORDER_PENDING);
		}
		if (updateStock) {
			adjustInventoryItems(session, order);
		}
		session.saveOrUpdate(order);
	}

	public void saveOrUpdate(PurchaseOrder order, boolean updateStock, boolean isAvailableUnitUpdate) {
		Transaction tx = null;
		try (final Session session = createNewSession()) {
			tx = session.beginTransaction();
			saveOrUpdate(order, updateStock, isAvailableUnitUpdate, session);
			tx.commit();
		}
	}

	public void saveOrUpdate(PurchaseOrder order, boolean updateStock, boolean isAvailableUnitUpdate, Session session) {

		if (order.getId() == null) {
			if (order.getCreatedDate() == null) {
				order.setCreatedDate(new Date());
			}
			order.setStatus(PurchaseOrder.ORDER_PENDING);
		}
		if (updateStock) {
			List<PurchaseOrderItem> orderItems = order.getOrderItems();
			if (orderItems != null) {
				Map<String, MenuItem> itemMap = new HashMap<String, MenuItem>();
				orderItems.forEach(orderItem -> {
					MenuItem menuItem = itemMap.get(orderItem.getMenuItemId());
					if (menuItem == null) {
						String menuItemId = orderItem.getMenuItemId();
						if (StringUtils.isNotBlank(menuItemId)) {
							menuItem = MenuItemDAO.getInstance().loadInitialized(menuItemId, session);
							if (menuItem != null) {
								itemMap.put(menuItem.getId(), menuItem);
							}
						}
					}
					String selectedUnitID = orderItem.getItemUnitName();
					IUnit iUnit = null;
					if (StringUtils.isNotBlank(orderItem.getUnitType()) && UnitType.match(orderItem.getUnitType(), UnitType.PACKAGING_UNIT)) {
						iUnit = (IUnit) InventoryStockUnitDAO.getInstance().get(selectedUnitID, session);
					}
					else {
						iUnit = (IUnit) DataProvider.get().getInventoryUnitById(selectedUnitID);
					}

					if (!PurchaseOrderType.match(order.getPurchaseOrderType(), PurchaseOrderType.RETURN)) {
						InventoryService.getInstance().adjustMenuItemCost(menuItem, orderItem.getUnitCost(), orderItem.getSalesPrice(), orderItem.getMrpPrice(),
								orderItem.getItemQuantity(), iUnit, session);
					}
				});
				if (itemMap.size() > 0) {
					for (MenuItem item : itemMap.values()) {
						MenuItemDAO.getInstance().saveOrUpdate(item, session);
					}
				}
			}

			adjustInventoryItems(session, order, isAvailableUnitUpdate);
		}
		session.saveOrUpdate(order);
	}

	private void adjustInventoryItems(Session session, PurchaseOrder order) {
		adjustInventoryItems(session, order, true);
	}

	private void adjustInventoryItems(Session session, PurchaseOrder order, boolean isAvailableUnitUpdate) {
		List<PurchaseOrderItem> orderItems = order.getOrderItems();
		if (orderItems == null) {
			return;
		}
		boolean fullyReceived = true;
		for (PurchaseOrderItem orderItem : orderItems) {
			orderItem.setQuantityReceived(orderItem.getQuantityReceived() + orderItem.getQuantityToReceive());
			InventoryTransaction stockInTrans = new InventoryTransaction();
			MenuItem menuItem = orderItem.getMenuItem();
			stockInTrans.setMenuItem(menuItem);
			stockInTrans.setQuantity(orderItem.getQuantityToReceive());
			Double unitCost = orderItem.getUnitCost();
			stockInTrans.setUnitCost(unitCost);
			//if (ProductType.match(order.getProductType(), ProductType.MEDICINE) || ProductType.match(order.getProductType(), ProductType.GOODS)) {
			stockInTrans.setBatchNumber(orderItem.getBatchNumber());
			stockInTrans.setMrpPrice(orderItem.getMrpPrice());
			stockInTrans.setUnitPrice(orderItem.getSalesPrice());
			stockInTrans.setUnitType(orderItem.getUnitType());
			//}
			//else {
			//	stockInTrans.setUnitType(orderItem.getUnitType());
			//}

			stockInTrans.setTransactionDate(StoreDAO.getServerTimestamp());
			stockInTrans.setUser(DataProvider.get().getCurrentUser());
			//stockInTrans.setOutletId(order.getOutletId());
			stockInTrans.setVendor(order.getVendor());
			stockInTrans.setUnit(orderItem.getItemUnitName());
			stockInTrans.setTotal(stockInTrans.getQuantity() * unitCost);
			if (order.getType().equals(PurchaseOrder.DEBIT)) {
				stockInTrans.setReason(InventoryTransaction.REASON_PURCHASE);
				stockInTrans.setTransactionType(InventoryTransactionType.IN);
				stockInTrans.setToInventoryLocation(order.getInventoryLocation());
				stockInTrans.setToOultetId(order.getOutletId());
			}
			else {
				stockInTrans.setReason(InventoryTransaction.REASON_RETURN);
				stockInTrans.setTransactionType(InventoryTransactionType.OUT);
				stockInTrans.setFromInventoryLocation(order.getInventoryLocation());
				stockInTrans.setOutletId(order.getOutletId());
			}

			InventoryTransactionDAO.getInstance().adjustInventoryStock(stockInTrans, session, isAvailableUnitUpdate, true, true, orderItem);
			if (orderItem.getQuantityReceived().doubleValue() < orderItem.getItemQuantity().doubleValue()) {
				fullyReceived = false;
			}
			orderItem.setQuantityToReceive(0.0);
			ActionHistoryDAO.logJournalForInventoryTrans(stockInTrans);
		}

		if (PurchaseOrderType.match(order.getPurchaseOrderType(), PurchaseOrderType.RETURN)) {
			order.setStatus(PurchaseOrder.ORDER_RETURNED);
		}
		else {
			if (fullyReceived) {
				order.setStatus(PurchaseOrder.ORDER_FULLY_RECEIVED);
			}
			else {
				order.setStatus(PurchaseOrder.ORDER_PARTIALLY_RECEIVED);
			}
		}
	}

	public String getNextOrderSequenceNumber() {
		DecimalFormat decimalFormat = new DecimalFormat("10000");
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.not(Restrictions.ilike(PurchaseOrder.PROP_ORDER_ID, "\\_", MatchMode.ANYWHERE))); //$NON-NLS-1$
			int rowCount = rowCount(criteria);
			String orderId = decimalFormat.format(rowCount + 1);
			while (exists(orderId, null)) {
				orderId = String.valueOf(Integer.valueOf(orderId) + 1);
			}
			return orderId;
		}
	}

	public boolean exists(String purchaseOrderNo, String id) {
		if (StringUtils.isBlank(purchaseOrderNo)) {
			return false;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(PurchaseOrder.PROP_ORDER_ID, purchaseOrderNo));
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(id)) {
				criteria.add(Restrictions.ne(PurchaseOrder.PROP_ID, id));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}

	public boolean isUniquePoNumber(String poNumber) {
		List<PurchaseOrder> list = findBy(poNumber, null, null, null);
		return !(list == null || list.isEmpty());
	}

	public List<PurchaseOrder> findBy(String poNumber, InventoryVendor vendor, Date from, Date to) {
		return findBy(poNumber, vendor, from, to, null);
	}

	public List<PurchaseOrder> findBy(String poNumber, InventoryVendor vendor, Date from, Date to, PurchaseOrderType poType) {
		return findBy(poNumber, vendor, from, to, null, DataProvider.get().getOutlet(), null);
	}

	public List<PurchaseOrder> findBy(String poNumber, InventoryVendor vendor, Date from, Date to, PurchaseOrderType poType, Outlet outlet,
			List<Integer> statusList) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			if (outlet != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_OUTLET_ID, outlet.getId()));
			}
			if (StringUtils.isNotEmpty(poNumber)) {
				criteria.add(Restrictions.ilike(PurchaseOrder.PROP_ORDER_ID, poNumber.trim(), MatchMode.START));
			}
			if (vendor != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_VENDOR, vendor));
			}
			//			if (productType != null) {
			//				criteria.add(Restrictions.eq(PurchaseOrder.PROP_PRODUCT_TYPE, productType.name()));
			//			}
			//			else {
			//				criteria.add(Restrictions.ne(PurchaseOrder.PROP_PRODUCT_TYPE, ProductType.MEDICINE.name()));
			//			}

			if (poType != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE, poType.name()));
			}
			else {
				criteria.add(Restrictions.or(Restrictions.ne(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE, PurchaseOrderType.RETURN.name()),
						Restrictions.isNull(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE)));
			}

			if (statusList != null && !statusList.isEmpty()) {
				Disjunction disjunction = Restrictions.disjunction();
				for (Integer status : statusList) {
					disjunction.add(Restrictions.eq(PurchaseOrder.PROP_STATUS, status));
				}
				criteria.add(disjunction);
			}

			if (from != null && to != null) {
				criteria.add(Restrictions.ge(PurchaseOrder.PROP_CREATED_DATE, from));
				criteria.add(Restrictions.le(PurchaseOrder.PROP_CREATED_DATE, to));
			}
			addDeletedFilter(criteria);
			criteria.addOrder(Order.desc(PurchaseOrder.PROP_CREATED_DATE));
			criteria.addOrder(Order.desc(PurchaseOrder.PROP_ORDER_ID));
			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void findPurchaseOrderBy(PaginationSupport model, String poNumber, InventoryVendor vendor, Date from, Date to, PurchaseOrderType poType) {
		findPurchaseOrderBy(model, poNumber, vendor, from, to, poType, null);
	}

	public void findPurchaseOrderBy(PaginationSupport model, String poNumber, InventoryVendor vendor, Date from, Date to, PurchaseOrderType poType,
			String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			if (StringUtils.isNotEmpty(poNumber)) {
				criteria.add(Restrictions.ilike(PurchaseOrder.PROP_ORDER_ID, poNumber.trim(), MatchMode.START));
			}
			if (vendor != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_VENDOR, vendor));
			}

			if (poType != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE, poType.name()));
			}
			else {
				criteria.add(Restrictions.or(Restrictions.ne(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE, PurchaseOrderType.RETURN.name()),
						Restrictions.isNull(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE)));
			}

			if (from != null && to != null) {
				criteria.add(Restrictions.ge(PurchaseOrder.PROP_CREATED_DATE, from));
				criteria.add(Restrictions.le(PurchaseOrder.PROP_CREATED_DATE, to));
			}
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_OUTLET_ID, outletId));
			}

			model.setNumRows(super.rowCount(criteria));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.addOrder(Order.desc(PurchaseOrder.PROP_CREATED_DATE));
			criteria.addOrder(Order.desc(PurchaseOrder.PROP_ORDER_ID));
			model.setRows(criteria.list());
		}
	}

	public List<PurchaseOrder> findPurchaseOrderBy(String poNumber, InventoryVendor vendor, Date from, Date to, ProductType productType,
			PurchaseOrderType poType) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			if (StringUtils.isNotEmpty(poNumber)) {
				criteria.add(Restrictions.ilike(PurchaseOrder.PROP_ORDER_ID, poNumber.trim(), MatchMode.START));
			}
			if (vendor != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_VENDOR, vendor));
			}
			if (productType != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_PRODUCT_TYPE, productType.name()));
			}
			else {
				criteria.add(Restrictions.ne(PurchaseOrder.PROP_PRODUCT_TYPE, ProductType.MEDICINE.name()));
			}

			if (poType != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE, poType.name()));
			}
			else {
				criteria.add(Restrictions.or(Restrictions.ne(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE, PurchaseOrderType.RETURN.name()),
						Restrictions.isNull(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE)));
			}

			if (from != null && to != null) {
				criteria.add(Restrictions.ge(PurchaseOrder.PROP_CREATED_DATE, from));
				criteria.add(Restrictions.le(PurchaseOrder.PROP_CREATED_DATE, to));
			}
			addDeletedFilter(criteria);

			criteria.addOrder(Order.desc(PurchaseOrder.PROP_CREATED_DATE));
			criteria.addOrder(Order.desc(PurchaseOrder.PROP_ORDER_ID));
			return criteria.list();
		}
	}

	//download and upload
	public void saveOrUpdatePurchaseOrders(List<PurchaseOrder> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Map<String, Object> existingItemMap = new HashMap<>();
		try (Session session = createNewSession()) {
			for (PurchaseOrder purchaseOrder : dataList) {
				PurchaseOrder existingOrder = PurchaseOrderDAO.getInstance().get(purchaseOrder.getId(), session);
				if (existingOrder != null) {
					existingItemMap.put(purchaseOrder.getId(), purchaseOrder);

					List<PurchaseOrderItem> orderItems = existingOrder.getOrderItems();
					for (PurchaseOrderItem purchaseOrderItem : orderItems) {
						existingItemMap.put(purchaseOrderItem.getId(), purchaseOrderItem);
					}
				}
			}
		}

		Transaction tx = null;
		try (Session session = createNewSession()) {
			for (Iterator<PurchaseOrder> iterator = dataList.iterator(); iterator.hasNext();) {
				PurchaseOrder purchaseOrder = (PurchaseOrder) iterator.next();

				try {
					tx = session.beginTransaction();

					List<PurchaseOrderItem> purchaseOrderItems = POSUtil.copyList(purchaseOrder.getOrderItems());
					POSUtil.clear(purchaseOrder.getOrderItems());
					purchaseOrder.setUpdateLastUpdateTime(updateLastUpdateTime);
					purchaseOrder.setUpdateSyncTime(updateSyncTime);

					PurchaseOrder existingPurchaseOrder = (PurchaseOrder) existingItemMap.get(purchaseOrder.getId());
					if (existingPurchaseOrder != null) {
						if (!BaseDataServiceDao.get().shouldSave(purchaseOrder.getLastUpdateTime(), existingPurchaseOrder.getLastUpdateTime())) {
							PosLog.debug(getClass(), purchaseOrder.getId() + " already updated"); //$NON-NLS-1$
							continue;
						}
						long version = existingPurchaseOrder.getVersion();
						purchaseOrder.setVersion(version);
					}
					else {
						save(purchaseOrder, session);
					}
					for (PurchaseOrderItem purchaseOrderItem : purchaseOrderItems) {
						purchaseOrderItem.setPurchaseOrder(purchaseOrder);

						PurchaseOrderItem existingPurchaseOrderItem = (PurchaseOrderItem) existingItemMap.get(purchaseOrderItem.getId());
						if (existingPurchaseOrderItem == null) {
							purchaseOrderItem.setVersion(0);
							PurchaseOrderItemDAO.getInstance().save(purchaseOrderItem, session);
						}
						else {
							purchaseOrderItem.setVersion(existingPurchaseOrderItem.getVersion());
							PurchaseOrderItemDAO.getInstance().update(purchaseOrderItem, session);
						}
					}
					if (purchaseOrder.getOrderItems() != null) {
						purchaseOrder.getOrderItems().addAll(purchaseOrderItems);
					}
					else {
						purchaseOrder.setOrderItems(purchaseOrderItems);
					}

					update(purchaseOrder, session);

					tx.commit();
				} catch (Exception e) {
					PosLog.error(getReferenceClass(), "Error saving purchase order " + purchaseOrder.getId());
					if (tx != null) {
						tx.rollback();
					}
					throw e;
				}
			}
		}
	}

	@SuppressWarnings("unchecked")
	public List<PurchaseOrder> getPurchaseOrderByInvLocation(InventoryLocation inventoryLocation, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(PurchaseOrder.PROP_INVENTORY_LOCATION, inventoryLocation));
		criteria.add(Restrictions.ne(PurchaseOrder.PROP_STATUS, PurchaseOrder.ORDER_FULLY_RECEIVED));
		criteria.setProjection(Projections.alias(Projections.property(PurchaseOrder.PROP_ID), PurchaseOrder.PROP_ID));
		addDeletedFilter(criteria);
		return criteria.setResultTransformer(Transformers.aliasToBean(getReferenceClass())).list();
	}

	public void loadPurchaseOrder(PaginatedListModel listModel) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			addDeletedFilter(criteria);

			listModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());

			criteria.addOrder(Order.desc(PurchaseOrder.PROP_CREATED_DATE));
			listModel.setData(criteria.list());
		}
	}

	public int countPurchaseOrderByCreatedDate(Date createdDate) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			addDeletedFilter(criteria);

			if (createdDate != null) {
				criteria.add(Restrictions.between(PurchaseOrder.PROP_CREATED_DATE, DateUtil.startOfDay(createdDate), DateUtil.endOfDay(createdDate)));
			}
			return rowCount(criteria);
		}
	}

	public PurchaseOrder saveAndGetBackOrder(boolean isConfirmedPatiallyReceived, PurchaseOrder purchaseOrder) {
		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();
			List<PurchaseOrderItem> orderItems = purchaseOrder.getOrderItems();
			PurchaseOrder clonePO = null;
			if (isConfirmedPatiallyReceived) {
				clonePO = (PurchaseOrder) SerializationUtils.clone(purchaseOrder);
				clonePO.setId(null);
				clonePO.setVersion(0);
				clonePO.setOrderItems(null);
				clonePO.setReceivingDate(null);
				clonePO.setProperties(null);

				for (PurchaseOrderItem purchaseOrderItem : orderItems) {
					PurchaseOrderItem clonePOItem = (PurchaseOrderItem) SerializationUtils.clone(purchaseOrderItem);
					if (clonePOItem.getItemQuantity() == 0) {
						continue;
					}

					double quantityReceived = clonePOItem.getQuantityReceived() + clonePOItem.getQuantityToReceive();
					double remainItemQty = Math.abs(clonePOItem.getItemQuantity() - quantityReceived);
					if (remainItemQty == 0) {
						continue;
					}
					clonePOItem.setItemQuantity(remainItemQty);
					clonePOItem.setId(null);
					clonePOItem.setVersion(0);
					clonePOItem.setPurchaseOrder(clonePO);
					clonePO.addToorderItems(clonePOItem);
				}
				clonePO.calculatePrice();

				String orderId = purchaseOrder.getOrderId();
				if (StringUtils.isNotBlank(orderId)) {
					clonePO.setOrderId(generateSubOrderNo(orderId));
				}

				clonePO.putBackOrderShortId(purchaseOrder.getOrderId());
				clonePO.putBackOrderLongId(purchaseOrder.getId());

				save(clonePO, session);
			}

			for (PurchaseOrderItem orderItem : orderItems) {
				double quantityReceived = orderItem.getQuantityReceived() + orderItem.getQuantityToReceive();
				orderItem.setItemQuantity(quantityReceived);
			}
			purchaseOrder.calculatePrice();

			Double totalAmount = purchaseOrder.getTotalAmount();
			Double paidAmount = purchaseOrder.getPaidAmount();
			if (paidAmount > totalAmount) {
				purchaseOrder.setDueValue(0);
				purchaseOrder.putFullPaid(true);
			}

			receivePurchaseOrders(session, purchaseOrder);

			if (clonePO != null) {
				createPaymentTransaction(session, purchaseOrder, clonePO);
			}

			transaction.commit();

			return clonePO;
		}
	}

	private String generateSubOrderNo(String orderId) {
		String[] split = orderId.split("_"); //$NON-NLS-1$
		if (split.length > 1) {
			try {
				String lastNumber = split[split.length - 1];
				int i = Double.valueOf(lastNumber).intValue() + 1;
				orderId = orderId.substring(0, orderId.lastIndexOf(lastNumber));
				orderId += i;
			} catch (Exception e) {
				orderId += "_1"; //$NON-NLS-1$
			}
		}
		else {
			orderId += "_1"; //$NON-NLS-1$
		}
		return PurchaseOrderDAO.getInstance().exists(orderId, null) ? generateSubOrderNo(orderId) : orderId;
	}

	public void receivePurchaseOrders(Session session, PurchaseOrder purchaseOrder) {
		InventoryVendorItemsDAO.getInstance().saveItems(purchaseOrder, session);
		PurchaseOrderDAO.getInstance().saveOrUpdate(purchaseOrder, true, StoreDAO.getRestaurant().isUpdateOnHandBlncForPORec(), session);

		LedgerEntryDAO.getInstance().savePurchaseOrderLedgerEntry(purchaseOrder, session);
		ActionHistoryDAO.createLogJournal(purchaseOrder, "Receive purchase order"); //$NON-NLS-1$
	}

	public void createPaymentTransaction(Session session, PurchaseOrder srcOrder, PurchaseOrder backOrder) {
		double totalPaidAmount = srcOrder.getPaidAmount();
		if (srcOrder.getTotalAmount() > totalPaidAmount) {
			return;
		}

		PurchaseTransaction carryOutTrans = new PurchaseTransaction();
		carryOutTrans.setEntityId(srcOrder.getId());
		carryOutTrans.setEntityType(EntityType.PURCHASE.name());
		carryOutTrans.setEntityNo(srcOrder.getOrderId());
		carryOutTrans.setOutletId(DataProvider.get().getCurrentOutletId());
		carryOutTrans.setTerminal(DataProvider.get().getCurrentTerminal());
		carryOutTrans.setTransactionTime(StoreDAO.getServerTimestamp());
		carryOutTrans.setTransactionType(TransactionType.CARRY_OUT.name());
		carryOutTrans.setPaymentType(PaymentType.CARRY_OUT);
		double carryOutAmount = totalPaidAmount - srcOrder.getTotalAmount();
		carryOutTrans.setAmount(NumberUtil.isZero(carryOutAmount) ? carryOutAmount : (-1) * carryOutAmount);

		PurchaseTransaction carryOverTrans = new PurchaseTransaction();
		carryOverTrans.setEntityId(backOrder.getId());
		carryOverTrans.setEntityType(EntityType.PURCHASE.name());
		carryOverTrans.setEntityNo(backOrder.getOrderId());
		carryOverTrans.putOriginPurchaseOrderId(srcOrder.getId());
		carryOverTrans.setOutletId(DataProvider.get().getCurrentOutletId());
		carryOverTrans.setTerminal(DataProvider.get().getCurrentTerminal());
		carryOverTrans.setTransactionTime(StoreDAO.getServerTimestamp());
		carryOverTrans.setTransactionType(TransactionType.CARRY_OVER.name());
		carryOverTrans.setPaymentType(PaymentType.CARRY_OVER);
		carryOverTrans.setAmount(carryOutAmount);

		double backorderDueAmount = backOrder.getDueValue() - carryOverTrans.getAmount();
		backOrder.setDueValue(backorderDueAmount < 0 ? 0 : backorderDueAmount);
		backOrder.setPaidAmount(carryOverTrans.getAmount());
		backOrder.putFullPaid(NumberUtil.isZero(backOrder.getDueAmount()));
		update(backOrder, session);

		PosTransactionDAO.getInstance().save(carryOutTrans, session);
		PosTransactionDAO.getInstance().save(carryOverTrans, session);

	}

	public List getPayablePurchaseOrder(InventoryVendor vendor) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			criteria.add(Restrictions.eq(PurchaseOrder.PROP_STATUS, PurchaseOrder.ORDER_FULLY_RECEIVED));
			criteria.add(Restrictions.eq(PurchaseOrder.PROP_VENDOR, vendor));
			criteria.addOrder(Order.asc(PurchaseOrder.PROP_CREATED_DATE));
			return criteria.list();
		}
	}
}