package com.floreantpos.report;

import java.math.BigDecimal;
import java.net.URL;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.swing.ImageIcon;

import org.apache.commons.lang.WordUtils;
import org.apache.commons.lang3.StringUtils;

import com.floreantpos.POSConstants;
import com.floreantpos.model.Address;
import com.floreantpos.model.BloodGroupType;
import com.floreantpos.model.Currency;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Department;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketDiscount;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.CurrencyDAO;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.ImageUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;
import pl.allegro.finance.tradukisto.MoneyConverters;

public class LabTestInvoiceUtil {

	public static JasperPrint createJasperPrint(Ticket ticket) throws JRException {
		return createJasperPrint(ticket, null, null);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, String copyFor) throws JRException {
		return createJasperPrint(ticket, null, copyFor);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, Map<String, Object> mapForPreview) throws JRException {
		return createJasperPrint(ticket, mapForPreview, null);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, Map<String, Object> mapForPreview, String copyFor) throws JRException {
		boolean isPharma = ticket.getTicketType() == TicketType.PHARMA;
		LabTestReportModel reportModel = createModels(ticket);

		HashMap<String, Object> map = new HashMap<String, Object>();
		Outlet outlet = DataProvider.get().getOutlet();

		PaperSize paperSize;
		if (mapForPreview == null) {
			PaperSize receiptPaperSize;
			if (ticket.getTicketType() == TicketType.PHARMA) {
				receiptPaperSize = outlet.getPharmacyReceiptPaperSize();
			}
			else {
				receiptPaperSize = outlet.getReceiptPaperSize();
			}
			paperSize = receiptPaperSize;
		}
		else {
			paperSize = (PaperSize) mapForPreview.get("paperSize"); //$NON-NLS-1$
		}

		ReportUtil.populateMedlogicsProperties(map, outlet, paperSize);

		JasperReport header;
		if (paperSize == PaperSize.A5) {
			header = ReportUtil.getReport("medlogics_report_customer_header_order_info_A5"); //$NON-NLS-1$	
		}
		else {
			header = ReportUtil.getReport("medlogics_report_customer_header_order_info"); //$NON-NLS-1$	
		}

		map.put("head", header); //$NON-NLS-1$
		map.put("reportTitle", "Lab test report"); //$NON-NLS-1$
		map.put("isVisibleLabID", !isPharma); //$NON-NLS-1$
		populateReportParameter(map, ticket, copyFor);

		int receiptReportHeaderMargin = outlet.getReceiptReportHeaderMargin();
		int receiptReportFooterMargin = outlet.getReceiptReportFooterMargin();
		JasperReport masterReport;
		if (mapForPreview != null) {
			map.putAll(mapForPreview);
			receiptReportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptHeaderMarginValue")); //$NON-NLS-1$
			receiptReportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptFooterMarginValue")); //$NON-NLS-1$
		}

		if (isPharma) {
			if (paperSize == PaperSize.A4) {
				masterReport = ReportUtil.adjustReportHeaderAndFooter("pharmacy_receipt", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
			}
			else if (paperSize == PaperSize.A5) {
				masterReport = ReportUtil.adjustReportHeaderAndFooter("pharmacy_receipt_A5", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
			}
			else {
				masterReport = ReportUtil.getReport("pharmacy-recipt-80mm"); //$NON-NLS-1$
			}
		}

		else {
			if (paperSize == PaperSize.A5) {
				if (ticket.getTicketType() == TicketType.IPD) {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("ipd_invoice_receipt_A5", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
				else {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-report_A5", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
			}
			else {
				if (ticket.getTicketType() == TicketType.IPD) {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("ipd_invoice_receipt", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
				else {
					masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-report", receiptReportFooterMargin, receiptReportHeaderMargin); //$NON-NLS-1$
				}
			}
		}

		JasperPrint jasperPrint = JasperFillManager.fillReport(masterReport, map, new JRTableModelDataSource(reportModel));
		return jasperPrint;
	}

	public static String getMoneyIntoWords(String input) {
		MoneyConverters converter = MoneyConverters.ENGLISH_BANKING_MONEY_VALUE;
		return WordUtils.capitalize(converter.asWords(new BigDecimal(input)));
	}

	private static void populateReportParameter(HashMap<String, Object> map, Ticket ticket, String copyFor) {
		Customer patient = ticket.getCustomer();
		Store store = DataProvider.get().getStore();
		String departmentName = ""; //$NON-NLS-1$
		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			String departmentId = DoctorDAO.getInstance().get(doctorId).getDoctorDepartmentId();
			Department department = DataProvider.get().getDepartmentById(departmentId);
			if (department != null) {
				departmentName = department.getName();
			}

		}

		String refName = "";
		String referrerId = ticket.getReferrerId();
		if (StringUtils.isNotBlank(referrerId)) {
			Customer referrer = CustomerDAO.getInstance().get(referrerId);
			if (referrer != null) {
				refName = referrer.getName();
			}
		}
		Outlet outlet = DataProvider.get().getOutlet();
		map.put("visibleReportHeader", outlet.isShowReceiptPrintHeader()); //$NON-NLS-1$
		map.put("visibleLabTestByCategory", outlet.isShowGroupLabTestByCategory()); //$NON-NLS-1$
		map.put("visibleLabTestPathologyRoomNo", outlet.isShowPathologyTestRoomNo()); //$NON-NLS-1$
		if (StringUtils.isNoneBlank(copyFor)) {
			map.put("bill", ticket.isVoided() ? "Voided " + "Bill (" + copyFor.trim() + ")" : "Bill (" + copyFor.trim() + ")"); //$NON-NLS-1$
		}
		else {
			map.put("bill", ticket.isVoided() ? "Voided " + "Bill" : "Bill"); //$NON-NLS-1$
		}
		map.put("barcode", ticket.getId()); //$NON-NLS-1$
		map.put("billNo", ticket.getId()); //$NON-NLS-1$
		map.put("date", DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(ticket.getCreateDate()), "dd MMM yy hh:mm a")); //$NON-NLS-1$
		//		map.put("dateReceived", "Delivery date: " + DateUtil.simplifyDateAndTime(ticket.getDeliveryDate(), "dd MMM yy hh:mm a")); //$NON-NLS-1$
		map.put("labId", store.getUuid()); //$NON-NLS-1$

		boolean isQrCodeVisible = false;
		if (patient != null) {
			map.put("patientId", patient.getId()); //$NON-NLS-1$
			map.put("ptName", patient.getName()); //$NON-NLS-1$
			map.put("phone", patient.getMobileNo()); //$NON-NLS-1$
			map.put("sex", patient.getPatientGender()); //$NON-NLS-1$
			Date dateOfBirth = patient.getDateOfBirth();
			if (dateOfBirth != null) {
				map.put("age", DateUtil.calculateAge(dateOfBirth)); //$NON-NLS-1$
			}

			String bloodGroup = patient.getBloodGroup();
			if (StringUtils.isNotBlank(bloodGroup)) {
				BloodGroupType bloodGroupType = BloodGroupType.fromNameString(bloodGroup);
				if (bloodGroupType != null) {
					bloodGroup = bloodGroupType.getDisplayString();
				}
				else {
					bloodGroup = "";
				}
			}
			map.put("bloodGroup", bloodGroup); //$NON-NLS-1$

			isQrCodeVisible = true;
		}

		String customerLoginURL = store.getCustomerLoginURL();
		if (StringUtils.isNotEmpty(customerLoginURL)) {
			if (StringUtils.isNotBlank(store.getStoreBaseLink()) && !customerLoginURL.startsWith(store.getStoreBaseLink())) {
				try {
					URL url = new URL(customerLoginURL);
					URL baseUrl = new URL(store.getStoreBaseLink());
					customerLoginURL = new URL(baseUrl.getProtocol(), baseUrl.getHost(), baseUrl.getPort(), url.getPath()).toExternalForm();
				} catch (Exception ignore) {
				}
			}
			map.put("qrCode", customerLoginURL + "/" + ticket.getCustomerId()); //$NON-NLS-1$
			map.put("qrCodeBottomMsg", "Scan the QR code to access the report");
		}
		else {
			isQrCodeVisible = false;
		}

		map.put("isQrCodeVisible", isQrCodeVisible);

		if (StringUtils.isNotBlank(doctorId)) {
			Doctor doctor = DoctorDAO.getInstance().get(doctorId);
			String doctorName = doctor.getName();
			String designation = doctor.getDoctorDesignation();
			if (StringUtils.isNotBlank(designation)) {
				doctorName = doctorName + " " + designation; //$NON-NLS-1$
			}
			map.put("consultant", doctorName); //$NON-NLS-1$
		}

		map.put("department", departmentName); //$NON-NLS-1$
		if (StringUtils.isNotBlank(refName)) {
			map.put("refBy", "Ref by: " + refName); //$NON-NLS-1$ //$NON-NLS-2$
		}

		Currency mainCurrency = CurrencyDAO.getInstance().getMainCurrencyByOutletId((ticket.getOutletId()));
		map.put("colNameOfTest", "<b>Item</b>"); //$NON-NLS-1$
		map.put("colQty", "<b>Quantity</b>"); //$NON-NLS-1$
		String currencySymbol = "<span style='font-family: Nirmala UI;'>" + mainCurrency.getSymbol() + "</span>";

		map.put("colAmount", "Amount" + " (" + currencySymbol + ")"); //$NON-NLS-1$

		map.put("grandTotal", "Grand total"); //$NON-NLS-1$
		map.put("colBillAmount", "Bill amount:"); //$NON-NLS-1$

		map.put("orderDiscount", "Discount on order:"); //$NON-NLS-1$
		double ticketDiscountAmount = 0d;
		List<TicketDiscount> discounts = ticket.getDiscounts();
		if (discounts != null && !discounts.isEmpty()) {
			ticketDiscountAmount = discounts.stream().mapToDouble(value -> value.getTotalDiscountAmount()).sum();
			map.put("orderDiscountAmount", //$NON-NLS-1$
					"-" + currencySymbol + "" + NumberUtil.formatAmount(ticketDiscountAmount < 0 ? ticketDiscountAmount * -1 : ticketDiscountAmount));
		}
		else {
			map.put("orderDiscountAmount", currencySymbol + "" + NumberUtil.formatAmount(ticketDiscountAmount)); //$NON-NLS-1$
		}
		map.put("billAmount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getSubtotalAmount() - ticket.getDiscountAmount() + ticketDiscountAmount)); //$NON-NLS-1$

		map.put("colTax", "Tax:"); //$NON-NLS-1$
		map.put("tax", currencySymbol + "" + NumberUtil.formatAmount(ticket.getTaxAmount())); //$NON-NLS-1$

		map.put("colSpecialDiscount", "Discount:"); //$NON-NLS-1$
		map.put("specialDiscount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getDiscountAmount())); //$NON-NLS-1$

		map.put("colTotalAmount", "Total Amount:"); //$NON-NLS-1$
		map.put("totalAmount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getTotalAmount())); //$NON-NLS-1$

		//		map.put("colReceviedAmount", "Received Amount  :");
		//		map.put("receviedAmount", NumberUtil.formatAmount(ticket.getPaidAmount()));

		map.put("colDueAmount", "Due Amount:");
		map.put("dueAmount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getDueAmount()));

		//map.put("colCancelAmount", "Cancel Amount  :");
		//map.put("cancelAmount", NumberUtil.formatAmount(ticket.getc())); 

		String serverName = "";
		User owner = ticket.getOwner();
		if (owner != null) {
			serverName = owner.getFullName();
		}
		else {
			serverName += ticket.getProperty("owner.firstname", "") + " " + ticket.getProperty("owner.lastname", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}
		map.put("colBillBy",
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd MMM yy hh:mm a") + " "
						+ "by" + " " + serverName);

		String payModeDtl = null;
		Set<PosTransaction> transactionList = ticket.getTransactions();
		if (transactionList != null) {
			List<PosTransaction> transactions = new ArrayList<>(transactionList.stream().filter(t -> !t.isVoided()).collect(Collectors.toList()));

			Comparator<PosTransaction> comparator = Comparator.comparing(PosTransaction::getTransactionTime, Comparator.nullsLast(Comparator.reverseOrder()));
			transactions.sort(comparator);

			if (transactions != null && !transactions.isEmpty()) {
				payModeDtl = "<div style='text-align: right;'>"; //$NON-NLS-1$
				for (PosTransaction posTransaction : transactions) {
					Double amount = posTransaction.getAmount();
					String nameDisplay = posTransaction.buildPaymentTypeDisplayName();
					String simplifyDateAndTime = DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(posTransaction.getTransactionTime()), "dd MMM yy");
					if (posTransaction instanceof RefundTransaction) {
						amount = (-1) * amount;
						nameDisplay += " " + POSConstants.REFUND_BUTTON_TEXT; //$NON-NLS-1$
					}
					User user = DataProvider.get().getUserById(posTransaction.getUserId(), posTransaction.getOutletId());
					if (ticket.getTicketType() == TicketType.PHARMA) {
						payModeDtl += simplifyDateAndTime + " " + nameDisplay + ": " + (amount < 0 ? "-" : "") + currencySymbol
								+ NumberUtil.getCurrencyFormatWithoutCurrencySymbol(amount < 0 ? amount * -1 : amount) + "<br>";
					}
					else {
						payModeDtl += simplifyDateAndTime + " " + nameDisplay + " (" + user.getFirstName().trim() + "): " + (amount < 0 ? "-" : "")
								+ currencySymbol + NumberUtil.getCurrencyFormatWithoutCurrencySymbol(amount < 0 ? amount * -1 : amount) + "<br>";
					}
				}
				payModeDtl += "</div>";
			}
		}

		map.put("colPayModeDtl", payModeDtl); //$NON-NLS-1$

		StringBuilder paymentInfo = new StringBuilder();
		if (StringUtils.isNotBlank(serverName)) {
			paymentInfo.append("Payments ----------------- "); //$NON-NLS-1$
		}
		paymentInfo.append(payModeDtl);
		if (StringUtils.isNotBlank(payModeDtl)) {
			map.put("additionalPaymentProperties", paymentInfo.toString()); //$NON-NLS-1$
		}

		//		map.put("colRemarks", "Remarks:");
		//		map.put("remarks", ticket.getNote());
		map.put("colPrint", "Print: " + DateUtil.formatDateWithTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()))); //$NON-NLS-1$

		Double totalAmount = ticket.getTotalAmount();
		String pAmount = totalAmount.toString();
		if (!NumberUtil.isFractional(totalAmount)) {
			pAmount = String.valueOf(totalAmount.intValue());
		}

		String words = getMoneyIntoWords(pAmount);
		map.put("colInWord", "In word: " + words); //$NON-NLS-1$ //$NON-NLS-2$

		if (ticket.isVoided()) {
			map.put("paidMsg", "VOIDED"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else if (NumberUtil.round(ticket.getDueAmount()) > 0) {
			map.put("paidMsg", "DUE"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else {
			map.put("paidMsg", "PAID"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		//		String labTestRoomNo = ""; //$NON-NLS-1$
		//		List<TicketItem> ticketItems = ticket.getTicketItems();
		//		for (TicketItem ticketItem : ticketItems) {
		//			if(StringUtils.isNotBlank(ticketItem.getLabTestRoomNo())){
		//				labTestRoomNo += (StringUtils.isBlank(labTestRoomNo) ? "" : " / ") + ticketItem.getLabTestRoomNo(); //$NON-NLS-1$ //$NON-NLS-2$
		//			}
		//		}
		//		map.put("labTestRoomNo", StringUtils.isBlank(labTestRoomNo) ? null : labTestRoomNo); //$NON-NLS-1$  
		map.put("roomNo", "Room No"); //$NON-NLS-1$//$NON-NLS-2$
		map.put("colPaidAmount", "Paid amount:"); //$NON-NLS-1$//$NON-NLS-2$
		map.put("paidAmount", currencySymbol + "" + NumberUtil.formatAmount(ticket.getPaidAmountAfterRefund())); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("visibleCategoryTotal", outlet.isShowCategoryTotal()); //$NON-NLS-1$

		map.put("colUnit", "Unit"); //$NON-NLS-1$ //$NON-NLS-2$

		StringBuilder extraOrderInfo = new StringBuilder();
		extraOrderInfo.append("Invoice No: " + ticket.getId());
		extraOrderInfo.append("<br>");
		if (StringUtils.isNotBlank(serverName)) {
			extraOrderInfo.append("Server: " + serverName);
			extraOrderInfo.append("<br>");
		}
		extraOrderInfo.append(
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd MMM yy hh:mm a"));
		map.put("additionalOrderInfo", extraOrderInfo.toString()); //$NON-NLS-1$

		map.put("ticketHeader", "Invoice");
		map.put("footerMessage", "Powerd by Medlogics");

		map.put("headerLine1", store != null ? store.getName() : ""); //$NON-NLS-1$
		boolean isPharma = ticket.getTicketType() == TicketType.PHARMA;
		addStoreInfo(map, store, outlet, isPharma);

		map.put("discount", "Discount");
	}

	public static void addStoreInfo(HashMap<String, Object> map, Store store, Outlet outlet, boolean isPharma) {
		StringBuilder storeInfo = new StringBuilder();
		if (outlet != null && outlet.getAddressLine2() != null) {
			storeInfo.append(outlet.getAddressLine1() + ", " + outlet.getAddressLine2());
			storeInfo.append("<br>");
		}
		Address address = outlet.getAddress();
		String websiteAddress = outlet.getWebsiteAddress();
		if (address != null || StringUtils.isNotBlank(websiteAddress)) {
			storeInfo.append("Hotline: " + address.getTelephone() + ", " + "Web: " + "<a style='color: black' href='" + websiteAddress + "'>" //$NON-NLS-1$
					+ websiteAddress + "</a>");
			storeInfo.append("<br>");
		}

		map.put("headerLine2", storeInfo.toString());

		if (outlet.isShowPrintReceiptStoreLogo() && !isPharma) {
			ImageIcon storeLogo = store.getStoreLogo();
			if (storeLogo == null) {
				return;
			}
			ImageIcon imageIcon = ImageUtil.pngToJpg(storeLogo);
			if (imageIcon != null) {
				map.put("storeLogo", imageIcon.getImage()); //$NON-NLS-1$
			}
		}
	}

	private static LabTestReportModel createModels(Ticket ticket) {
		LabTestReportModel reportModel = new LabTestReportModel();
		boolean showQuantityWithItemName = ticket.getTicketType() == TicketType.PHARMA
				&& DataProvider.get().getOutlet().getPharmacyReceiptPaperSize() == PaperSize.EIGHTY;
		reportModel.setShowQuantityWithItemName(showQuantityWithItemName);
		List<TicketItem> ticketItems = new ArrayList<>(ticket.getTicketItems());

		for (TicketItem item : ticketItems) {
			if (ProductType.match(item.getProductType(), ProductType.GOODS)) {
				item.setCategoryName(null);
			}
			else if (ProductType.match(item.getProductType(), ProductType.SERVICES)) {
				item.setCategoryName("Consumable services");
			}
		}

		Comparator<TicketItem> productTypeComparator = Comparator.comparing(TicketItem::getProductType, Comparator.nullsLast(Comparator.reverseOrder()));
		Comparator<TicketItem> comparator = Comparator.comparing(TicketItem::getCategoryName, Comparator.nullsFirst(Comparator.naturalOrder()));
		ticketItems.sort(productTypeComparator.thenComparing(comparator));
		//ticketItems.sort(comparator.thenComparing(productTypeComparator));

		List<TicketItem> services = ticketItems.stream().filter(item -> ProductType.match(item.getProductType(), ProductType.SERVICES))
				.collect(Collectors.toList());
		List<TicketItem> goods = ticketItems.stream().filter(item -> ProductType.match(item.getProductType(), ProductType.GOODS)).collect(Collectors.toList());

		ticketItems.removeAll(services);
		ticketItems.removeAll(goods);

		ticketItems.addAll(services);
		ticketItems.addAll(goods);

		reportModel.setRows(ticketItems);
		return reportModel;
	}
}
