package com.floreantpos.services.report;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.LogicalExpression;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosException;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashDropTransaction;
import com.floreantpos.model.CashInTransaction;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.DrawerType;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.GratuityPaymentHistory;
import com.floreantpos.model.LdfPayTransaction;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.RfPayTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.model.dao.GratuityDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class CashDrawerReportService {
	public final static String HEADERLINE1 = "headerLine1"; //$NON-NLS-1$
	public final static String SSREPORTTITLE = "SSReportTitle"; //$NON-NLS-1$
	public final static String SALESBALANCE = "salesBalance"; //$NON-NLS-1$
	public final static String ROWGROSSSALES = "rowGrossSales"; //$NON-NLS-1$
	public final static String ROWNETSALES = "rowNetSales"; //$NON-NLS-1$
	public final static String TITLECATBREAKOUT = "TitleCatBreakOut"; //$NON-NLS-1$
	public final static String ROWDISCOUNT = "rowDiscount"; //$NON-NLS-1$
	public final static String ROWRETURN = "rowReturn"; //$NON-NLS-1$
	public final static String ROWSALESTAX = "rowSalesTax"; //$NON-NLS-1$
	public final static String ROWSC = "rowSC"; //$NON-NLS-1$
	public final static String ROWDC = "rowDC"; //$NON-NLS-1$
	public final static String ROWTOTAL = "rowTotal"; //$NON-NLS-1$
	public final static String ROWTIPS = "rowTips"; //$NON-NLS-1$
	public final static String ROWFEE = "rowFee"; //$NON-NLS-1$
	public final static String ROWGRECEIVABLE = "rowGReceivable"; //$NON-NLS-1$
	public final static String ROWCRECEIPTS = "rowCReceipts"; //$NON-NLS-1$
	public final static String ROWCCARDS = "rowCCards"; //$NON-NLS-1$
	public final static String ROWDCARDS = "rowDCards"; //$NON-NLS-1$
	public final static String ROWMPAYMENTS = "rowMPayments"; //$NON-NLS-1$
	public final static String ROWCPAYMENTS = "rowCPayments"; //$NON-NLS-1$
	public final static String ROWGCERT = "rowGCert"; //$NON-NLS-1$
	public final static String ROWREFUNDPLUS = "rowRefundPlus"; //$NON-NLS-1$
	public final static String ROWTOLERANCE = "rowTolerance"; //$NON-NLS-1$
	public final static String ROWRECEIPTDIFF = "rowReceiptDiff"; //$NON-NLS-1$
	public final static String ROWCASHTIPS = "rowCashTips"; //$NON-NLS-1$
	public final static String ROWCHARGEDTIPS = "rowChargedTips"; //$NON-NLS-1$
	public final static String USER = "user"; //$NON-NLS-1$
	public final static String DATE = "date"; //$NON-NLS-1$
	public final static String TOTALVOID = "totalVoid"; //$NON-NLS-1$
	public final static String DECLAREDTIPS = "declaredTips"; //$NON-NLS-1$

	public final static String REPORTTITLE = "reportTitle"; //$NON-NLS-1$
	public final static String ROWGROSSRECEIPTS = "rowGrossReceipts"; //$NON-NLS-1$
	public final static String ROWGRETURNS = "rowGReturns"; //$NON-NLS-1$
	public final static String ROWGCCHANGE = "rowGCChange"; //$NON-NLS-1$
	public final static String ROWTIPSPAID = "rowTipsPaid"; //$NON-NLS-1$
	public final static String ROWTIPSPAIDBYOTHERTERMINAL = "rowTipsPaidByOtherTerminal"; //$NON-NLS-1$
	public final static String ROWTIPSDIFF = "rowTipsDiff"; //$NON-NLS-1$
	public final static String CASHBALANCE = "cashBalance"; //$NON-NLS-1$
	public final static String ROWCASH = "rowCash"; //$NON-NLS-1$
	public final static String ROWPAYOUT = "rowPayOut"; //$NON-NLS-1$
	public final static String ROWREFUNDMINUS = "rowRefundMinus"; //$NON-NLS-1$
	public final static String ROWBEGINCASH = "rowBeginCash"; //$NON-NLS-1$
	public final static String ROWCASHIN = "rowCashIn"; //$NON-NLS-1$
	public final static String ROWDBLEED = "rowDBleed"; //$NON-NLS-1$
	public final static String ROWDACC = "rowDAcc"; //$NON-NLS-1$
	public final static String ROWDTIPS = "rowDTips"; //$NON-NLS-1$
	public final static String ROWCTODIPO = "rowCToDipo"; //$NON-NLS-1$
	public final static String ROWCBREAKD = "rowCBreakD"; //$NON-NLS-1$
	public final static String ROWCSRAMOUNT = "rowCSRAmount"; //$NON-NLS-1$
	public final static String ROWVREXCEPTIONS = "rowVRExceptions"; //$NON-NLS-1$
	public final static String ROWVRTAX = "rowVRTax"; //$NON-NLS-1$
	public final static String ROWVRAMOUNT = "rowVRAmount"; //$NON-NLS-1$
	public final static String ROWVRTOTAL = "rowVRTotal"; //$NON-NLS-1$
	public final static String STARTTIMENAME = "startTime"; //$NON-NLS-1$
	public final static String CLOSETIMENAME = "closeTime"; //$NON-NLS-1$
	public final static String REPORTUSER = "reportUser"; //$NON-NLS-1$
	public final static String ROWTOTALTIPS = "rowTotalTips"; //$NON-NLS-1$
	public final static String ROWCASHDUEOROWEDTOSERVER = "rowCashDueOrOwedToServer"; //$NON-NLS-1$
	public final static String LABELROUNDING = "lblRounding"; //$NON-NLS-1$
	public final static String ROUNDINGAMOUNT = "roundingAmount"; //$NON-NLS-1$

	private CashDrawer cashDrawer;
	private Session session;

	public CashDrawerReportService(CashDrawer cashDrawer) {
		super();
		this.cashDrawer = cashDrawer;
	}

	public void populateReport() {
		GenericDAO dao = new GenericDAO();
		try {
			session = dao.createNewSession();
			populateReport(session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void populateSummaryOfCashdrawer() {
		GenericDAO dao = new GenericDAO();
		try {
			session = dao.createNewSession();
			populateSummaryOfCashdrawer(session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void populateSummaryOfCashdrawer(Session session) {
		this.session = session;
		calculateNetSales();
		calculateRefundAmount();
		calculateDrawerBleed();
		calculateCashTips();
		calculateChargedTips();

		double totalRev = cashDrawer.getNetSales() - cashDrawer.getRefundAmount() - cashDrawer.getDrawerBleedAmount() + cashDrawer.getSalesTax()
				+ cashDrawer.getServiceCharge();
		cashDrawer.setTotalRevenue(POSUtil.getDoubleAmount(totalRev));
		cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(cashDrawer.getTotalRevenue() + cashDrawer.getCashTips() + cashDrawer.getChargedTips()));
	}

	public void populateReport(Session session) {
		this.session = session;

		calculateNetSales();
		calculateRefundAmount();
		calculateDrawerBleed();
		calculateCashTips();
		calculateChargedTips();

		cashDrawer.setTotalTips(cashDrawer.getCashTips() + cashDrawer.getChargedTips());

		double totalRev = cashDrawer.getNetSales() - cashDrawer.getRefundAmount() + cashDrawer.getSalesTax() + cashDrawer.getServiceCharge();
		cashDrawer.setTotalRevenue(POSUtil.getDoubleAmount(totalRev));
		cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(cashDrawer.getTotalRevenue() + cashDrawer.getCashTips() + cashDrawer.getChargedTips()));

		calculateCashReceipt();
		calculateCreditReceipt();
		calculateDebitReceipt();
		calculateMemberPayment();

		calculateCustomPaymentWithoutPromotion();
		calculatePromotionAmount();
		calculateGiftCertReceipts();
		//calculateVoidAmount();
		calculateCashPayout();
		calculateCashRFAndLDFPay(RfPayTransaction.class);
		calculateCashRFAndLDFPay(LdfPayTransaction.class);
		calculateCashBack();

		calculateTipsPaid();
		calculateReceiptDiff();
		calculateTipsDiff();
		calculateCashIn();

		calculateDrawerAccountable();
		calculateTicketCount();
	}

	private void calculateTicketCount() {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.setProjection(Projections.countDistinct(PosTransaction.PROP_TICKET));
		Number rowCount = (Number) criteria.uniqueResult();
		int ticketCount = 0;
		if (rowCount != null) {
			ticketCount = rowCount.intValue();
		}
		cashDrawer.setTicketCount(ticketCount);
	}

	private void calculateMemberPayment() {
		Criteria criteria = getCriteriaForTransaction(CustomerAccountTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.MEMBER_ACCOUNT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setCustomerPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDrawerAccountable() {
		double drawerAccountableAmnt = cashDrawer.getCashReceiptAmount() - cashDrawer.getTipsPaid() - cashDrawer.getPayOutAmount() - cashDrawer.getCashBack()
				+ cashDrawer.getBeginCash() + cashDrawer.getCashInAmount() - cashDrawer.getDrawerBleedAmount() - cashDrawer.getRfAmount()
				- cashDrawer.getLdfAmount();
		cashDrawer.setDrawerAccountable(POSUtil.getDoubleAmount(drawerAccountableAmnt));
	}

	private void calculateTipsDiff() {
		double tipsDiff = cashDrawer.getCashTips() + cashDrawer.getChargedTips() - cashDrawer.getTipsPaid();
		cashDrawer.setTipsDifferential(POSUtil.getDoubleAmount(tipsDiff));
	}

	private void calculateReceiptDiff() {
		double receiptDiff = cashDrawer.getGrossReceipts() - cashDrawer.getCashReceiptAmount() - cashDrawer.getCreditCardReceiptAmount()
				- cashDrawer.getDebitCardReceiptAmount() - cashDrawer.getCustomerPaymentAmount() - cashDrawer.getCustomPaymentAmount()
				- cashDrawer.getPromotionAmount() - cashDrawer.getGiftCertChangeAmount() - cashDrawer.getGiftCertReturnAmount() + cashDrawer.getRefundAmount();
		cashDrawer.setReceiptDifferential(NumberUtil.round(receiptDiff));
	}

	private Criteria getCriteriaForTransaction(Class transactionClass) {
		Criteria criteria = session.createCriteria(transactionClass);
		if (transactionClass.equals(RefundTransaction.class)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
		}
		else if (transactionClass.equals(PayOutTransaction.class) || transactionClass.equals(CashDropTransaction.class)
				|| transactionClass.equals(LdfPayTransaction.class)) {
			LogicalExpression orExpression = Restrictions.or(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()),
					Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.OUT.name()));
			criteria.add(orExpression);
		}
		else if (transactionClass.equals(RfPayTransaction.class)) {
			//RfPayTransaction will add all TransactionType
		}
		else if (transactionClass.equals(CashInTransaction.class)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.IN.name()));
		}
		else {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		}
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		return criteria;
	}

	private void calculateNetSales() {
		Criteria criteria = getCriteriaForTransaction(PosTransaction.class);

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sum(PosTransaction.PROP_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_TAX_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_SERVICE_CHARGE_AMOUNT));
		criteria.setProjection(projectionList);

		List<Object[]> list = criteria.list();
		if (list == null || list.isEmpty()) {
			return;
		}
		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			Object[] objects = (Object[]) iterator.next();
			if (objects[0] == null) {
				return;
			}
			double salesAmnt = (objects[0] != null) ? ((Number) objects[0]).doubleValue() : 0;
			double taxAmnt = (objects[1] != null) ? ((Number) objects[1]).doubleValue() : 0;
			double tipsAmnt = (objects[2] != null) ? ((Number) objects[2]).doubleValue() : 0;
			double serviceChrgAmnt = (objects[3] != null) ? ((Number) objects[3]).doubleValue() : 0;

			double netSales = salesAmnt - taxAmnt - tipsAmnt - serviceChrgAmnt;
			cashDrawer.setNetSales(NumberUtil.round(netSales));
			cashDrawer.setSalesTax(NumberUtil.round(taxAmnt));
			cashDrawer.setServiceCharge(NumberUtil.round(serviceChrgAmnt));
		}
	}

	private void calculateCashTips() {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		cashDrawer.setCashTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateChargedTips() {
		Criteria criteria = getCriteriaForTransaction(PosTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		cashDrawer.setChargedTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashReceipt() {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setCashReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCreditReceipt() {
		Criteria criteria = getCriteriaForTransaction(CreditCardTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));

		cashDrawer.setCreditCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDebitReceipt() {
		Criteria criteria = getCriteriaForTransaction(DebitCardTransaction.class);
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		cashDrawer.setDebitCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	//	private void calculateCustomPayment() {
	//		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
	//		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
	//		cashDrawer.setCustomPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	//	}

	private void calculateCustomPaymentWithoutPromotion() {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		cashDrawer.setCustomPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculatePromotionAmount() {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		cashDrawer.setPromotionAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateGiftCertReceipts() {
		Criteria criteria = getCriteriaForTransaction(GiftCertificateTransaction.class);
		criteria.setProjection(Projections.sum(GiftCertificateTransaction.PROP_AMOUNT));
		cashDrawer.setGiftCertChangeAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateRefundAmount() {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class);
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setRefundAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateVoidAmount() {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.TRUE));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setCardVoidAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashPayout() {
		Criteria criteria = getCriteriaForTransaction(PayOutTransaction.class);
		criteria.add(Restrictions.not(Restrictions.ilike(PayOutTransaction.PROP_EXTRA_PROPERTIES, "\"expenses\":\"true\"", MatchMode.ANYWHERE)));
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));
		cashDrawer.setPayOutAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashRFAndLDFPay(Class transactionClass) {
		Criteria criteria = getCriteriaForTransaction(transactionClass);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));

		List<PosTransaction> list = criteria.list();

		double totalTodaysPay = 0;
		double totalDuePay = 0;
		for (PosTransaction posTransaction : list) {
			String transactionType = posTransaction.getTransactionType();
			if (TransactionType.IN.name().equals(transactionType)) {
				posTransaction.setAmount((-1) * posTransaction.getAmount());
			}

			String transTicketIds = posTransaction.getTransTicketIdsDisplay();
			if (StringUtils.isNotBlank(transTicketIds) && transTicketIds.contains(",")) {
				continue;
			}

			Date orderDate = DataProvider.get().getTransactionsTicketDate(transTicketIds, posTransaction.getOutletId());
			Date startOfDay = DateUtil.startOfDay(posTransaction.getTransactionTime());
			if (orderDate.before(startOfDay)) {
				totalDuePay += posTransaction.getAmount();
			}
			else {
				totalTodaysPay += posTransaction.getAmount();
			}
		}

		if (transactionClass.equals(RfPayTransaction.class)) {
			cashDrawer.setRfAmount(POSUtil.getDoubleAmount(totalTodaysPay));
			cashDrawer.setRfDuePay(POSUtil.getDoubleAmount(totalDuePay));
		}
		else if (transactionClass.equals(LdfPayTransaction.class)) {
			cashDrawer.setLdfAmount(POSUtil.getDoubleAmount(totalTodaysPay));
			cashDrawer.setLdfDuePay(POSUtil.getDoubleAmount(totalDuePay));
		}
	}

	private void calculateDrawerBleed() {
		Criteria criteria = getCriteriaForTransaction(CashDropTransaction.class);
		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));
		cashDrawer.setDrawerBleedAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashBack() {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setCashBack(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateTipsPaid() {
		if (cashDrawer.getDrawerType() == DrawerType.STAFF_BANK) {
			double paidGratuititesForStaffBank = findTotalPaidGratuititesForStaffBank(cashDrawer);
			cashDrawer.setTipsPaid(paidGratuititesForStaffBank);
			calculateTipsPaidByOtherTerminal();
		}
		else {
			Criteria criteria = session.createCriteria(GratuityPaymentHistory.class);
			criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_CASH_DRAWER, cashDrawer));
			criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));
			cashDrawer.setTipsPaid(POSUtil.getDoubleAmount(criteria.uniqueResult()));
		}

	}

	public double findTotalPaidGratuititesForStaffBank(CashDrawer staffBank) throws PosException {
		try (Session session = GratuityDAO.getInstance().createNewSession()) {
			DetachedCriteria detachedCriteria = DetachedCriteria.forClass(PosTransaction.class);
			detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			detachedCriteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, staffBank.getId()));
			detachedCriteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$
			detachedCriteria.setProjection(Projections.property("t.id")); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(Gratuity.class);
			criteria.add(Property.forName(Gratuity.PROP_TICKET_ID).in(detachedCriteria));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(Gratuity.PROP_AMOUNT));
			projectionList.add(Projections.sum(Gratuity.PROP_TIPS_PAID_AMOUNT));
			criteria.setProjection(projectionList);

			List list = criteria.list();
			double total = 0;
			if (list.size() > 0) {
				Object[] objects = (Object[]) list.get(0);
				if (objects.length > 0) {
					total = POSUtil.getDoubleAmount(objects[0]);
				}
				if (objects.length > 1) {
					total -= POSUtil.getDoubleAmount(objects[1]);
				}

			}
			criteria = session.createCriteria(GratuityPaymentHistory.class);
			criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_CASH_DRAWER, staffBank));
			criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));
			return total + POSUtil.getDoubleAmount(criteria.uniqueResult());
		}
	}

	private void calculateTipsPaidByOtherTerminal() throws PosException {
		DetachedCriteria detachedCriteria = DetachedCriteria.forClass(PosTransaction.class);
		detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		detachedCriteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
		detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		detachedCriteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$
		detachedCriteria.setProjection(Projections.property("t.id")); //$NON-NLS-1$

		Criteria criteria = session.createCriteria(Gratuity.class);
		criteria.add(Property.forName(Gratuity.PROP_TICKET_ID).in(detachedCriteria));

		criteria.setProjection(Projections.sum(Gratuity.PROP_TIPS_PAID_AMOUNT));

		double totalPaidTips = POSUtil.getDoubleAmount(criteria.uniqueResult());

		criteria = session.createCriteria(GratuityPaymentHistory.class);
		criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_CASH_DRAWER, cashDrawer));
		criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));
		double tipsPaidFromThisBank = POSUtil.getDoubleAmount(criteria.uniqueResult());

		cashDrawer.setTipsPaidByOtherTerminal(totalPaidTips - tipsPaidFromThisBank);
	}

	private void calculateCashIn() {
		Criteria criteria = getCriteriaForTransaction(CashInTransaction.class);
		criteria.setProjection(Projections.sum(CashInTransaction.PROP_AMOUNT));
		cashDrawer.setCashInAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	//	private void calculateGrossReceipts() {
	//		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
	//		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
	//		cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	//	}
	//	

	public DetailsOfCashdrawerDataProvider populateDetailOfCashdrawer() {
		GenericDAO dao = new GenericDAO();
		try {
			session = dao.createNewSession();

			return new DetailsOfCashdrawerDataProvider(getGrossReceiptCollectionList(false), getPayoutList(),
					getReferrerAndLDFPayTransactionList(false, RfPayTransaction.class), getReferrerAndLDFPayTransactionList(true, RfPayTransaction.class),
					getReferrerAndLDFPayTransactionList(false, LdfPayTransaction.class), getReferrerAndLDFPayTransactionList(true, LdfPayTransaction.class),
					getRefundTransactionList(), getGrossReceiptCollectionList(true));

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private List<CashDrawerDetailReport> getGrossReceiptCollectionList(boolean due) {
		List<CashDrawerDetailReport> collectionList = getDueAndGrossReceiptCollectionList(due, false);
		collectionList.addAll(getDueAndGrossReceiptCollectionList(due, true));
		return collectionList;
	}

	private List<CashDrawerDetailReport> getDueAndGrossReceiptCollectionList(boolean due, boolean isRefund) {

		Criteria criteria = getCriteriaForTransaction(isRefund ? RefundTransaction.class : PosTransaction.class);
		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TIME), CashDrawerDetailReport.TRANS_TIME);
		projectionList.add(Projections.property(PosTransaction.PROP_TICKET), CashDrawerDetailReport.PROP_TICKET);
		projectionList.add(Projections.property(PosTransaction.PROP_AMOUNT), CashDrawerDetailReport.TRANSACTION_AMOUNT);
		projectionList.add(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING), CashDrawerDetailReport.PAYMENT_TYPE);
		criteria.setProjection(projectionList);
		criteria.setResultTransformer(Transformers.aliasToBean(CashDrawerDetailReport.class));

		List<CashDrawerDetailReport> list = criteria.list();

		List<CashDrawerDetailReport> dueList = new ArrayList<>();
		List<CashDrawerDetailReport> todayPayList = new ArrayList<>();

		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			CashDrawerDetailReport detailReport = (CashDrawerDetailReport) iterator.next();
			Ticket ticket = detailReport.getTicket();
			Date orderDate = ticket.getCreateDate();
			Date startOfDay = DateUtil.startOfDay(detailReport.getTransTime());
			detailReport.initInvoiceDate();
			if (isRefund) {
				detailReport.setTransactionAmount((-1) * detailReport.getTransactionAmount());
			}

			if (orderDate.before(startOfDay)) {
				dueList.add(detailReport);
			}
			else {
				todayPayList.add(detailReport);
			}
		}

		return due ? dueList : todayPayList;
	}

	private List getPayoutList() {
		Criteria criteria = session.createCriteria(PayOutTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(PayOutTransaction.PROP_TRANSACTION_TIME), CashDrawerDetailReport.TRANS_TIME);
		projectionList.add(Projections.property(PayOutTransaction.PROP_RECEPIENT_ID), CashDrawerDetailReport.RECEPIENT_ID);
		projectionList.add(Projections.property(PayOutTransaction.PROP_AMOUNT), CashDrawerDetailReport.TRANSACTION_AMOUNT);
		projectionList.add(Projections.property(PayOutTransaction.PROP_PAYMENT_TYPE_STRING), CashDrawerDetailReport.PAYMENT_TYPE);
		criteria.setProjection(projectionList);

		criteria.setResultTransformer(Transformers.aliasToBean(CashDrawerDetailReport.class));

		List<CashDrawerDetailReport> list = criteria.list();
		for (CashDrawerDetailReport detailReport : list) {
			detailReport.initRecepientName();
		}
		return list;
	}

	private List getReferrerAndLDFPayTransactionList(boolean due, Class transactionClass) {
		Criteria criteria = getCriteriaForTransaction(transactionClass);

		List<PosTransaction> list = criteria.list();
		List<CashDrawerDetailReport> dueList = new ArrayList<>();
		List<CashDrawerDetailReport> todayPayList = new ArrayList<>();

		for (PosTransaction posTransaction : list) {
			String transactionType = posTransaction.getTransactionType();
			if (TransactionType.IN.name().equals(transactionType)) {
				posTransaction.setAmount((-1) * posTransaction.getAmount());
			}

			String transTicketIds = posTransaction.getTransTicketIdsDisplay();
			if (StringUtils.isNotBlank(transTicketIds) && transTicketIds.contains(",")) { //$NON-NLS-1$
				continue;
			}

			Date orderDate = DataProvider.get().getTransactionsTicketDate(transTicketIds, posTransaction.getOutletId());
			Date startOfDay = DateUtil.startOfDay(posTransaction.getTransactionTime());
			if (orderDate.before(startOfDay)) {
				dueList.add(new CashDrawerDetailReport(posTransaction, orderDate));
			}
			else {
				todayPayList.add(new CashDrawerDetailReport(posTransaction, orderDate));
			}
		}
		return due ? dueList : todayPayList;
	}

	private List getRefundTransactionList() {
		Criteria criteria = session.createCriteria(RefundTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(RefundTransaction.PROP_TRANSACTION_TIME), CashDrawerDetailReport.TRANS_TIME);
		projectionList.add(Projections.property(RefundTransaction.PROP_TICKET), CashDrawerDetailReport.PROP_TICKET);
		projectionList.add(Projections.property(RefundTransaction.PROP_AMOUNT), CashDrawerDetailReport.TRANSACTION_AMOUNT);
		projectionList.add(Projections.property(RefundTransaction.PROP_PAYMENT_TYPE_STRING), CashDrawerDetailReport.PAYMENT_TYPE);
		criteria.setProjection(projectionList);

		criteria.setResultTransformer(Transformers.aliasToBean(CashDrawerDetailReport.class));

		List<CashDrawerDetailReport> list = criteria.list();
		for (CashDrawerDetailReport detailReport : list) {
			detailReport.initInvoiceDate();
		}
		return list;
	}

	public class DetailsOfCashdrawerDataProvider {
		private List<CashDrawerDetailReport> detailTransactionList;
		private List<CashDrawerDetailReport> payoutList;
		private List<CashDrawerDetailReport> rfPayList;
		private List<CashDrawerDetailReport> rfDueList;
		private List<CashDrawerDetailReport> ldfPayList;
		private List<CashDrawerDetailReport> ldfDueList;
		private List<CashDrawerDetailReport> refundTransactionList;
		private List<CashDrawerDetailReport> dueCollectionList;

		public DetailsOfCashdrawerDataProvider(List<CashDrawerDetailReport> detailTransactionList, List<CashDrawerDetailReport> payoutList,
				List<CashDrawerDetailReport> rfPayList, List<CashDrawerDetailReport> rfDueList, List<CashDrawerDetailReport> ldfPayList,
				List<CashDrawerDetailReport> ldfDueList, List<CashDrawerDetailReport> refundTransactionList, List<CashDrawerDetailReport> dueCollectionList) {
			this.detailTransactionList = detailTransactionList;
			this.payoutList = payoutList;
			this.rfPayList = rfPayList;
			this.rfDueList = rfDueList;
			this.ldfPayList = ldfPayList;
			this.ldfDueList = ldfDueList;
			this.refundTransactionList = refundTransactionList;
			this.dueCollectionList = dueCollectionList;
		}

		public List<CashDrawerDetailReport> getDetailTransactionList() {
			return detailTransactionList;
		}

		public List<CashDrawerDetailReport> getPayoutList() {
			return payoutList;
		}

		public List<CashDrawerDetailReport> getRfPayList() {
			return rfPayList;
		}

		public List<CashDrawerDetailReport> getRfDueList() {
			return rfDueList;
		}

		public List<CashDrawerDetailReport> getLdfPayList() {
			return ldfPayList;
		}

		public List<CashDrawerDetailReport> getLdfDueList() {
			return ldfDueList;
		}

		public List<CashDrawerDetailReport> getRefundTransactionList() {
			return refundTransactionList;
		}

		public List<CashDrawerDetailReport> getDueCollectionList() {
			return dueCollectionList;
		}

	}

}
