package com.floreantpos.hl7;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.PosLog;
import com.floreantpos.hl7.model.Result;
import com.floreantpos.hl7.model.Test;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MachineResult;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemResult;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.MachineResultDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.floreantpos.model.ext.LabWorkStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.InventoryUnitConversionRule;
import com.google.gson.Gson;

public class HL7Util {

	private static ResultListener resultListener;
	private final static SimpleDateFormat mithicFormatter = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");

	public HL7Util() {
	}

	public static void addResultToTicketItem(Test test, TicketItem ticketItem) {
		Map<String, Result> machineResultmap = test.resultmap;
		List<TestItem> menuItemTestItems = getMenuItemTestItems(ticketItem.getMenuItemId());
		for (TestItem testItem : menuItemTestItems) {
			String machineCode = testItem.getMachineCode();
			if (StringUtils.isBlank(machineCode)) {
				continue;
			}

			Result machineResult = machineResultmap.get(machineCode);
			if (machineResult == null) {
				continue;
			}
			testItem.putOriginalMachineResult(machineResult.result);
			if (StringUtils.isBlank(testItem.getUnit())) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);
			testItem.putMachineName(test.machine);
			//mythic data
			if (StringUtils.isNotBlank(machineResult.normalLow) || StringUtils.isNotBlank(machineResult.normalHigh)) {
				testItem.setNormalValue(machineResult.normalLow + " - "+ machineResult.normalHigh);
			}
			machineResultmap.remove(machineCode);
		}

		int sortOrder = menuItemTestItems.size();
		List<Result> remainResults = new ArrayList<Result>(machineResultmap.values());
		for (int i = 0; i < remainResults.size(); i++) {
			Result machineResult = remainResults.get(i);

			String machineCode = machineResult.testName;
			TestItem testItem = TestItemDAO.getInstance().findByMachineCode(machineCode);
			if (testItem == null) {
				testItem = new TestItem();
				testItem.setName(machineCode);
				testItem.setMachineCode(machineCode);
			}
			else {
				testItem.setLastSyncTime(null);
				testItem.setLastUpdateTime(null);
			}

			testItem.putMachineName(test.machine);
			testItem.setId(machineResult.resultNo);
			testItem.putOriginalMachineResult(machineResult.result);
			String testItemUnit = testItem.getUnit();
			if (StringUtils.isBlank(testItemUnit)) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);
			
			//mythic data
			if (StringUtils.isNotBlank(machineResult.normalLow) || StringUtils.isNotBlank(machineResult.normalHigh)) {
				testItem.setNormalValue(machineResult.normalLow + " - "+ machineResult.normalHigh);
			}

			testItem.putShowTestItem(false);
			testItem.setSortOrder(++sortOrder);
			menuItemTestItems.add(testItem);
			//parsedResult += String.format("Result no: %s, name=%s, value: %s, unit: %s", result.resultNo, result.testName, result.result, result.unit);
			//parsedResult += "\n";
		}

		List<TestItemResult> testItemResults = ticketItem.getTestItemResults();
		testItemResults.clear();
		TestItemResult testItemResult = new TestItemResult();
		testItemResult.setSortOrder(testItemResults.size());
		testItemResult.setTestItems(menuItemTestItems);
		testItemResults.add(testItemResult);
		ticketItem.setTestItemResults(testItemResults);

		ticketItem.setTestItems(menuItemTestItems);
		ticketItem.putMachineResultReceived(true);
		ticketItem.putMachineResultTime(StoreDAO.getServerTimestamp());
		ticketItem.setLabWorkStatusValue(LabWorkStatus.RESULT_RECORDED);
	}
	
	public static void addResultToTicketItem(String testId, TicketItem ticketItem) {
		MachineResult machineResult = MachineResultDAO.getInstance().getOrderHistoryBySampleId(testId);
		addResultToTicketItem(ticketItem, machineResult);
	}

	public static void addResultToTicketItem(TicketItem ticketItem, MachineResult machineResult) {
		ticketItem.setTestItemResults(Collections.emptyList());
		if (machineResult != null && ticketItem != null) {
			String resultJson = machineResult.getResultJson();
			Test test = new Gson().fromJson(resultJson, Test.class);
			addResultToTicketItem(test, ticketItem);
			ticketItem.putMachineResultTime(machineResult.getResultDate());
		}
		else {
			ticketItem.putMachineResultReceived(false);
			ticketItem.putMachineResultTime(null);
			ticketItem.setLabWorkStatusValue(LabWorkStatus.RUN_TEST);
		}
	}

	public static List<TestItem> getMenuItemTestItems(String menuItemId) {
		MenuItem menuItem = MenuItemDAO.getInstance().loadInitialized(menuItemId);

		List<TestItem> testItems = menuItem.getTestItems();
		if (testItems != null) {
			int sortorder = 1;
			for (TestItem testItem : testItems) {
				if (testItem == null) {
					continue;
				}
				testItem.setSortOrder(sortorder);
				sortorder++;
			}
		}
		return testItems;
	}

	public static String cleanTestName(String testName) {
		//String[] chars = { "%", "#", "^" };
		String[] chars = { "^" }; //$NON-NLS-1$
		//System.out.println("cleaning: " + testName);
		for (int i = 0; i < chars.length; i++) {
			int index = testName.indexOf(chars[i]);
			if (index > 0) {
				return testName.substring(0, index);
			}
		}

		return testName;
	}

	public static void convertMachineResultToItemUnit(TestItem testItem, Result result) {
		try {
			String unitid = testItem.getUnit();
			if (org.apache.commons.lang.StringUtils.isBlank(unitid)) {
				return;
			}
			InventoryUnit inventoryUnit = InventoryUnitDAO.getInstance().get(unitid);
			if (inventoryUnit == null) {
				return;
			}
			if (inventoryUnit.getConversionRate() > 0) {
				double newRsult = Double.valueOf(result.result);

				String inventoryUnitConversionRuleName = inventoryUnit.getProperty(InventoryUnit.PROP_CONVERSION_RULE,
						InventoryUnitConversionRule.DIVISION.getName());
				InventoryUnitConversionRule inventoryUnitConversionRule = InventoryUnitConversionRule.fromName(inventoryUnitConversionRuleName);
				if (inventoryUnitConversionRule == InventoryUnitConversionRule.MULTIPLICATION) {
					newRsult = newRsult * 1D / inventoryUnit.getConversionRate();
				}
				else {
					newRsult = newRsult / inventoryUnit.getConversionRate();
				}
				result.result = String.valueOf(newRsult);
			}
		} catch (Exception e) {
			//ignore
		}
	}

	public static void updateUnit(Map<String, InventoryUnit> unitsMap, Result result, String unitName) {
		if (StringUtils.isNotBlank(unitName)) {
			InventoryUnitDAO instance = InventoryUnitDAO.getInstance();
			InventoryUnit inventoryUnit = unitsMap.get(unitName);
			if (inventoryUnit == null) {
				inventoryUnit = instance.findByName(unitName);
				if (inventoryUnit == null) {
					inventoryUnit = new InventoryUnit();
					inventoryUnit.setConversionRate(1.0);
					inventoryUnit.setName(unitName);
					if (unitName.length() > 10) {
						inventoryUnit.setCode(String.format("%.10s", unitName)); //$NON-NLS-1$
					}
					else {
						inventoryUnit.setCode(unitName);
					}
					instance.save(inventoryUnit);
				}
				unitsMap.put(unitName, inventoryUnit);
			}

			result.unit = inventoryUnit.getId();
			result.inventoryUnit = inventoryUnit;
		}
	}

	public static Test saveMachineResult(Test test, String postData) {

		String sampleId = test.sampleId;
		if (StringUtils.isBlank(sampleId)) {
			PosLog.error(HL7Util.class, "Sample id not found for machine data");
			sampleId = "00000";
		}
		PosLog.debug(HL7Util.class, String.format("Sample id: %s", sampleId));

		MachineResult machineResult = new MachineResult();
		if (StringUtils.isNotBlank(test.date)) {
			machineResult.setResultDate(convertToMithicFormatterDate(test.date, test.time));
		}
		else {
			machineResult.setResultDate(StoreDAO.getServerTimestamp());
		}
		machineResult.setSampleId(sampleId);
		machineResult.setOutletId(DataProvider.get().getCurrentOutletId());
		machineResult.setResultJson(new Gson().toJson(test));
		machineResult.putMachineRawData(postData);
		MachineResultDAO.getInstance().saveOrUpdate(machineResult);
		PosLog.debug(HL7Util.class, "Result has been stored.");

		if (resultListener != null) {
			resultListener.lastResult(machineResult);
		}

		TicketItem ticketItem = TicketItemDAO.getInstance().findByLabTest(sampleId);
		if (ticketItem == null) {
			PosLog.debug(HL7Util.class, String.format("Ticket item for sample %s not found", sampleId));
			return test;
		}

		addResultToTicketItem(test, ticketItem);
		TicketItemDAO.getInstance().saveOrUpdate(ticketItem);
		return test;
	}

	private static Date convertToMithicFormatterDate(String date, String time) {
		if (StringUtils.isNotBlank(date)) {
			String dateTimeString = date + " " + time;
			try {
				return mithicFormatter.parse(dateTimeString.trim());
			} catch (Exception e) {
			}
		}
		return StoreDAO.getServerTimestamp();
	}
	
	public static void setResultListener(ResultListener resultListener) {
		HL7Util.resultListener = resultListener;
	}

}
