package com.floreantpos.model.dao;

import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemGroup;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class TestItemDAO extends BaseTestItemDAO {

	public TestItemDAO() {
	}

	@Override
	public List<TestItem> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TestItem.class);
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(TestItem.PROP_NAME).ignoreCase());
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		TestItem testItem = (TestItem) obj;
		checkIfTestItemCanbeDeleted(testItem, s);
		testItem.setDeleted(true);
		update(testItem, s);
	}

	private void checkIfTestItemCanbeDeleted(TestItem testItem, Session session) {
		String details = ""; //$NON-NLS-1$;

		List<MenuItem> menuItems = MenuItemDAO.getInstance().getMenuItemsByTestUnitId(testItem.getId(), session);
		if (menuItems != null && !menuItems.isEmpty()) {
			details = String.format("Test unit %s is being used by the following tests:", testItem.getName());
			for (int i = 0; i < menuItems.size(); i++) {
				MenuItem menuItem = menuItems.get(i);
				details += String.format("\n%d. %s", (i + 1), menuItem.getName()); //$NON-NLS-1$
			}
		}

		if (StringUtils.isNotBlank(details)) {
			throw new PosException(String.format("Test unit <b>%s</b> is being used by other tests", testItem.getName()), details);
		}
	}

	public void findByName(PaginationSupport tableModel, String searchString) {
		findByNameAndGroup(tableModel, searchString, null, false);
	}

	public void findByNameAndGroup(PaginationSupport tableModel, String searchString, TestItemGroup itemGroup, boolean isIncludeMachineCodeSort) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(TestItem.class);

			addDeletedFilter(criteria);
			addCriteriaPropertyExcludeFilter(criteria, TestItem.PROP_PROPERTIES, TestItem.JESON_PROP_VISIBLE, false);

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction or = Restrictions.disjunction();
				or.add(Restrictions.ilike(TestItem.PROP_NAME, searchString.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(TestItem.PROP_MACHINE_CODE, searchString.trim(), MatchMode.ANYWHERE));
				criteria.add(or);
			}

			if (itemGroup != null) {
				criteria.add(Restrictions.eq(TestItem.PROP_TEST_ITEM_GROUP_ID, itemGroup.getId()));
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			if (isIncludeMachineCodeSort) {
				criteria.addOrder(Order.asc(TestItem.PROP_MACHINE_CODE).ignoreCase());
			}
			criteria.addOrder(Order.asc(TestItem.PROP_NAME).ignoreCase());
			criteria.addOrder(Order.asc(TestItem.PROP_SORT_ORDER));
			tableModel.setRows(criteria.list());
		}
	}

	public List<TestItem> findByGroupId(String groupId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TestItem.class);

			addDeletedFilter(criteria);
			addCriteriaPropertyExcludeFilter(criteria, TestItem.PROP_PROPERTIES, TestItem.JESON_PROP_VISIBLE, false);
			if (StringUtils.isNotBlank(groupId)) {
				criteria.add(Restrictions.eq(TestItem.PROP_TEST_ITEM_GROUP_ID, groupId));
			}
			else {
				criteria.add(Restrictions.eqOrIsNull(TestItem.PROP_TEST_ITEM_GROUP_ID, groupId));
			}
			return criteria.list();
		}

	}

	public TestItem findByName(String testUnitName) {
		try (Session session = createNewSession()) {
			return findByName(testUnitName, session);
		}
	}

	public TestItem findByName(String testUnitName, Session session) {
		Criteria criteria = session.createCriteria(TestItem.class);

		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(TestItem.PROP_NAME, testUnitName).ignoreCase());
		criteria.setMaxResults(1);
		return (TestItem) criteria.uniqueResult();
	}

	public TestItem findByMachineCode(String machineCode) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TestItem.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(TestItem.PROP_MACHINE_CODE, machineCode));
			criteria.setMaxResults(1);
			return (TestItem) criteria.uniqueResult();
		}
	}

	public void saveOrUpdateTestItemList(List<TestItem> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateTestItemList(dataList, updateLastUpdateTime, updateSyncTime, false, false);
	}

	public void saveOrUpdateTestItemList(List<TestItem> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean saveNewDataOnly,
			boolean forceUpdate) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<TestItem> iterator = dataList.iterator(); iterator.hasNext();) {
				TestItem item = (TestItem) iterator.next();
				TestItem existingItem = get(item.getId());
				if (existingItem != null) {
					if (!forceUpdate && (saveNewDataOnly || !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime()))) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

}