package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BalanceForwardTransaction;
import com.floreantpos.model.BalanceSubType;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.Customer;
import com.floreantpos.model.EndBalanceTransaction;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PayoutRecepient;
import com.floreantpos.model.PayoutSubReason;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Project;
import com.floreantpos.model.SalaryAdvanceTransaction;
import com.floreantpos.model.SalaryTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.UserSalaryDeductionModel;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.DefaultDataInserter;
import com.floreantpos.util.GsonUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class BalanceUpdateTransactionDAO extends BaseBalanceUpdateTransactionDAO {

	public BalanceUpdateTransactionDAO() {
	}

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private void adjustBalanceUpdateTransaction(BalanceUpdateTransaction balanceTransaction, Session session) {
		save(balanceTransaction, session);

		BalanceType balanceType = balanceTransaction.getBalanceType();
		if (balanceType == BalanceType.GIFT_CARD) {
			GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, balanceTransaction.getAccountNumber());
			if (giftCard == null) {
				//throw new PosException("No gift card found with account number " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
				return;
			}
			giftCard.setBalance(giftCard.getBalance() + balanceTransaction.getAmount());
			GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
			return;
		}

		Customer customer = CustomerDAO.getInstance().get(balanceTransaction.getAccountNumber(), session);
		if (customer == null) {
			//throw new PosException("No customer found with id " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
			return;
		}

		if (balanceType == BalanceType.CUSTOMER) {
			customer.setBalance(customer.getBalance() + balanceTransaction.getAmount());
		}
		else if (balanceType == BalanceType.LOYALTY) {
			customer.setLoyaltyPoint((int) (customer.getLoyaltyPoint() + balanceTransaction.getAmount()));
		}
		customer.setUpdateLastUpdateTime(false);
		customer.setUpdateSyncTime(false);
		CustomerDAO.getInstance().saveOrUpdate(customer, session);
	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, Session session) {
		saveBalanceUpdateTrans(balanceType, ticket, transaction, transactionType, accountNumber, amount, balanceBefore, subType, null, null, session);

	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, CustomPayment customPayment, String customPaymnetRef,
			Session session) {

		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(transaction == null ? PaymentType.CASH : transaction.getPaymentType(),
				transactionType, amount);
		if (customPayment != null) {
			balanceUpdateTransaction.setPaymentType(PaymentType.CUSTOM_PAYMENT);
			balanceUpdateTransaction.setCustomPaymnetId(customPayment.getId());
			balanceUpdateTransaction.setCustomPaymnetRef(customPaymnetRef);
		}
		balanceUpdateTransaction.setTransactionId(transaction == null ? null : transaction.getId());
		balanceUpdateTransaction.setTicketId(ticket == null ? null : ticket.getId());
		balanceUpdateTransaction.setAccountNumber(accountNumber);

		if (subType != null) {
			balanceUpdateTransaction.setTransactionSubType(subType.name());
			balanceUpdateTransaction.setDescription(subType.name());
		}
		if (!NumberUtil.isZero(balanceBefore)) {
			balanceBefore = NumberUtil.round(balanceBefore);
			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		}

		BalanceUpdateTransactionDAO.getInstance().save(balanceUpdateTransaction, session);
	}

	public void saveOrUpdateBalanceUpdateTransactions(List<BalanceUpdateTransaction> customerBalanceTransactions, boolean updateLastUpdateTime,
			boolean updateSyncTime) throws Exception {
		if (customerBalanceTransactions == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (BalanceUpdateTransaction transaction : customerBalanceTransactions) {
				BalanceUpdateTransaction existingTransaction = get(transaction.getId());
				if (existingTransaction != null) {
					continue;
				}
				transaction.setUpdateLastUpdateTime(updateLastUpdateTime);
				transaction.setUpdateSyncTime(updateSyncTime);
				adjustBalanceUpdateTransaction(transaction, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public double getTicketCommissionAmount(Session session, String ticketId, BalanceType balanceType) {
		return getTicketCommissionAmount(session, ticketId, "", balanceType); //$NON-NLS-1$
	}

	public double getTicketCommissionAmount(Session session, String ticketId, String accountNo, BalanceType balanceType) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
		criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TICKET_ID, ticketId));
		if (StringUtils.isNotBlank(accountNo)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
		}
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, balanceType.name()));
		double scFromItems = POSUtil.getDoubleAmount(criteria.uniqueResult());

		return scFromItems;
	}

	//	public void findBy(String accountNo, Date beginDate, Date currentDate, PaginationSupport tableModel) {
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
	//			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));
	//			tableModel.setNumRows(rowCount(criteria));
	//
	//			criteria.setFirstResult(tableModel.getCurrentRowIndex());
	//			criteria.setMaxResults(tableModel.getPageSize());
	//			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
	//			tableModel.setRows(criteria.list());
	//		}
	//	}

	public Map<String, Double> findSummaryBy(String accountNo, Date beginDate, Date currentDate) {
		Map<String, Double> map = new HashMap<String, Double>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
			//criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER), "accountNo"); //$NON-NLS-1$
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING), "balanceType"); //$NON-NLS-1$
			projectionList.add(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT), "total"); //$NON-NLS-1$
			criteria.setProjection(projectionList);

			List<Object[]> list = criteria.list();
			for (Object[] obj : list) {
				String balanceTypeString = (String) obj[1];
				double amount = (double) obj[2];
				map.put(balanceTypeString, amount);
			}
		}
		return map;
	}

	public void loadExpenses(PaginationSupport model, Date fromDate, Date toDate) {
		loadAccountsManagerPayment(model, fromDate, toDate, null, TransactionType.DEBIT, BalanceType.ACCOUNTS_MANAGER_BALANCE, true, null, null, null, null);
	}

	public List<BalanceUpdateTransaction> loadAccountsManagerPayment(Date fromDate, Date toDate, String accountNo, String recipientId, String categoryId,
			String projectId, String subCategoryId) {
		Pagination pagination = new Pagination<>(0, -1);
		loadAccountsManagerPayment(pagination, fromDate, toDate, accountNo, TransactionType.DEBIT, null, true, recipientId, categoryId, projectId,
				subCategoryId);
		return pagination.getDataList();
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, String accountNo) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountNo, null, BalanceType.ACCOUNTS_MANAGER_BALANCE);
	}

	public void loadStoreBalancePayment(PaginationSupport model, Date fromDate, Date toDate, String accountNo) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountNo, null, BalanceType.STORE_BALANCE);
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, String accountNo, TransactionType transactionType,
			BalanceType balanceType) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountNo, transactionType, balanceType, false, null, null);
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, String accountNo, TransactionType transactionType,
			BalanceType balanceType, boolean isExpenseOnly, String recipientId, String reasonId) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountNo, transactionType, balanceType, isExpenseOnly, recipientId, reasonId, null, null);
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, String accountNo, TransactionType transactionType,
			BalanceType balanceType, boolean isExpenseOnly, String recipientId, String reasonId, String projectId, String subCategoryId) {
		loadAccountsManagerPayment(model, fromDate, toDate, accountNo, transactionType, balanceType, isExpenseOnly, recipientId, reasonId, projectId,
				subCategoryId, null);
	}

	public void loadAccountsManagerPayment(PaginationSupport model, Date fromDate, Date toDate, String accountNo, TransactionType transactionType,
			BalanceType balanceType, boolean isExpenseOnly, String recipientId, String reasonId, String projectId, String subCategoryId, String memoNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			initAccountsManagerPaymentCriteria(fromDate, toDate, accountNo, transactionType, balanceType, isExpenseOnly, recipientId, reasonId, criteria,
					projectId, subCategoryId);

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());

			if (!(balanceType == BalanceType.STORE_BALANCE)) {
				criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER));
			}

			if (StringUtils.isNotBlank(memoNo)) {
				criteria.add(Restrictions.ilike(BalanceUpdateTransaction.PROP_EXTRA_PROPERTIES, String.format("memo_no\":\"%s", memoNo), MatchMode.ANYWHERE)); //$NON-NLS-1$
			}

			criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));

			criteria.setProjection(null);
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());

			List<BalanceUpdateTransaction> transactions = criteria.list();
			//			if (isExpenseOnly) {
			//				model.setRows(transactions);
			//				return;
			//			}

			List<BalanceUpdateTransaction> allTransactionList = new ArrayList<BalanceUpdateTransaction>();
			if (transactions != null) {
				String accountsManagerNumber = null;
				double totalAmount = 0;
				for (BalanceUpdateTransaction posTransaction : transactions) {
					if (balanceType == BalanceType.STORE_BALANCE) {
						if (StringUtils.isNotBlank(accountsManagerNumber)) {
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
						else {
							accountsManagerNumber = BalanceType.STORE_BALANCE.name();
							if (!isExpenseOnly) {
								allTransactionList.add(createForwardBalanceByTransction(accountsManagerNumber, posTransaction));
								totalAmount += posTransaction.getBalanceBefore();
							}
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
					}
					else {
						if (posTransaction.getAccountNumber().equals(accountsManagerNumber)) {
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
						else {
							if (StringUtils.isNotBlank(accountsManagerNumber)) {
								if (!isExpenseOnly) {
									allTransactionList.add(createEndBalanceByTransction(accountsManagerNumber, totalAmount));
								}
								totalAmount = 0;
							}
							accountsManagerNumber = posTransaction.getAccountNumber();
							if (!isExpenseOnly) {
								allTransactionList.add(createForwardBalanceByTransction(accountsManagerNumber, posTransaction));
								totalAmount += posTransaction.getBalanceBefore();
							}
							allTransactionList.add(posTransaction);
							totalAmount += posTransaction.getAmount();
						}
					}
				}
				if (!isExpenseOnly) {
					allTransactionList.add(createEndBalanceByTransction(accountsManagerNumber, totalAmount));
				}
			}
			model.setRows(allTransactionList);
		}
	}

	public double getTransferToStoreAmount(Date fromDate, Date toDate, String accountNo, TransactionType transactionType, BalanceType balanceType,
			BalanceSubType subtype, String recipientId, String reasonId, String projectId, String subCategoryId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			initAccountsManagerPaymentCriteria(fromDate, toDate, accountNo, transactionType, balanceType, false, recipientId, reasonId, criteria, projectId,
					subCategoryId);

			if (subtype != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, subtype.name()));
			}

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				return ((Number) uniqueResult).doubleValue();
			}

		}
		return 0;
	}

	private void initAccountsManagerPaymentCriteria(Date fromDate, Date toDate, String accountNo, TransactionType transactionType, BalanceType balanceType,
			boolean isExpenseOnly, String recipientId, String reasonId, Criteria criteria) {
		initAccountsManagerPaymentCriteria(fromDate, toDate, accountNo, transactionType, balanceType, isExpenseOnly, recipientId, reasonId, criteria, null,
				null);
	}

	private void initAccountsManagerPaymentCriteria(Date fromDate, Date toDate, String accountNo, TransactionType transactionType, BalanceType balanceType,
			boolean isExpenseOnly, String recipientId, String reasonId, Criteria criteria, String projectId, String subCategoryId) {
		if (balanceType != null) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, balanceType.name()));
		}
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));

		if (fromDate != null && toDate != null) {
			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
		}

		if (StringUtils.isNotBlank(accountNo)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
		}

		if (transactionType != null) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_TYPE, transactionType.name()));
		}

		if (isExpenseOnly) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.EXPENSE.name()));
		}

		if (StringUtils.isNotBlank(recipientId)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_RECEPIENT_ID, recipientId));
		}

		if (StringUtils.isNotBlank(reasonId)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_REASON_ID, reasonId));
		}

		if (StringUtils.isNotBlank(projectId)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_PROJECT_ID, projectId));
		}

		if (StringUtils.isNotBlank(subCategoryId)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_SUB_REASON_ID, subCategoryId));
		}
	}

	public double findTotalExpanses(Date fromDate, Date toDate, String accountNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			initAccountsManagerPaymentCriteria(fromDate, toDate, accountNo, null, BalanceType.ACCOUNTS_MANAGER_BALANCE, true, null, null, criteria);

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				return ((Number) uniqueResult).doubleValue();
			}
		}
		return 0;
	}

	private BalanceUpdateTransaction createForwardBalanceByTransction(String accountsManagerNumber, BalanceUpdateTransaction balanceUpdateTransaction) {
		BalanceForwardTransaction forwardTransaction = new BalanceForwardTransaction();
		if (balanceUpdateTransaction != null) {
			forwardTransaction.setAmount(balanceUpdateTransaction.getBalanceBefore());
			forwardTransaction.setPaymentTypeString(balanceUpdateTransaction.getPaymentTypeString());
			forwardTransaction.setTransactionType(balanceUpdateTransaction.getTransactionType());
			forwardTransaction.setAccountNumber(accountsManagerNumber);
			forwardTransaction.setOutletId(balanceUpdateTransaction.getOutletId());
			//forwardTransaction.setFromCashdrawerId(balanceUpdateTransaction.getFromCashdrawerId());
		}
		forwardTransaction.setTicketId("Balance forward");
		return forwardTransaction;
	}

	private BalanceUpdateTransaction createEndBalanceByTransction(String accountsManagerNumber, double totalAmount) {
		EndBalanceTransaction endBalanceTransaction = new EndBalanceTransaction();
		endBalanceTransaction.setAccountNumber(accountsManagerNumber);
		endBalanceTransaction.setAmount(totalAmount);
		endBalanceTransaction.setTicketId("End Balance");
		return endBalanceTransaction;
	}

	//	public List<BalanceUpdateTransaction> findPaymentsBy(Date startTime, Date endTime, Customer customer, boolean showAgentPayment) {
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
	//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
	//
	//			if (customer != null) {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, customer.getId()));
	//			}
	//
	//			if (showAgentPayment) {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
	//			}
	//			else {
	//				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.LAB_DOCTOR_FEE_PAYMENT.name()));
	//			}
	//			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(startTime), DateUtil.endOfDay(endTime)));
	//			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
	//			return criteria.list();
	//		}
	//	}

	public BalanceUpdateTransaction saveAccountsManagerAddBalanceTransaction(User accountsManager, Double balanceToAdd, String note) {
		Transaction tx = null;
		BalanceUpdateTransaction transaction;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			transaction = populateAccountsManagerTransaction(null, accountsManager, balanceToAdd, note, session);
			tx.commit();

		}

		LedgerEntryDAO.getInstance().saveCashDepositLedgerEntry(transaction, false, false);

		return transaction;
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Session session) {
		return populateAccountsManagerTransaction(cashDrawer, accountsManager, cashDrawer.getDrawerAccountable(), null, session);
	}

	public BalanceUpdateTransaction populateAccountsManagerTransaction(CashDrawer cashDrawer, User accountsManager, Double balanceToAdd, String note,
			Session session) {
		if (accountsManager == null) {
			return null;
		}
		UserDAO userDao = UserDAO.getInstance();
		userDao.refresh(accountsManager, session);

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT,
				balanceToAdd);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());

		balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
		if (cashDrawer != null) {
			balanceUpdateTransaction.setFromCashdrawerId(cashDrawer.getId());
		}
		if (StringUtils.isNotBlank(note)) {
			balanceUpdateTransaction.setDescription(note);
		}

		Double balanceBefore = accountsManager.getAccountsManagerAmount();
		accountsManager.putAccountsManagerAmount(NumberUtil.round(balanceBefore + balanceToAdd));

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);

		userDao.saveOrUpdate(accountsManager, session);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public void saveAccountsManagerExpenses(String batchNo, Date transactionTime, User accountsManager, User currentUser, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note) throws Exception {
		saveAccountsManagerExpenses(batchNo, transactionTime, accountsManager, currentUser, project, recepient, reason, subReason, expensesAmount, null, null,
				note);
	}

	public void saveAccountsManagerExpenses(String batchNo, Date transactionTime, User accountsManager, User currentUser, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String memoNo, String memoImageId, String note)
			throws Exception {
		saveAccountsManagerExpenses(batchNo, transactionTime, accountsManager, currentUser, project, recepient, reason, subReason, expensesAmount, memoNo,
				memoImageId, note, null);
	}

	public void saveAccountsManagerExpenses(String batchNo, Date transactionTime, User accountsManager, User currentUser, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String memoNo, String memoImageId, String note,
			String requisitionId) throws Exception {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String projectId = project == null ? null : project.getId();
			BalanceUpdateTransaction balanceUpdateTransaction = saveAccountsManagerExpensesTransaction(transactionTime, accountsManager, batchNo, projectId,
					recepient, reason, subReason, expensesAmount, note, session);
			balanceUpdateTransaction.putMemoNo(memoNo);
			balanceUpdateTransaction.putMemoImageId(memoImageId);
			balanceUpdateTransaction.putRequisitionId(requisitionId);

			ExpenseTransaction expenseTransaction = ExpenseTransactionDAO.getInstance().saveExpensesTransaction(currentUser, transactionTime, accountsManager,
					batchNo, project, recepient, reason, subReason, expensesAmount, note, balanceUpdateTransaction, session);

			expenseTransaction.setRequisitionId(requisitionId);
			balanceUpdateTransaction.putReferenceId(expenseTransaction.getId());
			update(balanceUpdateTransaction, session);
			tx.commit();

			LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction, true);

		}
	}

	public BalanceUpdateTransaction saveAccountsManagerExpensesTransaction(Date eventTime, User accountsManager, String batchNo, String projectId,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, Session session) {
		BalanceUpdateTransaction balanceUpdateTransaction;
		UserDAO userInstance = UserDAO.getInstance();
		userInstance.refresh(accountsManager, session);

		BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
		balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, expensesAmount);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.EXPENSE.name());
		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
		balanceUpdateTransaction.setBatchNo(batchNo);
		if (StringUtils.isNotBlank(projectId)) {
			balanceUpdateTransaction.setProjectId(projectId);
		}
		balanceUpdateTransaction.setRecepientId(recepient == null ? "" : recepient.getId());
		balanceUpdateTransaction.setReasonId(reason == null ? "" : reason.getId());
		if (subReason != null) {
			balanceUpdateTransaction.setSubReasonId(subReason.getId());
		}
		balanceUpdateTransaction.setDescription(note);

		Double balanceBefore = accountsManager.getAccountsManagerAmount();
		accountsManager.putAccountsManagerAmount(balanceBefore - expensesAmount);
		userInstance.saveOrUpdate(accountsManager, session);

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		save(balanceUpdateTransaction, session);
		return balanceUpdateTransaction;
	}

	public void saveStoreExpenses(String batchNo, Date transactionTime, Store store, User currentUser, Project project, PayoutRecepient recepient,
			PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note) throws Exception {

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			BalanceUpdateTransaction balanceUpdateTransaction = saveStoreExpensesTransaction(transactionTime, store, batchNo, project, recepient, reason,
					subReason, expensesAmount, note, session);

			ExpenseTransaction expenseTransaction = ExpenseTransactionDAO.getInstance().saveStoreExpensesTransaction(currentUser, transactionTime, store,
					batchNo, project, recepient, reason, subReason, expensesAmount, note, balanceUpdateTransaction, session);

			balanceUpdateTransaction.putReferenceId(expenseTransaction.getId());
			update(balanceUpdateTransaction, session);

			tx.commit();

			LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction, false);
		}
	}

	public BalanceUpdateTransaction saveStoreExpensesTransaction(Date eventTime, Store store, String batchNo, Project project, PayoutRecepient recepient,
			PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note, Session session) {
		BalanceUpdateTransaction balanceUpdateTransaction;
		BalanceType balanceType = BalanceType.STORE_BALANCE;
		balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, expensesAmount);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.EXPENSE.name());
		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		balanceUpdateTransaction.setAccountNumber(store.getId());
		balanceUpdateTransaction.setBatchNo(batchNo);
		if (project != null) {
			balanceUpdateTransaction.setProjectId(project.getId());
		}
		balanceUpdateTransaction.setRecepientId(recepient == null ? "" : recepient.getId());
		balanceUpdateTransaction.setReasonId(reason == null ? "" : reason.getId());
		if (subReason != null) {
			balanceUpdateTransaction.setSubReasonId(subReason.getId());
		}
		balanceUpdateTransaction.setDescription(note);

		Double storeCashBeforeBalance = store.getCashBalance();
		double storeCashBalance = NumberUtil.round(storeCashBeforeBalance - expensesAmount);
		updateStoreCashBalance(session, storeCashBalance);

		balanceUpdateTransaction.setBalanceBefore(storeCashBeforeBalance);
		save(balanceUpdateTransaction, session);
		return balanceUpdateTransaction;
	}

	public BalanceUpdateTransaction saveCashDrawerToStoreTransaction(CashDrawer cashDrawer, Double balanceToAdd, Session session) throws Exception {

		Store store = DataProvider.get().getStore();
		Double storeCashBeforeBalance = store.getCashBalance();

		double storeCashBalance = NumberUtil.round(storeCashBeforeBalance + balanceToAdd);
		updateStoreCashBalance(session, storeCashBalance);

		BalanceUpdateTransaction depositToStore = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToAdd);
		depositToStore.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());
		depositToStore.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
		depositToStore.setReasonId(DefaultDataInserter.addOrGetStoreReason(true).getId());
		depositToStore.setBalanceBefore(storeCashBeforeBalance);
		depositToStore.setAccountNumber(cashDrawer.getId());
		if (cashDrawer != null) {
			depositToStore.setFromCashdrawerId(cashDrawer.getId());
		}
		depositToStore.setDescription("Add balance to store from drawer");
		save(depositToStore, session);

		return depositToStore;
	}

	public void saveAccManagerToStoreTransaction(User accountsManager, Double balanceToTransfer, String note) throws Exception {

		if (accountsManager == null) {
			throw new PosException("Please select a accounts manager");
		}

		BalanceUpdateTransaction balanceUpdateTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			GenericDAO genericDAO = GenericDAO.getInstance();
			genericDAO.refresh(accountsManager, session);
			Double balanceBefore = accountsManager.getAccountsManagerAmount();

			Store store = DataProvider.get().getStore();
			Double storeCashBeforeBalance = store.getCashBalance();

			BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			balanceUpdateTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToTransfer);
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());

			balanceUpdateTransaction.setAccountNumber(accountsManager.getId());
			balanceUpdateTransaction.setDescription(note);

			accountsManager.putAccountsManagerAmount(NumberUtil.round(balanceBefore - balanceToTransfer));
			genericDAO.saveOrUpdate(accountsManager, session);

			double storeCashBalance = NumberUtil.round(storeCashBeforeBalance + balanceToTransfer);
			updateStoreCashBalance(session, storeCashBalance);

			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
			save(balanceUpdateTransaction, session);

			BalanceUpdateTransaction depositToStore = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToTransfer);
			depositToStore.setTransactionSubType(BalanceSubType.TRANSFER_IN.name());
			depositToStore.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
			depositToStore.setReasonId(DefaultDataInserter.addOrGetStoreReason(true).getId());
			depositToStore.setBalanceBefore(storeCashBeforeBalance);
			depositToStore.setAccountNumber(accountsManager.getId());
			depositToStore.setDescription("Add balance to store");
			depositToStore.setAccountProcessed(true);
			save(depositToStore, session);

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveAccManagerToStoreLedgerEntry(balanceUpdateTransaction);
	}

	private void updateStoreCashBalance(Session session, double storeCashBalance) {
		Query query = session.createQuery("update " + Store.REF + " set " + Store.PROP_CASH_BALANCE + " = " + storeCashBalance); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		query.executeUpdate();
	}

	public void saveWithdrawBalanceFromStoreTransaction(Double balanceToWithdraw, String note) throws Exception {

		BalanceUpdateTransaction balanceUpdateTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			Store store = DataProvider.get().getStore();
			Double storeCashBeforeBalance = store.getCashBalance();

			balanceUpdateTransaction = BalanceType.STORE_BALANCE.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToWithdraw);
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.WITHDRAW.name());
			balanceUpdateTransaction.setRecepientId(DefaultDataInserter.addOrGetStoreRecepient().getId());
			balanceUpdateTransaction.setReasonId(DefaultDataInserter.addOrGetStoreReason(false).getId());
			balanceUpdateTransaction.setBalanceBefore(storeCashBeforeBalance);
			balanceUpdateTransaction.setDescription(note);
			save(balanceUpdateTransaction, session);

			double storeCashBalance = NumberUtil.round(storeCashBeforeBalance - balanceToWithdraw);
			updateStoreCashBalance(session, storeCashBalance);

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveWithdrawBalanceFromStoreLedgerEntry(balanceUpdateTransaction);
	}

	public void saveBeginCashTransaction(CashDrawer cashDrawer) throws Exception {
		Double beginCashAmount = cashDrawer.getBeginCash();
		if (beginCashAmount <= 0) {
			return;
		}

		BalanceUpdateTransaction balanceUpdateTransaction;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			balanceUpdateTransaction = BalanceType.CASH_DRAWER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, beginCashAmount);
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.BEGIN_CASH_DEPOSIT.name());
			balanceUpdateTransaction.setAccountNumber(cashDrawer.getId());
			balanceUpdateTransaction.setDescription("Cash drawer begin cash");
			save(balanceUpdateTransaction, session);

			tx.commit();
		}

		LedgerEntryDAO.getInstance().saveBeginCashLedgerEntry(balanceUpdateTransaction);
	}

	public boolean isPresent(String batchNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BATCH_NO, batchNo));
			criteria.setMaxResults(1);
			List list = criteria.list();
			return list != null && !list.isEmpty();
		}
	}

	public BalanceUpdateTransaction saveSalaryExpenseTransaction(Session session, Date eventTime, String batchNo, String projectId, Object paymentMethod, String paymentRef,
			User accountsManager, User paidToUser, double expensesAmount, String note, Boolean isAMDepositMethod, PosTransaction salaryTransaction) throws Exception {
		String bankOrCustomPaymentId = getPaymentMethodId(paymentMethod);
		PaymentType paymentType = getPaymentType(paymentMethod);

		if (paymentMethod instanceof BankAccount) {
			return saveBankAccountExpenseTransaction(session, eventTime, paymentType, bankOrCustomPaymentId, note, batchNo, paidToUser, expensesAmount, salaryTransaction);
		}

		Store store = DataProvider.get().getStore();
		BalanceType balanceType;
		Double balanceBefore;

		if (isAMDepositMethod) {
			balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			balanceBefore = accountsManager.getAccountsManagerAmount();
		}
		else {
			balanceType = BalanceType.STORE_BALANCE;
			balanceBefore = store.getCashBalance();
		}

		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(paymentType, TransactionType.DEBIT, expensesAmount);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.EXPENSE.name());
		balanceUpdateTransaction.putSourceType(salaryTransaction.REF);
		balanceUpdateTransaction.putSourceId(salaryTransaction.getId());
		if (salaryTransaction instanceof SalaryTransaction) {
			balanceUpdateTransaction.putTransactionReason("Salary payment");
		}
		else if (salaryTransaction instanceof SalaryAdvanceTransaction) {
			balanceUpdateTransaction.putTransactionReason("Salary advance payment");
		}
		
		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		balanceUpdateTransaction.setBatchNo(batchNo);
		if (StringUtils.isNotBlank(projectId)) {
			balanceUpdateTransaction.setProjectId(projectId);
		}

		balanceUpdateTransaction.setDescription(note);

		double cashBalance = NumberUtil.round(balanceBefore - expensesAmount);

		if (isAMDepositMethod) {
			balanceUpdateTransaction.setAccountNumber(accountsManager.getId());

			accountsManager.putAccountsManagerAmount(cashBalance);
			UserDAO.getInstance().saveOrUpdate(accountsManager, session);
		}
		else {
			balanceUpdateTransaction.setAccountNumber(store.getId());

			updateStoreCashBalance(session, cashBalance);
		}

		if (paidToUser != null) {
			balanceUpdateTransaction.putSalaryPaidTo(paidToUser.getId());
		}

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public BalanceUpdateTransaction saveBankAccountExpenseTransaction(Session session, Date eventTime, PaymentType paymentType, String bankAccountId, String note,
			String batchNo, User paidToUser, double expenseAmount, PosTransaction salaryTransaction) throws Exception {
		BankAccount bankAccount = loadBankAccount(bankAccountId, session);

		Double balanceBefore = bankAccount.getBalance();

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.BANK_ACCOUNT_BALANCE.createTransaction(paymentType, TransactionType.DEBIT,
				expenseAmount);
		balanceUpdateTransaction.setTransactionSubType(BalanceSubType.EXPENSE.name());

		balanceUpdateTransaction.setAccountNumber(bankAccountId);
		balanceUpdateTransaction.putBankAccountId(bankAccountId);
		balanceUpdateTransaction.setDescription(note);
		balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		balanceUpdateTransaction.setAccountProcessed(true);
		balanceUpdateTransaction.setBatchNo(batchNo);
		balanceUpdateTransaction.putSourceType(salaryTransaction.REF);
		balanceUpdateTransaction.putSourceId(salaryTransaction.getId());

		if (eventTime != null) {
			balanceUpdateTransaction.setEventTime(eventTime);
		}

		if (paidToUser != null) {
			balanceUpdateTransaction.putSalaryPaidTo(paidToUser.getId());
		}

		bankAccount.setBalance(NumberUtil.round(balanceBefore - expenseAmount));

		save(balanceUpdateTransaction, session);
		BankAccountDAO.getInstance().saveOrUpdate(bankAccount, session);

		return balanceUpdateTransaction;
	}

	private BankAccount loadBankAccount(String bankAccountId, Session session) {
		if (StringUtils.isBlank(bankAccountId)) {
			throw new PosException("Please select a bank account");
		}

		BankAccount bankAccount = BankAccountDAO.getInstance().get(bankAccountId, session);
		if (bankAccount == null) {
			throw new PosException("Could not find bank account");
		}
		return bankAccount;
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, PaymentType paymentType, Double balanceToAdd, String note,
			String customPaymentInfo) {
		Transaction tx = null;
		BalanceUpdateTransaction transaction;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			transaction = saveBankAccountAddBalanceTransaction(bankAccount, paymentType, balanceToAdd, note, session, customPaymentInfo);
			tx.commit();
		}
		return transaction;
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, Double balanceToAdd, String note, Session session) {
		return saveBankAccountAddBalanceTransaction(bankAccount, PaymentType.CASH, balanceToAdd, note, session);
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, PaymentType paymentType, Double balanceToAdd, String note,
			Session session) {
		return saveBankAccountAddBalanceTransaction(bankAccount, paymentType, balanceToAdd, note, session, null);
	}

	public BalanceUpdateTransaction saveBankAccountAddBalanceTransaction(BankAccount bankAccount, PaymentType paymentType, Double balanceToAdd, String note,
			Session session, String customPaymentInfo) {
		return saveBankAccountTransaction(bankAccount, paymentType, null, balanceToAdd, note, session, customPaymentInfo);
	}

	public BalanceUpdateTransaction saveBankAccountTransaction(BankAccount bankAccount, PaymentType paymentType, PosTransaction posTransaction,
			Double balanceToAdd, String note, Session session, String customPaymentInfo) {
		if (bankAccount == null) {
			return null;
		}
		BankAccountDAO bankAccountDAO = BankAccountDAO.getInstance();
		bankAccountDAO.refresh(bankAccount, session);

		TransactionType transactionType = TransactionType.CREDIT;
		if (posTransaction != null) {
			TransactionType type = TransactionType.valueOf(posTransaction.getTransactionType());
			if (posTransaction.isVoided()) {
				if (type == TransactionType.CREDIT) {
					type = TransactionType.DEBIT;
				}
				else if (type == TransactionType.DEBIT) {
					type = TransactionType.CREDIT;
				}
				else if (type == TransactionType.IN) {
					type = TransactionType.OUT;
				}
				else if (type == TransactionType.OUT) {
					type = TransactionType.IN;
				}
			}
			transactionType = type;
		}

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.BANK_ACCOUNT_BALANCE.createTransaction(paymentType, transactionType, balanceToAdd);

		balanceUpdateTransaction.setAccountNumber(bankAccount.getId());

		if (posTransaction != null) {
			balanceUpdateTransaction.setTicketId(posTransaction.getTicketId());
			balanceUpdateTransaction.setTransactionId(posTransaction.getId());
			balanceUpdateTransaction.setProjectId(posTransaction.getProjectId());
			balanceUpdateTransaction.putEntityNo(posTransaction.getEntityNo());
			balanceUpdateTransaction.putEntityId(posTransaction.getEntityId());
			balanceUpdateTransaction.putEntityType(posTransaction.getEntityType());

			//TransactionType transactionType = TransactionType.valueOf(posTransaction.getTransactionType());
			if (posTransaction.isVoided()) {
				balanceUpdateTransaction.setTransactionSubType(BalanceSubType.WITHDRAW.name());
				balanceUpdateTransaction.putTransactionReason("Payment void");
			}
			else if (transactionType == TransactionType.CREDIT || transactionType == TransactionType.IN) {
				balanceUpdateTransaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());
				balanceUpdateTransaction.putTransactionReason("Payment receive");
			}
			else if (transactionType == TransactionType.DEBIT || transactionType == TransactionType.OUT) {
				balanceUpdateTransaction.setTransactionSubType(BalanceSubType.EXPENSE.name());
				balanceUpdateTransaction.putTransactionReason("Payment deposit");
			}
		}
		else {
			balanceUpdateTransaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());
			balanceUpdateTransaction.putTransactionReason("Add balance");
		}

		if (StringUtils.isNotBlank(note)) {
			balanceUpdateTransaction.setDescription(note);
		}

		Double balanceBefore = bankAccount.getBalance();
		bankAccount.setBalance(NumberUtil.round(balanceBefore + balanceUpdateTransaction.getAmount()));

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);

		if (StringUtils.isNotBlank(customPaymentInfo) && (posTransaction == null
				|| Arrays.asList(TransactionType.CREDIT.name(), TransactionType.IN.name()).contains(posTransaction.getTransactionType()))) {
			balanceUpdateTransaction.putCustomePaymentInfo(customPaymentInfo);
		}

		bankAccountDAO.saveOrUpdate(bankAccount, session);
		balanceUpdateTransaction.setAccountProcessed(true);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	private PaymentType getPaymentType(Object paymentMethod) {
		if (paymentMethod instanceof String) {
			return PaymentType.CASH;
		}
		if (paymentMethod instanceof CustomPayment) {
			return PaymentType.CUSTOM_PAYMENT;
		}
		if (paymentMethod instanceof BankAccount) {
			return PaymentType.BANK_ACCOUNT;
		}
		return PaymentType.CASH;
	}

	public String getPaymentMethodId(Object paymentMethod) {
		if (paymentMethod instanceof CustomPayment) {
			return ((CustomPayment) paymentMethod).getId();
		}
		if (paymentMethod instanceof BankAccount) {
			return ((BankAccount) paymentMethod).getId();
		}

		return null;
	}

	public List<BalanceUpdateTransaction> findBankAccountTransaction(Outlet outlet, Date from, Date to, String bankAccountId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			if (from != null && to != null) {
				criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_EVENT_TIME, from, to));

			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outlet.getId()));
			}
			criteria.add(Restrictions.eqOrIsNull(BalanceUpdateTransaction.PROP_VOIDED, false));

			if (StringUtils.isNotBlank(bankAccountId)) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, bankAccountId));
			}

			criteria.addOrder(Order.asc(BalanceUpdateTransaction.PROP_EVENT_TIME));

			List<BalanceUpdateTransaction> list = criteria.list();

			List<BalanceUpdateTransaction> allTransactionList = new ArrayList<BalanceUpdateTransaction>();

			for (int i = 0; i < list.size(); i++) {
				BalanceUpdateTransaction balanceUpdateTransaction = list.get(i);
				if (i == 0) {
					allTransactionList.add(createForwardBalanceByTransction(bankAccountId, balanceUpdateTransaction));
				}
				allTransactionList.add(balanceUpdateTransaction);
			}

			return allTransactionList;
		}
	}

	public BalanceUpdateTransaction createAccountsManagerTransaction(CashDrawer cashDrawer, User fromAccountsManager, User toAccountsManager,
			BalanceSubType balanceSubType, TransactionType transactionType, Double balanceToAdd, String note, Session session) {
		if (fromAccountsManager == null) {
			return null;
		}
		UserDAO userDao = UserDAO.getInstance();
		userDao.refresh(fromAccountsManager, session);

		BalanceUpdateTransaction balanceUpdateTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(PaymentType.CASH, transactionType,
				balanceToAdd);
		balanceUpdateTransaction.setTransactionSubType(balanceSubType.name());

		balanceUpdateTransaction.setAccountNumber(fromAccountsManager.getId());
		if (cashDrawer != null) {
			balanceUpdateTransaction.setFromCashdrawerId(cashDrawer.getId());
		}
		if (fromAccountsManager != null) {
			//balanceUpdateTransaction.set
		}
		if (StringUtils.isNotBlank(note)) {
			balanceUpdateTransaction.setDescription(note);
		}

		Double balanceBefore = fromAccountsManager.getAccountsManagerAmount();
		fromAccountsManager.putAccountsManagerAmount(NumberUtil.round(balanceBefore + balanceToAdd));

		balanceUpdateTransaction.setBalanceBefore(balanceBefore);

		userDao.saveOrUpdate(fromAccountsManager, session);
		save(balanceUpdateTransaction, session);

		return balanceUpdateTransaction;
	}

	public void transferBalanceToAccountsManager(User fromAccountsManager, User toAccountsManager, Double balanceToTransfer, String note) throws Exception {
		transferBalanceToAccountsManager(fromAccountsManager, toAccountsManager, balanceToTransfer, note, null);
	}

	public void transferBalanceToAccountsManager(User fromAccountsManager, User toAccountsManager, Double balanceToTransfer, String note, String requisitionId)
			throws Exception {

		if (fromAccountsManager == null) {
			throw new PosException("Please select accounts manager from whom balance to transfer");
		}
		if (toAccountsManager == null) {
			throw new PosException("Please select accounts manager to whom balance to transfer");
		}

		BalanceUpdateTransaction fromAccountTransaction = null;
		BalanceUpdateTransaction toAccountTransaction = null;

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			GenericDAO genericDAO = GenericDAO.getInstance();
			genericDAO.refresh(fromAccountsManager, session);
			genericDAO.refresh(toAccountsManager, session);

			Double fromAccountBalanceBefore = fromAccountsManager.getAccountsManagerAmount();
			Double toAccountBalanceBefore = toAccountsManager.getAccountsManagerAmount();

			BalanceType balanceType = BalanceType.ACCOUNTS_MANAGER_BALANCE;
			fromAccountTransaction = balanceType.createTransaction(PaymentType.CASH, TransactionType.DEBIT, balanceToTransfer);
			fromAccountTransaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());
			fromAccountTransaction.setAccountNumber(fromAccountsManager.getId());
			fromAccountTransaction.setDescription("Transfer to acm " + toAccountsManager.getFullName() + "\n" + note);

			genericDAO.saveOrUpdate(fromAccountsManager, session);

			double toAccountBalance = NumberUtil.round(toAccountBalanceBefore + balanceToTransfer);
			//updateStoreCashBalance(session, storeCashBalance);

			fromAccountTransaction.setBalanceBefore(fromAccountBalanceBefore);
			fromAccountTransaction.putBalanceTransferInfo(fromAccountsManager, toAccountsManager, balanceToTransfer);
			fromAccountTransaction.setAccountProcessed(true);
			fromAccountTransaction.putRequisitionId(requisitionId);
			save(fromAccountTransaction, session);

			toAccountTransaction = BalanceType.ACCOUNTS_MANAGER_BALANCE.createTransaction(PaymentType.CASH, TransactionType.CREDIT, balanceToTransfer);
			toAccountTransaction.setTransactionSubType(BalanceSubType.TRANSFER_IN.name());
			toAccountTransaction.setBalanceBefore(toAccountBalanceBefore);
			toAccountTransaction.setAccountNumber(toAccountsManager.getId());
			toAccountTransaction.setDescription("Transfer from acm " + fromAccountsManager.getFullName());
			toAccountTransaction.putBalanceTransferInfo(fromAccountsManager, toAccountsManager, balanceToTransfer);
			toAccountTransaction.setAccountProcessed(true);
			toAccountTransaction.setRecepientId(requisitionId);
			save(toAccountTransaction, session);

			fromAccountsManager.putAccountsManagerAmount(NumberUtil.round(fromAccountBalanceBefore - balanceToTransfer));
			UserDAO.getInstance().update(fromAccountsManager, session);

			toAccountsManager.putAccountsManagerAmount(toAccountBalance);
			UserDAO.getInstance().update(toAccountsManager, session);

			LedgerEntryDAO.getInstance().saveTransferToAcmLedgerEntry(fromAccountTransaction, toAccountTransaction, session);

			tx.commit();
		}
	}
}