package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.hibernate.type.DoubleType;
import org.hibernate.type.Type;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.COAAccountType;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.DirectionType;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.LedgerEntryType;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.SalaryAdvanceTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.report.model.BalanceSheetLedgerEntryModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class LedgerEntryDAO extends BaseLedgerEntryDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<LedgerEntry> findByChartOfAcountName(Date fromDate, Date toDate, String accountId, String orderId, String refNo) {
		Pagination<LedgerEntry> pagination = new Pagination<>(0, -1);
		findByChartOfAcountName(pagination, true, fromDate, toDate, accountId, orderId, refNo);
		return pagination.getDataList();
	}

	public void findByChartOfAcountName(PaginationSupport paginationSupport, Date fromDate, Date toDate, String accountId, String orderId, String refNo) {
		findByChartOfAcountName(paginationSupport, false, fromDate, toDate, accountId, orderId, refNo);
	}

	private void findByChartOfAcountName(PaginationSupport paginationSupport, boolean ignorePagination, Date fromDate, Date toDate, String accountId,
			String orderId, String refNo) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(LedgerEntry.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, orderId));
			}
			else {

				if (fromDate != null) {
					criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, fromDate));
				}
				if (toDate != null) {
					criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, toDate));
				}

				if (StringUtils.isNotBlank(accountId)) {
					criteria.add(Restrictions.eq(LedgerEntry.PROP_ACCOUNT_ID, accountId));
				}

				if (StringUtils.isNotBlank(refNo)) {
					criteria.add(Restrictions.ilike(LedgerEntry.PROP_GROUP_ID, refNo, MatchMode.ANYWHERE));
				}
			}

			criteria.setProjection(Projections.rowCount());

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(LedgerEntry.PROP_CREATE_DATE));
			criteria.addOrder(Order.asc(LedgerEntry.PROP_SORT_ORDER));
			if (!ignorePagination) {
				criteria.setFirstResult(paginationSupport.getCurrentRowIndex());
				criteria.setMaxResults(paginationSupport.getPageSize());
			}
			paginationSupport.setRows(criteria.list());
		}
	}

	//	public void saveTransactionLedgerEntry(Ticket ticket, PosTransaction transaction) {
	//		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
	//		try (Session session = instance.createNewSession()) {
	//			Transaction tx = session.beginTransaction();
	//
	//			ChartOfAccounts spendMethodCOA = getSpendMethodCOA(transaction.getPaymentType(), instance, session);
	//
	//			ChartOfAccounts receivableChartOfAccounts = instance.getCOAFromMap("5000", session);
	//			Date now = StoreDAO.getServerTimestamp();
	//			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("TRANS");
	//			Double transAmount = transaction.isVoided() ? (-1) * transaction.getAmount() : transaction.getAmount();
	//			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, spendMethodCOA, DirectionType.DEBIT, transAmount,
	//					"TRANS/" + transaction.getId(), now);
	//			createDebitLedgerEntry.setTransactionId(transaction.getId());
	//			createDebitLedgerEntry.setGroupId(groupId);
	//			createDebitLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
	//			createDebitLedgerEntry.setSortOrder(1);
	//			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
	//
	//			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableChartOfAccounts, DirectionType.CREDIT, transAmount,
	//					"TRANS/" + transaction.getId(), now);
	//			createCreditLedgerEntry.setTransactionId(transaction.getId());
	//			createCreditLedgerEntry.setGroupId(groupId);
	//			createCreditLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
	//			createCreditLedgerEntry.setSortOrder(2);
	//			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);
	//
	//			tx.commit();
	//		} catch (Exception e) {
	//			PosLog.error(getClass(), e);
	//		}
	//	}

	public void savePurchaseOrderLedgerEntry(PurchaseOrder purchaseOrder) {
		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();
			savePurchaseOrderLedgerEntry(purchaseOrder, session);
			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void savePurchaseOrderLedgerEntry(PurchaseOrder purchaseOrder, Session session) {
		DataProvider dataProvider = DataProvider.get();
		ChartOfAccounts accountPayableCOA = dataProvider.getCOAFromMap("800", session);
		ChartOfAccounts inventoryCOA = dataProvider.getCOAFromMap("630", session);
		Double purchaseAmount = purchaseOrder.getTotalAmount();
		if (!NumberUtil.isZero(purchaseAmount)) {
			String purchaseGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("PURCHASE");
			String orderId = purchaseOrder.getOrderId();
			String entityId = purchaseOrder.getEntityId();
			Date now = StoreDAO.getServerTimestamp();

			LedgerEntry createCRPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, accountPayableCOA, DirectionType.CREDIT, purchaseAmount,
					"PURCHASE/" + orderId, now);
			createCRPayableLedgerEntry.setGroupId(purchaseGroupId);
			createCRPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
			createCRPayableLedgerEntry.setSortOrder(1);
			createCRPayableLedgerEntry.setPurchaseOrderId(entityId);
			LedgerEntryDAO.getInstance().save(createCRPayableLedgerEntry, session);

			if (!NumberUtil.isZero(purchaseAmount)) {
				LedgerEntry createInvLedgerEntry = LedgerEntry.createLedgerEntry(null, inventoryCOA, DirectionType.DEBIT, purchaseAmount, "PURCHASE/" + orderId,
						now);
				createInvLedgerEntry.setGroupId(purchaseGroupId);
				createInvLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
				createInvLedgerEntry.setSortOrder(2);
				createInvLedgerEntry.setPurchaseOrderId(entityId);
				LedgerEntryDAO.getInstance().save(createInvLedgerEntry, session);
			}

			if (purchaseOrder.isFullPaid()) {
				purchaseOrder.setAccountProcessed(true);
				PurchaseOrderDAO.getInstance().saveOrUpdate(purchaseOrder, session);
			}
		}
	}

	public void savePurchaseBillLedgerEntry(PurchaseOrder purchaseOrder, PosTransaction transaction) {

		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts accountPayableCOA = DataProvider.get().getCOAFromMap("800", session);
			ChartOfAccounts spendMethodCOA = getSpendMethodCOA(transaction.getPaymentType(), DataProvider.get(), session);

			String entityId = purchaseOrder.getEntityId();
			String orderId = purchaseOrder.getOrderId();
			String transactionId = transaction.getId();
			Date now = StoreDAO.getServerTimestamp();
			Double transAmount = transaction.getAmount();
			if (!NumberUtil.isZero(transAmount)) {
				String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("PURCHASE_BILL");
				LedgerEntry createDBTPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, accountPayableCOA, DirectionType.DEBIT, transAmount,
						"PURCHASE_BILL/" + orderId, now);
				createDBTPayableLedgerEntry.setGroupId(groupId);
				createDBTPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
				createDBTPayableLedgerEntry.setSortOrder(1);
				createDBTPayableLedgerEntry.setPurchaseOrderId(entityId);
				createDBTPayableLedgerEntry.setTransactionId(transactionId);
				LedgerEntryDAO.getInstance().save(createDBTPayableLedgerEntry, session);

				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(null, spendMethodCOA, DirectionType.CREDIT, transAmount,
						"PURCHASE_BILL/" + orderId, now);
				createCashLedgerEntry.setGroupId(groupId);
				createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
				createCashLedgerEntry.setSortOrder(2);
				createCashLedgerEntry.setPurchaseOrderId(entityId);
				createCashLedgerEntry.setTransactionId(transactionId);
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	//	public void saveOrderLedgerEntry(Ticket ticket, PosTransaction transaction) {
	//		
	//	}
	public void saveOrderLedgerEntry(Ticket ticket, Session session) {
		saveOrderLedgerEntry(ticket, null, session);
	}

	public void saveOrderLedgerEntry(Ticket ticket, PosTransaction transaction, Session session) {
		saveOrderLedgerEntry(ticket, transaction, session, null);
	}

	public void saveOrderLedgerEntry(Ticket ticket, PosTransaction transaction, Session session, Date selectedDate) {
		boolean isRecalculateTicket = selectedDate != null;
		double maxRFFee = ticket.getReferrerFeeOnNetSales();
		double totalCost = 0;
		List<TicketItem> ticketItems = ticket.getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				if (isRecalculateTicket) {
					maxRFFee += ticketItem.getReferrerFeeOnReport();
					if (ticketItem.isVoided()) {
						continue;
					}

					if (ticketItem.isReturned()) {
						totalCost -= ticketItem.getUnitCost();
					}
					else {
						totalCost += ticketItem.getUnitCost();
					}
				}
				else {
					maxRFFee += ticketItem.getReferrerFeeOnReport();
					if (ticketItem.isVoided() || ticketItem.isReturned() || ticketItem.isReturnedSource()) {
						continue;
					}
					totalCost += ticketItem.getUnitCost();
				}
			}
		}

		DataProvider dataProvider = DataProvider.get();
		//		try (Session session = instance.createNewSession()) {
		//			Transaction tx = session.beginTransaction();

		ChartOfAccounts productsSalesCOA = dataProvider.getCOAFromMap("200", session);
		ChartOfAccounts taxCOA = dataProvider.getCOAFromMap("831", session);
		ChartOfAccounts rfPayableCOA = dataProvider.getCOAFromMap("832", session);
		ChartOfAccounts ldfPayableCOA = dataProvider.getCOAFromMap("833", session, false);
		//			ChartOfAccounts taxCOA = instance.findByAcountCode("506", session);
		//			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("507", session);
		//			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
		ChartOfAccounts receivableCOA = dataProvider.getCOAFromMap("5000", session);
		//ChartOfAccounts ownerADrawings = instance.findByAcountCode("880", session);

		String ticketId = ticket.getId();
		double leProductSalesAmount = getLedgerEntryAmount(productsSalesCOA.getId(), ticketId);
		double leTaxAmount = getLedgerEntryAmount(taxCOA.getId(), ticketId);
		double leRFAmount = getLedgerEntryAmount(rfPayableCOA.getId(), ticketId);
		double leLDFAmount = 0;
		if (ldfPayableCOA != null) {
			leLDFAmount = getLedgerEntryAmount(ldfPayableCOA.getId(), ticketId);
		}

		Date now = StoreDAO.getServerTimestamp();
		int sortOrder = 0;
		double previousProcessAmount = leProductSalesAmount + leTaxAmount + leRFAmount + leLDFAmount;
		double remainLedgerAmount = ticket.getTotalAmount() + previousProcessAmount;
		//Need to update adj ledger entry time 
		if (isRecalculateTicket) {
			now = selectedDate;
		}
		double maxTaxAmount = ticket.getTaxAmount();
		double maxLabDoctorFee = ticket.getLabDoctorFee();
		double otherAmount = maxTaxAmount + maxRFFee + maxLabDoctorFee;
		double maxSalesAmount = ticket.getTotalAmount() - otherAmount;
		//remainLedgerAmount = remainLedgerAmount - otherAmount;
		//double maxSalesAmount = ticket.getTotalAmount();
		double remainProductSalesAmount = maxSalesAmount + leProductSalesAmount;
		double remainTaxAmount = maxTaxAmount + leTaxAmount;
		double remainRFAmount = maxRFFee + leRFAmount;
		double remainLDFAmount = maxLabDoctorFee + leLDFAmount;

		String orderGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("ORDER");

		if (!NumberUtil.isZero(remainProductSalesAmount)) {
			LedgerEntry createPSLedgerEntry = LedgerEntry.createLedgerEntry(ticket, productsSalesCOA, DirectionType.CREDIT, remainProductSalesAmount,
					"ORDER/" + ticket.getId(), now);
			createPSLedgerEntry.setGroupId(orderGroupId);
			createPSLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createPSLedgerEntry.setSortOrder(++sortOrder);
			save(createPSLedgerEntry);
		}

		if (!NumberUtil.isZero(remainTaxAmount)) {
			LedgerEntry createTaxLedgerEntry = LedgerEntry.createLedgerEntry(ticket, taxCOA, DirectionType.CREDIT, remainTaxAmount, "ORDER/" + ticket.getId(),
					now);
			createTaxLedgerEntry.setGroupId(orderGroupId);
			createTaxLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createTaxLedgerEntry.setSortOrder(++sortOrder);
			save(createTaxLedgerEntry);
		}

		if (!NumberUtil.isZero(remainRFAmount)) {
			LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.CREDIT, remainRFAmount,
					"ORDER/" + ticket.getId(), now);
			createRFLedgerEntry.setGroupId(orderGroupId);
			createRFLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createRFLedgerEntry.setSortOrder(++sortOrder);
			save(createRFLedgerEntry);
		}

		if (!NumberUtil.isZero(remainLDFAmount)) {
			LedgerEntry createLDFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.CREDIT, remainLDFAmount,
					"ORDER/" + ticket.getId(), now);
			createLDFLedgerEntry.setGroupId(orderGroupId);
			createLDFLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createLDFLedgerEntry.setSortOrder(++sortOrder);
			save(createLDFLedgerEntry);
		}

		if (!NumberUtil.isZero(remainLedgerAmount)) {
			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableCOA, DirectionType.DEBIT, remainLedgerAmount,
					"ORDER/" + ticket.getId(), now);
			createDebitLedgerEntry.setGroupId(orderGroupId);
			createDebitLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createDebitLedgerEntry.setSortOrder(++sortOrder);
			save(createDebitLedgerEntry);
		}

		saveInventoryAdjLedgerEntry(dataProvider, session, orderGroupId, sortOrder, now, ticket, totalCost);
		sortOrder = sortOrder + 2;
		//}

		if (ticket.isPaid()) {
			ticket.setAccountProcessed(true);
		}

		if (transaction != null) {
			ChartOfAccounts spendMethodCOA = getSpendMethodCOA(transaction.getPaymentType(), dataProvider, session);

			Double transAmount = transaction.getAmount();
			if (transaction instanceof RefundTransaction || transaction.isVoided()) {
				transAmount = (-1) * transAmount;
			}
			if (!NumberUtil.isZero(transAmount)) {
				String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("TRANS");

				LedgerEntry createReceivableTransLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableCOA, DirectionType.CREDIT, transAmount,
						"TRANS/" + transaction.getId(), now);
				createReceivableTransLedgerEntry.setTransactionId(transaction.getId());
				createReceivableTransLedgerEntry.setGroupId(groupId);
				createReceivableTransLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
				createReceivableTransLedgerEntry.setSortOrder(++sortOrder);
				createReceivableTransLedgerEntry.setCashDrawerId(transaction.getCashDrawerId());
				save(createReceivableTransLedgerEntry);

				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, spendMethodCOA, DirectionType.DEBIT, transAmount,
						"TRANS/" + transaction.getId(), now);
				createCashLedgerEntry.setTransactionId(transaction.getId());
				createCashLedgerEntry.setGroupId(groupId);
				createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
				createCashLedgerEntry.setSortOrder(++sortOrder);
				createCashLedgerEntry.setCashDrawerId(transaction.getCashDrawerId());
				save(createCashLedgerEntry);
			}

			Set<PosTransaction> transactions = ticket.getTransactions();
			for (PosTransaction posTransaction : transactions) {
				if (posTransaction.getId().equals(transaction.getId())) {
					posTransaction.setAccountProcessed(true);
				}
			}
		}

		if (!isRecalculateTicket) {
			TicketDAO.getInstance().saveOrUpdate(ticket, session);
		}

		//			tx.commit();
		//		} catch (Exception e) {
		//			PosLog.error(getClass(), e);
		//		}
	}

	private ChartOfAccounts getSpendMethodCOA(PaymentType paymentType, DataProvider dataProvider, Session session) {

		if (PaymentType.BANK_ACCOUNT == paymentType || PaymentType.DEBIT_CARD == paymentType || PaymentType.CREDIT_CARD == paymentType) {
			return dataProvider.getCOAFromMap("10200", session); //Bank - Regular Checking
		}

		if (PaymentType.CUSTOM_PAYMENT == paymentType) {
			return dataProvider.getCOAFromMap("10500", session); //Regular Checking
		}

		return dataProvider.getCOAFromMap("10100", session); //Cash - Regular Checking
	}

	public void saveInventoryAdjLedgerEntry(DataProvider dataProvider, Session session, String groupId, int sortOrder, Date now, Ticket ticket,
			double totalCost) {
		ChartOfAccounts costOfGoodsSoldCOA = dataProvider.getCOAFromMap("310", session);
		ChartOfAccounts inventoryCOA = dataProvider.getCOAFromMap("630", session);

		double leCostOfGoodsSoldAmount = getLedgerEntryAmount(costOfGoodsSoldCOA.getId(), ticket.getId());
		double remainCostOfGoodsSoldAmount = totalCost - leCostOfGoodsSoldAmount;
		if (!NumberUtil.isZero(remainCostOfGoodsSoldAmount)) {
			LedgerEntry createInventoryLedgerEntry = LedgerEntry.createLedgerEntry(ticket, inventoryCOA, DirectionType.CREDIT, remainCostOfGoodsSoldAmount,
					"ORDER/" + ticket.getId(), now);
			createInventoryLedgerEntry.setGroupId(groupId);
			createInventoryLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createInventoryLedgerEntry.setSortOrder(++sortOrder);
			save(createInventoryLedgerEntry);

			LedgerEntry createCostOfGoodsSoldLedgerEntry = LedgerEntry.createLedgerEntry(ticket, costOfGoodsSoldCOA, DirectionType.DEBIT,
					remainCostOfGoodsSoldAmount, "ORDER/" + ticket.getId(), now);
			createCostOfGoodsSoldLedgerEntry.setGroupId(groupId);
			createCostOfGoodsSoldLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createCostOfGoodsSoldLedgerEntry.setSortOrder(++sortOrder);
			save(createCostOfGoodsSoldLedgerEntry);
		}
	}

	public double getLedgerEntryAmount(String accountId, String ticketId) {
		try (Session session = createNewSession()) {
			return getLedgerEntryAmount(accountId, ticketId, session);
		}
	}

	public double getLedgerEntryAmount(String accountId, String ticketId, Session session) {

		Criteria criteria = session.createCriteria(LedgerEntry.class);

		criteria.add(Restrictions.eq(LedgerEntry.PROP_ACCOUNT_ID, accountId));
		criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, ticketId));

		criteria.add(Restrictions.eq(LedgerEntry.PROP_LEDGER_ENTRY_TYPE, LedgerEntryType.ORDER_ENTRY.name()));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + LedgerEntry.PROP_AMOUNT, new String[] { LedgerEntry.PROP_AMOUNT },
				new Type[] { new DoubleType() }));
		criteria.setProjection(projectionList);
		Number number = (Number) criteria.uniqueResult();

		if (number != null) {
			return number.doubleValue();
		}
		return 0;
	}

	public void saveRFPayLedgerEntry(Map<Ticket, Double> ticketsMap, String transactionId, Session session) {
		Set<Ticket> keySet = ticketsMap.keySet();
		Date now = StoreDAO.getServerTimestamp();
		int sortOrder = 0;
		for (Ticket ticket : keySet) {
			DataProvider dataProvider = DataProvider.get();

			ChartOfAccounts cashChartOfAccounts = dataProvider.getCOAFromMap("10100", session);
			//ChartOfAccounts rfPayableCOA = instance.getCOAFromMap("507", session);
			ChartOfAccounts rfPayableCOA = dataProvider.getCOAFromMap("832", session);

			double transAmount = ticketsMap.get(ticket);
			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("RF");

			LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transactionId, now);
			createCashLedgerEntry.setTransactionId(transactionId);
			createCashLedgerEntry.setGroupId(groupId);
			createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.RF_PAY.name());
			createCashLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

			LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.DEBIT, transAmount, "TRANS/" + transactionId,
					now);
			createRFLedgerEntry.setTransactionId(transactionId);
			createRFLedgerEntry.setGroupId(groupId);
			createRFLedgerEntry.setLedgerEntryType(LedgerEntryType.RF_PAY.name());
			createRFLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createRFLedgerEntry, session);
		}

	}

	public void saveLDFPayLedgerEntry(Map<Ticket, Double> ticketsMap, String transactionId, Session session) {
		Set<Ticket> keySet = ticketsMap.keySet();
		Date now = StoreDAO.getServerTimestamp();
		int sortOrder = 0;
		for (Ticket ticket : keySet) {
			DataProvider dataProvider = DataProvider.get();

			ChartOfAccounts cashChartOfAccounts = dataProvider.getCOAFromMap("10100", session);
			//ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
			ChartOfAccounts ldfPayableCOA = dataProvider.getCOAFromMap("833", session);

			double transAmount = ticketsMap.get(ticket);
			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("LDF");

			LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transactionId, now);
			createCashLedgerEntry.setTransactionId(transactionId);
			createCashLedgerEntry.setGroupId(groupId);
			createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.LDF_PAY.name());
			createCashLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

			LedgerEntry createLDFPayLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.DEBIT, transAmount,
					"TRANS/" + transactionId, now);
			createLDFPayLedgerEntry.setTransactionId(transactionId);
			createLDFPayLedgerEntry.setGroupId(groupId);
			createLDFPayLedgerEntry.setLedgerEntryType(LedgerEntryType.LDF_PAY.name());
			createLDFPayLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createLDFPayLedgerEntry, session);
		}
	}

	public List getLedgerEntries(List<String> coaAccountTypeGroupIds) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			if (!coaAccountTypeGroupIds.isEmpty()) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, coaAccountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(LedgerEntry.class);

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			return criteria.list();

		}
	}

	public BalanceSheetLedgerEntryModel getRetainedEarning(Date startDate, Date endDate) {
		try (Session session = createNewSession()) {
			List<String> coaAccountTypeGroupIds = Arrays.asList("revenue", "expense");

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			if (!coaAccountTypeGroupIds.isEmpty()) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, coaAccountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(LedgerEntry.class);
			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$
			BalanceSheetLedgerEntryModel retainedEarning = new BalanceSheetLedgerEntryModel();
			retainedEarning.setName("Retained earning");
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				retainedEarning.setAmount(number.doubleValue());
			}
			return retainedEarning;

		}
	}

	public List getSumOfLedgerEntries(Date startDate, Date endDate) {
		try (Session session = createNewSession()) {
			Map<String, BalanceSheetLedgerEntryModel> map = new HashMap<String, BalanceSheetLedgerEntryModel>();
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.property(LedgerEntry.PROP_DIRECTION), LedgerEntry.PROP_DIRECTION);
			projectionList.add(Projections.sqlProjection("sum(amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_DIRECTION));
			criteria.setProjection(projectionList);

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));

			List<BalanceSheetLedgerEntryModel> list = criteria.list();
			if (list != null) {

				for (BalanceSheetLedgerEntryModel entryModel : list) {
					String coaId = entryModel.getCoaId();
					if (StringUtils.isBlank(coaId)) {
						continue;
					}

					BalanceSheetLedgerEntryModel model = map.get(coaId);
					if (model == null) {
						model = new BalanceSheetLedgerEntryModel();
						model.setCoaId(coaId);
						model.setDirection(null);
						map.put(coaId, model);
					}

					//double amount = model.getAmount();
					double amount = entryModel.getAmount() * entryModel.getDirection();
					//Show data based on Ledger entry direction
					DirectionType type = DirectionType.getByTypeNo(entryModel.getDirection());
					if (DirectionType.DEBIT == type) {
						model.setDebitAmount(amount);
					}
					else {
						model.setCreditAmount(amount);
					}
					model.setAmount(model.getDebitAmount() + model.getCreditAmount());

					//					//Show data based on  COA > COAAccountType > Account Type Group direction
					//					ChartOfAccounts chartOfAccounts = ChartOfAccountsDAO.getInstance().get(coaId);
					//					if (chartOfAccounts != null) {
					//						COAAccountType coaAccountType = COAAccountTypeDAO.getInstance().get(chartOfAccounts.getCoaAccountTypeId());
					//						COAAccountTypeGroup group = COAAccountTypeGroupDAO.getInstance().get(coaAccountType.getCoaAccountTypeGroupId());
					//						DirectionType type = DirectionType.getByTypeNo(group.getAccountDirection());
					//						if (DirectionType.DEBIT == type) {
					//							model.setDebitAmount(amount);
					//						}
					//						else {
					//							model.setCreditAmount(amount * (-1));
					//						}
					//						model.setAmount(amount);
					//					}
				}
			}
			return map.values().stream().filter(t -> !NumberUtil.isZero(t.getAmount())).collect(Collectors.toList());
		}
	}

	public double getTaxLedgerEntryAmount(Date startDate, Date endDate) {
		return getLedgerEntriesAmountByTypeId("expense", "506", startDate, endDate);
	}

	public double getLedgerEntriesAmountByTypeId(String coaAccountTypeId, Date startDate, Date endDate) {
		return getLedgerEntriesAmountByTypeId(coaAccountTypeId, "", startDate, endDate);
	}

	private double getLedgerEntriesAmountByTypeId(String coaAccountTypeId, String coaCode, Date startDate, Date endDate) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			if (StringUtils.isNotBlank(coaAccountTypeId)) {
				coaDetachedCriteria.add(Restrictions.eq(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID, coaAccountTypeId));
			}
			if (StringUtils.isNotBlank(coaCode)) {
				coaDetachedCriteria.add(Restrictions.eq(ChartOfAccounts.PROP_ACCOUNT_CODE, coaCode));
			}
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}

		}
		return 0;
	}

	/**
	 * @param startDate
	 * @param endDate
	 *  find all account type IDs with group id "expense"
	 *  find all ChartOfAccounts IDs with COAAccountType IDs without "direct_costs" account type
	 * 
	 * @return Expense amount by account type without cost
	 */
	public List getExpenseLedgerEntries(Date startDate, Date endDate) {
		return getCustomLedgerEntries(startDate, endDate, Arrays.asList("expense"), "direct_costs", "506");
	}

	public List getCustomLedgerEntries(Date startDate, Date endDate, List<String> accountTypeGroupIds, String withoutAccountTypeId, String withoutAccountCode) {
		return getCustomLedgerEntries(startDate, endDate, accountTypeGroupIds, Arrays.asList(withoutAccountTypeId), withoutAccountCode);
	}

	public List getCustomLedgerEntries(Date startDate, Date endDate, List<String> accountTypeGroupIds, List<String> withoutAccountTypeIds,
			String withoutAccountCode) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));

			if (accountTypeGroupIds != null) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, accountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$
			if (!withoutAccountTypeIds.isEmpty()) {
				coaDetachedCriteria.add(Restrictions.not(Restrictions.in(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID, withoutAccountTypeIds)));
			}
			if (StringUtils.isNotBlank(withoutAccountCode)) {
				coaDetachedCriteria.add(Restrictions.ne(ChartOfAccounts.PROP_ACCOUNT_CODE, withoutAccountCode));
			}
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			return criteria.list();

		}
	}

	public List getLedgerEntriesByTypeId(String coaAccountTypeId, Date startDate, Date endDate) {
		return getLedgerEntriesByTypeId(Arrays.asList(coaAccountTypeId), startDate, endDate);
	}

	public List getLedgerEntriesByTypeId(List<String> coaAccountTypeId, Date startDate, Date endDate) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaAccountTypeId)); //$NON-NLS-1$
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			criteria.addOrder(Order.desc("coaId"));
			return criteria.list();

		}
	}

	public void savePayoutLedgerEntry(PayOutTransaction payOutTransaction) {
		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();

			DataProvider dataProvider = DataProvider.get();
			ChartOfAccounts cashExpenseCOA = null;
			if (payOutTransaction.isExpenses()) {
				cashExpenseCOA = dataProvider.getCOAFromMap("430", session);
			}
			else {
				cashExpenseCOA = dataProvider.getCOAFromMap("429", session);
			}

			ChartOfAccounts cashCOA = dataProvider.getCOAFromMap("10100", session);
			Date now = StoreDAO.getServerTimestamp();
			Double payoutAmount = payOutTransaction.getAmount();
			if (!NumberUtil.isZero(payoutAmount)) {
				boolean isExpenses = payOutTransaction.isExpenses();
				String payoutGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef(isExpenses ? "EXPENSES" : "PAYOUT");
				String transId = payOutTransaction.getId();
				String description = StringUtils.isBlank(payOutTransaction.getNote()) ? ((isExpenses ? "EXPENSES/" : "PAYOUT/") + transId)
						: payOutTransaction.getNote();
				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(null, cashCOA, DirectionType.CREDIT, payoutAmount, description, now);
				createCashLedgerEntry.setGroupId(payoutGroupId);
				createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
				createCashLedgerEntry.setTransactionId(transId);
				createCashLedgerEntry.setCashDrawerId(payOutTransaction.getCashDrawerId());
				createCashLedgerEntry.setPayoutRecepientId(payOutTransaction.getRecepientId());
				createCashLedgerEntry.setSortOrder(2);
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

				LedgerEntry createCRPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, cashExpenseCOA, DirectionType.DEBIT, payoutAmount, description,
						now);
				createCRPayableLedgerEntry.setGroupId(payoutGroupId);
				createCRPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
				createCRPayableLedgerEntry.setTransactionId(transId);
				createCRPayableLedgerEntry.setCashDrawerId(payOutTransaction.getCashDrawerId());
				createCRPayableLedgerEntry.setPayoutRecepientId(payOutTransaction.getRecepientId());
				createCRPayableLedgerEntry.setSortOrder(1);
				LedgerEntryDAO.getInstance().save(createCRPayableLedgerEntry, session);

				payOutTransaction.setAccountProcessed(true);
				PayOutTransactionDAO.getInstance().saveOrUpdate(payOutTransaction, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveOrUpdateLedgerEntryList(List<LedgerEntry> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<LedgerEntry> iterator = dataList.iterator(); iterator.hasNext();) {
				LedgerEntry item = (LedgerEntry) iterator.next();
				LedgerEntry existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getChartOfAccountDisplay() + " already updated"); //$NON-NLS-1$
						continue;
					}
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public List<LedgerEntry> getLedgerEntry(String ticketId, Session session) {
		Criteria criteria = session.createCriteria(LedgerEntry.class);
		criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, ticketId));
		return criteria.list();
	}

	public void saveExpensesLedgerEntry(BalanceUpdateTransaction balanceUpdateTransaction, boolean isFromAM) {
		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();

			DataProvider dataProvider = DataProvider.get();

			ChartOfAccounts cashBookExpenseCOA = dataProvider.getCOAFromMap("430", session);
			ChartOfAccounts depositMethodCOA;
			if (isFromAM) {
				depositMethodCOA = dataProvider.getCOAFromMap("10300", session);
			}
			else {
				depositMethodCOA = dataProvider.getCOAFromMap("10400", session);
			}

			Double payoutAmount = Math.abs(balanceUpdateTransaction.getAmount());
			if (!NumberUtil.isZero(payoutAmount)) {
				String payoutGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("EXPENSES");
				String transId = balanceUpdateTransaction.getId();
				String description = StringUtils.isBlank(balanceUpdateTransaction.getDescription()) ? ("EXPENSES/" + transId)
						: balanceUpdateTransaction.getDescription();
				LedgerEntry createAccountsManagerEntry = LedgerEntry.createLedgerEntry(null, depositMethodCOA, DirectionType.CREDIT, payoutAmount, description,
						balanceUpdateTransaction.getEventTime());
				createAccountsManagerEntry.setGroupId(payoutGroupId);
				createAccountsManagerEntry.setLedgerEntryType(LedgerEntryType.EXPENSE.name());
				createAccountsManagerEntry.setTransactionId(transId);
				if (isFromAM) {
					createAccountsManagerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				}
				createAccountsManagerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createAccountsManagerEntry.setSortOrder(2);
				LedgerEntryDAO.getInstance().save(createAccountsManagerEntry, session);

				LedgerEntry createCashBookLedgerEntry = LedgerEntry.createLedgerEntry(null, cashBookExpenseCOA, DirectionType.DEBIT, payoutAmount, description,
						balanceUpdateTransaction.getEventTime());
				createCashBookLedgerEntry.setGroupId(payoutGroupId);
				createCashBookLedgerEntry.setLedgerEntryType(LedgerEntryType.EXPENSE.name());
				createCashBookLedgerEntry.setTransactionId(transId);
				if (isFromAM) {
					createCashBookLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				}
				createCashBookLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createCashBookLedgerEntry.setSortOrder(1);
				LedgerEntryDAO.getInstance().save(createCashBookLedgerEntry, session);

				balanceUpdateTransaction.setAccountProcessed(true);
				BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveCashDepositLedgerEntry(BalanceUpdateTransaction balanceUpdateTransaction, boolean isFormCashDrawer, boolean isStoreDeposit) {
		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();
			DataProvider dataProvider = DataProvider.get();
			Date now = StoreDAO.getServerTimestamp();

			ChartOfAccounts cashCOA;
			if (isFormCashDrawer) {
				cashCOA = dataProvider.getCOAFromMap("10100", session);
			}
			else {
				cashCOA = dataProvider.getCOAFromMap("880", session);
			}

			ChartOfAccounts depositMethodCOA;
			if (isStoreDeposit) {
				depositMethodCOA = dataProvider.getCOAFromMap("10400", session);
			}
			else {
				depositMethodCOA = dataProvider.getCOAFromMap("10300", session);
			}

			Double addAmount = balanceUpdateTransaction.getAmount();
			if (!NumberUtil.isZero(addAmount)) {
				String refKey = isStoreDeposit ? "STORE_DEPOSIT" : "ACCOUNTS_MANAGER";
				String prePaymentsId = SequenceNumberDAO.getInstance().getNextLedgerRef(refKey);
				String transId = balanceUpdateTransaction.getId();
				String description = StringUtils.isBlank(balanceUpdateTransaction.getDescription()) ? (refKey + "/" + transId)
						: balanceUpdateTransaction.getDescription();

				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(null, cashCOA, DirectionType.CREDIT, addAmount, description, now);
				createCashLedgerEntry.setGroupId(prePaymentsId);
				createCashLedgerEntry.setLedgerEntryType(isFormCashDrawer ? LedgerEntryType.TRANSFER.name() : LedgerEntryType.IN.name());
				createCashLedgerEntry.setTransactionId(transId);
				createCashLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createCashLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createCashLedgerEntry.setSortOrder(2);
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

				LedgerEntry createAccountsManagerLedgerEntry = LedgerEntry.createLedgerEntry(null, depositMethodCOA, DirectionType.DEBIT, addAmount,
						description, now);
				createAccountsManagerLedgerEntry.setGroupId(prePaymentsId);
				createAccountsManagerLedgerEntry.setLedgerEntryType(LedgerEntryType.TRANSFER.name());
				createAccountsManagerLedgerEntry.setTransactionId(transId);
				createAccountsManagerLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createAccountsManagerLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createAccountsManagerLedgerEntry.setSortOrder(1);
				LedgerEntryDAO.getInstance().save(createAccountsManagerLedgerEntry, session);

				balanceUpdateTransaction.setAccountProcessed(true);
				BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveAccManagerToStoreLedgerEntry(BalanceUpdateTransaction balanceUpdateTransaction) {
		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();
			DataProvider dataProvider = DataProvider.get();
			Date now = StoreDAO.getServerTimestamp();

			ChartOfAccounts accountsManagerCOA = dataProvider.getCOAFromMap("10300", session);
			ChartOfAccounts cashCOA = dataProvider.getCOAFromMap("10100", session);

			Double transferToStoreAmount = Math.abs(balanceUpdateTransaction.getAmount());
			if (!NumberUtil.isZero(transferToStoreAmount)) {
				String accountsManagerGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("ACCOUNTS_MANAGER");
				String transId = balanceUpdateTransaction.getId();
				String description = StringUtils.isBlank(balanceUpdateTransaction.getDescription()) ? ("ACCOUNTS_MANAGER/" + transId)
						: balanceUpdateTransaction.getDescription();
				LedgerEntry createAccountsManagerLedgerEntry = LedgerEntry.createLedgerEntry(null, accountsManagerCOA, DirectionType.CREDIT,
						transferToStoreAmount, description, now);
				createAccountsManagerLedgerEntry.setGroupId(accountsManagerGroupId);
				createAccountsManagerLedgerEntry.setLedgerEntryType(LedgerEntryType.TRANSFER.name());
				createAccountsManagerLedgerEntry.setTransactionId(transId);
				createAccountsManagerLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createAccountsManagerLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createAccountsManagerLedgerEntry.setSortOrder(2);
				LedgerEntryDAO.getInstance().save(createAccountsManagerLedgerEntry, session);

				LedgerEntry createCRPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, cashCOA, DirectionType.DEBIT, transferToStoreAmount, description,
						now);
				createCRPayableLedgerEntry.setGroupId(accountsManagerGroupId);
				createCRPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.TRANSFER.name());
				createCRPayableLedgerEntry.setTransactionId(transId);
				createCRPayableLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createCRPayableLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createCRPayableLedgerEntry.setSortOrder(1);
				LedgerEntryDAO.getInstance().save(createCRPayableLedgerEntry, session);

				balanceUpdateTransaction.setAccountProcessed(true);
				BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveWithdrawBalanceFromStoreLedgerEntry(BalanceUpdateTransaction balanceUpdateTransaction) {

		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();
			DataProvider dataProvider = DataProvider.get();
			Date now = StoreDAO.getServerTimestamp();

			ChartOfAccounts cashCOA = dataProvider.getCOAFromMap("10100", session);
			ChartOfAccounts ownerADrawingsCOA = dataProvider.getCOAFromMap("880", session);

			Double withdrawFromStoreAmount = Math.abs(balanceUpdateTransaction.getAmount());
			if (!NumberUtil.isZero(withdrawFromStoreAmount)) {
				String accountsManagerGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("ACCOUNTS_MANAGER");
				String transId = balanceUpdateTransaction.getId();
				String description = StringUtils.isBlank(balanceUpdateTransaction.getDescription()) ? ("ACCOUNTS_MANAGER/" + transId)
						: balanceUpdateTransaction.getDescription();

				LedgerEntry createAccountsManagerLedgerEntry = LedgerEntry.createLedgerEntry(null, cashCOA, DirectionType.CREDIT, withdrawFromStoreAmount,
						description, now);
				createAccountsManagerLedgerEntry.setGroupId(accountsManagerGroupId);
				createAccountsManagerLedgerEntry.setLedgerEntryType(LedgerEntryType.OUT.name());
				createAccountsManagerLedgerEntry.setTransactionId(transId);
				createAccountsManagerLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createAccountsManagerLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createAccountsManagerLedgerEntry.setSortOrder(2);
				LedgerEntryDAO.getInstance().save(createAccountsManagerLedgerEntry, session);

				LedgerEntry createCRPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, ownerADrawingsCOA, DirectionType.DEBIT, withdrawFromStoreAmount,
						description, now);
				createCRPayableLedgerEntry.setGroupId(accountsManagerGroupId);
				createCRPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.OUT.name());
				createCRPayableLedgerEntry.setTransactionId(transId);
				createCRPayableLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createCRPayableLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createCRPayableLedgerEntry.setSortOrder(1);
				LedgerEntryDAO.getInstance().save(createCRPayableLedgerEntry, session);

				balanceUpdateTransaction.setAccountProcessed(true);
				BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}

	}

	public void saveBeginCashLedgerEntry(BalanceUpdateTransaction balanceUpdateTransaction) {
		try (Session session = ChartOfAccountsDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();
			DataProvider dataProvider = DataProvider.get();
			Date now = StoreDAO.getServerTimestamp();

			ChartOfAccounts ownerDrawingCOA = dataProvider.getCOAFromMap("880", session);
			ChartOfAccounts cashCOA = dataProvider.getCOAFromMap("10100", session);

			Double beginCash = balanceUpdateTransaction.getAmount();
			if (!NumberUtil.isZero(beginCash)) {
				String beginCashGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("BEGIN_CASH");
				String transId = balanceUpdateTransaction.getId();
				String description = StringUtils.isBlank(balanceUpdateTransaction.getDescription()) ? ("BEGIN_CASH/" + transId)
						: balanceUpdateTransaction.getDescription();

				LedgerEntry createCRLedgerEntry = LedgerEntry.createLedgerEntry(null, ownerDrawingCOA, DirectionType.CREDIT, beginCash, description, now);
				createCRLedgerEntry.setGroupId(beginCashGroupId);
				createCRLedgerEntry.setLedgerEntryType(LedgerEntryType.IN.name());
				createCRLedgerEntry.setTransactionId(transId);
				createCRLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createCRLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createCRLedgerEntry.setSortOrder(2);
				LedgerEntryDAO.getInstance().save(createCRLedgerEntry, session);

				LedgerEntry createDELedgerEntry = LedgerEntry.createLedgerEntry(null, cashCOA, DirectionType.DEBIT, beginCash, description, now);
				createDELedgerEntry.setGroupId(beginCashGroupId);
				createDELedgerEntry.setLedgerEntryType(LedgerEntryType.IN.name());
				createDELedgerEntry.setTransactionId(transId);
				createDELedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
				createDELedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
				createDELedgerEntry.setSortOrder(1);
				LedgerEntryDAO.getInstance().save(createDELedgerEntry, session);

				balanceUpdateTransaction.setAccountProcessed(true);
				BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveSalaryLedgerEntry(Session session, BalanceUpdateTransaction balanceUpdateTransaction, boolean isAMDepositMethod) {
		DataProvider dataProvider = DataProvider.get();
		Date now = StoreDAO.getServerTimestamp();

		ChartOfAccounts wagesAndSalariesCOA = dataProvider.getCOAFromMap("477", session);
		ChartOfAccounts spendMethodCOA;
		if (PaymentType.BANK_ACCOUNT == balanceUpdateTransaction.getPaymentType()) {
			spendMethodCOA = dataProvider.getCOAFromMap("10200", session); //Bank - Regular Checking
		}
		else {
			if (isAMDepositMethod) {
				spendMethodCOA = dataProvider.getCOAFromMap("10300", session); //Accounts manager
			}
			else {
				spendMethodCOA = dataProvider.getCOAFromMap("10400", session); //Store deposit
			}
		}

		Double salaryAmount = Math.abs(balanceUpdateTransaction.getAmount());
		if (!NumberUtil.isZero(salaryAmount)) {
			String salaryGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("SALARY");
			String transId = balanceUpdateTransaction.getId();
			String description = StringUtils.isBlank(balanceUpdateTransaction.getDescription()) ? ("SALARY/" + transId)
					: balanceUpdateTransaction.getDescription();
			LedgerEntry createStoreOrAMLedgerEntry = LedgerEntry.createLedgerEntry(null, spendMethodCOA, DirectionType.CREDIT, salaryAmount, description, now);
			createStoreOrAMLedgerEntry.setGroupId(salaryGroupId);
			createStoreOrAMLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
			createStoreOrAMLedgerEntry.setTransactionId(transId);
			createStoreOrAMLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
			createStoreOrAMLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
			createStoreOrAMLedgerEntry.setSortOrder(2);
			LedgerEntryDAO.getInstance().save(createStoreOrAMLedgerEntry, session);

			LedgerEntry createSalaryPaymentLedgerEntry = LedgerEntry.createLedgerEntry(null, wagesAndSalariesCOA, DirectionType.DEBIT, salaryAmount,
					description, now);
			createSalaryPaymentLedgerEntry.setGroupId(salaryGroupId);
			createSalaryPaymentLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
			createSalaryPaymentLedgerEntry.setTransactionId(transId);
			createSalaryPaymentLedgerEntry.setAccountsManagerId(balanceUpdateTransaction.getAccountNumber());
			createSalaryPaymentLedgerEntry.setPayoutRecepientId(balanceUpdateTransaction.getRecepientId());
			createSalaryPaymentLedgerEntry.setSortOrder(1);
			LedgerEntryDAO.getInstance().save(createSalaryPaymentLedgerEntry, session);

			balanceUpdateTransaction.setAccountProcessed(true);
			BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(balanceUpdateTransaction, session);
		}
	}

	public void saveSalaryAdvanceLedgerEntry(Session session, SalaryAdvanceTransaction salaryAdvanceTransaction) {
		if (salaryAdvanceTransaction == null) {
			return;
		}
		DataProvider dataProvider = DataProvider.get();
		Date now = StoreDAO.getServerTimestamp();

		ChartOfAccounts wagesAndSalariesCOA = dataProvider.getCOAFromMap("477", session);
		ChartOfAccounts spendMethodCOA = dataProvider.getCOAFromMap("10500", session); //Cash Checking

		Double salaryAmount = Math.abs(salaryAdvanceTransaction.getAmount());
		if (!NumberUtil.isZero(salaryAmount)) {
			String salaryGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("SALARY");
			String transId = salaryAdvanceTransaction.getId();
			String description = StringUtils.isBlank(salaryAdvanceTransaction.getNote()) ? ("Loan/Salary advance") : salaryAdvanceTransaction.getNote();
			LedgerEntry createStoreOrAMLedgerEntry = LedgerEntry.createLedgerEntry(null, spendMethodCOA, DirectionType.CREDIT, salaryAmount, description, now);
			createStoreOrAMLedgerEntry.setGroupId(salaryGroupId);
			createStoreOrAMLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
			createStoreOrAMLedgerEntry.setTransactionId(transId);
			createStoreOrAMLedgerEntry.setAccountsManagerId(salaryAdvanceTransaction.getAccountManagerId());
			createStoreOrAMLedgerEntry.setPayoutRecepientId(salaryAdvanceTransaction.getRecepientId());
			createStoreOrAMLedgerEntry.setSortOrder(2);
			LedgerEntryDAO.getInstance().save(createStoreOrAMLedgerEntry, session);

			LedgerEntry createSalaryPaymentLedgerEntry = LedgerEntry.createLedgerEntry(null, wagesAndSalariesCOA, DirectionType.DEBIT, salaryAmount,
					description, now);
			createSalaryPaymentLedgerEntry.setGroupId(salaryGroupId);
			createSalaryPaymentLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
			createSalaryPaymentLedgerEntry.setTransactionId(transId);
			createSalaryPaymentLedgerEntry.setAccountsManagerId(salaryAdvanceTransaction.getAccountManagerId());
			createSalaryPaymentLedgerEntry.setPayoutRecepientId(salaryAdvanceTransaction.getRecepientId());
			createSalaryPaymentLedgerEntry.setSortOrder(1);
			LedgerEntryDAO.getInstance().save(createSalaryPaymentLedgerEntry, session);

			salaryAdvanceTransaction.setAccountProcessed(true);
			BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(salaryAdvanceTransaction, session);

		}

	}

	public void saveTransferToAcmLedgerEntry(BalanceUpdateTransaction debitTransaction, BalanceUpdateTransaction creditTransaction, Session session) {
		Date now = StoreDAO.getServerTimestamp();

		ChartOfAccounts accountsManagerCOA = DataProvider.get().getCOAFromMap("10300", session);
		//ChartOfAccounts cashCOA = instance.getCOAFromMap("10100", session);

		Double transferAmount = Math.abs(debitTransaction.getAmount());
		if (!NumberUtil.isZero(transferAmount)) {
			String accountsManagerGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("ACCOUNTS_MANAGER");
			String transId = debitTransaction.getId();

			LedgerEntry debitLedgerEntry = LedgerEntry.createLedgerEntry(null, accountsManagerCOA, DirectionType.DEBIT, transferAmount,
					debitTransaction.getDescription(), now);
			debitLedgerEntry.setGroupId(accountsManagerGroupId);
			debitLedgerEntry.setLedgerEntryType(LedgerEntryType.TRANSFER.name());
			debitLedgerEntry.setTransactionId(transId);
			debitLedgerEntry.setAccountsManagerId(debitTransaction.getAccountNumber());
			debitLedgerEntry.setPayoutRecepientId(creditTransaction.getAccountNumber());
			debitLedgerEntry.setSortOrder(2);
			LedgerEntryDAO.getInstance().save(debitLedgerEntry, session);

			LedgerEntry creditLedgerEntry = LedgerEntry.createLedgerEntry(null, accountsManagerCOA, DirectionType.CREDIT, transferAmount,
					creditTransaction.getDescription(), now);
			creditLedgerEntry.setGroupId(accountsManagerGroupId);
			creditLedgerEntry.setLedgerEntryType(LedgerEntryType.TRANSFER.name());
			creditLedgerEntry.setTransactionId(creditTransaction.getId());
			creditLedgerEntry.setAccountsManagerId(creditTransaction.getAccountNumber());
			creditLedgerEntry.setPayoutRecepientId(debitTransaction.getAccountNumber());
			creditLedgerEntry.setSortOrder(1);
			LedgerEntryDAO.getInstance().save(creditLedgerEntry, session);
		}
	}
}