package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;

import org.hibernate.Session;
import org.hibernate.Transaction;

import com.floreantpos.PosException;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.SalaryAdvanceBackTransaction;
import com.floreantpos.model.SalaryAdvanceTransaction;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.NumberUtil;

public class SalaryAdvanceBackTransactionDAO extends BaseSalaryAdvanceBackTransactionDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public SalaryAdvanceBackTransaction saveSalaryAdvanceBackTransaction(Date eventTime, String batchNo, String projectId, Object paymentMethod,
			String paymentRef, User accountsManager, User paidToUser, double returnAmount, String note, Session session) throws Exception {

		Double beforeSalaryAdvancedAmount = paidToUser.getSalaryAdvanced();

		if (returnAmount > beforeSalaryAdvancedAmount) {
			throw new PosException(String.format("Cannot return more than the advance salary amount for employee %s", paidToUser.getFullName()));
		}

		SalaryAdvanceBackTransaction salaryAdvanceBackTransaction = createTransaction(paymentMethod, paymentRef, TransactionType.IN, returnAmount);
		salaryAdvanceBackTransaction.setEventTime(eventTime);
		salaryAdvanceBackTransaction.setBatchNo(batchNo);
		salaryAdvanceBackTransaction.setProjectId(projectId);
		// set AccountNo to Server
		salaryAdvanceBackTransaction.setServer(accountsManager);
		salaryAdvanceBackTransaction.putAccountManagerId(accountsManager == null ? null : accountsManager.getId());
		//paid salary to
		salaryAdvanceBackTransaction.setUser(paidToUser);
		salaryAdvanceBackTransaction.putSalaryPaidTo(paidToUser == null ? null : paidToUser.getId());

		salaryAdvanceBackTransaction.setNote(note);
		salaryAdvanceBackTransaction.setAccountProcessed(true);

		save(salaryAdvanceBackTransaction, session);

		double salaryAdvancedAmount = NumberUtil.round(beforeSalaryAdvancedAmount - returnAmount);
		paidToUser.setSalaryAdvanced(salaryAdvancedAmount);
		UserDAO.getInstance().saveOrUpdate(paidToUser, session);

		return salaryAdvanceBackTransaction;
	}

	private SalaryAdvanceBackTransaction createTransaction(Object paymentMethod, String paymentRef, TransactionType transactionType, double amount) {
		SalaryAdvanceBackTransaction salaryAdvanceBackTransaction = new SalaryAdvanceBackTransaction();

		PaymentType paymentType = SalaryTransactionDAO.initPaymentMethod(paymentMethod, paymentRef, salaryAdvanceBackTransaction);

		salaryAdvanceBackTransaction.setPaymentType(paymentType);
		salaryAdvanceBackTransaction.setTransactionType(transactionType.name());
		salaryAdvanceBackTransaction.setTransactionTime(DataProvider.get().getServerTimestamp());
		salaryAdvanceBackTransaction.setAmount(transactionType == TransactionType.DEBIT ? -Math.abs(amount) : Math.abs(amount));
		salaryAdvanceBackTransaction.setOutletId(DataProvider.get().getOutlet().getId());
		return salaryAdvanceBackTransaction;
	}

	public void saveSalaryExpenses(Date eventTime, String batchNo, String projectId, Object paymentMethod, String paymentRef, User accountsManager,
			User paidToUser, double salaryAmount, String note, String deductionGsonData, Boolean isAMDepositMethod) throws Exception {

		boolean isCustomPayment = paymentMethod instanceof CustomPayment;

		Transaction tx = null;
		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			accountsManager = session.get(User.class, accountsManager);
			paidToUser = session.get(User.class, paidToUser);
			
			tx = session.beginTransaction();
			
			if (paymentMethod != null && "Cash".equalsIgnoreCase(paymentMethod.toString())) {
				if (accountsManager.getAccountsManagerAmount() < salaryAmount) {
					throw new PosException("Not enough balance in account manager's account");
				}
			}
			
			SalaryAdvanceTransaction salaryAdvanceTransaction = SalaryAdvanceTransactionDAO.getInstance().saveSalaryAdvanceTransaction(eventTime, batchNo,
					projectId, paymentMethod, paymentRef, accountsManager, paidToUser, salaryAmount, note, session);

			LedgerEntryDAO.getInstance().saveSalaryAdvanceLedgerEntry(session, salaryAdvanceTransaction);

			if (!isCustomPayment) {
				BalanceUpdateTransaction balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().saveSalaryExpenseTransaction(session, eventTime,
						batchNo, projectId, paymentMethod, paymentRef, accountsManager, paidToUser, salaryAmount, note, isAMDepositMethod,
						salaryAdvanceTransaction);

				LedgerEntryDAO.getInstance().saveSalaryLedgerEntry(session, balanceUpdateTransaction, true);
			}

			tx.commit();

		}
	}

}