package com.floreantpos.model.util;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.Customer;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemInventoryStatus;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketDiscount;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.util.NumberUtil;

public class TicketUtil {

	public static void setCustomer(Ticket ticket, Customer customer) {
		if (customer == null) {
			ticket.removeCustomer();
			return;
		}

		ticket.setCustomerId(customer.getId());
		ticket.addProperty(Ticket.CUSTOMER_ID, String.valueOf(customer.getId()));
		ticket.addProperty(Ticket.CUSTOMER_NAME, customer.getFirstName());
		ticket.addProperty(Ticket.CUSTOMER_LAST_NAME, customer.getLastName());
		ticket.addProperty(Ticket.CUSTOMER_PHONE, customer.getMobileNo());
		ticket.addProperty(Ticket.CUSTOMER_ZIP_CODE, customer.getZipCode());
		ticket.addProperty(Ticket.CUSTOMER_TAX_EXEMPT, customer.isTaxExempt().toString());
	}

	public static void validateTicket(Ticket ticket, String validationMessage) {
		if (ticket == null) {
			return;
		}

		if (StringUtils.isBlank(validationMessage)) {
			validationMessage = Messages.getString("IS_NOT_AVAILABLE_FOR_REORDER"); //$NON-NLS-1$
		}

		List<TicketItem> ticketItems = ticket.getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				TicketItemUtil.validateTicketItem(ticketItem, ticketItem.getTicket(), validationMessage);

			}
		}
	}

	public static boolean hasAvailableStock(TicketItem selectedTicketItem, MenuItem menuItem, double selectedQty) {
		return hasAvailableStock(selectedTicketItem, selectedTicketItem.getTicket(), menuItem, selectedQty);
	}

	public static boolean hasAvailableStock(TicketItem selectedTicketItem, Ticket oldTicket, MenuItem menuItem, double selectedQty) {
		if (menuItem == null || menuItem.isHasVariant())
			return true;
		double qty = calculateCartQuantity(oldTicket, menuItem, selectedQty);

		if (menuItem.isComboItem()) {

			List<TicketItem> comboItems = selectedTicketItem.getComboItems();
			if (comboItems != null) {
				for (TicketItem comboTicketItem : comboItems) {
					MenuItem comboMenuItem = MenuItemDAO.getInstance().get(comboTicketItem.getMenuItemId());

					Double unitQuantity = selectedQty * (comboTicketItem.getQuantity() / selectedTicketItem.getQuantity());
					qty = calculateCartQuantity(oldTicket, comboMenuItem, unitQuantity);
					if (comboMenuItem.isInventoryItem() && comboMenuItem.isDisableWhenStockAmountIsZero() && comboMenuItem.getUnitOnHand() < qty) {
						throw new PosException(doGenerateMessage(comboMenuItem));
					}
				}
			}

		}

		else if (menuItem.isInventoryItem() && menuItem.isDisableWhenStockAmountIsZero() && menuItem.getUnitOnHand() < qty) {
			throw new PosException(doGenerateMessage(menuItem));
		}
		return true;
	}

	private static double calculateCartQuantity(Ticket oldTicket, MenuItem menuItem, double selectedQty) {
		double qty = 0;
		if (oldTicket != null) {
			qty = oldTicket.countMenuItemQuantity(menuItem);
		}
		qty += selectedQty;
		return qty;
	}

	private static String doGenerateMessage(MenuItem menuItem) {

		MenuItemInventoryStatus stockStatus = DataProvider.get().getMenuItemStockStatus(menuItem);
		Double availableUnit = stockStatus.getUnitOnHand();

		if (availableUnit <= 0) {
			return (menuItem.getDisplayName() + Messages.getString("TicketUtil.0")); //$NON-NLS-1$
		}

		else {
			return (Messages.getString("TicketUtil.1") + menuItem.getDisplayName() + Messages.getString("TicketUtil.2") //$NON-NLS-1$ //$NON-NLS-2$
					+ NumberUtil.formatNumberIfNeeded(availableUnit));
		}
	}

	public static Ticket doGenerateDuplicateOrder(Ticket ticket) {
		Ticket cloneTicket = Ticket.clone(ticket);
		Ticket newTicket = new Ticket();
		newTicket.setOutlet(com.floreantpos.model.util.DataProvider.get().getOutlet());
		newTicket.setCreateDate(StoreDAO.getServerTimestamp());

		List<TicketItem> items = new ArrayList<>();
		List<TicketItem> cloneTicketItems = cloneTicket.getTicketItems();
		for (TicketItem ticketItem : cloneTicketItems) {
			if (ticketItem.isVoided() || ticketItem.isItemReturned() || StringUtils.isBlank(ticketItem.getMenuItemId())) {
				continue;
			}

			ticketItem.setId(null);
			ticketItem.setCreateDate(StoreDAO.getServerTimestamp());
			ticketItem.setTicket(newTicket);
			ticketItem.setPaid(false);
			ticketItem.setReturned(null);
			ticketItem.setReturnedSource(null);
			ticketItem.setItemReturned(null);
			ticketItem.setVoidReason(null);
			ticketItem.setVoidDate(null);
			ticketItem.setVoided(null);
			ticketItem.setVoidItem(null);
			ticketItem.setVoidedItemId(null);
			ticketItem.setIncludeVoidQuantity(null);
			ticketItem.setInventoryAdjustQty(null);
			ticketItem.setDiscounts(null);

			items.add(ticketItem);
		}
		newTicket.setTicketItems(items);

		List<TicketDiscount> discounts = new ArrayList<>();
		for (TicketDiscount ticketDiscount : cloneTicket.getDiscounts()) {
			ticketDiscount.setTicket(newTicket);
			discounts.add(ticketDiscount);
		}
		newTicket.setDiscounts(discounts);

		newTicket.setProjectId(cloneTicket.getProjectId());
		newTicket.setCustomer(cloneTicket.getCustomer());
		newTicket.setOrderType(cloneTicket.getOrderType());
		newTicket.putTermsAndConditionsId(cloneTicket.getTermsAndConditionsId());
		newTicket.setReferrerId(cloneTicket.getReferrerId());

		return newTicket;
	}
}
