package com.floreantpos.util;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.Date;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.id.IdentifierGenerator;

import com.floreantpos.PosLog;
import com.floreantpos.model.SequenceNumber;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.dao.SequenceNumberDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.util.DataProvider;

public class TicketIdGenerator implements IdentifierGenerator {
	@Override
	public Serializable generate(SessionImplementor session, Object object) throws HibernateException {
		Class<? extends Object> clazz = object.getClass();
		Serializable generatedId = null;
		try {
			Method method = clazz.getMethod("getId", (Class<?>[]) null); //$NON-NLS-1$
			if (method != null) {
				Object id = method.invoke(object, (Object[]) null);
				if (id != null) {
					generatedId = (Serializable) id;
				}
			}

			if (generatedId == null) {
				generatedId = generateTicketId((Session) session, DataProvider.get().getOutletId());
			}
		} catch (Exception e) {
			PosLog.error(TicketIdGenerator.class, e);
		}

		return generatedId;
	}

	private static String generateTicketId(Session session, String outletId) {
		return generateTicketId(session, outletId, new Date(), false);
	}

	public static String generateTicketId(Ticket ticket) {
		try (Session session = TicketDAO.getInstance().createNewSession()) {
			String outletId = ticket.getOutletId();
			if (StringUtils.isBlank(outletId)) {
				outletId = DataProvider.get().getOutletId();
			}
			return generateTicketId(session, outletId, ticket.getCreateDate(), ticket.getTicketType() == TicketType.PHARMA);
		}
	}

	public static String generateTicketId(Session session, String outletId, Date createDate, boolean ispharma) {
		if (StringUtils.isBlank(outletId)) {
			outletId = DataProvider.get().getOutletId();
		}

		if (createDate == null) {
			createDate = new Date();
		}

		String prefix = SequenceNumber.yearMonthDayFormat.format(createDate);
		if (ispharma) {
			prefix = "P" + prefix;
		}
		else {
			prefix = DataProvider.get().getTicketIdPrefex() + prefix;
		}
		String ticketId = prefix + SequenceNumber.threeDigitDecimalFormat
				.format(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.TICKET_SEQUENCE_NUMBER, prefix));
		while (TicketDAO.getInstance().get(ticketId, outletId, session) != null) {
			ticketId = prefix + SequenceNumber.threeDigitDecimalFormat
					.format(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.TICKET_SEQUENCE_NUMBER, prefix));
		}

		return ticketId;
	}
}
