package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.BankAccount;
import com.floreantpos.model.DepositMethod;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.GlobalConfig;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.GsonUtil;
import com.floreantpos.util.POSUtil;
import com.google.common.reflect.TypeToken;
import com.orocube.medlogics.Module;

public class GlobalConfigDAO extends BaseGlobalConfigDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public GlobalConfigDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void addProperty(String key, String value) {
		GlobalConfig globalConfig = findByKey(key);
		if (globalConfig == null) {
			globalConfig = new GlobalConfig();
			globalConfig.setKey(key);
		}
		globalConfig.setValue(value);
		saveOrUpdate(globalConfig);
	}

	public String getProperty(String key) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(GlobalConfig.PROP_VALUE));
			criteria.add(Restrictions.eq(GlobalConfig.PROP_KEY, key));
			return (String) criteria.uniqueResult();
		}
	}

	public String getProperty(String key, String defaultValue) {
		String propValue = getProperty(key);
		if (StringUtils.isBlank(propValue)) {
			return defaultValue;
		}
		return propValue;
	}

	public GlobalConfig findByKey(String key) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(GlobalConfig.PROP_KEY, key));
			criteria.setMaxResults(1);
			return (GlobalConfig) criteria.uniqueResult();
		}
	}

	public static boolean getBooleanProperty(String key, boolean defaultValue) {
		return POSUtil.getBoolean(getInstance().getProperty(key), defaultValue);
	}

	public static Boolean isShouldSentAppointmentSMS() {
		return getBooleanProperty("appointment.sent.sms", false); //$NON-NLS-1$
	}

	public static void setSentAppointmentSMS(boolean sentSms) {
		getInstance().addProperty("appointment.sent.sms", String.valueOf(sentSms)); //$NON-NLS-1$
	}

	public void deleteByKey(String key) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(GlobalConfig.class);
			criteria.add(Restrictions.eq(GlobalConfig.PROP_KEY, key));
			criteria.setMaxResults(1);

			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null) {
				Transaction transaction = session.beginTransaction();
				session.delete(uniqueResult);
				transaction.commit();
			}
		}
	}

	public boolean isDoctorsCanChangePrescriptionPad() {
		return getBooleanProperty("doctors_can_change_prescription_pad", false); //$NON-NLS-1$
	}

	public void putDoctorsCanChangePrescriptionPad(boolean doctorsCanChangePrescriptionPad) {
		addProperty("doctors_can_change_prescription_pad", String.valueOf(doctorsCanChangePrescriptionPad)); //$NON-NLS-1$
	}

	public void putEnabledModules(List<Module> enableKeyList) {
		addProperty("module.enabled.list", GsonUtil.createGson().toJson(enableKeyList));
	}

	public List<Module> getEnabledModules() {
		String enableModulesNames = getProperty("module.enabled.list", "[" + Module.DIAGNOSTICS_LAB.name() + "," + Module.FRONT_DESK.name() + "]");
		return GsonUtil.createGson().fromJson(enableModulesNames, new TypeToken<List<Module>>() {
		}.getType());
	}

	public static List<Module> getEnabledStandAloneModules() {
		List<Module> enabledModules = DataProvider.get().getEnabledModules();
		List<Module> standAloneModules = new ArrayList<Module>();
		for (Module module : enabledModules) {
			if (module == null) {
				continue;
			}
			if (module.isStandalone()) {
				standAloneModules.add(module);
			}
		}
		Comparator<Module> comparator = Comparator.comparing(Module::toString, Comparator.nullsLast(String.CASE_INSENSITIVE_ORDER));
		standAloneModules.sort(comparator);
		return standAloneModules;
	}

	public static boolean isEnabledInventoryPlugin() {
		return getInstance().isEnabledPlugin(Module.INVENTORY);
	}

	public static Boolean isEnabledRefSystemPlugin() {
		return getInstance().isEnabledPlugin(Module.REFERRAL);
	}

	public static boolean isEnabledFrontDeskPlugin() {
		return getInstance().isEnabledPlugin(Module.FRONT_DESK);
	}

	public static boolean isEnabledIPDPlugin() {
		return getInstance().isEnabledPlugin(Module.IPD);
	}

	public static boolean isEnabledDiagnosticsLabPlugin() {
		return getInstance().isEnabledPlugin(Module.DIAGNOSTICS_LAB);
	}

	public static boolean isEnabledPharmacyPlugin() {
		return getInstance().isEnabledPlugin(Module.PHARMACY);
	}

	public static boolean isEnabledAppointmentPlugin() {
		return getInstance().isEnabledPlugin(Module.OPD);
	}

	public static boolean isEnabledChamberServicePlugin() {
		return getInstance().isEnabledPlugin(Module.CHAMBER_SERVICE);
	}

	public boolean isEnabledPlugin(Module module) {
		//return getInstance().getEnabledModules().contains(module);

		return DataProvider.get().getEnabledModules().contains(module);
	}

	public void setLastSelectedModule(String terminalKey, Module selectedDepartment) {
		addProperty(terminalKey + ".selected_department", selectedDepartment.name());
	}

	public Module getLastSelectedModule(String terminalKey) {
		String selectedDepartment = getProperty(terminalKey + ".selected_department");
		if (StringUtils.isNotBlank(selectedDepartment)) {
			return Module.valueOf(selectedDepartment);
		}
		return null;
	}

	public String getMachineIntegrationSecretKey() {
		return getProperty("machine_integration.secret_key", "");
	}

	public void putMachineIntegrationSecretKey(String secretKey) {
		addProperty("machine_integration.secret_key", secretKey);
	}

	public void setLastSelectedLabTechnician(String terminalKey, String lastSelectedLabTechnicianId) {
		//addProperty(terminalKey + ".last_lab_technician_id", lastSelectedLabTechnicianId);
		addProperty("last_lab_technician_id", lastSelectedLabTechnicianId);
	}

	public User getLastSelectedLabTechnician(String terminalKey) {
		String selectedLabTechnicianId = getProperty("last_lab_technician_id");
		//String selectedLabTechnicianId = getProperty(terminalKey + ".last_lab_technician_id");
		if (StringUtils.isNotBlank(selectedLabTechnicianId)) {
			return UserDAO.getInstance().get(selectedLabTechnicianId, DataProvider.get().getCurrentOutletId());
		}
		return null;
	}

	public void setLastSelectedLabIncharge(String terminalKey, String lastSelectedLabInchargeId) {
		//addProperty(terminalKey + ".last_lab_incharge_id", lastSelectedLabInchargeId);
		addProperty("last_lab_incharge_id", lastSelectedLabInchargeId);
	}

	public User getLastSelectedLabIncharge(String terminalKey) {
		//String selectedLabInchargeId = getProperty(terminalKey + ".last_lab_incharge_id");
		String selectedLabInchargeId = getProperty("last_lab_incharge_id");
		if (StringUtils.isNotBlank(selectedLabInchargeId)) {
			return UserDAO.getInstance().get(selectedLabInchargeId, DataProvider.get().getCurrentOutletId());
		}
		return null;
	}

	public void setLastSelectedLabDoctor(String terminalKey, String lastSelectedLabDoctorId) {
		//addProperty(terminalKey + ".last_lab_doctor_id", lastSelectedLabDoctorId);
		addProperty("last_lab_doctor_id", lastSelectedLabDoctorId);
	}

	public Doctor getLastSelectedLabDoctor(String terminalKey) {
		//String selectedLabDocrtorId = getProperty(terminalKey + ".last_lab_doctor_id");
		String selectedLabDocrtorId = getProperty("last_lab_doctor_id");
		if (StringUtils.isNotBlank(selectedLabDocrtorId)) {
			return (Doctor) DataProvider.get().getObjectOf(Doctor.class, selectedLabDocrtorId);
		}
		return null;
	}

	public void setLastSelectedTestMethod(String lastSelectedTestMethod) {
		addProperty("last_test_method", lastSelectedTestMethod);
	}

	public String getLastSelectedTestMethod() {
		return getProperty("last_test_method");
	}

	public void setLastSelectedAccountManager(String lastSelectedAccountManagerId) {
		User currentUser = DataProvider.get().getCurrentUser();
		String key = "last_selected_account_manager";
		if (currentUser != null) {
			key += "_" + currentUser.getId();
		}
		addProperty(key, lastSelectedAccountManagerId);
	}

	public User getLastSelectedAccountManager() {
		User currentUser = DataProvider.get().getCurrentUser();
		String key = "last_selected_account_manager";
		if (currentUser != null) {
			key += "_" + currentUser.getId();
		}
		String accountManagerId = getProperty(key);
		return DataProvider.get().getUserById(accountManagerId, DataProvider.get().getOutletId());
	}

	public void setLastSelectedExpenseFrom(String lastSelectedExpenseFromId) {
		User currentUser = DataProvider.get().getCurrentUser();
		String key = "last_selected_expense_from";
		if (currentUser != null) {
			key += "_" + currentUser.getId();
		}
		addProperty(key, lastSelectedExpenseFromId);
	}

	public Object getLastSelectedExpenseFrom() {
		User currentUser = DataProvider.get().getCurrentUser();
		String key = "last_selected_expense_from";
		if (currentUser != null) {
			key += "_" + currentUser.getId();
		}
		String value = getProperty(key);
		if ("1".equals(value)) {
			return DataProvider.get().getStore();
		}
		return DataProvider.get().getUserById(value, DataProvider.get().getOutletId());
	}

	public void setLastSelectedDepositMethod(String lastSelectedDepositMethodName) {
		addProperty("last_selected_deposit_method", lastSelectedDepositMethodName);
	}

	public DepositMethod getLastSelectedDepositMethod() {
		String depositMethodName = getProperty("last_selected_deposit_method");
		if (StringUtils.isNotBlank(depositMethodName)) {
			return DepositMethod.valueOf(depositMethodName);
		}
		return DepositMethod.STORE;
	}

	public String getLongProperty(String key) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(GlobalConfig.PROP_LONG_VALUE));
			criteria.add(Restrictions.eq(GlobalConfig.PROP_KEY, key));
			return (String) criteria.uniqueResult();
		}
	}

	public String getLongProperty(String key, String defaultValue) {
		String longPropValue = getLongProperty(key);
		if (StringUtils.isBlank(longPropValue)) {
			return defaultValue;
		}
		return longPropValue;
	}

	public void addLongProperty(String key, String longValue) {
		GlobalConfig globalConfig = findByKey(key);
		if (globalConfig == null) {
			globalConfig = new GlobalConfig();
			globalConfig.setKey(key);
		}
		globalConfig.setLongValue(longValue);
		saveOrUpdate(globalConfig);
	}

	public String getPayoutEntryState(boolean isExpense) {
		String key = isExpense ? "expense_entry_state" : "payout_entry_state";
		String json = getLongProperty(key);

		if (StringUtils.isBlank(json)) {
			return StringUtils.EMPTY;
		}
		return json;
	}

	public void putPayoutEntryState(String json, boolean isExpense) {
		String key = isExpense ? "expense_entry_state" : "payout_entry_state";
		addLongProperty(key, json);
	}

	public boolean isSampleIdMatchWithInvoice() {
		return getBooleanProperty("sampleid_match_with_invoice", false); //$NON-NLS-1$
	}

	public void putSampleIdMatchWithInvoice(boolean matchWithInvoice) {
		addProperty("sampleid_match_with_invoice", String.valueOf(matchWithInvoice)); //$NON-NLS-1$
	}

	public void setLastSelectedBankAccount(String customPaymentKey, String lastSelectedBankId) {
		addProperty(customPaymentKey + ".last_bank_account_id", lastSelectedBankId);
	}

	public BankAccount getLastSelectedBankAccount(String customPaymentKey) {
		String selectedBankAccountId = getProperty(customPaymentKey + ".last_bank_account_id");
		if (StringUtils.isNotBlank(selectedBankAccountId)) {
			return (BankAccount) DataProvider.get().getObjectOf(BankAccount.class, selectedBankAccountId);
		}
		return null;
	}

}