package com.floreantpos.print;

import java.awt.Image;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JPanel;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.PosLog;
import com.floreantpos.main.Application;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.ComboTicketItem;
import com.floreantpos.model.ITicketItem;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.ReceiptParam;
import com.floreantpos.model.Store;
import com.floreantpos.model.TerminalPrinters;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.report.AbstractReportDataSource;
import com.floreantpos.report.KitchenTicketDataSource;
import com.floreantpos.report.ReceiptPrintService;
import com.floreantpos.report.TicketDataSource;
import com.floreantpos.report.TicketPrintProperties;
import com.floreantpos.util.POSUtil;
import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Paragraph;

import io.github.escposjava.PrinterService;
import io.github.escposjava.print.Commands;
import io.github.escposjava.print.NetworkPrinter;
import io.github.escposjava.print.SerialPortPrinter;
import net.sf.jasperreports.engine.JRParameter;

public class EscPosPrintService extends ReceiptPrintServiceProvider {
	public static final int COL_RIGHT_FIELD_LENGTH = 13;
	public int totalCharLength;
	public int colLeftFieldLength;

	@Override
	public void printTicket(Ticket ticket, HashMap map, String deviceName) throws Exception {
		doPrint(deviceName, map, ticket, new TicketDataSource(ticket, false));
	}

	@Override
	public void printTransaction(Ticket ticket, HashMap map, PosTransaction transaction, String deviceName) throws Exception {
		doPrint(deviceName, map, ticket, transaction, new TicketDataSource(ticket, false));
	}

	@Override
	public void printRefundTicket(Ticket ticket, HashMap map, String deviceName) throws Exception {
		doPrint(deviceName, map, ticket, new TicketDataSource(ticket, false));
	}

	@Override
	public void printVoidTicket(Ticket ticket, HashMap map, String deviceName) throws Exception {
		doPrint(deviceName, map, ticket, new TicketDataSource(ticket, false));
	}

	@Override
	public void printKitchenTicket(KitchenTicket kitchenTicket, HashMap map, Ticket ticket, String printerName, String deviceName) throws Exception {
		doPrint(deviceName, map, ticket, new KitchenTicketDataSource(kitchenTicket));
	}

	@Override
	public void printVoidKitchenTicket(KitchenTicket kitchenTicket, HashMap map, String printerName, String deviceName) throws Exception {
		doPrint(deviceName, map, null, new KitchenTicketDataSource(kitchenTicket));
	}

	@Override
	public void rendererTicketReceipt(Ticket ticket, JPanel reportPanel) throws Exception {
		TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true); //$NON-NLS-1$ //$NON-NLS-2$
		HashMap map = ReceiptPrintService.populateTicketProperties(ticket, printProperties, null);
		ITextPrinter printer = ITextPrinter.create();
		printer.createReport(null, map, new TicketDataSource(ticket));
		printer.rendererReport(reportPanel);
	}

	@Override
	public void rendererPosTransactionReceipt(PosTransaction transaction, JPanel reportPanel) throws Exception {
		try {
			Ticket ticket = transaction.getTicket();
			TicketPrintProperties printProperties = new TicketPrintProperties(Messages.getString("ReceiptPrintService.3"), true, true, true); //$NON-NLS-1$
			printProperties.setPrintCookingInstructions(false);
			HashMap map = ReceiptPrintService.populateTicketProperties(ticket, printProperties, transaction);
			map.put(JRParameter.IS_IGNORE_PAGINATION, true);
			if (transaction != null && transaction.isCard()) {
				CardReader cardReader = CardReader.fromString(transaction.getCardReader());
				if (cardReader == CardReader.EXTERNAL_TERMINAL) {
					return;
				}
				/*map.put("cardPayment", true); //$NON-NLS-1$
				map.put("copyType", Messages.getString("ReceiptPrintService.4")); //$NON-NLS-1$ //$NON-NLS-2$
				ITextPrinter printer = ITextPrinter.create(null, map, new TicketDataSource(ticket));
				printer.populateReport(reportPanel);*/

				map.put("copyType", Messages.getString("ReceiptPrintService.5")); //$NON-NLS-1$ //$NON-NLS-2$
				ITextPrinter printer = ITextPrinter.create();
				printer.createReport(null, map, new TicketDataSource(ticket));
				printer.rendererReport(reportPanel);
			}
			else {
				ITextPrinter printer = ITextPrinter.create();
				printer.createReport(null, map, new TicketDataSource(ticket));
				printer.rendererReport(reportPanel);
			}
		} catch (Exception e) {
			ReceiptPrintService.getLogger().error(com.floreantpos.POSConstants.PRINT_ERROR, e);
		}
	}

	@Override
	public void rendererKitchenReceipt(Ticket ticket, JPanel reportPanel) throws Exception {
		List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, false);
		for (KitchenTicket kitchenTicket : kitchenTickets) {
			kitchenTicket.setParentTicket(ticket);
			Map map = ReceiptPrintService.populateKitchenTicketProperties(kitchenTicket, "", "", true); //$NON-NLS-1$ //$NON-NLS-2$
			ITextPrinter printer = ITextPrinter.create();
			printer.createReport(null, map, new KitchenTicketDataSource(kitchenTicket));
			printer.rendererReport(reportPanel);
			break;
		}
	}

	private void doPrint(String deviceName, Map<String, Object> params, Ticket ticket, AbstractReportDataSource dataSource) throws Exception {
		doPrint(deviceName, params, ticket, null, dataSource);
	}

	private void doPrint(String deviceName, Map<String, Object> params, Ticket ticket, PosTransaction transaction, AbstractReportDataSource dataSource)
			throws Exception {
		calculateTextLength(params);
		PrinterService ps = createPrintService(deviceName, params);
		//if (dataSource instanceof KitchenTicketDataSource) {
		//	ps.write(Commands.TXT_FONT_BOLD_MEDIUM);
		//}
		addBoldText(ps, params, ReceiptPrintService.HEADER_LINE1, Commands.TXT_ALIGN_CT);
		addText(ps, params, ReceiptPrintService.TICKET_HEADER, Commands.TXT_ALIGN_CT);
		br(ps);
		addText(ps, params, ReceiptPrintService.ADDITIONAL_ORDER_INFO, Commands.TXT_ALIGN_LT);

		if (dataSource instanceof KitchenTicketDataSource) {
			addSeparatorLeftAlign(ps);
		}
		else {
			addSeparatorRightAlign(ps);
		}

		Store store = DataProvider.get().getStore();
		List rows = dataSource.getRows();
		if (rows != null) {
			if (dataSource instanceof TicketDataSource) {
				TicketDataSource ticketDataSource = (TicketDataSource) dataSource;
				for (Iterator iterator = rows.iterator(); iterator.hasNext();) {
					ITicketItem item = (ITicketItem) iterator.next();
					if (StringUtils.isEmpty(item.getNameDisplay())) {
						continue;
					}
					renderTextColor(ps, ticketDataSource.getColorCode(store, item));
					String name = getText(item.getNameDisplay(), ReceiptPrintService.LEFT, colLeftFieldLength);
					String displayString = name;
					String subTotalAmountDisplay = item.getSubTotalAmountDisplay();
					if (StringUtils.isNotEmpty(subTotalAmountDisplay)) {
						if (item instanceof ComboTicketItem) {
							int comboItemNameIndex = name.indexOf("\n"); //$NON-NLS-1$
							if (comboItemNameIndex != -1) {
								ps.printRight(wrapLine(getText(name.substring(0, comboItemNameIndex), ReceiptPrintService.LEFT, colLeftFieldLength),
										colLeftFieldLength, subTotalAmountDisplay));
								br(ps);
								String[] comboItems = name.substring(comboItemNameIndex).split("\n"); //$NON-NLS-1$
								if (comboItems.length > 0) {
									for (String comboItem : comboItems) {
										if (StringUtils.isBlank(comboItem)) {
											continue;
										}
										ps.printRight(wrapLine(getText(comboItem, ReceiptPrintService.LEFT, colLeftFieldLength), colLeftFieldLength, " ")); //$NON-NLS-1$
										br(ps);
									}
								}
								continue;
							}
						}
						if (name.length() > colLeftFieldLength) {
							displayString = wrapLine(name, colLeftFieldLength, subTotalAmountDisplay);
						}
						else {
							displayString += getText(subTotalAmountDisplay, ReceiptPrintService.RIGHT, COL_RIGHT_FIELD_LENGTH);
						}
					}
					else {
						displayString += getText(" ", ReceiptPrintService.RIGHT, COL_RIGHT_FIELD_LENGTH); //$NON-NLS-1$
					}
					ps.printRight(displayString);
					br(ps);
				}
			}
			else if (dataSource instanceof KitchenTicketDataSource) {
				KitchenTicketDataSource kitchenTicketDataSource = (KitchenTicketDataSource) dataSource;
				int lineCount = 0;
				for (Iterator iterator = rows.iterator(); iterator.hasNext();) {
					KitchenTicketItem item = (KitchenTicketItem) iterator.next();
					if (StringUtils.isEmpty(item.getMenuItemNameDisplay())) {
						continue;
					}
					if (!item.isModifierItem() && !item.isCookingInstruction() && lineCount > 0) {
						ps.write(Commands.TXT_NORMAL);
						addSeparatorLeftAlign(ps);
					}
					ps.write(new byte[] { 0x1B, 0x21, 0x16 });
					renderTextColor(ps, kitchenTicketDataSource.getColorCode(item, item.getKitchenTicket().getOrderTypeId()));
					ps.printLeft(item.getMenuItemNameDisplay());
					ps.lineBreak();
					lineCount += 1;
				}
				ps.write(Commands.TXT_NORMAL);
				addSeparatorLeftAlign(ps);
			}
		}
		ps.setTextColor(Commands.COLOR_BLACK);
		ps.setTextNormal();
		if (dataSource instanceof TicketDataSource) {
			addSeparatorRightAlign(ps);
			addRow(ps, params, ReceiptPrintService.TOTAL_TEXT, ReceiptPrintService.GRAND_SUBTOTAL, false);
			addRow(ps, params, ReceiptPrintService.DISCOUNT_TEXT, ReceiptPrintService.DISCOUNT_AMOUNT);
			boolean isShowTaxBreakdown = Boolean.parseBoolean(String.valueOf(params.get(ReceiptPrintService.IS_SHOW_TAX_BREAKDOWN))); //$NON-NLS-1$
			if (!isShowTaxBreakdown) {
				addRow(ps, params, ReceiptPrintService.TAX_TEXT, ReceiptPrintService.TAX_AMOUNT);
			}
			else {
				params.put(ReceiptPrintService.TAX_BREAKDOWN_TEXT,
						ReceiptPrintService.getTaxBreakdown(ticket, colLeftFieldLength, COL_RIGHT_FIELD_LENGTH, true, store));
				addText(ps, params, ReceiptPrintService.TAX_BREAKDOWN_TEXT, Commands.TXT_ALIGN_RT);
			}
			addRow(ps, params, ReceiptPrintService.SERVICE_CHARGE_TEXT, ReceiptPrintService.SERVICE_CHARGE);
			addRow(ps, params, ReceiptPrintService.DELIVERY_CHARGE_TEXT, ReceiptPrintService.DELIVERY_CHARGE);
			addSeparatorRightAlign(ps);
			addRow(ps, params, ReceiptPrintService.NET_AMOUNT_TEXT, ReceiptPrintService.NET_AMOUNT, false);
			addRow(ps, params, ReceiptPrintService.PAID_AMOUNT_TEXT, ReceiptPrintService.PAID_AMOUNT, false);
			addRow(ps, params, ReceiptPrintService.REFUND_AMOUNT_TEXT, ReceiptPrintService.REFUND_AMOUNT);
			addRow(ps, params, ReceiptPrintService.TENDER_AMOUNT_TEXT, ReceiptPrintService.TENDER_AMOUNT);
			addRow(ps, params, ReceiptPrintService.DUE_AMOUNT_TEXT, ReceiptPrintService.DUE_AMOUNT, false);
			if (params.get(ReceiptPrintService.TENDER_AMOUNT) != null) {
				addRow(ps, params, ReceiptPrintService.CHANGE_AMOUNT_TEXT, ReceiptPrintService.CHANGED_AMOUNT);
			}
			addRow(ps, params, ReceiptPrintService.TIPS_TEXT, ReceiptPrintService.TIP_AMOUNT);
			if (params.get(ReceiptPrintService.SHOW_TIPS) != null) {
				br(ps);
				addText(ps, params, ReceiptPrintService.SHOW_TIPS, Commands.TXT_ALIGN_CT);
			}
			addRow(ps, params, ReceiptPrintService.FEE_AMOUNT_TEXT, ReceiptPrintService.FEE_AMOUNT);

			if (Application.getInstance().getTerminal().isEnableMultiCurrency()) {
				if (transaction != null) {
					Set<PosTransaction> transactions = new HashSet<>();
					transactions.add(transaction);
					StringBuilder multiCurrencyBreakdownCashBack = ReceiptPrintService.buildAllMultiCurrency(ticket, transactions, true);
					if (multiCurrencyBreakdownCashBack != null) {
						params.put(ReceiptPrintService.ADDITIONAL_PROPERTIES, multiCurrencyBreakdownCashBack.toString()); //$NON-NLS-1$
					}
				}
				else {
					StringBuilder multiCurrencyTotalAmount = ReceiptPrintService.buildAllMultiCurrency(ticket, ticket.getTransactions(), true);
					if (multiCurrencyTotalAmount != null)
						params.put(ReceiptPrintService.ADDITIONAL_PROPERTIES, multiCurrencyTotalAmount.toString()); //$NON-NLS-1$

				}
			}
			addText(ps, params, ReceiptPrintService.ADDITIONAL_PROPERTIES, Commands.TXT_ALIGN_RT);

			if (params.get(ReceiptPrintService.ADDITIONAL_PAYMENT_PROPERTIES) != null) {
				params.put(ReceiptPrintService.ADDITIONAL_PAYMENT_PROPERTIES, ReceiptPrintService.buildPayments(ticket, true));
			}
			addText(ps, params, ReceiptPrintService.ADDITIONAL_PAYMENT_PROPERTIES, Commands.TXT_ALIGN_RT);

			boolean cardPayment = Boolean.parseBoolean(String.valueOf(params.get(ReceiptPrintService.CARD_PAYMENT)));
			boolean showTipsSection = cardPayment && Boolean.parseBoolean(String.valueOf(params.get(ReceiptPrintService.SHOW_TIPS_BLOCK)));

			if (showTipsSection) {
				addTextWithUnderline(ps, getText(Messages.getString("EscPosPrintService.4"), ReceiptPrintService.LEFT, colLeftFieldLength)); //$NON-NLS-1$
				addTextWithUnderline(ps, getText(Messages.getString("EscPosPrintService.5"), ReceiptPrintService.LEFT, colLeftFieldLength)); //$NON-NLS-1$
				addTextWithUnderline(ps, getText(Messages.getString("EscPosPrintService.6"), ReceiptPrintService.LEFT, colLeftFieldLength)); //$NON-NLS-1$
			}
			if (cardPayment) {
				addText(ps, params, ReceiptPrintService.APPROVAL_CODE, Commands.TXT_ALIGN_LT);
			}
			addText(ps, params, ReceiptPrintService.BOTTOM_MESSAGE, Commands.TXT_ALIGN_LT);
			br(ps);

			addText(ps, params, ReceiptPrintService.COPY_TYPE, Commands.TXT_ALIGN_CT);
			addText(ps, params, ReceiptPrintService.FOOTER_MESSAGE, Commands.TXT_ALIGN_CT);
		}
		else {
			/*if (dataSource instanceof KitchenTicketDataSource) {
				ps.write(Commands.TXT_FONT_BOLD_MEDIUM);
			}*/
			addText(ps, params, ReceiptPrintService.PROP_PRINTER_NAME, Commands.TXT_ALIGN_LT);
			addText(ps, params, ReceiptPrintService.BOTTOM_MESSAGE, Commands.TXT_ALIGN_LT);
			addText(ps, params, ReceiptPrintService.FOOTER_MESSAGE, Commands.TXT_ALIGN_CT);
		}
		ps.lineBreak(3);
		close(ps);
	}

	public PrinterService createPrintService(String deviceName, Map<String, Object> params) {
		String printerType = valueOf(params.get(TerminalPrinters.PRINTER_TYPE));

		PrinterService ps = null;
		if (printerType.equals(PrinterType.NETWORK.getName())) {
			ps = new PrinterService(getNetworkPrinter(params));
		}
		else if (printerType.equals(PrinterType.SERIAL.getName())) {
			ps = new PrinterService(new SerialPortPrinter(String.valueOf(params.get(TerminalPrinters.SERIAL_PORT))));
		}
		else {
			ps = new PrinterService(new AwtReceiptPrinter(deviceName));
		}
		ps.open();
		ps.init();
		return ps;
	}

	public static NetworkPrinter getNetworkPrinter(Map params) {
		String printerIp = valueOf(params.get(TerminalPrinters.IP_ADDRESS));
		int printerPort = POSUtil.parseInteger(String.valueOf(params.get(TerminalPrinters.IP_PORT)));
		return new NetworkPrinter(printerIp, printerPort);
	}

	public void close(PrinterService ps) throws Exception {
		ps.cutFull();
		ps.close();
	}

	private void renderTextColor(PrinterService ps, String color) {
		if (StringUtils.isEmpty(color)) {
			ps.setTextColor(Commands.COLOR_BLACK);
			return;
		}
		if (color.equalsIgnoreCase("#ff0000")) { //$NON-NLS-1$
			ps.setTextColor(Commands.COLOR_RED);
		}
		else {
			ps.setTextColor(Commands.COLOR_BLACK);
		}
	}

	private String wrapLine(String line, int lineLength, String subtotalAmount) {
		String linebreak = "\n"; //$NON-NLS-1$
		if (line.length() == 0)
			return ""; //$NON-NLS-1$
		if (line.length() <= lineLength)
			return line + getText(subtotalAmount, ReceiptPrintService.RIGHT, COL_RIGHT_FIELD_LENGTH);
		String[] words = line.split(" "); //$NON-NLS-1$
		StringBuilder allLines = new StringBuilder();
		StringBuilder trimmedLine = new StringBuilder();
		for (String word : words) {
			if (trimmedLine.length() + 1 + word.length() <= lineLength) {
				trimmedLine.append(word).append(" "); //$NON-NLS-1$
			}
			else {
				allLines.append(getText(trimmedLine.toString(), ReceiptPrintService.LEFT, colLeftFieldLength)
						+ getText(subtotalAmount, ReceiptPrintService.RIGHT, COL_RIGHT_FIELD_LENGTH)).append(linebreak);
				trimmedLine = new StringBuilder();
				trimmedLine.append(word).append(" "); //$NON-NLS-1$
				subtotalAmount = " "; //$NON-NLS-1$
			}
		}
		if (trimmedLine.length() > 0) {
			allLines.append(getText(trimmedLine.toString(), ReceiptPrintService.LEFT, colLeftFieldLength)
					+ getText(subtotalAmount, ReceiptPrintService.RIGHT, COL_RIGHT_FIELD_LENGTH));
		}
		//allLines.append(linebreak);
		return allLines.toString();
	}

	private void calculateTextLength(Map<String, Object> params) {
		totalCharLength = POSUtil.getInteger(params.get(TerminalPrinters.TEXT_LENGTH));
		if (totalCharLength <= 0) {
			totalCharLength = 42;
		}
		colLeftFieldLength = totalCharLength - COL_RIGHT_FIELD_LENGTH;
		/*if (params.get(ReceiptPrintService.NET_AMOUNT) != null) {
			int priceLength = params.get(ReceiptPrintService.NET_AMOUNT).toString().length();
			if (priceLength > colPriceLength) {
				colPriceLength = priceLength;
				colSummaryFieldLength = colPriceLength;
				colSummaryLabelLength = length - colSummaryFieldLength;
			}
			colNameLength = length - colPriceLength;
		}*/
	}

	private void br(PrinterService ps) {
		ps.lineBreak();
	}

	private void addImage(Map<String, Object> parameters, PrinterService service) {
		Image is = (Image) parameters.get(ReceiptPrintService.STORE_LOGO_IMAGE);
		if (is != null) {
			try {
				service.printImage(POSUtil.convertBlackAndWhiteImage(is));
			} catch (Exception e) {
				PosLog.error(ReceiptPrintService.class, e);
			}
		}
	}

	private void addBarcode(Map<String, Object> parameters, PrinterService service) {
		Object barcode = parameters.get(ReceiptParam.BARCODE.getParamName());
		if (barcode != null) {
			try {
				service.printBarcode(barcode.toString(), "CODE128", 255, 2, "OFF", "A"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			} catch (Exception e) {
				PosLog.error(ReceiptPrintService.class, e);
			}
		}
	}

	private void addRow(PrinterService service, Map<String, Object> parameters, String labelKey, String valueKey) {
		addRow(service, parameters, labelKey, valueKey, true);
	}

	private void addRow(PrinterService service, Map<String, Object> parameters, String labelKey, String valueKey, boolean removeNullOrBlank) {
		Object value = parameters.get(valueKey);
		if (removeNullOrBlank && (value == null || value.equals("null") || value.equals("0.00"))) { //$NON-NLS-1$ //$NON-NLS-2$
			return;
		}
		service.printRight(parameters.get(labelKey) + getText(String.valueOf(parameters.get(valueKey)), ReceiptPrintService.RIGHT, COL_RIGHT_FIELD_LENGTH));
		br(service);
	}

	public String getText(String txt, String align, int length) {
		if (txt == null) {
			txt = ""; //$NON-NLS-1$
		}
		if (align.equals(ReceiptPrintService.CENTER)) {
			int space = (length - txt.length()) / 2;
			for (int i = 1; i <= space; i++) {
				txt = " " + txt + " "; //$NON-NLS-1$ //$NON-NLS-2$
			}
			return txt;
		}
		else if (align.equals(ReceiptPrintService.RIGHT)) {
			int space = (length - txt.length());
			for (int i = 1; i <= space; i++) {
				txt = " " + txt; //$NON-NLS-1$
			}
			return txt;
		}
		else if (align.equals(ReceiptPrintService.LEFT)) {
			int space = (length - txt.length());
			for (int i = 1; i <= space; i++) {
				txt = txt + " "; //$NON-NLS-1$
			}
			return txt;
		}
		return null;
	}

	private void addSeparatorRightAlign(PrinterService ps) {
		String sep = ""; //$NON-NLS-1$
		for (int i = 0; i < totalCharLength; i++) {
			sep += "-"; //$NON-NLS-1$
		}
		ps.printLn(Commands.TXT_ALIGN_RT, sep);
	}

	private void addSeparatorLeftAlign(PrinterService ps) {
		String sep = ""; //$NON-NLS-1$
		for (int i = 0; i < totalCharLength; i++) {
			sep += "-"; //$NON-NLS-1$
		}
		ps.printLn(Commands.TXT_ALIGN_LT, sep);
	}

	public void addBoldText(PrinterService service, Map<String, Object> parameters, String key, byte[] align) {
		service.setTextTypeBold();
		addText(service, parameters, key, align, true);
		service.setTextTypeNormal();
		br(service);
	}

	public void addText(PrinterService service, Map<String, Object> parameters, String key, byte[] align) {
		addText(service, parameters, key, align, false);
	}

	public void addText(PrinterService service, Map<String, Object> parameters, String key, byte[] align, boolean bold) {
		String text = String.valueOf(parameters.get(key));
		if (StringUtils.isEmpty(text) || text.equals("null")) { //$NON-NLS-1$
			return;
		}
		if (text.contains("<strike>")) { //$NON-NLS-1$
			text = text.replaceAll("<div><span><strike>", Messages.getString("EscPosPrintService.25")); //$NON-NLS-1$ //$NON-NLS-2$
		}
		String[] splitTexts = text.split("<br>"); //$NON-NLS-1$
		if (splitTexts.length > 0) {
			for (int i = 0; i < splitTexts.length; i++) {
				String p = splitTexts[i];
				if (StringUtils.isEmpty(p)) {
					service.print("\n"); //$NON-NLS-1$
					continue;
				}
				if (p.contains("<storeLogo")) { //$NON-NLS-1$
					addImage(parameters, service);
				}
				else if (p.contains("<barcode>")) { //$NON-NLS-1$
					addText(service, align, bold, p, false);
					addBarcode(parameters, service);
				}
				else {
					addText(service, align, bold, p);
				}
			}
		}
	}

	private void addText(PrinterService service, byte[] align, boolean bold, String text) {
		addText(service, align, bold, text, true);
	}

	private void addText(PrinterService service, byte[] align, boolean bold, String text, boolean newLine) {
		if (text.contains("<b>") || bold) { //$NON-NLS-1$
			service.setTextTypeBold();
		}
		String displayText = text.replaceAll("<.*?>", "").replaceAll("</", "").replaceAll("&nbsp;", " "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
		if (StringUtils.isBlank(displayText)) {
			//displayText += "\n";
			return;
		}
		if (newLine) {
			service.printLn(align, displayText);
		}
		else {
			service.print(align, displayText);
		}
		if (text.contains("</b>")) { //$NON-NLS-1$
			service.setTextTypeNormal();
		}
	}

	private void addTextWithUnderline(PrinterService service, String text) {
		br(service);
		service.setTextTypeUnderline();
		service.printLn(Commands.TXT_ALIGN_RT, text);
		service.setTextTypeNormal();
		br(service);
	}

	public void addLine(Document document, int number) throws DocumentException {
		for (int i = 0; i < number; i++) {
			document.add(new Paragraph(" ")); //$NON-NLS-1$
		}
	}

	@Override
	public void testPrinter(HashMap<String, Object> map, String deviceName) throws Exception {
		PrinterService pService = createPrintService(deviceName, map);
		addBoldText(pService, map, ReceiptPrintService.TITLE, Commands.TXT_ALIGN_CT);
		addText(pService, map, ReceiptPrintService.DATA, Commands.TXT_ALIGN_LT);
		close(pService);
	}

	public static String valueOf(Object object) {
		if (object == null) {
			return ""; //$NON-NLS-1$
		}
		return object.toString();
	}

}
