package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.Project;
import com.floreantpos.model.SalaryAdvanceTransaction;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;

public class SalaryAdvanceTransactionDAO extends BaseSalaryAdvanceTransactionDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public SalaryAdvanceTransaction saveSalaryAdvanceTransaction(Date eventTime, String batchNo, Project project, Object paymentMethod, String paymentRef,
			User accountsManager, User paidToUser, double advanceAmount, String note, Session session) throws Exception {

		if (Math.abs(advanceAmount) == 0) {
			return null;
		}

		Double beforeSalaryAdvancedAmount = paidToUser.getSalaryAdvanced();

		SalaryAdvanceTransaction salaryAdvanceTransaction = createTransaction(paymentMethod, paymentRef, TransactionType.OUT, advanceAmount);
		salaryAdvanceTransaction.setEventTime(eventTime);
		salaryAdvanceTransaction.setBatchNo(batchNo);
		salaryAdvanceTransaction.setProject(project);
		// set AccountNo to Server
		salaryAdvanceTransaction.setServer(accountsManager);
		salaryAdvanceTransaction.setAccountManager(accountsManager == null ? null : accountsManager);
		//paid salary to
		salaryAdvanceTransaction.setUser(paidToUser);
		salaryAdvanceTransaction.putSalaryPaidTo(paidToUser == null ? null : paidToUser.getId());
		if (paymentMethod.toString().equalsIgnoreCase("Cash")&& accountsManager != null) {
			salaryAdvanceTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_ACM);
		}
		else {
			salaryAdvanceTransaction.setSourceType(paymentMethod.toString());
		}
		salaryAdvanceTransaction.setNote(note);
		salaryAdvanceTransaction.setAccountProcessed(true);

		save(salaryAdvanceTransaction, session);

		double salaryAdvancedAmount = NumberUtil.round(beforeSalaryAdvancedAmount + advanceAmount);
		paidToUser.setSalaryAdvanced(salaryAdvancedAmount);
		UserDAO.getInstance().saveOrUpdate(paidToUser, session);

		return salaryAdvanceTransaction;
	}

	public SalaryAdvanceTransaction createTransaction(Object paymentMethod, String paymentRef, TransactionType transactionType, double amount) {
		SalaryAdvanceTransaction salaryAdvanceTransaction = new SalaryAdvanceTransaction();

		PaymentType paymentType = SalaryTransactionDAO.initPaymentMethod(paymentMethod, paymentRef, salaryAdvanceTransaction);

		salaryAdvanceTransaction.setPaymentType(paymentType);
		salaryAdvanceTransaction.setTransactionType(transactionType.name());
		salaryAdvanceTransaction.setTransactionTime(DataProvider.get().getServerTimestamp());
		salaryAdvanceTransaction.setAmount(transactionType == TransactionType.DEBIT ? -Math.abs(amount) : Math.abs(amount));
		salaryAdvanceTransaction.setOutletId(DataProvider.get().getOutlet().getId());
		return salaryAdvanceTransaction;
	}

	public void loadSalaryAdvancePayment(PaginationSupport model, String salaryUserId, String projectId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(SalaryAdvanceTransaction.class);

			criteria.add(Restrictions.eq(SalaryAdvanceTransaction.PROP_VOIDED, false));
			//criteria.add(Restrictions.between(SalaryAdvanceTransaction.PROP_EVENT_TIME, startOfMonth, endOfMonth));
			if (StringUtils.isNotBlank(salaryUserId)) {
				criteria.add(Restrictions.eq(SalaryAdvanceTransaction.PROP_USER_ID, salaryUserId));
			}

			if (StringUtils.isNotBlank(projectId)) {
				criteria.add(Restrictions.eq(SalaryAdvanceTransaction.PROP_PROJECT_ID, projectId));
			}

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());
			
			criteria.addOrder(Order.desc(SalaryAdvanceTransaction.PROP_TRANSACTION_TIME));

			criteria.setProjection(null);
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());

			List<SalaryAdvanceTransaction> transactions = criteria.list();
			model.setRows(transactions);
		}
	}
	
	public void saveAdvaceSalaryRevertExpenses(SalaryAdvanceTransaction sourceSATransaction, String note, User revertedByUser) throws Exception {

		PaymentType paymentType = sourceSATransaction.getPaymentType();

		boolean isCustomPayment = false;
		Object paymentMethod = null;
		if (PaymentType.CUSTOM_PAYMENT == paymentType) {
			isCustomPayment = true;
		}
		else if (PaymentType.BANK_ACCOUNT == paymentType) {
			String bankAccountId = sourceSATransaction.getBankAccountId();
			if (StringUtils.isNotBlank(bankAccountId)) {
				paymentMethod = BankAccountDAO.getInstance().get(bankAccountId);
			}
		}

		Transaction tx = null;
		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Date now = StoreDAO.getServerTimestamp();
			tx = session.beginTransaction();
			
			User accountsManager = UserDAO.getInstance().get(sourceSATransaction.getAccountManagerId(), sourceSATransaction.getOutletId(), session);
			User paidToUser = UserDAO.getInstance().get(sourceSATransaction.getSalaryPaidTo(), sourceSATransaction.getOutletId(), session);

			Double salaryAmount = sourceSATransaction.getAmount();
			sourceSATransaction.setVoidedByUser(revertedByUser);
			sourceSATransaction.setVoided(true);
			sourceSATransaction.setVoidDate(now);
			sourceSATransaction.putRevertReason(note);

			String batchNo = sourceSATransaction.getBatchNo();
			Project project = sourceSATransaction.getProject();
			String paymentRef = sourceSATransaction.getCustomPaymentRef();

			PosTransactionDAO.getInstance().saveOrUpdate(sourceSATransaction, session);

			BalanceUpdateTransaction balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().saveSalaryExpenseTransaction(session, now, batchNo,
					project, paymentMethod, paymentRef, accountsManager, paidToUser, salaryAmount, note, true, sourceSATransaction, true);

			LedgerEntryDAO.getInstance().saveSalaryAdvanceLedgerEntry(session, sourceSATransaction);

			if (!isCustomPayment) {
				LedgerEntryDAO.getInstance().saveSalaryLedgerEntry(session, balanceUpdateTransaction, true);
			}

			tx.commit();

		}
	}

}