/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.swing;

import java.awt.Color;
import java.awt.Component;
import java.beans.PropertyDescriptor;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.border.Border;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;

import com.floreantpos.PosLog;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;

/**
 * A table model where each row represents one instance of a Java bean.
 * When the user edits a cell the model is updated.
 * 
 * @author Lennart Schedin
 *
 * @param <M> The type of model
 */
@SuppressWarnings("serial")
public class BeanTableModel<M> extends AbstractTableModel implements PaginationSupport {
	private int numRows;
	private int currentRowIndex;
	private int pageSize = 100;

	private List<M> rows = new ArrayList<M>();
	private List<BeanColumn> columns = new ArrayList<BeanColumn>();
	private Class<?> beanClass;

	private String sortBy;
	private boolean ascOrder;

	public enum DataType {
		MONEY, NUMBER, DATE;
	}

	public BeanTableModel(Class<?> beanClass) {
		this(beanClass, 50);
	}

	public BeanTableModel(Class<?> beanClass, int pageSize) {
		this.beanClass = beanClass;
		this.pageSize = pageSize;
	}

	public void setDoubleTextFieldLength(int beforeDecimal, int afterDecimal) {
	}

	public void addColumn(String columnGUIName, String beanAttribute, EditMode editable) {
		addColumn(columnGUIName, beanAttribute, editable, SwingConstants.LEADING, null);
	}

	public void addColumn(String columnGUIName, String beanAttribute, EditMode editable, int horizontalAlignment, DataType dataType) {
		try {
			PropertyDescriptor descriptor = new PropertyDescriptor(beanAttribute, beanClass);
			columns.add(new BeanColumn(columnGUIName, editable, descriptor, horizontalAlignment, dataType));
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void addColumn(String columnGUIName, String beanAttribute) {
		addColumn(columnGUIName, beanAttribute, EditMode.NON_EDITABLE);
	}

	public void addColumn(String columnGUIName, String beanAttribute, int horizontalAlignment, DataType dataType) {
		addColumn(columnGUIName, beanAttribute, EditMode.NON_EDITABLE, horizontalAlignment, dataType);
	}

	public void addRow(M row) {
		rows.add(row);
		fireTableDataChanged();
	}

	public void removeRow(M row) {
		rows.remove(row);
		fireTableDataChanged();
	}

	public void removeRow(int index) {
		rows.remove(index);
		fireTableRowsDeleted(index, index);
	}

	public void removeAll() {
		rows.clear();
		fireTableDataChanged();
	}

	public void addRows(List<M> rows) {
		if (rows == null) {
			return;
		}
		for (M row : rows) {
			addRow(row);
		}
		fireTableDataChanged();
	}

	public void setRows(List rows) {
		this.rows = rows;
		fireTableDataChanged();
	}

	public int getColumnCount() {
		return columns.size();

	}

	public int getRowCount() {
		if (rows == null) {
			return 0;
		}
		return rows.size();
	}

	public Object getValueAt(int rowIndex, int columnIndex) {

		BeanColumn column = columns.get(columnIndex);
		M row = rows.get(rowIndex);

		Object result = null;
		try {
			result = column.descriptor.getReadMethod().invoke(row);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return result;
	}

	public void setValueAt(Object value, int rowIndex, int columnIndex) {
		M row = rows.get(rowIndex);
		BeanColumn column = columns.get(columnIndex);

		try {
			column.descriptor.getWriteMethod().invoke(row, value);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public Class<?> getColumnClass(int columnIndex) {
		BeanColumn column = columns.get(columnIndex);
		Class<?> returnType = column.descriptor.getReadMethod().getReturnType();
		return returnType;
	}

	public String getColumnName(int column) {
		return columns.get(column).columnGUIName;
	}

	public BeanColumn getColumn(int column) {
		return columns.get(column);
	}

	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return columns.get(columnIndex).editable == EditMode.EDITABLE;
	}

	public void setRow(int index, M row) {
		getRows().set(index, row);
		fireTableRowsUpdated(index, index);
	}

	public M getRow(int index) {
		return getRows().get(index);
	}

	public List<M> getRows() {
		return rows;
	}

	public enum EditMode {
		NON_EDITABLE, EDITABLE;
	}

	public int getNumRows() {
		return numRows;
	}

	public void setNumRows(int numRows) {
		this.numRows = numRows;
	}

	public int getCurrentRowIndex() {
		return currentRowIndex;
	}

	public void setCurrentRowIndex(int currentRowIndex) {
		this.currentRowIndex = currentRowIndex;
	}

	public int getPageSize() {
		return pageSize;
	}

	public void setPageSize(int pageSize) {
		this.pageSize = pageSize;
	}

	public boolean hasNext() {
		return (currentRowIndex + pageSize) < numRows;
	}

	public boolean hasPrevious() {
		return currentRowIndex > 0;
	}

	public int getNextRowIndex() {
		if (numRows == 0) {
			return 0;
		}

		return getCurrentRowIndex() + getPageSize();
	}

	public int getPreviousRowIndex() {
		int i = getCurrentRowIndex() - getPageSize();
		if (i < 0) {
			i = 0;
		}

		return i;
	}

	/** One column in the table */
	public static class BeanColumn {
		private String columnGUIName;
		private EditMode editable;
		private PropertyDescriptor descriptor;
		private int horizontalAlignment = SwingConstants.LEADING;
		private DataType dataType;

		//		public BeanColumn(String columnGUIName, EditMode editable, PropertyDescriptor descriptor) {
		//			this.columnGUIName = columnGUIName;
		//			this.editable = editable;
		//			this.descriptor = descriptor;
		//		}

		public BeanColumn(String columnGUIName, EditMode editable, PropertyDescriptor descriptor, int horizontalAlignment, DataType dataType) {
			this.columnGUIName = columnGUIName;
			this.editable = editable;
			this.descriptor = descriptor;
			this.horizontalAlignment = horizontalAlignment;
			this.dataType = dataType;
		}

		public boolean isEditable() {
			if (editable == null) {
				return false;
			}
			if (editable == EditMode.EDITABLE) {
				return true;
			}
			return false;
		}

		public int getHorizontalAlignment() {
			return horizontalAlignment;
		}

		public void setHorizontalAlignment(int horizontalAlignment) {
			this.horizontalAlignment = horizontalAlignment;
		}

		public DataType getDataType() {
			return dataType;
		}
	}

	public static class BeanTableCellRenderer extends DefaultTableCellRenderer {
		private Border unselectedBorder = null;
		private Border selectedBorder = null;

		public BeanTableCellRenderer() {
		}

		public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
			BeanTableModel model = (BeanTableModel) table.getModel();
			BeanColumn beanColumn = model.getColumn(table.convertColumnIndexToModel(column));
			setHorizontalAlignment(beanColumn.getHorizontalAlignment());
			if (selectedBorder == null) {
				selectedBorder = BorderFactory.createMatteBorder(5, 5, 5, 5, table.getSelectionBackground());
			}
			if (unselectedBorder == null) {
				unselectedBorder = BorderFactory.createMatteBorder(5, 5, 5, 5, table.getBackground());
			}

			if (value instanceof byte[]) {
				byte[] imageData = (byte[]) value;
				ImageIcon image = new ImageIcon(imageData);
				image = ImageResource.getScaledImageIcon(image, 40);

				JLabel l = new JLabel(image);
				if (isSelected) {
					l.setBorder(selectedBorder);
				}
				else {
					l.setBorder(unselectedBorder);
				}
				l.setHorizontalAlignment(SwingConstants.CENTER);
				return l;
			}

			if (value instanceof Color) {
				JLabel l = new JLabel();

				Color newColor = (Color) value;
				l.setOpaque(true);
				l.setBackground(newColor);
				if (isSelected) {
					l.setBorder(selectedBorder);
				}
				else {
					l.setBorder(unselectedBorder);
				}
				return l;
			}

			if (value instanceof Date) {
				String valueStr = DateUtil.formatFullDateAndTimeWithoutYearAsString((Date) value);
				return super.getTableCellRendererComponent(table, valueStr, isSelected, hasFocus, row, column);
			}

			DataType dataType = beanColumn.getDataType();
			if (dataType == null) {
				return super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
			}
			switch (dataType) {
				case MONEY:
					try {
						value = NumberUtil.getCurrencyFormat(value);
					} catch (Exception e) {
					}
					break;

				case NUMBER:
					try {
						value = NumberUtil.format6DigitNumber((Double) value);
					} catch (Exception e) {
					}
					break;
				default:
					value = "<html>" + value + "</html>"; //$NON-NLS-1$ //$NON-NLS-2$
					break;
			}
			return super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
		}
	}

	public void setSortBy(String propertyName, boolean ascOrder) {
		this.sortBy = propertyName;
		this.ascOrder = ascOrder;
	}

	public boolean isAscOrder() {
		return ascOrder;
	}

	public String getSortBy() {
		return sortBy;
	}
}