/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemGroup;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.util.DataProvider;

public class UpdateDBTo437 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo437(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		int totalTestItems = rowCount();
		PosLog.info(UpdateDBTo416.class, "Found total test item: " + totalTestItems); //$NON-NLS-1$

		int increment = 10;
		int count = 0;
		while (count < totalTestItems) {
			try (Session session = sessionFactory.openSession()) {
				Transaction tx = session.beginTransaction();

				Pagination<TestItem> pagination = new Pagination<TestItem>(count, increment);
				pagination.setNumRows(totalTestItems);
				findTestItems(pagination, session);

				List<TestItem> testItems = pagination.getDataList();
				for (TestItem testItem : testItems) {
					StringBuilder sb = new StringBuilder();

					String testItemGroupId = testItem.getTestItemGroupId();
					if (StringUtils.isNotBlank(testItemGroupId)) {
						sb.append("\ntestItemGroupId: " + testItemGroupId);
						TestItemGroup testItemGroup = (TestItemGroup) DataProvider.get().getObjectOf(TestItemGroup.class, testItemGroupId);
						if (testItemGroup != null) {
							testItem.setTestItemGroupName(testItemGroup.getName());

							sb.append(", ");
							sb.append("TestItem GroupName: " + testItemGroup.getName());
						}
					}

					String message = sb.toString();
					if (StringUtils.isNotBlank(message)) {
						PosLog.info(getClass(),
								"TestItem (" + testItem.getName() + "_" + testItem.getId() + ") is updating from field to property: " + message);
						TestItemDAO.getInstance().saveOrUpdate(testItem, session);
					}

				}
				tx.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

	private void findTestItems(Pagination<TestItem> pagination, Session session) {
		Criteria criteria = createTestItemCriteria(session);
		criteria.addOrder(Order.asc(TestItem.PROP_NAME));
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

	public int rowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createTestItemCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createTestItemCriteria(Session session) {
		Criteria criteria = session.createCriteria(TestItem.class);
		criteria.add(Restrictions.isNotNull(TestItem.PROP_TEST_ITEM_GROUP_ID)); //$NON-NLS-1$
		return criteria;
	}

}