/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.report;

import java.util.List;

import javax.swing.table.AbstractTableModel;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.util.NumberUtil;

public class SalesReportModel extends AbstractTableModel {

	private String[] columnNames = { "barcode", "name", "unit", "qty", "price", "cost", "costPercentage", "grossTotal", "tax", "netTotal", "group", "discount", //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$//$NON-NLS-5$//$NON-NLS-6$//$NON-NLS-7$//$NON-NLS-8$//$NON-NLS-9$//$NON-NLS-10$//$NON-NLS-11$ //$NON-NLS-12$
			"serviceCharge", "grossSale", "returnAmount", "profit", "profitPercentage", "userName", "outletId", "qtyAndUnit", "taxName", "labDoctorFee", //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$//$NON-NLS-5$//$NON-NLS-6$//$NON-NLS-7$//$NON-NLS-8$//$NON-NLS-9$//$NON-NLS-10$
			"agentFee", "rfOnNetsalesAmount" };//$NON-NLS-1$
	private List<ReportItem> items;

	private double gndTotalGrossSales;
	private double gndTotalDiscount;
	private double gndTotalRFCommission;
	private double gndTotalLabDoctorFee;
	private double gndTotalReturn;
	private double gndTotalNetSales;
	private double gndTotalServCharge;
	private double gndTotalTax;
	private double gndTotalTotal;
	private double gndTotalCost;
	private double gndTotalProfit;
	private double gndTotalQty;

	public SalesReportModel() {
		super();
	}

	public int getRowCount() {
		if (items == null) {
			return 0;
		}

		return items.size();
	}

	public int getColumnCount() {
		return columnNames.length;
	}

	@Override
	public String getColumnName(int column) {
		return columnNames[column];
	}

	public Object getValueAt(int rowIndex, int columnIndex) {
		ReportItem item = items.get(rowIndex);

		switch (columnIndex) {
			case 0:
				return item.getBarcode() != null ? item.getBarcode() : item.getId();
			case 1:
				return item.getName();

			case 2:
				return item.getUnit();

			case 3:
				return item.getQuantity();

			case 4:
				return item.getPrice();

			case 5:
				return item.getCost();

			case 6:
				if (item.getAdjustedPrice() > 0)
					return (item.getCost() / item.getAdjustedPrice()) * 100;
				return 0.0;

			case 7:
				return item.getGrossTotal();

			case 8:
				return item.getTaxTotal();

			case 9:
				return item.getNetTotal();

			case 10:
				if (item.getGroupName() != null) {
					return Messages.getString("SalesReportModel.1") + item.getGroupName(); //$NON-NLS-1$
				}
				return Messages.getString("SalesReportModel.2"); //$NON-NLS-1$

			case 11:
				return item.getDiscount();

			case 12:
				return item.getServiceCharge();

			case 13:
				//net sales
				return item.getGrossSale();

			case 14:
				return item.getReturnAmount();

			case 15:
				//gross profit = total - cost of goods
				return item.getProfit();

			case 16:
				//profit margin percentage calculation:
				//profit percentage = gross profit / total * 100
				return item.getProfitPrecentage();

			case 17:
				return item.getUserName();

			case 18:
				return item.getOutletId();

			case 19:
				String qtyAndUnit = NumberUtil.formatNumberIfNeeded(item.getQuantity());
				if (StringUtils.isNotBlank(item.getUnit())) {
					qtyAndUnit += "" + item.getUnit(); //$NON-NLS-1$
				}
				return qtyAndUnit;

			case 20:
				return item.getTaxName();

			case 21:
				return item.getLabDoctorFeeAmount();

			case 22:
				return item.getAgentFeeAmount();

			case 23:
				return item.getRfOnNetsalesAmount();

		}

		return null;
	}

	public List<ReportItem> getItems() {
		return items;
	}

	public void setItems(List<ReportItem> items) {
		this.items = items;
	}

	public double getGndTotalGrossSales() {
		return gndTotalGrossSales;
	}

	public void setGndTotalGrossSales(double gndTotalGrossSales) {
		this.gndTotalGrossSales = gndTotalGrossSales;
	}

	public double getGndTotalDiscount() {
		return gndTotalDiscount;
	}

	public void setGndTotalDiscount(double gndTotalDiscount) {
		this.gndTotalDiscount = gndTotalDiscount;
	}

	public double getGndTotalRFCommission() {
		return gndTotalRFCommission;
	}

	public void setGndTotalRFCommission(double gndTotalRFCommission) {
		this.gndTotalRFCommission = gndTotalRFCommission;
	}

	public double getGndTotalLabDoctorFee() {
		return gndTotalLabDoctorFee;
	}

	public void setGndTotalLabDoctorFee(double gndTotalLabDoctorFee) {
		this.gndTotalLabDoctorFee = gndTotalLabDoctorFee;
	}

	public double getGndTotalReturn() {
		return gndTotalReturn;
	}

	public void setGndTotalReturn(double gndTotalReturn) {
		this.gndTotalReturn = gndTotalReturn;
	}

	public double getGndTotalNetSales() {
		return gndTotalNetSales;
	}

	public void setGndTotalNetSales(double gndTotalNetSales) {
		this.gndTotalNetSales = gndTotalNetSales;
	}

	public double getGndTotalServCharge() {
		return gndTotalServCharge;
	}

	public void setGndTotalServCharge(double gndTotalServCharge) {
		this.gndTotalServCharge = gndTotalServCharge;
	}

	public double getGndTotalTax() {
		return gndTotalTax;
	}

	public void setGndTotalTax(double gndTotalTax) {
		this.gndTotalTax = gndTotalTax;
	}

	public double getGndTotalTotal() {
		return gndTotalTotal;
	}

	public void setGndTotalTotal(double gndTotalTotal) {
		this.gndTotalTotal = gndTotalTotal;
	}

	public double getGndTotalCost() {
		return gndTotalCost;
	}

	public void setGndTotalCost(double gndTotalCost) {
		this.gndTotalCost = gndTotalCost;
	}

	public double getGndTotalProfit() {
		return gndTotalProfit;
	}

	public void setGndTotalProfit(double gndTotalProfit) {
		this.gndTotalProfit = gndTotalProfit;
	}

	public double getGndTotalQty() {
		return gndTotalQty;
	}

	public void setGndTotalQty(double gndTotalQty) {
		this.gndTotalQty = gndTotalQty;
	}

	public void calculateGrandTotal() {
		if (items == null) {
			return;
		}
		double grossSale = 0;
		double discount = 0;
		double rfCommission = 0;
		double labDoctorFee = 0;
		double returnAmount = 0;
		double netSales = 0;
		double servCharge = 0;
		double totaltax = 0;
		double grossTotal = 0;
		double totalCost = 0;
		double totalProfit = 0;
		double totalQty = 0;

		for (ReportItem item : items) {
			grossSale += item.getGrossSale();
			discount += item.getDiscount();
			rfCommission += item.getAgentFeeAmount();
			labDoctorFee += item.getLabDoctorFeeAmount();
			returnAmount += item.getReturnAmount();
			netSales += item.getNetTotal();
			servCharge += item.getServiceCharge();
			totaltax += item.getTaxTotal();
			grossTotal += item.getGrossTotal();
			totalCost += item.getCost();
			totalProfit += item.getGrossTotal() - item.getCost();
			totalQty += item.getQuantity();
		}

		setGndTotalGrossSales(grossSale);
		setGndTotalDiscount(discount);
		setGndTotalRFCommission(rfCommission);
		setGndTotalLabDoctorFee(labDoctorFee);
		setGndTotalReturn(returnAmount);
		setGndTotalNetSales(netSales);
		setGndTotalServCharge(servCharge);
		setGndTotalTax(totaltax);
		setGndTotalTotal(grossTotal);
		setGndTotalCost(totalCost);
		setGndTotalProfit(totalProfit);
		setGndTotalQty(totalQty);
	}
}
