/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;

import com.floreantpos.PosLog;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.PurchaseOrderDAO;
import com.floreantpos.model.util.DataProvider;

public class UpdateDBTo467 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo467(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$
		
		updatePOFields();
		updateTransactionFields();
		
		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

	public void updateTransactionFields() {
		

		int totalTransactions = rowCount();
		PosLog.info(UpdateDBTo467.class, "Found total transactions: " + totalTransactions); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalTransactions) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<PosTransaction> pagination = new Pagination<PosTransaction>(count, increment);
				pagination.setNumRows(totalTransactions);
				findPosTransactions(pagination, session);

				List<PosTransaction> posTransactions = pagination.getDataList();
				for (PosTransaction posTransaction : posTransactions) {
					String coaCode = posTransaction.getProperty("chartOfAccountsCode", "");
					if (StringUtils.isNotBlank(coaCode)) {
						ChartOfAccounts chartOfAccounts = DataProvider.get().getCOAFromMap(coaCode);
						posTransaction.setChartOfAccounts(chartOfAccounts);
					}
				}
				transaction.commit();

				count += increment;
			}
		}

	
	}

	private void findPosTransactions(Pagination<PosTransaction> pagination, Session session) {
		Criteria criteria = createTransactionsCriteria(session);
		criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

	public int rowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createTransactionsCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createTransactionsCriteria(Session session) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		return criteria;
	}

	public void updateOutletId(String tableName, String outletID) {
		try (Session session = sessionFactory.openSession()) {
			Transaction transaction = session.beginTransaction();

			String hqlString = "update " + schemaName + ".%s set outlet_id = '%s'";
			hqlString = String.format(hqlString, tableName, outletID);

			SQLQuery sqlQuery = session.createSQLQuery(hqlString); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

			PosLog.info(getClass(), "updated query: " + sqlQuery + ""); //$NON-NLS-1$ //$NON-NLS-2$
			int update = sqlQuery.executeUpdate();
			PosLog.info(getClass(), "updated " + update + " " + tableName); //$NON-NLS-1$ //$NON-NLS-2$
			transaction.commit();
		}

	}

	public void updatePOFields() {


		int totalPO = poRowCount();
		PosLog.info(UpdateDBTo467.class, "Found total PO: " + totalPO); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalPO) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<PurchaseOrder> pagination = new Pagination<PurchaseOrder>(count, increment);
				pagination.setNumRows(totalPO);
				findPO(pagination, session);

				List<PurchaseOrder> orders = pagination.getDataList();
				for (PurchaseOrder purchaseOrder : orders) {

					if (!purchaseOrder.isAdjustedInventory() && !purchaseOrder.hasFixedAssetItems()) {
						purchaseOrder.putCOAId("5021");
					}

					String coaId = purchaseOrder.getCOAId();
					if (StringUtils.isNotBlank(coaId)) {
						ChartOfAccounts chartOfAccounts = DataProvider.get().getCOAFromMap(coaId);
						List<PosTransaction> poTransactions = purchaseOrder.getTransactions();
						if (poTransactions != null) {
							for (PosTransaction potran : poTransactions) {
								potran.setChartOfAccounts(chartOfAccounts);
								PosTransactionDAO.getInstance().saveOrUpdate(potran, session);
								
							}
						}
						PurchaseOrderDAO.getInstance().saveOrUpdate(purchaseOrder, session);
					}
				}
				transaction.commit();

				count += increment;
			}
		}

	}

	public int poRowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createPOCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createPOCriteria(Session session) {
		Criteria criteria = session.createCriteria(PurchaseOrder.class);
		return criteria;
	}

	private void findPO(Pagination<PurchaseOrder> pagination, Session session) {
		Criteria criteria = createPOCriteria(session);
		criteria.addOrder(Order.asc(PurchaseOrder.PROP_CREATED_DATE));
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

}