package com.floreantpos.services.report;

import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.ImageIcon;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.model.Address;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PatientAllergy;
import com.floreantpos.model.Prescription;
import com.floreantpos.model.PrescriptionItem;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Store;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.floreantpos.model.dao.ImageResourceDAO;
import com.floreantpos.model.dao.PatientDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.ReportUtil;
import com.floreantpos.swing.ListTableModel;
import com.floreantpos.util.ImageUtil;
import com.floreantpos.util.POSUtil;

import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;

public class PrescriptionService {
	public JasperPrint getPrescriptionAsJasperPrint(Prescription prescription, String patientPortalLink) throws Exception {
		return getPrescriptionAsJasperPrint(prescription, null, patientPortalLink);
	}

	public JasperPrint getPrescriptionAsJasperPrint(Prescription prescription, Doctor doctor, String patientPortalLink) throws Exception {
		return getPrescriptionAsJasperPrint(prescription, doctor, null, patientPortalLink);
	}

	public JasperPrint getPrescriptionAsJasperPrint(Prescription prescription, Doctor doctor, Map<String, Object> mapForPreview, String patientPortalLink)
			throws Exception {
		HashMap<String, Object> map = new HashMap<String, Object>();

		Outlet outlet = DataProvider.get().getOutlet();

		JasperReport topHeader = outlet.isShowPrintPrescriptionStoreLogo() ? ReportUtil.getReport("top_header_for_prescription_with_logo")
				: ReportUtil.getReport("top_header_for_prescription");
		map.put("topHead", topHeader); //$NON-NLS-1$

		Store store = StoreDAO.getRestaurant();
		if (store != null) {
			map.put("labName", store.getName()); //$NON-NLS-1$
			String labInfo = StringUtils.EMPTY;
			if (outlet != null) {
				labInfo += outlet.getAddressLine1() + "<br>"; //$NON-NLS-1$
				if (StringUtils.isNotBlank(outlet.getAddressLine2())) {
					labInfo += outlet.getAddressLine2() + "<br>"; //$NON-NLS-1$
				}
				if (StringUtils.isNotBlank(outlet.getAddressLine3())) {
					labInfo += outlet.getAddressLine3() + "<br>"; //$NON-NLS-1$
				}

				StringBuilder hotlineOrWebAddressBuilder = new StringBuilder();
				Address address = outlet.getAddress();
				if (address != null) {
					String telephone = address.getTelephone();
					if (StringUtils.isNotBlank(telephone)) {
						hotlineOrWebAddressBuilder.append("Hotline: " + telephone + "<br>"); //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
				String websiteAddress = outlet.getWebsiteAddress();
				if (StringUtils.isNotBlank(websiteAddress)) {
					hotlineOrWebAddressBuilder.append("Web: " + "<a style='color: black' href='" + websiteAddress + "'>" + websiteAddress + "</a>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				}
				labInfo += hotlineOrWebAddressBuilder.toString();
			}
			map.put("labInfo", labInfo); //$NON-NLS-1$
		}

		JasperReport header = ReportUtil.getReport("patient_header_for_prescription"); //$NON-NLS-1$
		map.put("head", header); //$NON-NLS-1$
		map.put("prescriptionId", prescription.getId()); //$NON-NLS-1$
		map.put("barcode", prescription.getId()); //$NON-NLS-1$

		Customer patient = prescription.getPatient();
		if (patient == null && StringUtils.isNotBlank(prescription.getPatientId())) {
			patient = PatientDAO.getInstance().get(prescription.getPatientId());
		}
		if (patient != null) {
			map.put("patientId", patient.getId()); //$NON-NLS-1$
			String patientName = patient.getName();
			if (patientName != null) {
				patientName = patientName.trim();
			}
			map.put("ptName", patientName); //$NON-NLS-1$
			map.put("address", StringUtils.isNotBlank(patient.getPatientAddressLine()) ? patient.getPatientAddressLine().trim() : ""); //$NON-NLS-1$ //$NON-NLS-2$
			map.put("phone", patient.getMobileNo() == null ? "" : patient.getMobileNo()); //$NON-NLS-1$
			map.put("sex", patient.getGender()); //$NON-NLS-1$
			Date dateOfBirth = patient.getDateOfBirth();
			if (dateOfBirth != null) {
				map.put("age", patient.getApproxAge()); //$NON-NLS-1$
			}
		}
		else {
			map.put("patientId", ""); //$NON-NLS-1$
			map.put("ptName", ""); //$NON-NLS-1$
			map.put("address", ""); //$NON-NLS-1$ 
			map.put("phone", ""); //$NON-NLS-1$
			map.put("sex", ""); //$NON-NLS-1$
			map.put("age", ""); //$NON-NLS-1$
		}
		if (prescription.getCreatedDate() != null) {
			map.put("date", DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(prescription.getCreatedDate()), "dd/MM/yy")); //$NON-NLS-1$
		}
		else {
			map.put("date", DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd/MM/yy")); //$NON-NLS-1$
		}

		map.put("qrCode", patientPortalLink); //$NON-NLS-1$
		map.put("qrCodeBottomText", "Scan for online view"); //$NON-NLS-1$

		StringBuilder patientInfo = new StringBuilder();

		String heightInCM = prescription.getHeightInCM();
		if (StringUtils.isNotBlank(heightInCM)) {
			patientInfo.append("Height: ");
			patientInfo.append(String.format("%s cm<br>", heightInCM));
		}

		String weightInKG = prescription.getWeightInKG();
		if (StringUtils.isNotBlank(weightInKG)) {
			patientInfo.append("Weight: ");
			patientInfo.append(weightInKG + " KG <br>");
		}

		String temparature = prescription.getTemparature();
		if (StringUtils.isNotBlank(temparature)) {
			patientInfo.append("Body temparature: ");
			patientInfo.append(temparature + "<span>'F</span><br>");
		}

		String bpUp = prescription.getBpUp();
		String bpDown = prescription.getBpDown();
		if (StringUtils.isNotBlank(bpUp)) {
			patientInfo.append("Blood pressure: ");
			patientInfo.append(bpUp + "/" + bpDown + " mmHg <br>");
		}

		String bpm = prescription.getBpm();
		if (StringUtils.isNotBlank(bpm)) {
			patientInfo.append("Heart bit: ");
			patientInfo.append(bpm + "<br>"); //$NON-NLS-1$
		}

		List<String> problems = prescription.getProblems();
		if (!problems.isEmpty()) {
			for (String problem : problems) {
				patientInfo.append("<div> * " + problem + "</div>");
			}
			patientInfo.append("<br>");
		}

		else if (StringUtils.isNotBlank(prescription.getProperty("notes"))) {
			patientInfo.append("<br>");
			patientInfo.append(prescription.getProperty("notes") + "<br>");
		}

		Boolean showBanglaTextOnPrescription = outlet.isShowReportsInBangla();
		String testName = "";
		String medicines = "";
		if (prescription.getItems() != null) {
			List<String> medicineAndServiceType = Arrays.asList(ProductType.MEDICINE.name(), ProductType.SERVICES.name());
			for (Iterator<PrescriptionItem> iterator = prescription.getItems().iterator(); iterator.hasNext();) {
				PrescriptionItem presItem = (PrescriptionItem) iterator.next();
				if (presItem.getProductType() == null || !medicineAndServiceType.contains(presItem.getProductType())) {
					testName += outlet.isShowReportsInBangla() ? "<div> * " + presItem.getName() + "</div>" //$NON-NLS-1$ //$NON-NLS-2$
							: "<span style=\"font-family: 'SansSerif';\"> * " + presItem.getName() + " <br></span>"; //$NON-NLS-1$ //$NON-NLS-2$
				}
				else {
					medicines += outlet.isShowReportsInBangla() ? "<div> * " + presItem.getName() + "</div>" //$NON-NLS-1$ //$NON-NLS-2$
							: "<span style=\"font-family: 'SansSerif';\"> * " + presItem.getName() + "</span>"; //$NON-NLS-1$ //$NON-NLS-2$

					String instructions = presItem.getInstructionDisplay(false, showBanglaTextOnPrescription);

					if (StringUtils.isNotBlank(instructions)) {
						medicines += "<div>" + instructions + "</div>";
					}

					String notes = presItem.getProperty("medicine.time.notes");
					if (StringUtils.isNotBlank(notes)) {
						medicines += "<div>" + notes + "</div>";
					}
					medicines += "<p></p>"; //$NON-NLS-1$
				}
			}
		}

		String comment = prescription.getComments();
		Date nextFollowUpDate = prescription.getNextFollowUpDate();
		String nextDate = nextFollowUpDate != null ? ("Next follow up date: " + DateUtil.formatAsShortDate(nextFollowUpDate)) : StringUtils.EMPTY;

		StringBuilder patientAllergy = new StringBuilder();

		List<PatientAllergy> patientAllergyList = prescription.getPatientAllergyList();
		if (!patientAllergyList.isEmpty()) {
			for (PatientAllergy allergy : patientAllergyList) {
				patientAllergy.append("<div> * " + allergy.getAllergyName() + "</div>"); //$NON-NLS-1$ //$NON-NLS-2$
			}
			patientAllergy.append("<br>"); //$NON-NLS-1$
		}
		map.put("visibleAllergy", patientAllergyList.size() != 0); //$NON-NLS-1$

		PTableModel model = new PTableModel(patientInfo.toString(), patientAllergy.toString(), testName, medicines, comment, nextDate);

		PrescriptionDataSource source = new PrescriptionDataSource();
		source.setRows(Arrays.asList(model));
		JRDataSource dataSource = new JRTableModelDataSource(source);

		boolean visiblePrescriptionPadHeader = true;
		boolean isDoctorPrescriptionEditable = false;
		int topMargin = 0;
		String doctorId = prescription.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			if (doctor == null) {
				doctor = DoctorDAO.getInstance().get(doctorId);
			}
			if (GlobalConfigDAO.getInstance().isDoctorsCanChangePrescriptionPad()) {
				isDoctorPrescriptionEditable = true;
				visiblePrescriptionPadHeader = POSUtil.getBoolean(doctor.getProperty("prescription.header"), true);
				topMargin = POSUtil.parseInteger(doctor.getProperty("prescription.top_padding"));
			}
			map.put("visiblePrescriptionPadHeader", visiblePrescriptionPadHeader); //$NON-NLS-1$
			map.put("drName", doctor.getName()); //$NON-NLS-1$
			String doctorDesignation = doctor.getDoctorDesignation();
			if (StringUtils.isNotBlank(doctorDesignation)) {
				doctorDesignation = doctorDesignation.replaceAll("\n", "<br>");
				map.put("doctorDesignation", doctorDesignation); //$NON-NLS-1$
			}
			if (doctor.isAllowDigitalSignature() && StringUtils.isNotBlank(doctor.getSignatureImageId())) {
				ImageResource imageResource = ImageResourceDAO.getInstance().get(doctor.getSignatureImageId());
				if (imageResource != null) {
					map.put("signature", imageResource.getImage()); //$NON-NLS-1$
				}
			}
		}

		//Additional 20 is temporary value
		int lastPagefooterMargin = 0;
		int footerMargin = 0;
		if (topMargin > 0) {
			topMargin += 20;
			lastPagefooterMargin += 20;
			footerMargin += 20;
		}

		if (visiblePrescriptionPadHeader && topMargin < 36) {
			topMargin = 36;
		}

		map.put("colPrintedBy", //$NON-NLS-1$
				"Printed by Medlogics on " //$NON-NLS-1$
						+ DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd/MM/yy hh:mm a") + " " //$NON-NLS-1$//$NON-NLS-2$
		);

		if (outlet.isShowPrintPrescriptionStoreLogo()) {
			ImageIcon storeLogo = outlet.getStoreLogo();
			if (storeLogo != null) {
				ImageIcon imageIcon = ImageUtil.pngToJpg(storeLogo);
				if (imageIcon != null) {
					if (outlet.isShowPrintPrescriptionStoreLogo()) {
						map.put("storeLogo", imageIcon.getImage()); //$NON-NLS-1$
					}
				}
			}
		}

		if (outlet.isShowPadImageInPrescription()) {
			String imageId = outlet.getPreviewPrescriptionImageId();
			if (StringUtils.isNotBlank(imageId)) {
				ImageResource imageResource = ImageResourceDAO.getInstance().get(imageId);
				map.put("imagePath", imageResource == null ? null : imageResource.getImage()); //$NON-NLS-1$
			}
		}

		if (mapForPreview != null) {
			map.putAll(mapForPreview);
		}

		if (mapForPreview == null) {
			map.put("visiblePrescriptionPadHeader", isDoctorPrescriptionEditable ? visiblePrescriptionPadHeader : outlet.isShowPrescriptionPrintHeader()); //$NON-NLS-1$
		}

		JasperReport prescriptionJr = ReportUtil.adjustReportHeaderAndFooter("prescription", outlet.getPrescriptionFooterMargin(),
				isDoctorPrescriptionEditable ? topMargin : outlet.getPrescriptionHeaderMargin(), outlet.getPrescriptionFooterMargin());
		JasperPrint jasperPrint = JasperFillManager.fillReport(prescriptionJr, map, dataSource);
		return jasperPrint;
	}

	class PrescriptionDataSource extends ListTableModel<PTableModel> {

		public PrescriptionDataSource() {
			super(new String[] { "patient_condition", "left_side", "right_side", "comment", "next_date", "patient_allergy" });
		}

		@Override
		public Object getValueAt(int rowIndex, int columnIndex) {
			PTableModel data = rows.get(rowIndex);
			switch (columnIndex) {
				case 0:
					return data.getPatientCondition();

				case 1:
					return data.getLeftSide();

				case 2:
					return data.getRightSide();

				case 3:
					return data.getComment();

				case 4:
					return data.getNextDate();

				case 5:
					return data.getPatientAllergy();
			}
			return null;
		}

	}

	public class PTableModel {
		private String patientCondition;
		private String patientAllergy;
		private String leftSide;
		private String rightSide;
		private String comment;
		private String nextDate;

		public PTableModel(String patientCondition, String patientAllergy, String leftSide, String rightSide, String comment, String nextDate) {
			this.patientCondition = patientCondition;
			this.patientAllergy = patientAllergy;
			this.leftSide = leftSide;
			this.rightSide = rightSide;
			this.comment = comment;
			this.nextDate = nextDate;
		}

		public String getPatientCondition() {
			return patientCondition;
		}

		public String getPatientAllergy() {
			return patientAllergy;
		}

		public String getLeftSide() {
			return leftSide;
		}

		public String getRightSide() {
			return rightSide;
		}

		public String getComment() {
			return comment;
		}

		public String getNextDate() {
			return nextDate;
		}

	}
}