/**
 * ************************************************************************

 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import static com.floreantpos.util.NumberUtil.convertToBigDecimal;
import static com.floreantpos.util.NumberUtil.round;

import java.beans.Transient;
import java.io.IOException;
import java.io.StringReader;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.base.BaseTicket;
import com.floreantpos.model.dao.ActionHistoryDAO;
import com.floreantpos.model.dao.ShopTableDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.DiscountUtil;
import com.floreantpos.util.JsonUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.NumericGlobalIdGenerator;
import com.floreantpos.util.POSUtil;
import com.floreantpos.util.ShiftUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.orocube.common.util.TicketStatus;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "properties", "customer", "tables", "terminal", "createDateFormatted", "title", "outlet", "customerName",
		"customerEmail", "customerMobileNo", "bartabName", "gratuityAmount", "serviceChargePaidAmount", "serviceChargeDueAmount", "owner", "cashier", "shift",
		"customer", "department", "salesArea", "assignedDriver", "voidedBy", "terminal", "beverageCount", "beverageItemCount", "ticketStatus" })
@XmlSeeAlso({ PosTransaction.class, TicketItem.class })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement(name = "ticket")
public class Ticket extends BaseTicket implements TimedModel, PropertyContainer {
	public static final String CUSTOMER_TAX_EXEMPT = "customer.taxExempt"; //$NON-NLS-1$

	private static final long serialVersionUID = 1L;

	public static final String JSON_PROP_TICKET_DISC_AMOUNT = "ticketDiscAmount"; //$NON-NLS-1$
	public static final String JSON_PROP_ITEM_DISC_AMOUNT = "itemDiscAmount"; //$NON-NLS-1$

	public final static String PROP_TRANSACTIONS = "transactions"; //$NON-NLS-1$
	public final static String PROP_TICKET_ITEMS = "ticketItems"; //$NON-NLS-1$

	public final static String PROPERTY_ONLINE_ID = "onlineOrderId"; //$NON-NLS-1$
	public final static String PROPERTY_CARD_TRANSACTION_ID = "card_transaction_id"; //$NON-NLS-1$
	public final static String PROPERTY_CARD_TRACKS = "card_tracks"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_NAME = "card_name"; //$NON-NLS-1$
	public static final String PROPERTY_PAYMENT_METHOD = "payment_method"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_READER = "card_reader"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_NUMBER = "card_number"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_EXP_YEAR = "card_exp_year"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_EXP_MONTH = "card_exp_month"; //$NON-NLS-1$
	public static final String PROPERTY_ADVANCE_PAYMENT = "advance_payment"; //$NON-NLS-1$
	public static final String PROPERTY_CARD_AUTH_CODE = "card_auth_code"; //$NON-NLS-1$
	public static final String PROPERTY_SUB_ORDER_TYPE = "ticket.sub_order_type"; //$NON-NLS-1$
	public static final String PROPERTY_LOYALTY_ADDED = "loyalty"; //$NON-NLS-1$
	public static final String PROPERTY_EXTRA_SEATS = "extra_seats"; //$NON-NLS-1$
	public static final String PROPERTY_OUTLET_SC_RATE = "storeScRate"; //$NON-NLS-1$
	private static final String PROPERTY_OUTLET_GRATUITY_RATE = "storeGratuityRate"; //$NON-NLS-1$

	public static final String CUSTOMER_PHONE = "CUSTOMER_MOBILE"; //$NON-NLS-1$
	public static final String CUSTOMER_NAME = "CUSTOMER_NAME"; //$NON-NLS-1$
	public static final String CUSTOMER_EMAIL = "CUSTOMER_EMAIL"; //$NON-NLS-1$
	public static final String JSON_PROP_BARTAB_NAME = "BARTAB_NAME"; //$NON-NLS-1$
	public static final String CUSTOMER_LAST_NAME = "CUSTOMER_LAST_NAME"; //$NON-NLS-1$
	public static final String CUSTOMER_ID = "CUSTOMER_ID"; //$NON-NLS-1$
	public static final String CUSTOMER_ZIP_CODE = "CUSTOMER_ZIP_CODE"; //$NON-NLS-1$
	public static final String MANAGER_INSTRUCTION = "MANAGER_INSTRUCTION"; //$NON-NLS-1$
	public static final String PHONE_EXTENSION = "PHONE_EXTENSION"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_ADDRESS = "CUSTOMER_ADDRESS"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_COUNTRY = "CUSTOMER_COUNTRY"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_STATE = "CUSTOMER_STATE"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_CITY = "CUSTOMER_CITY"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_ZIP = "CUSTOMER_ZIP"; //$NON-NLS-1$

	public static final String CUSTOMER_TOTAL_LOYALTY_POINT = "CUSTOMER_LOYALTY_POINT"; //$NON-NLS-1$

	public static final String DRIVER_OUT_TIME = "OUT_AT"; //$NON-NLS-1$

	public static final String PROPERTY_DISCOUNT = "DISCOUNT"; //$NON-NLS-1$

	public static final int ORDER_PENDING = 0;
	public static final int ORDER_VERIFIED = 1;
	public static final int ORDER_FULLY_INVOICED = 2;
	public static final int ORDER_PARTIALLY_INVOICED = 3;
	public static final int ORDER_FULLY_SHIPMENT = 4;
	public static final int ORDER_PARTIALLY_SHIPMENT = 5;
	public static final int ORDER_PARTIALLY_SHIPMENT_AND_INVOICED = 6;
	public static final int ORDER_CLOSED = 7;
	public static final int ORDER_CANCELLED = 8;

	private transient Map<String, ActionHistory> events = new LinkedHashMap<>();

	//@formatter:off
	public static final String[] ORDER_STATUS = { "Pending", //$NON-NLS-1$
			"Verified", //$NON-NLS-1$
			"Fully Invoiced", //$NON-NLS-1$
			"Partially Invoiced", //$NON-NLS-1$
			"Fully Shipped", //$NON-NLS-1$
			"Partially Shipped", //$NON-NLS-1$
			"Partially Shipped and Invoiced", //$NON-NLS-1$
			"Closed", //$NON-NLS-1$
			"Cancelled" //$NON-NLS-1$
	};
	//@formatter:on

	public static final String ORDER_VERIFIED_BY = "Verified_By"; //$NON-NLS-1$
	public static final String ORDER_SENT_BY = "Sent_By"; //$NON-NLS-1$
	public static final String ORDER_SHIPMENT_BY = "Shipping_By"; //$NON-NLS-1$
	public static final String ORDER_INVOICED_BY = "Invoiced_By"; //$NON-NLS-1$
	public static final String ORDER_CLOSED_BY = "Closed_By"; //$NON-NLS-1$

	public static final String DEBIT = "DEBIT"; //$NON-NLS-1$
	public static final String CREDIT = "CREDIT"; //$NON-NLS-1$

	public static final String SPLIT = "split"; //$NON-NLS-1$
	public static final String SPLIT_TICKET_ID = "split_ticket_id"; //$NON-NLS-1$
	public static final String SPLIT_SEAT_NUMBER = "split_seat_number"; //$NON-NLS-1$
	public static final String SPLIT_TYPE = "SPLIT_TYPE"; //$NON-NLS-1$
	public static final int SPLIT_EQUALLY = 0;
	public static final int SPLIT_BY_SEAT = 1;
	public static final int SPLIT_MANUALLY = 2;

	/**
	 * This is the total discount amount applied to this ticket, not to any item.
	 * That is, this total of TicketDiscount, not TicketItemDiscount.
	 */
	private double voidSubtotal;
	private double voidTotal;
	private double itemDiscountAmount = 0;
	private double ticketDiscountAmount = 0;
	private List deletedItems;
	private String sortOrder;
	private transient Customer customer;
	private transient Outlet outlet;
	private transient Department department;
	private transient SalesArea salesArea;
	private List<TicketDiscount> discounts;
	private List<Integer> tableNumbers;
	private List<ShopTable> tables;
	private boolean shouldUpdateTableStatus;
	private transient boolean shouldUpdateStock;
	private double toleranceAmount;
	private Boolean refundableItem;

	private transient OrderType orderType;
	private transient com.google.gson.JsonObject propertiesContainer;
	private boolean shouldPublishMqtt = true;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private static final String jSON_PROP_TABLE_NAMES = "tableNames"; //$NON-NLS-1$

	public static final String PROP_ID = "id";
	public static final String PROP_OUTLET_ID = "outletId";

	public Ticket() {
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public Ticket(String ticketId, String outletId) {
		super(ticketId, outletId);
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public Ticket(boolean populateDefaultProperties) {
		super();
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
		if (populateDefaultProperties) {
			populateDefaultProperties();
		}
	}

	public TicketType getTicketType() {
		return TicketType.getByTypeNo(super.getType());
	}

	public void setTicketType(TicketType type) {
		if (type != null) {
			super.setType(type.getTypeNo());
		}
	}

	public void addTable(int tableNumber) {
		List<Integer> numbers = getTableNumbers();
		if (numbers == null) {
			numbers = new ArrayList<Integer>();
		}

		numbers.add(tableNumber);
		setTableNumbers(numbers);
	}

	@Override
	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	@Override
	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	@Override
	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	@Override
	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@XmlTransient
	public double getGratuityAmount() {
		Gratuity gratuity = getGratuity();
		if (gratuity != null) {
			return gratuity.getAmount();
		}
		return 0;
	}

	public void updateGratuityInfo() {
		Gratuity gratuity = getGratuity();
		if (gratuity == null || gratuity.getAmount() == 0.0) {
			return;
		}
		Store store = DataProvider.get().getStore();
		User cashier = getCashier();
		User driver = getAssignedDriver();
		if (driver != null) {
			TipsReceivedBy tipsReceiverForDelivery = store.getTipsReceivedByForDeliveryOrder();
			if (tipsReceiverForDelivery == TipsReceivedBy.Driver) {
				gratuity.setOwnerId(driver != null ? driver.getId() : getOwner().getId());
			}
			else if (tipsReceiverForDelivery == TipsReceivedBy.Cashier) {
				gratuity.setOwnerId(cashier != null ? cashier.getId() : getOwner().getId());
			}
			else {
				gratuity.setOwnerId(getOwnerId());
			}

		}
		else {
			TipsReceivedBy tipsReceiverForNonDelivery = store.getTipsReceivedByForNonDeliveryOrder();

			if (tipsReceiverForNonDelivery == TipsReceivedBy.Cashier) {
				gratuity.setOwnerId(cashier != null ? cashier.getId() : getOwner().getId());
			}
			else {
				gratuity.setOwnerId(getOwnerId());
			}
		}
		gratuity.setTicketId(getId());
	}

	public void setGratuityAmount(double amount) {
		if (amount == 0 && getGratuity() == null) {
			return;
		}
		Gratuity gratuity = createGratuity();
		gratuity.setAutoCalculated(false);
		gratuity.setAmount(NumberUtil.round(amount));
		setGratuity(gratuity);
	}

	public Gratuity createGratuity() {
		Gratuity gratuity = getGratuity();
		if (gratuity == null) {
			gratuity = new Gratuity();
			gratuity.setTicketId(getId());
			Terminal terminal = Application.getInstance().getTerminal();
			if (terminal != null) {
				gratuity.setTerminalId(terminal.getId());
			}
			gratuity.setOutletId(getOutletId());
			gratuity.setPaid(false);
		}

		return gratuity;
	}

	public boolean hasGratuity() {
		Gratuity gratuity = getGratuity();
		return gratuity != null && gratuity.getAmount() > 0;
	}

	@Override
	public void setCreateDate(Date createDate) {
		super.setCreateDate(createDate);
		super.setActiveDate(createDate);
	}

	@Override
	public List<TicketItem> getTicketItems() {
		List<TicketItem> items = super.getTicketItems();

		if (items == null) {
			items = new ArrayList<TicketItem>();
			super.setTicketItems(items);
		}
		return items;
	}

	@XmlElement(name = "transactionList")
	@Override
	public Set<PosTransaction> getTransactions() {
		Set<PosTransaction> items = super.getTransactions();

		if (items == null) {
			items = new HashSet<PosTransaction>();
			super.setTransactions(items);
		}
		return items;
	}

	@Override
	public Integer getNumberOfGuests() {
		Integer guests = super.getNumberOfGuests();
		if (guests == null || guests.intValue() == 0) {
			return Integer.valueOf(1);
		}
		return guests;
	}

	public String getCreateDateFormatted() {
		return DateUtil.formatDateWithTime(getCreateDate());
	}

	public String getTitle() {
		String title = ""; //$NON-NLS-1$
		if (getId() != null) {
			title += "#" + getId(); //$NON-NLS-1$
		}
		title += "" + ": " + getOwner(); //$NON-NLS-1$ //$NON-NLS-2$
		title += "" + ":" + getCreateDateFormatted(); //$NON-NLS-1$ //$NON-NLS-2$
		title += "" + ": " + NumberUtil.formatNumber(getTotalAmountWithTips()); //$NON-NLS-1$ //$NON-NLS-2$

		return title;
	}

	@XmlTransient
	@JsonIgnore
	public int getBeverageCount() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null)
			return 0;

		int count = 0;
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isBeverage()) {
				count += ticketItem.getQuantity();
			}
		}
		return count;
	}

	public void calculatePrice() {
		if (getDataVersion() == 2) {
			calculatePriceV2();
		}
		else {
			calculatePriceV1();
		}
	}

	@Deprecated
	private void calculatePriceV1() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return;
		}
		if (!isShouldCalculatePrice()) {
			setDiscountAmount(calculateTicketDiscount(getSubtotalAmount(), 0, 0));
			return;
		}
		//		debug(getClass(), "Price calculation start----------------"); //$NON-NLS-1$

		BigDecimal subtotal = convertToBigDecimal(0);
		BigDecimal discount = convertToBigDecimal(0);
		BigDecimal itemsServiceChargeAmount = convertToBigDecimal(0);
		BigDecimal taxAmount = convertToBigDecimal(0);
		BigDecimal itemDiscountAmountWithVoidItems = convertToBigDecimal(0);
		voidSubtotal = 0;
		voidTotal = 0;
		itemDiscountAmount = 0;
		ticketDiscountAmount = 0;
		double taxAmountProperty = 0;

		setRefundableItem(false);
		//@formatter:off
		for (TicketItem ticketItem : ticketItems) {
			if (!isPriceApplicable(ticketItem)) {
				 if (ticketItem.isVoided()) {
	                    setRefundableItem(ticketItem.getVoidItem() != null);
	                    if (!ticketItem.isItemReturned()) {
	                        voidSubtotal += Math.abs(ticketItem.getSubtotalAmount());
	                        voidTotal += Math.abs(ticketItem.getSubtotalAmount()) + Math.abs(ticketItem.getTaxAmount());
	                    }
	                }
				continue;
			}
			ticketItem.calculatePrice();

			subtotal = round(subtotal.add(convertToBigDecimal(ticketItem.getSubtotalAmount())));
			itemsServiceChargeAmount = round(itemsServiceChargeAmount.add(convertToBigDecimal(ticketItem.getServiceCharge())));

			taxAmount = round(taxAmount.add(convertToBigDecimal(ticketItem.getTaxAmount())));
			taxAmountProperty += ticketItem.getTaxExemptAmount();

			itemDiscountAmountWithVoidItems = round(itemDiscountAmountWithVoidItems.add(convertToBigDecimal(ticketItem.getDiscountAmount())));

			if (ticketItem.isVoided()) {
				setRefundableItem(true);
				if (ticketItem.getVoidedItemId() != null) {
					voidSubtotal += Math.abs(ticketItem.getSubtotalAmount());
					voidTotal += Math.abs(ticketItem.getSubtotalAmount()) + Math.abs(ticketItem.getTaxAmount());
				}
			}
			else {
				itemDiscountAmount += ticketItem.getDiscountAmount();
			}
		}
		
		this.addProperty(AppConstants.TAX_EXEMPT_AMOUNT, String.valueOf(taxAmountProperty));
		
		ticketDiscountAmount = calculateTicketDiscount(subtotal.doubleValue(), voidSubtotal, itemDiscountAmount);
		if (ticketDiscountAmount > subtotal.doubleValue()) {
			ticketDiscountAmount = subtotal.doubleValue();
		}
		discount = convertToBigDecimal(
				calculateTotalDiscountAmount(subtotal.doubleValue(), voidSubtotal, ticketDiscountAmount, itemDiscountAmountWithVoidItems.doubleValue()));

		double deliveryChargeAmount = getDeliveryCharge();
//		double ticketServiceChargeAmount = calculateServiceCharge(subtotalWithoutReturn, discount.doubleValue());
		double totalServiceCharge = itemsServiceChargeAmount.doubleValue();

		BigDecimal totalAmount = convertToBigDecimal(0);

		if (ticketDiscountAmount > 0) {
			BigDecimal itemDiscountedSubtotal = subtotal.subtract(convertToBigDecimal(itemDiscountAmount));
			BigDecimal discountSubtotal = itemDiscountedSubtotal.subtract(convertToBigDecimal(ticketDiscountAmount));
			BigDecimal newTax = convertToBigDecimal(0);
			if (itemDiscountedSubtotal.compareTo(BigDecimal.ZERO) > 0) {
				newTax = discountSubtotal.multiply(taxAmount).divide(itemDiscountedSubtotal, 4, RoundingMode.HALF_UP);
			}
			taxAmount = round(newTax);
			if (isTaxIncluded()) {
				totalAmount = totalAmount.add(discountSubtotal);
				//taxAmount = convertToBigDecimal(0);
			}
			else {
				totalAmount = totalAmount.add(discountSubtotal).add(NumberUtil.convertToBigDecimal(newTax.doubleValue()));
			}
		}
		else {
			if (isTaxIncluded()) {
				totalAmount = totalAmount.add(subtotal).subtract(discount);
			}
			else {
				totalAmount = totalAmount.add(subtotal).subtract(discount).add(taxAmount);
			}
		}
		
		calculateGratuity(subtotal.doubleValue(), discount.doubleValue());
		double gratuityAmount = getGratuityAmount();
		
		BigDecimal deliveryChargeBG = convertToBigDecimal(deliveryChargeAmount);
		BigDecimal totalServiceChargeBG = convertToBigDecimal(totalServiceCharge);
		BigDecimal feeAmountBG = convertToBigDecimal(getFeeAmount());
		
		totalAmount = round(totalAmount.add(deliveryChargeBG).add(totalServiceChargeBG).add(feeAmountBG)).subtract(convertToBigDecimal(getOldDataTolerance()));
		
		BigDecimal gratuityBG = NumberUtil.convertToBigDecimal(gratuityAmount);
		BigDecimal paidAmountBG = convertToBigDecimal(getPaidAmount());
		
		BigDecimal dueAmount = round(totalAmount.add(gratuityBG).subtract(paidAmountBG));
		dueAmount = calculateToleranceAmount(dueAmount);

		setSubtotalAmount(round(subtotal.doubleValue()));
		setDiscountAmount(round(discount.doubleValue()));
		setServiceCharge(round(totalServiceCharge));
		setDeliveryCharge(round(deliveryChargeAmount));
		setTaxAmount(round(taxAmount.doubleValue()));
		setTotalAmount(totalAmount.doubleValue());
		setDueAmount(dueAmount.doubleValue());

		setTicketDiscountAmount(ticketDiscountAmount);
		if (ticketDiscountAmount <= 0) {
			return;
		}

		double newServiceCharge = 0;
		double newTaxAmount = 0;
		for (TicketItem ticketItem : ticketItems) {
			if (!isPriceApplicable(ticketItem)) {
				continue;
			}
			ticketItem.calculateAdjustedPrice();
			newServiceCharge += ticketItem.getServiceCharge();
			newTaxAmount += ticketItem.getAdjustedTax();
		}
		totalAmount = round(totalAmount.subtract(convertToBigDecimal(totalServiceCharge)).add(convertToBigDecimal(newServiceCharge)));
		dueAmount = round(totalAmount.add(gratuityBG).subtract(paidAmountBG));
		dueAmount = calculateToleranceAmount(dueAmount);
		setTaxAmount(newTaxAmount);
		setTotalAmount(totalAmount.doubleValue());
		setDueAmount(dueAmount.doubleValue());
		if (NumberUtil.isZero(getDueAmount())) {
			setDueAmount(0.0);
		}
		setServiceCharge(newServiceCharge);
		//@formatter:on
	}

	private void calculatePriceV2() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return;
		}
		if (!isShouldCalculatePrice()) {
			setDiscountAmount(round(calculateTicketDiscountV2(getSubtotalAmount())));
			return;
		}

		double ticketDiscountAmount = 0;
		itemDiscountAmount = 0;
		double subtotal = 0.0;
		double ticketDiscountableSubtotal = 0;
		double totalDiscount = 0.0;
		double serviceCharge = 0.0;
		double itemServiceCharge = 0.0;
		double taxAmount = 0.0;
		double taxExemptAmount = 0;

		for (TicketItem ticketItem : ticketItems) {
			ticketItem.setTicket(this);
			if (ticketItem.isVoided()) {
				setRefundableItem(ticketItem.getVoidItem() != null);
				continue;
			}

			ticketItem.calculatePriceV2();

			subtotal += ticketItem.getSubtotalAmount();
			itemDiscountAmount += ticketItem.getDiscountAmount();
			itemServiceCharge += ticketItem.getServiceCharge();
			taxAmount += ticketItem.getTaxAmount();

			taxExemptAmount += ticketItem.getTaxExemptAmount();
			if (ticketItem.isTicketDiscountApplicable()) {
				ticketDiscountableSubtotal += ticketItem.getSubtotalAmount();
			}
		}
		subtotal = round(subtotal);
		itemDiscountAmount = round(itemDiscountAmount);
		double ticketServiceCharge = calculateTicketServiceCharge();
		serviceCharge = round(itemServiceCharge + ticketServiceCharge);

		taxAmount = round(taxAmount);
		ticketDiscountAmount = round(calculateTicketDiscountV2(ticketDiscountableSubtotal - itemDiscountAmount));
		totalDiscount = round(ticketDiscountAmount + itemDiscountAmount);

		calculateGratuity(subtotal, totalDiscount);

		double deliveryChargeAmount = getDeliveryCharge();
		double gratuityAmount = getGratuityAmount();
		double feeAmount = getFeeAmount();
		double totalAmount = 0.0;
		if (isTaxIncluded()) {
			totalAmount = round(subtotal - totalDiscount + serviceCharge + deliveryChargeAmount + feeAmount);
		}
		else {
			totalAmount = round(subtotal - totalDiscount + taxAmount + serviceCharge + deliveryChargeAmount + feeAmount);
		}

		calculatePaidAmount();

		double dueAmount = round((totalAmount + gratuityAmount) - getPaidAmount() + getRefundAmount());
		double roundedAmount = getRoundedAmount();

		dueAmount = round(calculateDueAfterTolerance(dueAmount + roundedAmount));
		totalAmount = round(totalAmount);
		if (NumberUtil.isZero(dueAmount)) {
			dueAmount = 0;
		}

		setSubtotalAmount(subtotal);
		setDiscountAmount(totalDiscount);
		setServiceCharge(serviceCharge);
		setTicketServiceCharge(ticketServiceCharge);
		setDeliveryCharge(deliveryChargeAmount);
		setTaxAmount(taxAmount);
		setTotalAmount(totalAmount);
		setDueAmount(dueAmount);
		setItemDiscountAmount(itemDiscountAmount);
		setTicketDiscountAmount(ticketDiscountAmount);

		this.addProperty(AppConstants.TAX_EXEMPT_AMOUNT, String.valueOf(taxExemptAmount));

		if (ticketDiscountAmount <= 0) {
			return;
		}
		calculateAdjustedPriceV2();
	}

	private void calculateAdjustedPriceV2() {
		double subtotalAfterDiscount = 0.0;
		double serviceCharge = 0.0;
		double itemServiceCharge = 0.0;
		double taxAmount = 0.0;
		double taxExemptAmount = 0;
		double discountAmount = 0;

		List<TicketItem> ticketItems = getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			ticketItem.setTicket(this);
			if (ticketItem.isVoided()) {
				setRefundableItem(ticketItem.getVoidItem() != null);
				continue;
			}

			ticketItem.calculateAdjustedPriceV2();

			subtotalAfterDiscount += ticketItem.getAdjustedSubtotal();
			discountAmount += ticketItem.getAdjustedDiscount();
			itemServiceCharge += ticketItem.getServiceCharge();
			taxAmount += ticketItem.getAdjustedTax();
			taxExemptAmount += ticketItem.getTaxExemptAmount();
		}

		//adjust discount fraction start
		double adjustableAmount = round((getTicketDiscountAmount() + getItemDiscountAmount()) - discountAmount);
		if (!NumberUtil.isZero(adjustableAmount)) {
			PosLog.debug(Ticket.class, "Adjustable discount amount: " + adjustableAmount); //$NON-NLS-1$
			int totalAdjustable = (int) Math.abs((adjustableAmount * 100));
			List<TicketItem> adjustedTicketItems = getTicketItems();

			boolean isNagative = (adjustableAmount < 0);
			int adjustedAmount = 0;
			for (int i = 0; i < totalAdjustable; i++) {
				if (adjustedAmount == totalAdjustable) {
					break;
				}

				for (int j = 0; j < adjustedTicketItems.size(); j++) {
					if (adjustedAmount == totalAdjustable) {
						break;
					}

					TicketItem ticketItem = adjustedTicketItems.get(j);
					double adjAmount = 0.01;
					if (isNagative) {
						adjAmount = -0.01;
					}

					PosLog.debug(Ticket.class, "Item name: " + ticketItem.getName() + ",Before adjusted discount: " + ticketItem.getAdjustedDiscount()); //$NON-NLS-1$ //$NON-NLS-2$

					ticketItem.setAdjustedDiscount(ticketItem.getAdjustedDiscount() + adjAmount);
					discountAmount += adjAmount;
					subtotalAfterDiscount -= adjAmount;
					adjustedAmount += 1;

					StringBuilder builder = new StringBuilder();
					builder.append("Item name: " + ticketItem.getName()); //$NON-NLS-1$
					builder.append(", "); //$NON-NLS-1$
					builder.append("After adjusted discount: " + ticketItem.getAdjustedDiscount()); //$NON-NLS-1$
					builder.append(", "); //$NON-NLS-1$
					builder.append("discount amount: " + discountAmount); //$NON-NLS-1$
					builder.append(", "); //$NON-NLS-1$
					builder.append("subtotal after discount: " + subtotalAfterDiscount); //$NON-NLS-1$
					PosLog.debug(Ticket.class, builder.toString());
				}
			}
		}

		//adjust discount fraction End

		subtotalAfterDiscount = round(subtotalAfterDiscount);
		discountAmount = round(discountAmount);
		itemServiceCharge = round(itemServiceCharge);
		double ticketServiceCharge = calculateTicketServiceCharge();
		serviceCharge = round(itemServiceCharge + ticketServiceCharge);
		taxAmount = round(taxAmount);

		calculateGratuity(subtotalAfterDiscount, discountAmount);

		double deliveryChargeAmount = getDeliveryCharge();
		double gratuityAmount = getGratuityAmount();
		double feeAmount = getFeeAmount();
		double totalAmount = 0.0;
		if (isTaxIncluded()) {
			totalAmount = round(subtotalAfterDiscount + serviceCharge + deliveryChargeAmount + feeAmount);
		}
		else {
			totalAmount = round(subtotalAfterDiscount + taxAmount + serviceCharge + deliveryChargeAmount + feeAmount);
		}
		double dueAmount = round((totalAmount + gratuityAmount) - getPaidAmount() + getRefundAmount());
		double roundedAmount = getRoundedAmount();
		dueAmount = round(calculateDueAfterTolerance(dueAmount + roundedAmount));
		totalAmount = round(totalAmount);

		if (NumberUtil.isZero(dueAmount)) {
			dueAmount = 0;
		}

		setDiscountAmount(discountAmount);
		setServiceCharge(serviceCharge);
		setTicketServiceCharge(ticketServiceCharge);
		setDeliveryCharge(deliveryChargeAmount);
		setTaxAmount(taxAmount);
		setTotalAmount(totalAmount);
		setDueAmount(dueAmount);

		this.addProperty(AppConstants.TAX_EXEMPT_AMOUNT, String.valueOf(taxExemptAmount));
	}

	public double getRepricableAmount() {
		double repricableAmount = getTotalAmount() - getDeliveryCharge() - getFeeAmount();
		if (!isDiscountOnSerivceCharge()) {
			repricableAmount -= getServiceCharge();
		}
		return repricableAmount;
	}

	@Deprecated
	private BigDecimal calculateToleranceAmount(BigDecimal dueAmount) {
		toleranceAmount = 0;
		if (getPaidAmount() <= 0) {
			return dueAmount;
		}
		double dueAmountDouble = dueAmount.doubleValue();
		Store store = DataProvider.get().getStore();
		double roundedDueAmount = 0;
		if (store != null) {
			if (store.isAllowPenyRounding()) {
				roundedDueAmount = Math.round(dueAmountDouble * 100.0 / 5.0) * 5.0 / 100.0;
			}
			else {
				roundedDueAmount = dueAmountDouble;
			}
		}

		toleranceAmount = roundedDueAmount - dueAmountDouble;

		double toleranceAmountFactor = 0.0;
		Currency mainCurrency = CurrencyUtil.getMainCurrency();
		if (mainCurrency != null) {
			toleranceAmountFactor = mainCurrency.getTolerance();
		}
		if (Math.abs(roundedDueAmount) <= toleranceAmountFactor) {
			if (toleranceAmount < 0 || roundedDueAmount > 0) {
				toleranceAmount += -roundedDueAmount;
			}
			else {
				toleranceAmount += roundedDueAmount;
			}
			return BigDecimal.ZERO;
		}
		//
		if (roundedDueAmount == 0.0 || roundedDueAmount <= 0.05) {
			return convertToBigDecimal(roundedDueAmount);
		}
		return dueAmount;
	}

	private double calculateDueAfterTolerance(double dueAmount) {
		toleranceAmount = 0;
		if (getPaidAmount() <= 0) {
			return dueAmount;
		}

		if (Math.abs(dueAmount) <= getToleranceFactor()) {
			if (toleranceAmount < 0) {
				toleranceAmount = dueAmount;
			}
			else if (dueAmount != 0) {
				toleranceAmount = -1 * dueAmount;
			}
			return 0;
		}

		return dueAmount;
	}

	public double getToleranceAmount() {
		return toleranceAmount;
	}

	public double getToleranceFactor() {
		if (hasProperty("toleranceFactor")) { //$NON-NLS-1$
			return POSUtil.parseDouble(getProperty("toleranceFactor")); //$NON-NLS-1$
		}
		Currency mainCurrency = DataProvider.get().getMainCurrency();
		if (mainCurrency != null) {
			return mainCurrency.getTolerance();
		}
		return 0;
	}

	public void setToleranceFactor(double tf) {
		addProperty("toleranceFactor", String.valueOf(tf)); //$NON-NLS-1$
	}

	public double getSubtotalAmountWithVoidItems() {
		return getSubtotalAmount() + getVoidSubtotal();
	}

	private double calculateTotalDiscountAmount(double subtotal, double voidSubtotal, double ticketDiscountAmount, double itemDiscountAmountWithVoidItems) {
		//commented because of calculation error with voidSubtotal
		voidSubtotal = 0;

		if (getDiscounts() == null || getDiscounts().size() == 0) {
			return itemDiscountAmountWithVoidItems;
		}
		//final double subtotalAmountWithVoidItems = subtotal + voidSubtotal;
		double totalDiscountAmount = 0;
		totalDiscountAmount += itemDiscountAmountWithVoidItems;
		totalDiscountAmount += ticketDiscountAmount;

		//consider voided items...
		/*if (subtotalAmountWithVoidItems != 0) {
			totalDiscountAmount -= (ticketDiscountAmount * Math.abs(voidSubtotal)) / subtotalAmountWithVoidItems;
		}*/

		if (subtotal > 0 && totalDiscountAmount > subtotal) {
			totalDiscountAmount = subtotal;
		}
		if (ticketDiscountAmount > subtotal) {
			ticketDiscountAmount = subtotal;
		}
		if (totalDiscountAmount < 0) {
			totalDiscountAmount = 0;
		}

		return round(totalDiscountAmount);
	}

	@Deprecated
	private double calculateTicketDiscount(final double subtotal, final double voidSubtotal, final double itemDiscount) {
		//final double subtotalWithVoidItems = subtotal + voidSubtotal;
		final double subtotalAfterItemDiscount = subtotal - itemDiscount;
		double ticketDisount = 0;
		List<TicketDiscount> ticketCouponAndDiscounts = getDiscounts();
		if (ticketCouponAndDiscounts != null) {
			for (TicketDiscount ticketDiscount : ticketCouponAndDiscounts) {
				double discountForThisCoupon = 0;
				Integer discountType = ticketDiscount.getType();
				if (discountType == Discount.DISCOUNT_TYPE_REPRICE) {
					discountForThisCoupon = DiscountUtil.calculateRepriceDiscount(this, ticketDiscount.getValue());
				}
				else if (discountType == Discount.DISCOUNT_TYPE_LOYALTY) {
					discountForThisCoupon = ticketDiscount.getValue() * ticketDiscount.getCouponQuantity();
				}
				else if (discountType == Discount.DISCOUNT_TYPE_PERCENTAGE) {
					Double calculateDiscountAmount = DiscountUtil.calculateDiscountAmount(subtotal - itemDiscountAmount, ticketDiscount);
					discountForThisCoupon = calculateDiscountAmount * ticketDiscount.getCouponQuantity();
				}
				else {
					Double calculateDiscountAmount = DiscountUtil.calculateDiscountAmount(subtotalAfterItemDiscount, ticketDiscount);
					discountForThisCoupon = calculateDiscountAmount * ticketDiscount.getCouponQuantity();
				}

				if (subtotal < ticketDiscount.getMinimumAmount()) {
					discountForThisCoupon = 0;
				}

				ticketDiscount.setTotalDiscountAmount(discountForThisCoupon);
				ticketDisount += discountForThisCoupon;
			}
		}
		buildDiscounts();
		return round(ticketDisount);
	}

	private double calculateTicketDiscountV2(final double totalAmount) {
		double ticketDiscountAmount = 0;
		List<TicketDiscount> ticketCouponAndDiscounts = getDiscounts();
		if (ticketCouponAndDiscounts != null) {
			for (TicketDiscount ticketDiscount : ticketCouponAndDiscounts) {
				double discountForThisCoupon = 0;
				Integer discountType = ticketDiscount.getType();
				if (discountType == Discount.DISCOUNT_TYPE_REPRICE) {
					discountForThisCoupon = DiscountUtil.calculateRepriceDiscount(this, ticketDiscount.getValue());
				}
				else if (discountType == Discount.DISCOUNT_TYPE_LOYALTY) {
					discountForThisCoupon = ticketDiscount.getValue() * ticketDiscount.getCouponQuantity();
				}
				else if (discountType == Discount.DISCOUNT_TYPE_PERCENTAGE) {
					Double calculateDiscountAmount = 0.0;
					if (ticketDiscount.getOriginalType() == Discount.DISCOUNT_TYPE_REPRICE) {
						calculateDiscountAmount = DiscountUtil.calculateDiscountAmount(totalAmount - ticketDiscountAmount, ticketDiscount);
					}
					else {
						calculateDiscountAmount = DiscountUtil.calculateDiscountAmount(totalAmount - ticketDiscountAmount, ticketDiscount);
					}
					discountForThisCoupon = calculateDiscountAmount;
				}
				else {
					Double calculateDiscountAmount = DiscountUtil.calculateDiscountAmount(totalAmount, ticketDiscount);
					discountForThisCoupon = calculateDiscountAmount * ticketDiscount.getCouponQuantity();
				}

				if (totalAmount < ticketDiscount.getMinimumAmount()) {
					discountForThisCoupon = 0;
				}

				ticketDiscount.setTotalDiscountAmount(discountForThisCoupon);
				ticketDiscountAmount += discountForThisCoupon;
			}
		}
		buildDiscounts();
		return ticketDiscountAmount;
	}

	public double getAmountByType(TicketDiscount discount) {
		switch (discount.getType()) {
			case Discount.DISCOUNT_TYPE_AMOUNT:
				return discount.getValue();

			case Discount.DISCOUNT_TYPE_PERCENTAGE:
				return (discount.getValue() * getSubtotalAmount()) / 100;

			default:
				break;
		}
		return 0;
	}

	public static TicketDiscount convertToTicketDiscount(Discount discount, Ticket ticket) {
		TicketDiscount ticketDiscount = new TicketDiscount();
		ticketDiscount.setDiscountId(discount.getId());
		ticketDiscount.setName(discount.getName());
		ticketDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
		ticketDiscount.setMinimumAmount(discount.getMinimumBuy());
		ticketDiscount.setOriginalType(discount.getType());
		ticketDiscount.setOriginalValue(discount.getOriginalValue());

		if (discount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
			ticketDiscount.setType(Discount.DISCOUNT_TYPE_AMOUNT);
			ticketDiscount.setValue(discount.getValue());
		}
		else if (discount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
			double percentageForTotal = DiscountUtil.calculatePercentageForTotal(discount.getValue(), ticket.getTotalAmount());
			ticketDiscount.setValue(percentageForTotal);
		}
		else {
			ticketDiscount.setValue(discount.getValue());
		}
		ticketDiscount.setCouponQuantity(1D);
		ticketDiscount.setTicket(ticket);
		ticketDiscount.setCouponBarcode(discount.getBarcode());
		return ticketDiscount;
	}

	public static TicketDiscount buildLoyaltyDiscount(Ticket ticket) {
		TicketDiscount loyaltyDiscount = new TicketDiscount();
		loyaltyDiscount.setDiscountId(Discount.getLoyaltyDiscountId());
		loyaltyDiscount.setName(Discount.LOYALTY_DISCOUNT_NAME);
		loyaltyDiscount.setType(Discount.DISCOUNT_TYPE_LOYALTY);
		loyaltyDiscount.setMinimumAmount(1D);
		loyaltyDiscount.setValue(null);
		loyaltyDiscount.setCouponQuantity(1D);
		loyaltyDiscount.setTicket(ticket);
		return loyaltyDiscount;
	}

	public void removeLoyaltyDiscount() {
		List<TicketDiscount> discounts = getDiscounts();
		for (Iterator<TicketDiscount> iterator = discounts.iterator(); iterator.hasNext();) {
			TicketDiscount ticketDiscount = (TicketDiscount) iterator.next();
			if (ticketDiscount.getType().equals(Discount.DISCOUNT_TYPE_LOYALTY)) {
				iterator.remove();
				ActionHistoryDAO.addDiscountRemovedActionHistory(this, ticketDiscount);
				break;
			}
		}
	}

	public double calculateDiscountFromType(TicketDiscount coupon, double subtotal) {
		List<TicketItem> ticketItems = getTicketItems();

		double discount = 0;
		int type = coupon.getType();
		double couponValue = coupon.getValue();

		switch (type) {
			case Discount.FIXED_PER_ORDER:
				discount += couponValue;
				break;

			case Discount.FIXED_PER_CATEGORY:
				HashSet<String> categoryIds = new HashSet<String>();
				for (TicketItem item : ticketItems) {
					String itemId = item.getMenuItemId();
					if (!categoryIds.contains(itemId)) {
						discount += couponValue;
						categoryIds.add(itemId);
					}
				}
				break;

			case Discount.FIXED_PER_ITEM:
				for (TicketItem item : ticketItems) {
					discount += (couponValue * item.getQuantity());
				}
				break;

			case Discount.PERCENTAGE_PER_ORDER:
				discount += ((subtotal * couponValue) / 100.0);
				break;

			case Discount.PERCENTAGE_PER_CATEGORY:
				categoryIds = new HashSet<String>();
				for (TicketItem item : ticketItems) {
					String itemId = item.getMenuItemId();
					if (!categoryIds.contains(itemId)) {
						discount += ((item.getUnitPrice() * couponValue) / 100.0);
						categoryIds.add(itemId);
					}
				}
				break;

			case Discount.PERCENTAGE_PER_ITEM:
				for (TicketItem item : ticketItems) {
					discount += ((item.getSubtotalAmountWithoutModifiers() * couponValue) / 100.0);
				}
				break;

			case Discount.FREE_AMOUNT:
				discount += couponValue;
				break;
		}
		return discount;
	}

	public void addDeletedItems(Object o) {
		if (deletedItems == null) {
			deletedItems = new ArrayList();
		}

		deletedItems.add(o);
	}

	public List getDeletedItems() {
		return deletedItems;
	}

	public void clearDeletedItems() {
		if (deletedItems != null) {
			deletedItems.clear();
		}

		deletedItems = null;
	}

	public int countItem(TicketItem ticketItem) {
		List<TicketItem> items = getTicketItems();
		if (items == null) {
			return 0;
		}

		int count = 0;
		for (TicketItem ticketItem2 : items) {
			if (ticketItem.getMenuItemId().equals(ticketItem2.getMenuItemId())) {
				++count;
			}
		}

		return count;
	}

	public boolean needsKitchenPrint() {
		if (getDeletedItems() != null && getDeletedItems().size() > 0) {
			return true;
		}

		List<TicketItem> ticketItems = getTicketItems();
		for (TicketItem item : ticketItems) {
			if (item.isShouldPrintToKitchen() && !item.isPrintedToKitchen()) {
				return true;
			}
		}

		return false;
	}

	public boolean isCoreAmountPaid() {
		if (getPaidAmount() < getTotalAmount()) {
			return false;
		}
		return true;
	}

	public double getRemainingGratuity() {
		if (!hasGratuity()) {
			return 0;
		}
		Set<PosTransaction> transactions = getTransactions();
		double paidTips = 0;
		for (PosTransaction posTransaction : transactions) {
			paidTips += posTransaction.getTipsAmount();
		}
		Gratuity gratuity = getGratuity();
		return gratuity.getAmount() - paidTips;
	}

	private double calculateTicketServiceCharge() {
		if (!isServiceChargeApplicable()) {
			return 0.0;
		}

		double serviceCharge = 0;
		ServiceChargeType serviceChargeType = getServiceChargeType();
		switch (serviceChargeType) {
			case PERCENTAGE:
			case EMPTY:
				serviceCharge = 0;
				break;

			case FIXEDAMOUNT:
				serviceCharge = getOutletServiceChargeRate();

			default:
				break;
		}

		return serviceCharge;
	}

	public void calculateGratuity(double subtotalAmount, double discountAmount) {
		Double gratuityPercentage = getOutletGratuityRate();
		if (gratuityPercentage <= 0) {
			return;
		}
		Gratuity gratuity = getGratuity();
		if (gratuity != null && !gratuity.isAutoCalculated()) {
			return;
		}
		if (gratuity == null) {
			gratuity = new Gratuity();
			gratuity.setAutoCalculated(true);
			gratuity.setOutletId(getOutletId());
		}

		double gratuityAmount = 0.0;
		if (gratuityPercentage > 0.0) {
			gratuityAmount = (subtotalAmount - discountAmount) * (gratuityPercentage / 100.0);
		}
		if (gratuityAmount > 0) {
			gratuity.setAmount(round(gratuityAmount));
		}
		else {
			gratuity.setAmount(0.0);
		}
		setGratuity(gratuity);
	}

	@Override
	public void addProperty(String name, String value) {
		if (StringUtils.isEmpty(value)) {
			return;
		}
		this.initPropertiesContainer();
		propertiesContainer.addProperty(name, value);
		super.setExtraProperties(propertiesContainer.toString());
	}

	@Override
	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	@Override
	public String getProperty(String key) {
		return this.getProperty(key, null);
	}

	private void initPropertiesContainer() {
		if (this.propertiesContainer == null) {
			String properties = super.getExtraProperties();
			this.propertiesContainer = StringUtils.isBlank(properties) ? new com.google.gson.JsonObject()
					: new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		}
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		this.initPropertiesContainer();
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}

		return defaultValue;
	}

	@Override
	public void removeProperty(String propertyName) {
		this.initPropertiesContainer();
		propertiesContainer.remove(propertyName);
		super.setExtraProperties(propertiesContainer.toString());
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);
		return POSUtil.getBoolean(property);
	}

	public String toURLForm() {
		String s = "ticket_id=" + getId(); //$NON-NLS-1$

		List<TicketItem> items = getTicketItems();
		if (items == null || items.size() == 0) {
			return s;
		}

		for (int i = 0; i < items.size(); i++) {
			TicketItem ticketItem = items.get(i);
			s += "&items[" + i + "][id]=" + ticketItem.getId(); //$NON-NLS-1$ //$NON-NLS-2$
			s += "&items[" + i + "][name]=" + POSUtil.encodeURLString(ticketItem.getName()); //$NON-NLS-1$ //$NON-NLS-2$
			s += "&items[" + i + "][price]=" + ticketItem.getSubtotalAmount(); //$NON-NLS-1$ //$NON-NLS-2$
		}

		s += "&tax=" + getTaxAmount(); //$NON-NLS-1$
		s += "&subtotal=" + getSubtotalAmount(); //$NON-NLS-1$
		s += "&grandtotal=" + getTotalAmountWithTips(); //$NON-NLS-1$

		return s;
	}

	public void updateCustomer(Customer customer) {
		this.customer = customer;
	}

	public void setCustomer(Customer customer) {
		String oldCustomerId = this.getCustomerId();

		this.customer = customer;
		removeCustomer();
		if (customer == null) {
			return;
		}

		setCustomerId(customer.getId());
		addProperty(Ticket.CUSTOMER_ID, String.valueOf(customer.getId()));
		addProperty(Ticket.CUSTOMER_NAME, customer.getFirstName());
		addProperty(Ticket.CUSTOMER_LAST_NAME, customer.getLastName());
		addProperty(Ticket.CUSTOMER_PHONE, customer.getMobileNo());
		addProperty(Ticket.CUSTOMER_ZIP_CODE, customer.getZipCode());
		addProperty(Ticket.CUSTOMER_TAX_EXEMPT, customer.isTaxExempt().toString());
		addProperty(Ticket.CUSTOMER_EMAIL, customer.getEmail());

		setTaxExempt(customer.isTaxExempt());

		if (this.getId() == null) {
			return;
		}
		String action = ActionHistory.CUSTOMER_REMOVED;
		String customerId = null;
		if (customer != null) {
			action = ActionHistory.CUSTOMER_SET;
			customerId = customer.getId();
		}
		if (customer.getId() != null) {
			ActionHistoryDAO.addActionHistory(this, action, String.format("Old customer : %s, New customer : %s", oldCustomerId, customerId)); //$NON-NLS-1$
		}
	}

	/**
	 * To be used in menugreat. If a customer does not want to register then this method to be called.
	 * 
	 * @param customer
	 */
	public void setCustomerInfo(String name, String email, String phone) {
		if (StringUtils.isNotBlank(name)) {
			name = name.trim();

			int firstSpaceIndex = name.indexOf(" ");
			String firstName = name;
			String lastName = "";

			if (firstSpaceIndex > 0) {
				firstName = name.substring(0, firstSpaceIndex);
				lastName = name.substring(firstSpaceIndex);
			}
			addProperty(Ticket.CUSTOMER_NAME, firstName);
			addProperty(Ticket.CUSTOMER_LAST_NAME, lastName);
		}

		addProperty(Ticket.CUSTOMER_EMAIL, email); //$NON-NLS-1$
		if (StringUtils.isBlank(phone)) {
			removeProperty(CUSTOMER_PHONE);
		}
		else {
			addProperty(Ticket.CUSTOMER_PHONE, phone);
		}
	}

	public void updateCustomerRef(Customer customer) {
		this.customer = customer;
	}

	@XmlTransient
	public Customer getCustomer() {
		if (this.customer != null) {
			return this.customer;
		}
		String customerId = getCustomerId();
		if (StringUtils.isEmpty(customerId)) {
			return null;
		}
		customer = DataProvider.get().getCustomer(customerId);
		return customer;
	}

	public void removeCustomer() {
		removeProperty(Ticket.CUSTOMER_ID);
		removeProperty(Ticket.CUSTOMER_NAME);
		removeProperty(Ticket.CUSTOMER_LAST_NAME);
		removeProperty(Ticket.CUSTOMER_PHONE);
		removeProperty(Ticket.CUSTOMER_ZIP_CODE);
		removeProperty(Ticket.CUSTOMER_TAX_EXEMPT);
		removeProperty(Ticket.CUSTOMER_EMAIL);
	}

	public String getSortOrder() {
		if (sortOrder == null) {
			return ""; //$NON-NLS-1$
		}
		return sortOrder;
	}

	public void setSortOrder(String sortOrder) {
		this.sortOrder = sortOrder;
	}

	public void setTicketStatus(TicketStatus ticketStatus) {
		super.setStatus(ticketStatus.name());
		String statusUpdateTimeKey = "delivery." + ticketStatus.name() + ".time"; //$NON-NLS-1$ //$NON-NLS-2$
		if (hasProperty(statusUpdateTimeKey)) {
			return;
		}
		addProperty(statusUpdateTimeKey, DateUtil.formatDateWithDefaultTimeAndSec(new Date()));
	}

	public Date getTicketStatusUpdatedTime(TicketStatus ticketStatus) {
		String statusUpdateTimeKey = "delivery." + ticketStatus.name() + ".time"; //$NON-NLS-1$ //$NON-NLS-2$
		if (!hasProperty(statusUpdateTimeKey)) {
			return null;
		}
		try {
			return DateUtil.formatDateWithDefaultTimeAndSec(getProperty(statusUpdateTimeKey));
		} catch (Exception e) {
		}
		return null;
	}

	@XmlTransient
	public TicketStatus getTicketStatus() {
		try {
			String status = super.getStatus();
			if (StringUtils.isBlank(status)) {
				return TicketStatus.Unknown;
			}

			return TicketStatus.valueOf(status);
		} catch (Exception e) {
			return TicketStatus.Unknown;
		}
	}

	public String getOrderStatusDisplay() {
		return ORDER_STATUS[getOrderStatus()];
	}

	public void setOrderStatusDisplay(String orderStatusDisplay) {
		//this.orderStatusDisplay = orderStatusDisplay;
	}

	public void consolidateTicketItems() {
		if (hasSeat()) {
			return;
		}
		List<TicketItem> ticketItems = getTicketItems();

		Map<String, List<TicketItem>> itemMap = new LinkedHashMap<String, List<TicketItem>>();

		for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
			TicketItem newItem = (TicketItem) iterator.next();
			String menuItemId = newItem.getMenuItemId() == null ? newItem.getName() : newItem.getMenuItemId();
			if (StringUtils.isEmpty(menuItemId)) {
				continue;
			}
			List<TicketItem> itemListInMap = itemMap.get(menuItemId);

			if (itemListInMap == null) {
				List<TicketItem> list = new ArrayList<TicketItem>();
				list.add(newItem);

				itemMap.put(menuItemId.toString(), list);
			}
			else {
				boolean merged = false;
				for (TicketItem itemInMap : itemListInMap) {
					if (itemInMap.isMergable(newItem, false)) {
						itemInMap.merge(newItem);
						merged = true;
						break;
					}
				}

				if (!merged) {
					itemListInMap.add(newItem);
				}
			}
		}

		getTicketItems().clear();
		Collection<List<TicketItem>> values = itemMap.values();
		for (List<TicketItem> list : values) {
			if (list != null) {
				getTicketItems().addAll(list);
			}
		}
		List<TicketItem> ticketItemList = getTicketItems();
		if (getOrderType().isAllowSeatBasedOrder()) {
			Collections.sort(ticketItemList, new Comparator<TicketItem>() {

				@Override
				public int compare(TicketItem o1, TicketItem o2) {
					if (o1.getId() == null || o2.getId() == null) {
						return 0;
					}
					return o1.getId().compareTo(o2.getId());
				}
			});
			Collections.sort(ticketItemList, new Comparator<TicketItem>() {

				@Override
				public int compare(TicketItem o1, TicketItem o2) {
					return o1.getSeatNumber() - o2.getSeatNumber();
				}

			});
		}
		calculatePrice();
	}

	public void consolidateKitchenTicketItems() {
		boolean isFilterKitchenPrintedItems = true;
		consolidateKitchenTicketItems(isFilterKitchenPrintedItems);
	}

	public void consolidateKitchenTicketItems(boolean isFilterTicket) {
		consolidateKitchenTicketItems(isFilterTicket, null);
	}

	public void consolidateKitchenTicketItems(boolean isFilterTicket, List<TicketItem> ticketItems) {

		if (hasSeat() || isSourceOnline()) {
			return;
		}
		if (ticketItems == null) {
			ticketItems = getTicketItems();
		}
		Map<String, List<TicketItem>> itemMap = new LinkedHashMap<String, List<TicketItem>>();

		for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
			TicketItem newItem = (TicketItem) iterator.next();
			if ((!newItem.isShouldPrintToKitchen() && !newItem.isPrintKitchenSticker())) {
				continue;
			}
			if (isFilterTicket && newItem.isPrintedToKitchen()) {
				continue;
			}
			String menuItemId = newItem.getMenuItemId();
			if (StringUtils.isEmpty(menuItemId)) {
				menuItemId = newItem.getName();
			}
			List<TicketItem> itemListInMap = itemMap.get(menuItemId);

			if (itemListInMap == null) {
				List<TicketItem> list = new ArrayList<TicketItem>();
				list.add(newItem);

				itemMap.put(menuItemId.toString(), list);
			}
			else {
				/*boolean merged = false;
				for (TicketItem itemInMap : itemListInMap) {
					if (itemInMap.isMergable(newItem, false)) {
						itemInMap.merge(newItem);
						merged = true;
						break;
					}
				}*/

				//if (!merged) {
				itemListInMap.add(newItem);
				//}
			}
		}

		ticketItems.clear();
		Collection<List<TicketItem>> values = itemMap.values();
		for (List<TicketItem> list : values) {
			if (list != null) {
				ticketItems.addAll(list);
			}
		}
		List<TicketItem> ticketItemList = ticketItems;
		if (getOrderType().isAllowSeatBasedOrder()) {
			Collections.sort(ticketItemList, new Comparator<TicketItem>() {

				@Override
				public int compare(TicketItem o1, TicketItem o2) {
					if (o1.getId() == null || o2.getId() == null)
						return 0;
					return o1.getId().compareTo(o2.getId());
				}
			});
			Collections.sort(ticketItemList, new Comparator<TicketItem>() {

				@Override
				public int compare(TicketItem o1, TicketItem o2) {
					return o1.getSeatNumber() - o2.getSeatNumber();
				}

			});
		}
		calculatePrice();
	}

	/**
	 * Mark ticket items, modifiers, add-ons as printed to kitchen
	 */
	public void markPrintedToKitchen() {
		List<TicketItem> ticketItems = getTicketItems();
		markPrintedToKitchen(ticketItems);
	}

	public void markPrintedToKitchen(List<TicketItem> ticketItems) {
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isPrintedToKitchen() || !ticketItem.isShouldPrintToKitchen()) {
				continue;
			}

			List<Printer> printers = ticketItem.getPrinters(getOrderType());
			if (printers == null) {
				continue;
			}
			ticketItem.setPrintedToKitchen(true);
			ticketItem.setKitchenStatusValue(KitchenStatus.WAITING);
			List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
			if (ticketItemModifiers != null) {
				for (TicketItemModifier itemModifier : ticketItemModifiers) {
					itemModifier.setPrintedToKitchen(true);
				}
			}
			List<TicketItemCookingInstruction> cookingInstructions = ticketItem.getCookingInstructions();
			if (cookingInstructions != null) {
				for (TicketItemCookingInstruction ticketItemCookingInstruction : cookingInstructions) {
					ticketItemCookingInstruction.setPrintedToKitchen(true);
				}
				ticketItem.buildCoookingInstructions();
			}
		}
	}

	public TicketSource getTicketSource() {
		return TicketSource.fromName(super.getSource());
	}

	public void setTicketSource(TicketSource ticketSource) {
		super.setSource(ticketSource.name());
	}

	public boolean isSourceOnline() {
		return getTicketSource() == TicketSource.Online;
	}

	public boolean isReservation() {
		if (getTicketType() != null) {
			TicketType ticketType = getTicketType();
			if (ticketType.equals(TicketType.RESERVATION)) {
				return true;
			}
		}
		return false;
	}

	public static Ticket clone(Ticket source) {
		return (Ticket) SerializationUtils.clone(source);
	}

	/**
	 * If clone fails return null else return New ticket.
	 * @param ticket
	 * @param newOutlet
	 * 
	 * @return New ticket
	 */
	public Ticket clone(Ticket ticket, Outlet newOutlet) {
		try {
			Ticket newTicket = new Ticket();
			PropertyUtils.copyProperties(newTicket, ticket);
			newTicket.setId(NumericGlobalIdGenerator.generateGlobalId());
			newTicket.setOutlet(newOutlet);

			//TicketItem
			List<TicketItem> newTicketItems = new ArrayList<TicketItem>();
			for (TicketItem ticketItem : ticket.getTicketItems()) {

				TicketItem newTicketItem = new TicketItem();
				if (ticketItem instanceof ModifiableTicketItem) {
					newTicketItem = new ModifiableTicketItem();
				}
				PropertyUtils.copyProperties(newTicketItem, ticketItem);
				newTicketItem.setId(null);
				newTicketItem.setTicket(newTicket);
				if (ticketItem.isHasModifiers()) {
					List<TicketItemModifier> newTicketItemModifiers = new ArrayList<TicketItemModifier>();
					for (TicketItemModifier ticketItemModifier : ticketItem.getTicketItemModifiers()) {
						TicketItemModifier newTicketItemModi = new TicketItemModifier();
						PropertyUtils.copyProperties(newTicketItemModi, ticketItemModifier);
						newTicketItemModi.setId(null);
						newTicketItemModi.setTicketItem(newTicketItem);
						newTicketItemModifiers.add(newTicketItemModi);
					}
					newTicketItem.setTicketItemModifiers(newTicketItemModifiers);
				}
				newTicketItems.add(newTicketItem);
			}
			newTicket.setTicketItems(newTicketItems);

			//Gratuity
			Gratuity gratuity = ticket.getGratuity();
			if (gratuity != null) {
				Gratuity newGratuity = new Gratuity();
				PropertyUtils.copyProperties(newGratuity, gratuity);
				newGratuity.setId(null);
				newGratuity.setOutletId(newOutlet.getId());
				newTicket.setGratuity(newGratuity);
			}

			//Transaction
			Set<PosTransaction> transactionList = ticket.getTransactions();
			if (transactionList != null && transactionList.size() > 0) {
				ArrayList<PosTransaction> newTransactionList = new ArrayList<>();
				ArrayList<PosTransaction> transactions = new ArrayList<>(transactionList);
				for (PosTransaction posTransaction : transactions) {
					PosTransaction newPosTransaction = new PosTransaction();
					PropertyUtils.copyProperties(newPosTransaction, posTransaction);
					newPosTransaction.setId(null);
					newPosTransaction.setOutletId(newOutlet.getId());
					newTransactionList.add(newPosTransaction);
				}
				newTicket.setTransactions(transactionList);
			}

			return newTicket;
		} catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
			return null;
		}
	}

	public String getNumberToDisplay() {
		return Messages.getString("Ticket.11") + getTokenNo(); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public Outlet getOutlet() {
		if (this.outlet != null) {
			return this.outlet;
		}
		String outletId = getOutletId();
		if (outletId == null)
			return null;
		outlet = (Outlet) DataProvider.get().getObjectOf(Outlet.class, outletId);
		return outlet;
	}

	@Transient
	public void setOutlet(com.floreantpos.model.Outlet outlet) {
		this.outlet = outlet;
		if (outlet == null) {
			setOutletId(null);
		}
		else {
			setOutletId(outlet.getId());
		}
	}

	@XmlTransient
	public Department getDepartment() {
		String departmentId = getDepartmentId();
		if (departmentId == null)
			return null;
		if (department == null)
			department = (Department) DataProvider.get().getObjectOf(Department.class, departmentId);
		return department;
	}

	public void setDepartment(com.floreantpos.model.Department department) {
		this.department = department;
		if (department == null) {
			setDepartmentId(null);
		}
		else {
			setDepartmentId(department.getId());
		}
	}

	@XmlTransient
	public SalesArea getSalesArea() {
		if (this.salesArea != null) {
			return this.salesArea;
		}
		String salesAreaId = getSalesAreaId();
		if (salesAreaId == null)
			return null;
		salesArea = DataProvider.get().getSalesArea(salesAreaId);
		return salesArea;
	}

	public void setSalesArea(com.floreantpos.model.SalesArea salesArea) {
		this.salesArea = salesArea;
		if (salesArea == null) {
			setSalesAreaId(null);
		}
		else {
			setSalesAreaId(salesArea.getId());
		}
	}

	public boolean isSplited() {
		return Boolean.valueOf(getProperty(Ticket.SPLIT));
	}

	public void updateTicketItemPriceByOrderType(String orderTypeOption) {
		List<TicketItem> ticketItems = getTicketItems();
		OrderType orderType = getOrderType();
		TaxGroup taxGroup = null;

		if (ticketItems == null) {
			return;
		}
		if (orderTypeOption.equals(SubOrderType.FOR_HERE.getName())) {
			taxGroup = orderType.getForHereTaxGroup();
			addProperty(Ticket.PROPERTY_SUB_ORDER_TYPE, SubOrderType.FOR_HERE.getName());
		}
		else if (orderTypeOption.equals(SubOrderType.TO_GO.getName())) {
			taxGroup = orderType.getToGoTaxGroup();
			addProperty(Ticket.PROPERTY_SUB_ORDER_TYPE, SubOrderType.TO_GO.getName());
		}

		for (TicketItem ticketItem : ticketItems) {
			MenuItem.setItemTaxes(ticketItem, taxGroup, getOrderType());
		}
		calculatePrice();
	}

	public String getSplitId() {
		return getProperty(Ticket.SPLIT_TICKET_ID);
	}

	public void setSplitOrder(int order) {
		addProperty("SPLIT_ORDER", String.valueOf(order)); //$NON-NLS-1$
	}

	@JsonIgnore
	public String getDiffWithCrntTime() {
		if (getCreateDate() == null) {
			return null;
		}
		return DateUtil.getElapsedTime(getCreateDate(), new Date());
	}

	public void setDiffWithCrntTime(String diffWithCrntTime) {
	}

	public String getTableNames() {
		return getProperty(jSON_PROP_TABLE_NAMES);
	}

	@XmlTransient
	public String getBartabName() {
		if (getOrderType() != null && getOrderType().isBarTab()) {
			String barTabName = getProperty(Ticket.JSON_PROP_BARTAB_NAME);
			if (StringUtils.isBlank(barTabName)) {
				barTabName = getProperty(Ticket.CUSTOMER_NAME);
			}
			if (StringUtils.isNotBlank(barTabName)) {
				return barTabName;
			}
		}
		return null;
	}

	public String getTableNameDisplay() {
		String tableNames = getTableNames();
		if (StringUtils.isBlank(tableNames)) {
			return POSUtil.toFormattedString(getTableNumbers());
		}
		return tableNames;
	}

	public String getTableNameOrNumberDisplay() {
		Store store = DataProvider.get().getStore();
		if (store.isShowTableNameOnTable()) {
			return this.getTableNameDisplay();
		}
		else {
			return this.getTableNumbersDisplay();
		}
	}

	public Double getTotalAmountWithTips() {
		return super.getTotalAmount() + getGratuityAmount();
	}

	public void calculateRefundAmount() {
		double refundAmount = 0;
		double voidAmount = 0;
		Set<PosTransaction> transactions = getTransactions();
		if (transactions != null) {
			for (PosTransaction t : transactions) {
				if (t instanceof RefundTransaction && !t.isVoided()) {
					refundAmount += t.getAmount();
				}
			}
		}
		setRefundAmount(refundAmount + voidAmount);
	}

	public String getTableNumbersDisplay() {
		List<Integer> numbers = getTableNumbers();
		if (numbers == null || numbers.size() <= 0) {
			return ""; //$NON-NLS-1$
		}
		String tableNumbers = numbers.toString().replaceAll("[\\[\\]]", ""); //$NON-NLS-1$ //$NON-NLS-2$
		return tableNumbers;
	}

	public boolean isPrintedToKitchenOrInventoryAdjusted() {
		for (TicketItem ticketItem : getTicketItems()) {
			if (ticketItem.isPrintedToKitchen() || ticketItem.isInventoryAdjusted() || (!ticketItem.isReturned() && ticketItem.isVoided())) {
				return true;
			}
			if (ticketItem.isHasModifiers()) {
				for (TicketItemModifier modifier : ticketItem.getTicketItemModifiers()) {
					if (modifier.isPrintedToKitchen())
						return true;
				}
			}
		}
		return false;
	}

	public boolean hasRepriceDiscount() {
		List<TicketDiscount> discounts = getDiscounts();
		if (discounts == null || discounts.isEmpty()) {
			return false;
		}

		for (TicketDiscount ticketDiscount : discounts) {
			if (ticketDiscount.getOriginalType() == Discount.DISCOUNT_TYPE_REPRICE) {
				return true;
			}
		}

		return false;
	}

	public Boolean hasRefundableItem() {
		return refundableItem == null ? false : refundableItem;
	}

	public void setRefundableItem(boolean refundableItem) {
		this.refundableItem = refundableItem;
	}

	public double getVoidSubtotal() {
		return voidSubtotal;
	}

	public double getVoidTotal() {
		return voidTotal;
	}

	@Override
	public void setTotalAmount(Double totalAmount) {
		super.setTotalAmount(totalAmount + 0.0);
	}

	@Override
	public void setDueAmount(Double dueAmount) {
		super.setDueAmount(dueAmount + 0.0);
	}

	@Override
	public void setDiscountAmount(Double discountAmount) {
		super.setDiscountAmount(discountAmount + 0.0);
	}

	@Override
	public void setTaxAmount(Double taxAmount) {
		super.setTaxAmount(taxAmount + 0.0);
	}

	@XmlTransient
	public OrderType getOrderType() {
		if (orderType == null) {
			orderType = DataProvider.get().getOrderTypeById(getOrderTypeId(), getOutletId());
		}
		return orderType;
	}

	public void setOrderType(OrderType orderType) {
		this.orderType = orderType;
		if (orderType != null) {
			setServiceChargeApplicable(orderType.isServiceChargeApplicable());
			setServiceChargeType(orderType.getServiceChargeType());
			setOutletServiceChargeRate(orderType.getServiceChargeRate());
			super.setOrderTypeId(orderType.getId());
		}
		else {
			try {
				throw new PosException("Order type is null!");
			} catch (Exception e) {
				PosLog.error(getClass(), e);
			}
		}
	}

	@XmlTransient
	public Shift getShift() {
		return DataProvider.get().getShiftById(getShiftId(), getOutletId());
	}

	public void setShift(Shift shift) {
		String shiftId = null;
		if (shift != null) {
			shiftId = shift.getId();
		}
		super.setShiftId(shiftId);
	}

	@XmlTransient
	public User getOwner() {
		return DataProvider.get().getUserById(getOwnerId(), getOutletId());
	}

	public void setOwner(User owner) {
		String ownerId = null;
		String ownerTypeId = null;
		if (owner != null) {
			ownerId = owner.getId();
			ownerTypeId = owner.getUserTypeId();

			if (getCreatedByUserId() == null) {
				setCreatedByUserId(ownerId);
			}

			addProperty("owner.firstname", owner.getFirstName()); //$NON-NLS-1$
			addProperty("owner.lastname", owner.getLastName()); //$NON-NLS-1$
		}
		super.setOwnerId(ownerId);
		super.setOwnerTypeId(ownerTypeId);
	}

	@XmlTransient
	public User getCashier() {
		return DataProvider.get().getUserById(getCashierId(), getOutletId());
	}

	public void setCashier(User cashier) {
		String cashierId = null;
		if (cashier != null) {
			cashierId = cashier.getId();
		}
		super.setCashierId(cashierId);
	}

	@XmlTransient
	public User getAssignedDriver() {
		return DataProvider.get().getUserById(getAssignedDriverId(), getOutletId());
	}

	public void setAssignedDriver(User assignedDriver) {
		String assignedDriverId = null;
		if (assignedDriver != null) {
			assignedDriverId = assignedDriver.getId();
			addProperty("driver.firstname", assignedDriver.getFirstName()); //$NON-NLS-1$
			addProperty("driver.lastname", assignedDriver.getLastName()); //$NON-NLS-1$
		}
		else {
			removeProperty("driver.firstname"); //$NON-NLS-1$
			removeProperty("driver.lastname"); //$NON-NLS-1$
		}
		super.setAssignedDriverId(assignedDriverId);
	}

	@XmlTransient
	public String getDriverNameFromProperty() {
		String firstName = getProperty("driver.firstname"); //$NON-NLS-1$
		String lastName = getProperty("driver.lastname"); //$NON-NLS-1$

		return firstName + " " + lastName; //$NON-NLS-1$
	}

	@XmlTransient
	public User getVoidedBy() {
		return DataProvider.get().getUserById(getVoidedById(), getOutletId());
	}

	public void setVoidedBy(User voidBy) {
		String voidById = null;
		if (voidBy != null) {
			voidById = voidBy.getId();
		}
		super.setVoidedById(voidById);
	}

	@XmlTransient
	public Terminal getTerminal() {
		return DataProvider.get().getTerminalById(getTerminalId(), getOutletId());
	}

	public void setTerminal(Terminal terminal) {
		Integer terminalId = null;
		if (terminal != null) {
			terminalId = terminal.getId();
		}
		super.setTerminalId(terminalId);
	}

	/*
	 * Discount part build as json and put in properties
	 */
	public void setDiscounts(List<TicketDiscount> discounts) {
		this.discounts = discounts;
		if (this.discounts == null) {
			this.discounts = new ArrayList<>();
		}
	}

	public List<TicketDiscount> getDiscounts() {
		if (discounts == null) {
			discounts = new ArrayList<TicketDiscount>();
			convertDiscountPropertyToList();
		}
		return discounts;
	}

	public void convertDiscountPropertyToList() {
		String property = super.getDiscountsProperty();
		if (StringUtils.isNotEmpty(property)) {
			JsonReader jsonParser = Json.createReader(new StringReader(property));
			JsonArray jsonArray = jsonParser.readArray();
			jsonParser.close();
			for (int i = 0; i < jsonArray.size(); i++) {
				JsonObject dObj = (JsonObject) jsonArray.get(i);
				TicketDiscount td = new TicketDiscount();
				td.setTicket(this);
				td.setId(JsonUtil.getString(dObj, TicketDiscount.PROP_ID));
				td.setLoyaltyPoint(JsonUtil.getInt(dObj, TicketDiscount.PROP_LOYALTY_POINT));
				td.setLoyaltyCharged(JsonUtil.getBoolean(dObj, TicketDiscount.PROP_LOYALTY_CHARGED));
				td.setDiscountId(JsonUtil.getString(dObj, TicketDiscount.PROP_DISCOUNT_ID));
				td.setName(JsonUtil.getString(dObj, TicketDiscount.PROP_NAME));
				td.setType(JsonUtil.getInt(dObj, TicketDiscount.PROP_TYPE));
				td.setAutoApply(JsonUtil.getBoolean(dObj, TicketDiscount.PROP_AUTO_APPLY));
				td.setCouponQuantity(JsonUtil.getDouble(dObj, TicketDiscount.PROP_COUPON_QUANTITY));
				td.setMinimumAmount(JsonUtil.getDouble(dObj, TicketDiscount.PROP_MINIMUM_AMOUNT));
				td.setValue(JsonUtil.getDouble(dObj, TicketDiscount.PROP_VALUE));
				td.setTotalDiscountAmount(JsonUtil.getDouble(dObj, TicketDiscount.PROP_TOTAL_DISCOUNT_AMOUNT));
				td.setCouponBarcode(JsonUtil.getString(dObj, "couponBarcode")); //NON-NLS-1
				Integer originalType = JsonUtil.getInt(dObj, TicketDiscount.PROP_ORIGINAL_TYPE);
				if (originalType == null) {
					originalType = td.getType();
				}
				td.setOriginalType(originalType);
				Double originalValue = JsonUtil.getDouble(dObj, TicketDiscount.PROP_ORIGINAL_VALUE);
				if (originalValue == null) {
					originalValue = td.getValue();
				}
				td.setOriginalValue(originalValue);
				discounts.add(td);
			}
		}
	}

	public void addTodiscounts(TicketDiscount ticketDiscount) {
		discounts = getDiscounts();
		discounts.add(ticketDiscount);
		buildDiscounts();
		ActionHistoryDAO.addDiscountAddedActionHistory(this, ticketDiscount);
	}

	public void buildDiscounts() {
		JSONArray jsonArray = new JSONArray();
		for (TicketDiscount ticketDiscount : discounts) {
			jsonArray.put(ticketDiscount.toJson());
		}
		setDiscountsProperty(jsonArray.toString());
	}

	public String getAppliedCouponNumberDisplay() {
		if (getDiscountAmount() == 0 || getDiscounts() == null || getDiscounts().isEmpty()) {
			return ""; //$NON-NLS-1$
		}
		TicketDiscount ticketDiscount = getDiscounts().get(0);
		String couponBarcode = ticketDiscount.getCouponBarcode();
		if (StringUtils.isBlank(couponBarcode)) {
			couponBarcode = ticketDiscount.getNameDisplay();
		}
		return " (" + couponBarcode + ")"; //$NON-NLS-1$ //$NON-NLS-2$
	}

	//	@XmlTransient
	//	public Map<String, String> getProperties() {
	//		Map<String, String> properties = super.getProperties();
	//		if (properties == null) {
	//			properties = new HashMap<>();
	//			super.setProperties(properties);
	//		}
	//		return properties;
	//	}

	@Override
	public String getTicketTableNumbers() {
		String ticketTableNumbers = ""; //$NON-NLS-1$
		List<Integer> tableNumbers = getTableNumbers();
		if (tableNumbers != null && tableNumbers.size() > 0) {
			for (Iterator iterator = tableNumbers.iterator(); iterator.hasNext();) {
				Integer tableNo = (Integer) iterator.next();
				ticketTableNumbers += tableNo;
				if (iterator.hasNext())
					ticketTableNumbers += ","; //$NON-NLS-1$
			}
		}
		return ticketTableNumbers;
	}

	@Override
	public void setTicketTableNumbers(String ticketTableNumbers) {
		tableNumbers = new ArrayList<Integer>();
		if (StringUtils.isNotEmpty(ticketTableNumbers)) {
			String[] tableNums = ticketTableNumbers.split(","); //$NON-NLS-1$
			for (String tbl : tableNums) {
				int tableNo = POSUtil.parseInteger(tbl);
				if (tableNo > 0)
					tableNumbers.add(tableNo);
			}
		}
		super.setTicketTableNumbers(ticketTableNumbers);
	}

	public List<Integer> getTableNumbers() {
		return tableNumbers;
	}

	public void setTableNumbers(List<Integer> tableNumbers) {
		this.tableNumbers = tableNumbers;

		if (tableNumbers == null || tableNumbers.size() <= 0) {
			removeProperty(jSON_PROP_TABLE_NAMES);
			return;
		}
		List<String> tableNames = ShopTableDAO.getInstance().getTableNames(tableNumbers);
		if (tableNames != null && !tableNames.isEmpty()) {
			addProperty(jSON_PROP_TABLE_NAMES, POSUtil.toFormattedString(tableNames));
		}
		else {
			addProperty(jSON_PROP_TABLE_NAMES, POSUtil.toFormattedString(tableNumbers));
		}
	}

	public List<ShopTable> getTables() {
		List<Integer> tableNumbers = getTableNumbers();
		if (tableNumbers == null || tableNumbers.size() == 0) {
			return new ArrayList<>(1);
		}
		if (tables == null) {
			tables = new ArrayList<>(1);
		}
		for (Integer tableId : tableNumbers) {
			ShopTable table = new ShopTable(tableId, getOutletId());
			if (!tables.contains(table)) {
				table = ShopTableDAO.getInstance().loadWithSeats(tableId, table.getOutletId());
				if (table != null) {
					tables.add(table);
				}
			}
		}
		return tables = tables.stream().filter(st -> tableNumbers.contains(st.getId())).collect(Collectors.toList());
	}

	public void voidItem(TicketItem ticketItem, String voidReason, boolean itemWasted) {
		voidItem(ticketItem, voidReason, itemWasted, ticketItem.getQuantity());
	}

	public void voidItem(TicketItem ticketItem, String voidReason, boolean itemWasted, double quantity) {
		if (ticketItem.isTreatAsSeat()) {
			return;
		}
		if (StringUtils.isEmpty(voidReason)) {
			throw new PosException(Messages.getString("Ticket.23")); //$NON-NLS-1$
		}
		if (quantity == 0) {
			throw new PosException(Messages.getString("Ticket.24")); //$NON-NLS-1$
		}
		if (quantity > (Math.abs(ticketItem.getQuantity()))) {
			throw new PosException(Messages.getString("Ticket.25")); //$NON-NLS-1$
		}
		double ticketItemQuantity = ticketItem.getQuantity();
		boolean fullVoid = ticketItem.getQuantity() == quantity;

		TicketItem voidTicketItem = null;
		if (fullVoid) {
			ticketItem.setVoided(true);
			ticketItem.setQuantity(-quantity);
			ticketItem.setVoidedItemId(ticketItem.getId());
			voidTicketItem = ticketItem;
		}
		else {
			ticketItem.setQuantity(ticketItem.getQuantity() - quantity);
			voidTicketItem = ticketItem.cloneAsNew();
			voidTicketItem.setId(null);
			voidTicketItem.setQuantity(-1 * quantity);
		}
		Boolean printedToKitchen = ticketItem.isPrintedToKitchen();
		voidTicketItem.setDiscounts(null);
		voidTicketItem.setPrintedToKitchen(false);
		voidTicketItem.setVoided(true);
		voidTicketItem.setCloudSynced(false);
		voidTicketItem.setVoidDate(StoreDAO.getServerTimestamp());
		voidTicketItem.setVoidedItemId(ticketItem.getId());
		voidTicketItem.setInventoryAdjustQty(itemWasted ? -quantity : 0.0);

		if (voidTicketItem.isHasModifiers()) {
			List<TicketItemModifier> ticketItemModifiers = voidTicketItem.getTicketItemModifiers();
			if (ticketItemModifiers != null) {
				for (TicketItemModifier modifier : ticketItemModifiers) {
					if (modifier.isInfoOnly()) {
						continue;
					}
					modifier.setItemQuantity(-modifier.getItemQuantity());
				}
			}
		}
		//VoidItem voidItem = ticketItem.createVoidItem(voidReason, itemWasted, quantity);
		voidTicketItem.setVoidProperties(voidReason, itemWasted, quantity, printedToKitchen);
		if (ticketItem.isComboItem()) {
			if (voidTicketItem.getComboItems() != null) {
				for (TicketItem comboTicketItem : voidTicketItem.getComboItems()) {
					double toBeVoidComboQuantity = (comboTicketItem.getQuantity() / ticketItemQuantity) * quantity;
					comboTicketItem.setQuantity(-toBeVoidComboQuantity);
					voidTicketItem.setInventoryAdjustQty(itemWasted ? -toBeVoidComboQuantity : 0.0);
					comboTicketItem.setVoidProperties(voidReason, itemWasted, toBeVoidComboQuantity);
					//VoidItem comboVoidItem = comboTicketItem.createVoidItem(voidReason, itemWasted, toBeVoidComboQuantity);
					comboTicketItem.setVoided(true);
					//comboTicketItem.setVoidItem(comboVoidItem);

					List<TicketItemModifier> ticketItemModifiers = comboTicketItem.getTicketItemModifiers();
					if (ticketItemModifiers != null && ticketItemModifiers.size() > 0) {
						for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
							ticketItemModifier.setItemQuantity(-1 * ticketItemModifier.getItemQuantity());
							ticketItemModifier.calculatePrice();
						}
					}
					comboTicketItem.calculatePrice();
				}
			}
			if (!fullVoid) {
				if (ticketItem.getComboItems() != null) {
					for (TicketItem comboTicketItem : ticketItem.getComboItems()) {
						comboTicketItem.setQuantity(comboTicketItem.getQuantity() - ((comboTicketItem.getQuantity() / ticketItemQuantity) * quantity));
					}
				}
			}
		}
		voidTicketItem.calculatePrice();

		if (voidTicketItem != ticketItem) {
			addToticketItems(voidTicketItem);
		}
	}

	public void voidReturnedItem(TicketItem voidTicketItem, String voidReason, boolean itemWasted) {
		if (voidTicketItem.isTreatAsSeat()) {
			return;
		}
		voidTicketItem.markReturnedItemVoided(voidReason, itemWasted, voidTicketItem.getQuantity());
		if (voidTicketItem.isComboItem()) {
			if (voidTicketItem.getComboItems() != null) {
				for (TicketItem comboTicketItem : voidTicketItem.getComboItems()) {
					comboTicketItem.markReturnedItemVoided(voidReason, itemWasted, comboTicketItem.getQuantity());
				}
			}
		}
	}

	public TicketItem undoVoidItem(TicketItem voidedTicketItem) {
		TicketItem origTicketItem = getTicketItem(voidedTicketItem.getVoidedItemId());
		if (origTicketItem == null) {
			return null;
		}
		double quantity = Math.abs(voidedTicketItem.getQuantity());
		if (origTicketItem.getVoidedItemId() != null) {
			origTicketItem = voidedTicketItem;
		}
		boolean itemWasted = POSUtil.getBoolean(voidedTicketItem.getProperty(TicketItem.JSON_PROP_WASTED));
		double ticketItemQuantity = origTicketItem.getQuantity();
		boolean fullVoid = voidedTicketItem == origTicketItem;

		if (fullVoid) {
			origTicketItem.setQuantity(quantity);
		}
		else {
			origTicketItem.setQuantity(origTicketItem.getQuantity() + quantity);
		}
		origTicketItem.setPrintedToKitchen(POSUtil.getBoolean(voidedTicketItem.getProperty(TicketItem.JSON_PROP_PRINTED_TO_KITCHEN)));
		origTicketItem.setVoided(false);
		origTicketItem.setVoidDate(null);
		origTicketItem.setVoidedItemId(null);
		origTicketItem.setCloudSynced(false);
		origTicketItem.setHasSyncError(false);
		origTicketItem.removeProperty(TicketItem.JSON_PROP_RETURNED);
		if (voidedTicketItem.isInventoryAdjusted() && !itemWasted) {
			origTicketItem.setInventoryAdjustQty(Math.abs(origTicketItem.getInventoryAdjustQty()) - quantity);
		}
		else {
			origTicketItem.setInventoryAdjustQty(Math.abs(fullVoid ? origTicketItem.getQuantity() : origTicketItem.getQuantity() + quantity));
		}
		if (origTicketItem.isHasModifiers()) {
			List<TicketItemModifier> ticketItemModifiers = origTicketItem.getTicketItemModifiers();
			if (ticketItemModifiers != null) {
				for (TicketItemModifier modifier : ticketItemModifiers) {
					if (modifier.isInfoOnly()) {
						continue;
					}
					modifier.setItemQuantity(Math.abs(modifier.getItemQuantity()));
				}
			}
		}
		origTicketItem.removeVoidProperties();
		if (voidedTicketItem.isComboItem()) {
			if (voidedTicketItem.getComboItems() != null) {
				for (TicketItem comboTicketItem : voidedTicketItem.getComboItems()) {
					double toBeVoidComboQuantity = Math.abs((comboTicketItem.getQuantity() / ticketItemQuantity) * quantity);
					comboTicketItem.setQuantity(toBeVoidComboQuantity);
					voidedTicketItem.setInventoryAdjustQty(itemWasted ? -toBeVoidComboQuantity : 0.0);
					comboTicketItem.removeVoidProperties();
					comboTicketItem.setVoided(false);
				}
			}
			if (!fullVoid) {
				if (origTicketItem.getComboItems() != null) {
					for (TicketItem comboTicketItem : origTicketItem.getComboItems()) {
						comboTicketItem.setQuantity(comboTicketItem.getQuantity() + ((comboTicketItem.getQuantity() / ticketItemQuantity) * quantity));
					}
				}
			}
		}
		origTicketItem.calculatePrice();
		return origTicketItem;
	}

	private TicketItem getTicketItem(String ticketItemId) {
		if (StringUtils.isBlank(ticketItemId)) {
			return null;
		}
		for (TicketItem ticketItem : getTicketItems()) {
			if (ticketItem.getId() == null) {
				continue;
			}
			if (ticketItem.getId().equals(ticketItemId)) {
				return ticketItem;
			}
		}
		return null;
	}

	private void populateDefaultProperties() {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		setTerminal(terminal);
		setCreateDate(StoreDAO.getServerTimestamp());

		StoreSession storeSession = DataProvider.get().getStoreSession();
		if (storeSession != null) {
			setStoreSessionId(storeSession.getId());
		}

		Department department = terminal.getDepartment();
		if (department != null)
			setDepartmentId(department.getId());

		Outlet outlet = terminal.getOutlet();
		if (outlet != null) {
			setOutletId(outlet.getId());
			setOutletServiceChargeRate(outlet.getServiceChargePercentage());
			setOutletGratuityRate(outlet.getDefaultGratuityPercentage());
		}

		setOwner(Application.getCurrentUser());
		setShift(ShiftUtil.getCurrentShift());
		setShouldIncludeInSales(true);
		Currency mainCurrency = DataProvider.get().getMainCurrency();
		setToleranceFactor(mainCurrency.getTolerance());

		//setApplyFloridaTaxRule(DataProvider.get().getStore().isEnableFloridaTaxRule());

		setDataVersion(2);
	}

	public boolean isShouldUpdateTableStatus() {
		return shouldUpdateTableStatus;
	}

	public void setShouldUpdateTableStatus(boolean isUpdateTable) {
		this.shouldUpdateTableStatus = isUpdateTable;
	}

	public PosTransaction getBartabTransaction() {
		String bartabTransactionId = getProperty("bartab.transaction.id"); //$NON-NLS-1$
		if (StringUtils.isNotEmpty(bartabTransactionId)) {
			for (PosTransaction transaction : getTransactions()) {
				if (!transaction.isVoided() && bartabTransactionId.equals(String.valueOf(transaction.getId()))) {
					return transaction;
				}
			}
		}

		return null;
	}

	@Override
	public String getExtraProperties() {
		this.initPropertiesContainer();
		return this.propertiesContainer.toString();
	}

	@Override
	public void setExtraProperties(String properties) {
		super.setExtraProperties(properties);
		this.propertiesContainer = null;
		this.initPropertiesContainer();
	}

	public void addExtraProperty(String key, String value) {
		this.initPropertiesContainer();
		propertiesContainer.addProperty(key, value);
		super.setExtraProperties(propertiesContainer.toString());
	}

	public String getExtraProperty(String key) {
		this.initPropertiesContainer();
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	@Override
	public String toString() {
		return "Ticket [ticketDiscountAmount=" + ", itemDiscountAmount=" + getItemDiscountAmount() + ", voidSubtotal=" + voidSubtotal //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				+ ", voidTotal=" + voidTotal + ", deletedItems=" + deletedItems + ", sortOrder=" + sortOrder //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				+ ", customer=" + customer + ", outlet=" + outlet + ", department=" + department + ", salesArea=" + salesArea + ", discounts=" + discounts //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
				+ ", tableNumbers=" + tableNumbers + ", tables=" + tables + ", shouldUpdateTableStatus=" + shouldUpdateTableStatus + ", roundedAmount=" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				+ getRoundedAmount() + ", toleranceAmount=" + getToleranceAmount() + ", refundableItem=" + refundableItem + ", getTicketItems()=" //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
				+ getTicketItems() + "]"; //$NON-NLS-1$
	}

	public boolean isShouldPublishMqtt() {
		return shouldPublishMqtt;
	}

	public void setShouldPublishMqtt(boolean shouldPublishMqtt) {
		this.shouldPublishMqtt = shouldPublishMqtt;
	}

	public String getOwnerName() {
		if (getOwner() != null) {
			return getOwner().getFullName();
		}
		return ""; //$NON-NLS-1$
	}

	@XmlTransient
	public String getOwnerNameFromProperty() {
		String firstName = getProperty("owner.firstname"); //$NON-NLS-1$
		String lastName = getProperty("owner.lastname"); //$NON-NLS-1$
		if (StringUtils.isBlank(lastName)) {
			return firstName;
		}

		return firstName + " " + lastName; //$NON-NLS-1$
	}

	public void setOwnerName(String ownerName) {
	}

	@XmlTransient
	public String getCustomerName() {
		if (getCustomer() != null) {
			return getCustomer().getName();
		}
		String customerName = getProperty(CUSTOMER_NAME);
		if (propertiesContainer.has(CUSTOMER_LAST_NAME)) {
			customerName = getProperty(CUSTOMER_NAME) + " " + getProperty(CUSTOMER_LAST_NAME); //$NON-NLS-1$
		}
		return customerName;
	}

	public void setCustomerName(String customerName) {
		addProperty(CUSTOMER_NAME, customerName);
	}

	@XmlTransient
	public String getCustomerEmail() {
		if (getCustomer() != null) {
			return getCustomer().getEmail();
		}
		return getProperty(Ticket.CUSTOMER_EMAIL); //$NON-NLS-1$
	}

	public void setCustomerEmail(String customerEmail) {
		addProperty(Ticket.CUSTOMER_EMAIL, customerEmail); //$NON-NLS-1$
	}

	@XmlTransient
	public String getCustomerMobileNo() {
		String phone = getProperty(CUSTOMER_PHONE, "");//$NON-NLS-1$

		if (StringUtils.isBlank(phone) && getCustomer() != null) {
			return getCustomer().getMobileNo();
		}

		return phone;
	}

	public void setCustomerMobileNo(String customerMobileNo) {
		addProperty(CUSTOMER_PHONE, customerMobileNo);
	}

	public void makeXmlTransient() {
		//		setProperties(null);
		setDiscounts(null);
		for (TicketItem ticketItem : getTicketItems()) {
			ticketItem.setTicket(null);
		}
		if (getTransactions() != null) {
			for (PosTransaction transaction : getTransactions()) {
				transaction.setTicket(null);
			}
		}
	}

	public void mergeTicket(Ticket otherTicket) {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems != null && ticketItems.size() > 0) {
			for (TicketItem ticketItem : ticketItems) {
				if (ticketItem.isHasModifiers()) {
					if (ticketItem.getTicketItemModifiers() != null && ticketItem.getTicketItemModifiers().size() > 0) {
						for (TicketItemModifier ticketItemModifier : ticketItem.getTicketItemModifiers()) {
							ticketItemModifier.setTicketItem(ticketItem);
						}
					}
				}
				List<TicketItemDiscount> ticketItemDiscounts = ticketItem.getDiscounts();
				if (ticketItemDiscounts != null && ticketItemDiscounts.size() > 0) {
					for (TicketItemDiscount ticketItemDiscount : ticketItemDiscounts) {
						ticketItemDiscount.setTicketItem(ticketItem);
					}
				}
			}
		}
		mergeTicketItems(otherTicket.getTicketItems());
		mergeTransactions(otherTicket.getTransactions());

		calculatePrice();
	}

	private void mergeTicketItems(List<TicketItem> otherTicketItems) {
		if (otherTicketItems.isEmpty())
			return;

		List<TicketItem> thisTicketItems = getTicketItems();
		for (TicketItem localTicketItem : otherTicketItems) {
			int idx = thisTicketItems.indexOf(localTicketItem);
			if (idx != -1) {
				TicketItem thisTicketItem = thisTicketItems.get(idx);
				if (localTicketItem.isPrintedToKitchen()) {
					thisTicketItem.setPrintedToKitchen(true);
				}
				if (localTicketItem.getQuantity() != thisTicketItem.getQuantity()) {
					//to be fix
				}
			}
			else {
				localTicketItem.setTicket(this);
				addToticketItems(localTicketItem);
			}
		}
	}

	private void mergeTransactions(Set<PosTransaction> otherTransactions) {
		if (otherTransactions == null || otherTransactions.isEmpty())
			return;

		List<PosTransaction> thisTransactions = new ArrayList<PosTransaction>(getTransactions());
		for (PosTransaction otherTransaction : otherTransactions) {
			int idx = thisTransactions.indexOf(otherTransaction);
			if (idx == -1) {
				otherTransaction.setTicket(this);
				addTotransactions(otherTransaction);
			}
		}
	}

	private void calculatePaidAmount() {
		double totalCredit = 0;
		String bartabTransactionId = getProperty("bartab.transaction.id"); //$NON-NLS-1$
		if (bartabTransactionId == null) {
			bartabTransactionId = ""; //$NON-NLS-1$
		}
		Set<PosTransaction> transactionsAfterMerged = super.getTransactions();
		if (transactionsAfterMerged != null && transactionsAfterMerged.size() > 0) {
			for (PosTransaction t : transactionsAfterMerged) {
				if (t.isVoided()) {
					continue;
				}
				if (bartabTransactionId.equals(t.getId()) && !t.isCaptured()) {
					//pre authorized transaction for bartab. Should not be considered as paid. (OR-2865)
					continue;
				}

				if (t.getTransactionType().equals(TransactionType.CREDIT.toString())) {
					totalCredit += t.getAmount();
				}
			}
		}
		setPaidAmount(NumberUtil.round(totalCredit));
		calculateRefundAmount();
		if (getRefundAmount() > 0) {
			setRefunded(true);
		}
	}

	private boolean isPriceApplicable(TicketItem ticketItem) {
		if (ticketItem.isVoided() && !ticketItem.isItemReturned()) {
			return false;
		}
		return true;
	}

	public void setHasGiftCard(boolean hasGiftCard) {
		addProperty(AppConstants.HAS_GIFT_CARD, String.valueOf(hasGiftCard));
	}

	public boolean hasGiftCard() {
		return Boolean.valueOf(getProperty(AppConstants.HAS_GIFT_CARD));
	}

	public void setSubOrderType(SubOrderType subOrderType) {
		if (subOrderType != null) {
			addProperty(PROPERTY_SUB_ORDER_TYPE, subOrderType.getName());
		}
	}

	public SubOrderType getSubOrderType() {
		return SubOrderType.fromName(getExtraProperty(Ticket.PROPERTY_SUB_ORDER_TYPE));
	}

	public boolean hasSeat() {
		for (TicketItem ticketItem : getTicketItems()) {
			if (ticketItem.isTreatAsSeat()) {
				return true;
			}
		}
		return false;
	}

	public boolean isKitchenPrintable() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return false;
		}
		for (TicketItem ticketItem : ticketItems) {
			if (!ticketItem.isTreatAsSeat() && ticketItem.isShouldPrintToKitchen() && !ticketItem.isPrintedToKitchen()) {
				return true;
			}
		}
		return false;
	}

	public boolean isDiscountable() {
		List<TicketItem> ticketItems = this.getTicketItems();
		if (ticketItems == null || ticketItems.isEmpty()) {
			return Boolean.FALSE;
		}
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isTreatAsSeat()) {
				continue;
			}
			return Boolean.TRUE;
		}

		return Boolean.FALSE;
	}

	public boolean isVoidable() {
		List<TicketItem> ticketItems = this.getTicketItems();
		if (ticketItems == null || ticketItems.isEmpty()) {
			return Boolean.FALSE;
		}
		return Boolean.TRUE;
	}

	public List<ShopSeat> getExtraSeats() {
		List<ShopSeat> shopSeats = null;
		String shopSeatJSON = this.getProperty(PROPERTY_EXTRA_SEATS, null);
		if (StringUtils.isEmpty(shopSeatJSON)) {
			shopSeats = new ArrayList<>();
		}
		else {
			ObjectMapper mapper = new ObjectMapper();
			try {
				shopSeats = mapper.readValue(shopSeatJSON, new TypeReference<List<ShopSeat>>() {
				});
			} catch (IOException e) {
				PosLog.error(getClass(), e);
			}
		}
		return shopSeats;
	}

	public void addExtraSeats(List<ShopSeat> shopSeats) {
		String shopSeatJSON = null;
		if (shopSeats != null) {
			ObjectMapper mapper = new ObjectMapper();
			try {
				shopSeatJSON = mapper.writeValueAsString(shopSeats);
			} catch (JsonProcessingException e) {
				PosLog.error(getClass(), e);
			}
		}
		this.addProperty(PROPERTY_EXTRA_SEATS, shopSeatJSON);
	}

	public Map<String, ActionHistory> getEvents() {
		if (this.events == null) {
			this.events = new LinkedHashMap<>();
		}
		return this.events;
	}

	public boolean hasLoyaltyDiscount() {
		List<TicketDiscount> discounts = getDiscounts();
		if (discounts == null) {
			return false;
		}
		for (TicketDiscount ticketDiscount : discounts) {
			String discountId = ticketDiscount.getDiscountId();
			if (StringUtils.isNotBlank(discountId) && discountId.equals("loyalty_discount")) {//$NON-NLS-1$
				return true;
			}
		}
		return false;
	}

	@XmlTransient
	public int getBeverageItemCount() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null) {
			return 0;
		}

		int beverageCount = 0;
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem == null || ticketItem.isTreatAsSeat()) {
				continue;
			}
			if (ticketItem.isBeverage()) {
				beverageCount += Math.abs(ticketItem.getQuantity());
			}
		}

		return beverageCount;
	}

	public void validateTicketBeforeSave() {
		if (getTicketItems() == null || getTicketItems().size() == 0) {
			throw new PosException(POSConstants.TICKET_IS_EMPTY_);
		}

		if (orderType.isBeverageMandatory()) {
			Integer numberOfGuests = getNumberOfGuests();
			int beverageItemCount = getBeverageItemCount();
			if (beverageItemCount < numberOfGuests) {
				throw new PosException(String.format(Messages.getString("Ticket.2"), numberOfGuests)); //$NON-NLS-1$
			}
		}
	}

	public double getTaxAmountFromProperty() {
		return POSUtil.parseDouble(this.getProperty(AppConstants.TAX_EXEMPT_AMOUNT));
	}

	public String getCustomerNameByProperty() {
		String firstName = this.getProperty(Ticket.CUSTOMER_NAME) == null ? "" : this.getProperty(Ticket.CUSTOMER_NAME); //$NON-NLS-1$
		String lastName = this.getProperty(Ticket.CUSTOMER_LAST_NAME) == null ? "" : this.getProperty(Ticket.CUSTOMER_LAST_NAME); //$NON-NLS-1$
		if (StringUtils.isBlank(lastName)) {
			return firstName;
		}

		return firstName + " " + lastName; //$NON-NLS-1$
	}

	public Double getOutletServiceChargeRate() {
		String scRateStr = getProperty(PROPERTY_OUTLET_SC_RATE);
		if (StringUtils.isBlank(scRateStr)) {
			return 0.0;
		}
		try {
			return NumberUtil.parse(scRateStr).doubleValue();
		} catch (ParseException e) {
			//PosLog.error(getClass(), e);
		}
		return 0.0;
	}

	public void setOutletServiceChargeRate(Double rate) {
		addProperty(PROPERTY_OUTLET_SC_RATE, String.valueOf(rate));
	}

	public Double getOutletGratuityRate() {
		String gratuityRate = getProperty(PROPERTY_OUTLET_GRATUITY_RATE);
		if (StringUtils.isBlank(gratuityRate)) {
			return 0.0;
		}
		try {
			return NumberUtil.parse(gratuityRate).doubleValue();
		} catch (ParseException e) {
			PosLog.error(getClass(), e);
		}
		return 0.0;
	}

	public void setOutletGratuityRate(Double rate) {
		addProperty(PROPERTY_OUTLET_GRATUITY_RATE, String.valueOf(rate));
	}

	public Double getSubtotalWithoutIncludedTax() {
		if (isTaxIncluded()) {
			return getSubtotalAmount() - getTaxAmount();
		}
		return getSubtotalAmount();
	}

	public void closeIfApplicable() {
		if (NumberUtil.isZero(getDueAmount())) {
			OrderType orderType = getOrderType();

			if (orderType != null && (orderType.isCloseOnPaid() || orderType.isBarTab())) {//fix
				setClosed(true);
				setClosingDate(StoreDAO.getServerTimestamp());
			}
		}
	}

	public boolean isShouldUpdateStock() {
		return shouldUpdateStock;
	}

	public void setShouldUpdateStock(boolean shouldUpdateStock) {
		this.shouldUpdateStock = shouldUpdateStock;
	}

	@JsonIgnore
	public double getServiceChargePaidAmount() {
		Set<PosTransaction> transactions = getTransactions();
		if (transactions == null || transactions.isEmpty()) {
			return 0;
		}

		double scPaidAmount = 0.0;
		for (PosTransaction posTransaction : transactions) {
			if (posTransaction.isVoided() || (posTransaction instanceof RefundTransaction)) {
				continue;
			}
			scPaidAmount += posTransaction.getServiceChargeAmount();
		}

		return scPaidAmount;
	}

	@JsonIgnore
	public double getServiceChargeDueAmount() {
		return getServiceCharge() - getServiceChargePaidAmount();
	}

	public double getTotalDiscountPercentageRate() {
		List<TicketDiscount> discounts = getDiscounts();
		if (discounts == null || discounts.isEmpty()) {
			return 0.0;
		}

		double percentage = 0.0;
		for (TicketDiscount ticketDiscount : discounts) {
			if (ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY || ticketDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
				Double totalAmount = getSubtotalAmount() - itemDiscountAmount;
				Double discountValue = ticketDiscount.getValue() * ticketDiscount.getCouponQuantity();
				double discountPercent = (100.0 * discountValue) / totalAmount;
				percentage += discountPercent;
			}
			else {
				percentage += (ticketDiscount.getValue() * ticketDiscount.getCouponQuantity());
			}
		}
		if (percentage > 100) {
			percentage = 100;
		}
		return percentage / 100.0;
	}

	public double getDiscountPercentageRate(TicketDiscount ticketDiscount, double ticketAmountAfterDiscount) {
		Double minimumAmount = ticketDiscount.getMinimumAmount();
		double subtotalAfterItemDiscount = getSubtotalAmount() - getItemDiscountAmount();
		if (minimumAmount > 0 && subtotalAfterItemDiscount < minimumAmount) {
			return 0;
		}
		double percentage = 0.0;
		if (ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY || ticketDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
			Double discountValue = ticketDiscount.getValue() * ticketDiscount.getCouponQuantity();
			double discountPercent = (100.0 * discountValue) / ticketAmountAfterDiscount;
			percentage += discountPercent;
		}
		else {
			percentage += (ticketDiscount.getValue() * ticketDiscount.getCouponQuantity());
		}
		if (percentage > 100) {
			percentage = 100;
		}
		return percentage / 100.0;
	}

	public void setItemDiscountAmount(double amount) {
		addProperty(JSON_PROP_ITEM_DISC_AMOUNT, String.valueOf(amount));
	}

	public double getItemDiscountAmount() {
		return getDoubleProperty(JSON_PROP_ITEM_DISC_AMOUNT);
	}

	public double getTicketDiscountAmount() {
		return getDoubleProperty(JSON_PROP_TICKET_DISC_AMOUNT);
	}

	public void setTicketDiscountAmount(double amount) {
		addProperty(JSON_PROP_TICKET_DISC_AMOUNT, String.valueOf(amount));
	}

	public double getOldDataTolerance() {
		return getDoubleProperty("oldDataTolerance"); //$NON-NLS-1$
	}

	public void setOldDataTolerance(double amount) {
		addProperty("oldDataTolerance", String.valueOf(amount)); //$NON-NLS-1$
	}

	@Override
	public com.google.gson.JsonObject getPropertyStore() {
		return propertiesContainer;
	}

	public void setDiscountOnSerivceCharge(boolean value) {
		addProperty("discOnSc", String.valueOf(value)); //$NON-NLS-1$
	}

	public boolean isDiscountOnSerivceCharge() {
		return POSUtil.getBoolean(getProperty("discOnSc")); //$NON-NLS-1$
	}

	public void setServiceChargeApplicable(boolean value) {
		addProperty("scApplicable", String.valueOf(value)); //$NON-NLS-1$
	}

	public boolean isServiceChargeApplicable() {
		return POSUtil.getBoolean(getProperty("scApplicable")); //$NON-NLS-1$
	}

	public void setApplyFloridaTaxRule(boolean value) {
		addProperty("applyFlTaxRule", String.valueOf(value)); //$NON-NLS-1$
	}

	public boolean isApplyFloridaTaxRule() {
		return POSUtil.getBoolean(getProperty("applyFlTaxRule")); //$NON-NLS-1$
	}

	/*public void setPenyRoundingRule(boolean value) {
		addProperty("applyPenyRoundingRule", String.valueOf(value)); //$NON-NLS-1$
	}
	
	public boolean isApplyPenyRoundingRule() {
		return POSUtil.getBoolean(getProperty("applyPenyRoundingRule")); //$NON-NLS-1$
	}
	
	public void set10CentRoundingRule(boolean value) {
		addProperty("apply10CentRoundingRule", String.valueOf(value)); //$NON-NLS-1$
	}
	
	public boolean isApply10CentRoundingRule() {
		return POSUtil.getBoolean(getProperty("apply10CentRoundingRule")); //$NON-NLS-1$
	}*/

	public boolean isItemReturnable(TicketItem item, double toBeReturnedQuantity) {
		double totalItemCount = 0;
		double totalReturnedCount = 0;
		List<TicketItem> ticketItems = getTicketItems();
		String keyByMenuItemId = getKeyByMenuItemId(item);
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isTreatAsSeat()) {
				continue;
			}
			String menuItemId = getKeyByMenuItemId(ticketItem);
			if (menuItemId != null && menuItemId.equals(keyByMenuItemId)) {
				if (ticketItem.isItemReturned()) {
					totalReturnedCount += ticketItem.getQuantity();
				}
				else if (!ticketItem.isVoided()) {
					totalItemCount += ticketItem.getQuantity();
				}
			}
		}
		return Math.abs(totalReturnedCount) + toBeReturnedQuantity <= totalItemCount;
	}

	private String getKeyByMenuItemId(TicketItem ticketItem) {
		String key = ticketItem.getMenuItemId();
		if (ticketItem.isComboItem()) {
			List<TicketItem> comboItems = ticketItem.getComboItems();
			if (comboItems != null) {
				for (TicketItem comboTicketItem : comboItems) {
					key += "_" + comboTicketItem.getMenuItemId(); //$NON-NLS-1$
				}
			}
		}
		return key;
	}

	public boolean hasReturnedItem() {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				if (ticketItem.isItemReturned()) {
					return Boolean.TRUE;
				}
			}
		}
		return Boolean.FALSE;
	}

	public String getDeliveryZipCode() {
		return getProperty("deliAddressZipCode"); //$NON-NLS-1$
	}

	public void setDeliveryAddressZipCode(String addressDetail) {
		addProperty("deliAddressZipCode", addressDetail); //$NON-NLS-1$
	}

	public String getDeliveryFlatNo() {
		return getProperty("flatNo", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void setDeliveryFlatNo(String flatNo) {
		addProperty("flatNo", flatNo); //$NON-NLS-1$
	}

	public double getMinimumDeliveryAmount() {
		return getDoubleProperty(Store.PROP_DELIVERY_MINIMUM_AMOUNT);
	}

	public void setMinimumDeliveryAmount(double deliveryMinimumAmount) {
		addProperty(Store.PROP_DELIVERY_MINIMUM_AMOUNT, String.valueOf(deliveryMinimumAmount));
	}

	public String getStoreId() {
		return getProperty("storeId"); //$NON-NLS-1$
	}

	public void setStoreId(String storeId) {
		addProperty("storeId", storeId); //$NON-NLS-1$
	}

	public String getEstimatedDeliveryTime() {
		return getProperty("estimated_delivery_time"); //$NON-NLS-1$
	}

	public void setEstimatedDeliveryTime(String deliveryTimeInMin) {
		addProperty("estimated_delivery_time", deliveryTimeInMin); //$NON-NLS-1$
	}

	public boolean isDeliveryASAP() {
		return getBooleanProperty("delivery_asap", Boolean.FALSE); //$NON-NLS-1$
	}

	public void setDeliveryASAP(boolean delivery_asap) {
		addProperty("delivery_asap", String.valueOf(delivery_asap)); //$NON-NLS-1$
	}

	public void setServiceChargeType(ServiceChargeType serviceChargeType) {
		addProperty("serviceChargeType", serviceChargeType.name()); //$NON-NLS-1$
	}

	@JsonIgnore
	public ServiceChargeType getServiceChargeType() {
		String property = getProperty("serviceChargeType"); //$NON-NLS-1$
		if (StringUtils.isNotBlank(property)) {
			try {
				return ServiceChargeType.valueOf(property);
			} catch (Exception e) {
				return ServiceChargeType.EMPTY;
			}
		}
		return ServiceChargeType.EMPTY;
	}

	public void setOrginalDeliveryDate(Date orginalDeliveryDate) {
		if (orginalDeliveryDate == null) {
			removeProperty("OrginalDeliveryDate"); //$NON-NLS-1$
			return;
		}
		addProperty("OrginalDeliveryDate", DateUtil.formatDateWithDefaultTimeAndSec(orginalDeliveryDate)); //$NON-NLS-1$
	}

	@JsonIgnore
	public Date getOrginalDeliveryDate() {
		String orginalDate = getProperty("OrginalDeliveryDate");
		try {
			return DateUtil.formatDateWithDefaultTimeAndSec(orginalDate);
		} catch (ParseException e) {

		}
		return null;
	}

	public void checkVoidable() throws PosException {
		List<TicketItem> ticketItems = this.getTicketItems();
		if (ticketItems == null || ticketItems.isEmpty()) {
			throw new PosException(POSConstants.TICKET_IS_EMPTY_);
		}
		if (hasReturnedItem()) {
			throw new PosException(Messages.getString("POSConstants.27")); //$NON-NLS-1$
		}
		if (this.getBooleanProperty("split", false)) { //$NON-NLS-1$
			throw new PosException(Messages.getString("Ticket.4")); //$NON-NLS-1$
		}

		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isGiftCard() && ticketItem.getGiftCardPaidAmount() > 0) {
				throw new PosException(Messages.getString("Ticket.3")); //$NON-NLS-1$
			}
		}
	}

	public boolean hasAnyAdditionalService(String serviceKey) {
		List<TicketItem> ticketItems = getTicketItems();
		if (ticketItems == null || ticketItems.isEmpty()) {
			return false;
		}

		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.getBooleanProperty(serviceKey, Boolean.FALSE)) {
				return Boolean.TRUE;
			}
		}

		return false;
	}

	public String getDiscountCodeDisplay() {
		if (getDiscountAmount() == 0 || getDiscounts() == null || getDiscounts().isEmpty()) {
			return ""; //$NON-NLS-1$
		}
		TicketDiscount ticketDiscount = getDiscounts().get(0);
		if (StringUtils.isBlank(ticketDiscount.getCouponBarcode())) {
			return ""; //$NON-NLS-1$
		}
		return "(" + ticketDiscount.getCouponBarcode() + ")"; //$NON-NLS-1$ //$NON-NLS-2$
	}

	public String getInterval() {
		List<TicketItem> ticketItems = getTicketItems();
		String priceInterval = null;
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.getMenuItem() == null) {
				continue;
			}
			if (ticketItem.isService() || ticketItem.getMenuItem().isEditablePrice()) {
				continue;
			}
			priceInterval = ticketItem.getProperty("price_interval"); //$NON-NLS-1$
			if (StringUtils.isNotBlank(priceInterval)) {
				return priceInterval;
			}
		}
		if (isServiceItems(ticketItems)) {
			return OneTimePrice.ONE_TIME;
		}
		return OneTimePrice.YEAR;
	}

	private boolean isServiceItems(List<TicketItem> ticketItems) {
		for (TicketItem ticketItem : ticketItems) {
			if (!ticketItem.isService()) {
				return false;
			}
		}
		return true;
	}
}