/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.util;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.SessionFactory;
import org.hibernate.boot.Metadata;
import org.hibernate.boot.MetadataSources;
import org.hibernate.boot.registry.StandardServiceRegistry;
import org.hibernate.boot.registry.StandardServiceRegistryBuilder;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.hbm2ddl.SchemaUpdate;
import org.hibernate.tool.schema.TargetType;

import com.floreantpos.Database;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.db.update.UpdateDBTo256;
import com.floreantpos.model.Store;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao._RootDAO;

public class DatabaseUtil {
	public static final Integer DATABASE_VERSION = DatabaseVersionHistory.DATABASE_VERSION;

	public static void checkConnection(String connectionString, String hibernateDialect, String hibernateConnectionDriverClass, String user, String password)
			throws DatabaseConnectionException {
		try {

			Class.forName(hibernateConnectionDriverClass);
			DriverManager.getConnection(connectionString, user, password);

		} catch (Exception e) {
			throw new DatabaseConnectionException(Messages.getString("DatabaseUtil.0") + e.getMessage()); //$NON-NLS-1$
		}
	}

	public static void initialize() throws DatabaseConnectionException {
		_RootDAO.initialize();
	}

	public static void initialize(String hibernanetCfgFile) throws DatabaseConnectionException {
		_RootDAO.initialize(hibernanetCfgFile);
	}

	public static void initialize(String hibernanetCfgFile, String hibernateDialectClass, String driverClass, String connectString, String databaseUser,
			String databasePassword) {
		_RootDAO.initialize(hibernanetCfgFile, hibernateDialectClass, driverClass, connectString, databaseUser, databasePassword);
	}

	public static void initialize(String hibernanetCfgFile, Map<String, String> properties) {
		_RootDAO.initialize(hibernanetCfgFile, properties);
	}

	public static void initialize(Map<String, String> properties) {
		_RootDAO.initialize("cloudpos.hibernate.cfg.xml", properties); //$NON-NLS-1$
	}

	public static boolean createDatabase(String hibernateCfgFileName, String connectionString, String hibernateDialect, String hibernateConnectionDriverClass,
			String user, String password, boolean exportSampleData, boolean testData) {
		try {
			_RootDAO.releaseConnection();

			StandardServiceRegistryBuilder registryBuilder = new StandardServiceRegistryBuilder().configure(hibernateCfgFileName);
			registryBuilder.applySetting("hibernate.dialect", hibernateDialect); //$NON-NLS-1$
			registryBuilder.applySetting("hibernate.connection.driver_class", hibernateConnectionDriverClass); //$NON-NLS-1$
			registryBuilder.applySetting("hibernate.connection.url", connectionString); //$NON-NLS-1$
			registryBuilder.applySetting("hibernate.connection.username", user); //$NON-NLS-1$
			registryBuilder.applySetting("hibernate.connection.password", password); //$NON-NLS-1$
			registryBuilder.applySetting("hibernate.hbm2ddl.auto", "create"); //$NON-NLS-1$ //$NON-NLS-2$
			registryBuilder.applySetting("hibernate.c3p0.checkoutTimeout", "0"); //$NON-NLS-1$ //$NON-NLS-2$
			registryBuilder.applySetting("hibernate.cache.use_second_level_cache", "false"); //$NON-NLS-1$ //$NON-NLS-2$

			StandardServiceRegistry standardRegistry = registryBuilder.build();
			Metadata metaData = new MetadataSources(standardRegistry).getMetadataBuilder().build();

			SchemaExport schemaExport = new SchemaExport();
			EnumSet<TargetType> enumSet = EnumSet.of(TargetType.DATABASE);
			schemaExport.create(enumSet, metaData);

			_RootDAO.setSessionFactory(metaData.buildSessionFactory());

			DefaultDataInserter dataInserter = new DefaultDataInserter();
			dataInserter.insertDefaultData(DatabaseVersionHistory.DATABASE_VERSION, false, false);

			if (!exportSampleData) {
				StandardServiceRegistryBuilder.destroy(standardRegistry);
				return true;
			}
			if (testData) {
				dataInserter.createSampleTestData();
			}
			else {
				dataInserter.createSampleData(false);
			}

			StandardServiceRegistryBuilder.destroy(standardRegistry);
			return true;
		} catch (Exception e) {
			throw e;
		}
	}

//	public static boolean updateDatabase(String hibernateCfgFileName, String connectionString, String hibernateDialect, String hibernateConnectionDriverClass,
//			String user, String password) {
//		try {
//			_RootDAO.releaseConnection();
//
//			StandardServiceRegistryBuilder registryBuilder = new StandardServiceRegistryBuilder().configure(hibernateCfgFileName);
//			registryBuilder.applySetting("hibernate.dialect", hibernateDialect); //$NON-NLS-1$
//			registryBuilder.applySetting("hibernate.connection.driver_class", hibernateConnectionDriverClass); //$NON-NLS-1$
//			registryBuilder.applySetting("hibernate.connection.url", connectionString); //$NON-NLS-1$
//			registryBuilder.applySetting("hibernate.connection.username", user); //$NON-NLS-1$
//			registryBuilder.applySetting("hibernate.connection.password", password); //$NON-NLS-1$
//			registryBuilder.applySetting("hibernate.max_fetch_depth", "3"); //$NON-NLS-1$ //$NON-NLS-2$
//			registryBuilder.applySetting("hibernate.connection.isolation", String.valueOf(Connection.TRANSACTION_READ_UNCOMMITTED)); //$NON-NLS-1$
//			registryBuilder.applySetting("hibernate.cache.use_second_level_cache", "false"); //$NON-NLS-1$ //$NON-NLS-2$
//
//			StandardServiceRegistry standardRegistry = registryBuilder.build();
//			Metadata metaData = new MetadataSources(standardRegistry).getMetadataBuilder().build();
//
//			SchemaUpdate schemaUpdate = new SchemaUpdate();
//			EnumSet<TargetType> enumSet = EnumSet.of(TargetType.DATABASE, TargetType.STDOUT);
//			schemaUpdate.execute(enumSet, metaData);
//
//			SessionFactory sessionFactory = metaData.buildSessionFactory();
//			_RootDAO.setSessionFactory(sessionFactory);
//
//			Store store = StoreDAO.getRestaurant();
//			if (store != null) {
//				Integer databaseVersion = store == null ? null : store.getDatabaseVersion();
//				if (databaseVersion == null || databaseVersion < 256) {
//					PosLog.debug(DatabaseUtil.class, "updating db to 256");
//					UpdateDBTo256 dataConvertionUtil = new UpdateDBTo256(connectionString, user, password);
//					dataConvertionUtil.update();
//				}
//				store.setDatabaseVersion(DatabaseVersionHistory.DATABASE_VERSION);
//				StoreDAO.getInstance().update(store);
//			}
//			else {
//				DefaultDataInserter.createStore(DatabaseVersionHistory.DATABASE_VERSION, false);
//			}
//			sessionFactory.close();
//			StandardServiceRegistryBuilder.destroy(standardRegistry);
//			return true;
//		} catch (Exception e) {
//			PosLog.error(DatabaseUtil.class, e);
//			logger.error(e);
//			return false;
//		}
//	}

//	public static boolean isDbUpdateNeeded() {
//		try {
//			Store store = StoreDAO.getRestaurant();
//			if (store == null) {
//				throw new PosException(Messages.getString("DatabaseUtil.3") + Messages.getString("DatabaseUtil.4")); //$NON-NLS-1$ //$NON-NLS-2$
//			}
//			Integer databaseVersion = store.getDatabaseVersion();
//			if (databaseVersion != null && databaseVersion >= DatabaseVersionHistory.DATABASE_VERSION) {
//				return false;
//			}
//			return true;
//		} catch (SQLGrammarException e) {
//			Throwable cause = e.getCause();
//			if (cause != null) {
//				String message = cause.getMessage();
//				if (message != null) {
//					Pattern pattern = Pattern.compile("Column .* is either not in any table"); //$NON-NLS-1$
//					boolean found = pattern.matcher(message).find();
//					if (found) {
//						return true;
//					}
//				}
//
//			}
//
//			logger.error(e);
//			return false;
//		}
//	}

	public static boolean isDbUpdateNeeded(Integer databaseVersionInDb) {
		if (databaseVersionInDb != null && databaseVersionInDb >= DatabaseVersionHistory.DATABASE_VERSION) {
			return false;
		}
		return true;
	}

	public static Map<String, String> getStoreProperties(String connectionString, String hibernateConnectionDriverClass, String user, String password)
			throws Exception {
		Class.forName(hibernateConnectionDriverClass);
		java.sql.Connection connection = null;

		try {
			String errorMessage = Messages.getString("DatabaseUtil.6"); //$NON-NLS-1$

			connection = DriverManager.getConnection(connectionString, user, password);
			DatabaseMetaData metaData = connection.getMetaData();
			String actualTableName = getActualTableName(metaData, "store_properties"); //$NON-NLS-1$
			if (actualTableName == null) {
				throw new PosException(errorMessage);
			}

			String propertyNameCol = getActualColumnName(metaData, actualTableName, "property_name"); //$NON-NLS-1$
			String propertyValueCol = getActualColumnName(metaData, actualTableName, "property_value"); //$NON-NLS-1$

			if (propertyNameCol == null || propertyValueCol == null) {
				throw new PosException(errorMessage);
			}

			String sql = String.format("select * from %s", actualTableName); //$NON-NLS-1$
			Map<String, String> storePropertes = new HashMap<>();
			Statement statement = connection.createStatement();
			ResultSet resultSet = statement.executeQuery(sql);
			while (resultSet.next()) {
				String propertyName = resultSet.getString(propertyNameCol);
				String propertyValue = resultSet.getString(propertyValueCol);
				storePropertes.put(propertyName, propertyValue);
			}
			resultSet.close();
			statement.close();

			return storePropertes;
		} finally {
			if (connection != null) {
				connection.close();
			}
		}
	}

	public static boolean hasTable(DatabaseMetaData metaData, String tableName) throws SQLException {
		ResultSet resultSet = null;
		try {
			resultSet = metaData.getTables(null, null, null, new String[] { "TABLE" }); //$NON-NLS-1$
			ArrayList<String> tableNameList = new ArrayList<>();

			while (resultSet.next()) {
				String tableName2 = resultSet.getString("TABLE_NAME"); //$NON-NLS-1$
				tableNameList.add(tableName2.toLowerCase());
			}

			if (tableNameList.contains(tableName.toLowerCase())) {
				return true;
			}
			return false;
		} finally {
			if (resultSet != null) {
				resultSet.close();
			}
		}
	}

	public static String getActualTableName(DatabaseMetaData metaData, String tableName) throws SQLException {
		ResultSet resultSet = null;
		try {
			resultSet = metaData.getTables(null, null, null, new String[] { "TABLE" }); //$NON-NLS-1$
			while (resultSet.next()) {
				String tableName2 = resultSet.getString("TABLE_NAME"); //$NON-NLS-1$
				if (tableName2.equalsIgnoreCase(tableName)) {
					return tableName2;
				}
			}

			return null;
		} finally {
			if (resultSet != null) {
				resultSet.close();
			}
		}
	}

	public static String getActualColumnName(DatabaseMetaData metaData, String actutalTableName, String virtualColumnName) throws SQLException {
		ResultSet sourcecolumns = null;
		try {
			sourcecolumns = metaData.getColumns(null, null, actutalTableName, null);
			while (sourcecolumns.next()) {
				String columnName = sourcecolumns.getString("COLUMN_NAME"); //$NON-NLS-1$
				if (columnName.equalsIgnoreCase(virtualColumnName)) {
					return columnName;
				}
			}
			return null;
		} finally {
			if (sourcecolumns != null) {
				sourcecolumns.close();
			}
		}
	}
	
	public static void updateDbIfNeeded(String connectString, Database database, String username, String password) throws Exception {
		PosLog.info(DatabaseUtil.class, "checking Db update"); //$NON-NLS-1$
		Map<String, String> storeProperties = DatabaseUtil.getStoreProperties(connectString, database.getHibernateConnectionDriverClass(), username, password);
		Integer databaseVersion = null;
		try {
			databaseVersion = Integer.parseInt(storeProperties.get("database.version")); //$NON-NLS-1$
		} catch (Exception e) {
		}

		PosLog.info(DatabaseUtil.class, "Database version no: " + databaseVersion); //$NON-NLS-1$

		boolean dbUpdateNeeded = DatabaseUtil.isDbUpdateNeeded(databaseVersion);
		if (dbUpdateNeeded) {
			PosLog.info(DatabaseUtil.class, "Db update needed"); //$NON-NLS-1$

			List<String> schemaList = new ArrayList<>();

			try (Connection connection = DriverManager.getConnection(connectString, username, password)) {
				try (Statement statement = connection.createStatement()) {
					try (ResultSet rs = statement.executeQuery("select store_id from siiopa_customer")) { //$NON-NLS-1$
						while (rs.next()) {
							String schemaName = rs.getString(1);
							if (StringUtils.isNotEmpty(schemaName)) {
								schemaList.add(schemaName);
							}
						}
					}
				}
			} catch (Exception e) {
				PosLog.error(DatabaseUtil.class, e);
			}

			try {
				updateSchema(database, connectString, "public", username, password); //$NON-NLS-1$
				for (String schemaName : schemaList) {
					try {
						updateSchema(database, connectString, schemaName, username, password);
					} catch (Exception e) {
						PosLog.error(DatabaseUtil.class, e);
					}
				}
			} catch (Exception e) {
				PosLog.error(DatabaseUtil.class, e);
			}

		}
		PosLog.info(DatabaseUtil.class, "Db updated"); //$NON-NLS-1$
	}
	
	private static void updateSchema(Database database, String connectionString, final String schema, String user, String password) throws Exception {
		PosLog.info(DatabaseUtil.class, "updating schema: " + schema); //$NON-NLS-1$

		StandardServiceRegistryBuilder registryBuilder = new StandardServiceRegistryBuilder().configure("cloudpos.hibernate.cfg.xml"); //$NON-NLS-1$
		registryBuilder.applySetting(AvailableSettings.DIALECT, database.getHibernateDialect());
		registryBuilder.applySetting(AvailableSettings.DRIVER, database.getHibernateConnectionDriverClass()); //$NON-NLS-1$
		registryBuilder.applySetting(AvailableSettings.URL, connectionString); //$NON-NLS-1$
		registryBuilder.applySetting(AvailableSettings.USER, user); //$NON-NLS-1$
		registryBuilder.applySetting(AvailableSettings.PASS, password); //$NON-NLS-1$
		registryBuilder.applySetting(AvailableSettings.MAX_FETCH_DEPTH, "3"); //$NON-NLS-1$ //$NON-NLS-2$
		registryBuilder.applySetting(AvailableSettings.ISOLATION, String.valueOf(Connection.TRANSACTION_READ_UNCOMMITTED)); //$NON-NLS-1$
		registryBuilder.applySetting(AvailableSettings.USE_SECOND_LEVEL_CACHE, "false"); //$NON-NLS-1$ //$NON-NLS-2$
		registryBuilder.applySetting(AvailableSettings.DEFAULT_SCHEMA, schema); //$NON-NLS-1$ //$NON-NLS-2$

		registryBuilder.applySetting("hibernate.search.default.indexBase", _RootDAO.LUCENE_INDEX_DIR); //$NON-NLS-1$ //$NON-NLS-2$
		registryBuilder.applySetting("hibernate.search.autoregister_listeners", "false"); //$NON-NLS-1$ //$NON-NLS-2$
		registryBuilder.applySetting("hibernate.search.indexing_strategy", "manual"); //$NON-NLS-1$ //$NON-NLS-2$

		StandardServiceRegistry standardRegistry = registryBuilder.build();
		Metadata metaData = new MetadataSources(standardRegistry).getMetadataBuilder().build();
		SchemaUpdate schemaUpdate = new SchemaUpdate();
		EnumSet<TargetType> enumSet = EnumSet.of(TargetType.DATABASE, TargetType.STDOUT);
		schemaUpdate.execute(enumSet, metaData);

		SessionFactory sessionFactory = metaData.buildSessionFactory();
		_RootDAO.setSessionFactory(sessionFactory);

		Store store = StoreDAO.getRestaurant();
		if (store != null) {
			Integer databaseVersion = store == null ? null : store.getDatabaseVersion();
			if (databaseVersion == null || databaseVersion < 256) {
				PosLog.debug(DatabaseUtil.class, "updating db to 256");
				UpdateDBTo256 dataConvertionUtil = new UpdateDBTo256(connectionString, user, password);
				dataConvertionUtil.update();
			}
			store.setDatabaseVersion(DatabaseVersionHistory.DATABASE_VERSION);
			StoreDAO.getInstance().update(store);
		}
		sessionFactory.close();
		StandardServiceRegistryBuilder.destroy(standardRegistry);
		PosLog.info(DatabaseUtil.class, schema + " schema update complete."); //$NON-NLS-1$
	}
}
