package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.Customer;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.stripe.model.Invoice;
import com.stripe.model.InvoiceLineItem;
import com.stripe.model.Subscription;

public class BalanceUpdateTransactionDAO extends BaseBalanceUpdateTransactionDAO {

	public BalanceUpdateTransactionDAO() {
	}

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private void adjustBalanceUpdateTransaction(BalanceUpdateTransaction balanceTransaction, Session session) {
		save(balanceTransaction, session);

		BalanceType balanceType = balanceTransaction.getBalanceType();
		if (balanceType == BalanceType.GIFT_CARD) {
			GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, balanceTransaction.getAccountNumber());
			if (giftCard == null) {
				//throw new PosException("No gift card found with account number " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
				return;
			}
			giftCard.setBalance(giftCard.getBalance() + balanceTransaction.getAmount());
			GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
			return;
		}

		Customer customer = CustomerDAO.getInstance().get(balanceTransaction.getAccountNumber(), session);
		if (customer == null) {
			//throw new PosException("No customer found with id " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
			return;
		}

		if (balanceType == BalanceType.CUSTOMER) {
			customer.setBalance(customer.getBalance() + balanceTransaction.getAmount());
		}
		else if (balanceType == BalanceType.LOYALTY) {
			customer.setLoyaltyPoint((int) (customer.getLoyaltyPoint() + balanceTransaction.getAmount()));
		}
		customer.setUpdateLastUpdateTime(false);
		customer.setUpdateSyncTime(false);
		CustomerDAO.getInstance().saveOrUpdate(customer, session);
	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, Session session) {
		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(transaction == null ? PaymentType.CASH : transaction.getPaymentType(),
				transactionType, amount);
		balanceUpdateTransaction.setTransactionId(transaction == null ? null : transaction.getId());
		balanceUpdateTransaction.setTicketId(ticket == null ? null : ticket.getId());
		balanceUpdateTransaction.setAccountNumber(accountNumber);

		if (subType != null) {
			balanceUpdateTransaction.setTransactionSubType(subType.name());
			balanceUpdateTransaction.setDescription(subType.name());
		}
		if (!NumberUtil.isZero(balanceBefore)) {
			balanceBefore = NumberUtil.round(balanceBefore);
			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		}

		BalanceUpdateTransactionDAO.getInstance().save(balanceUpdateTransaction, session);
	}

	public void saveOrUpdateBalanceUpdateTransactions(List<BalanceUpdateTransaction> customerBalanceTransactions, boolean updateLastUpdateTime,
			boolean updateSyncTime) throws Exception {
		if (customerBalanceTransactions == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (BalanceUpdateTransaction transaction : customerBalanceTransactions) {
				BalanceUpdateTransaction existingTransaction = get(transaction.getId());
				if (existingTransaction != null) {
					continue;
				}
				transaction.setUpdateLastUpdateTime(updateLastUpdateTime);
				transaction.setUpdateSyncTime(updateSyncTime);
				adjustBalanceUpdateTransaction(transaction, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void createBalanceUpdateTransactionForPayment(String ticketId, String paymentGateway, int count) {
		info(String.format("<h3>%s Ticket %s ------------------------------------------------------------</h3>", paymentGateway, (count++))); //$NON-NLS-1$
		info("Ticket id : " + ticketId); //$NON-NLS-1$
		Ticket ticket = TicketDAO.getInstance().findByOrderId(ticketId);
		if (ticket == null) {
			info("No ticket found with order id " + ticketId); //$NON-NLS-1$
			return;
		}
		TicketDAO.getInstance().loadFullTicket(ticket);
		Set<PosTransaction> transactions = ticket.getTransactions();
		if (transactions != null && transactions.size() > 0) {
			info("Ticket transaction count " + transactions.size()); //$NON-NLS-1$
			Transaction tx = null;
			try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
				tx = session.beginTransaction();
				for (PosTransaction posTransaction : transactions) {
					BalanceUpdateTransactionDAO.getInstance().createBalanceUpdateTransactionForPayment(posTransaction, paymentGateway, session);
				}
				tx.commit();
			} catch (Exception e) {
				if (tx != null) {
					tx.rollback();
				}
				throw e;
			}
		}
	}

	public void createBalanceUpdateTransactionForPayment(PosTransaction transaction, String paymentGateway, Session session) {
		info(String.format("%s trans, id: %s, amount: %s, date: %s", paymentGateway, transaction.getId(), transaction.getAmount(),
				transaction.getTransactionTime()));

		Ticket ticket = transaction.getTicket();
		BalanceUpdateTransaction balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().findByTransactionId(transaction.getId(),
				TransactionType.CREDIT);
		if (balanceUpdateTransaction != null) {
			info(String.format("Balance update transaction already exists for %s transaction %s", paymentGateway, transaction.getId())); //$NON-NLS-1$
			return;
		}
		balanceUpdateTransaction = new BalanceUpdateTransaction();
		balanceUpdateTransaction.setTransactionId(transaction.getId());
		balanceUpdateTransaction.setBalanceType(BalanceType.OROSTORE_PAYMENT);
		balanceUpdateTransaction.setPaymentType(PaymentType.CREDIT_CARD);
		balanceUpdateTransaction.setPaymentGateway(paymentGateway);
		balanceUpdateTransaction.setTransactionType(TransactionType.CREDIT.name());
		balanceUpdateTransaction.setTicketId(ticket.getId());
		balanceUpdateTransaction.addProperty("resellerCommission", ticket.getProperty("commission")); //$NON-NLS-1$ //$NON-NLS-2$
		balanceUpdateTransaction.setOutletId(ticket.getProperty("resellerId")); //$NON-NLS-1$
		balanceUpdateTransaction.setAccountNumber(transaction.getCustomerId());
		balanceUpdateTransaction.setTransactionTime(transaction.getTransactionTime());
		balanceUpdateTransaction.setAmount(transaction.getAmount());
		balanceUpdateTransaction.addProperty("discountAmount", String.valueOf(ticket.getDiscountAmount())); //$NON-NLS-1$
		loadCommonProperties(balanceUpdateTransaction, null, null);
		String products = ""; //$NON-NLS-1$
		List<TicketItem> items = ticket.getTicketItems();
		for (Iterator<TicketItem> iterator = items.iterator(); iterator.hasNext();) {
			TicketItem item = (TicketItem) iterator.next();
			if (item == null || StringUtils.isBlank(item.getMenuItemId())) {
				continue;
			}
			products += item.getMenuItemId();
			if (iterator.hasNext()) {
				products += ","; //$NON-NLS-1$
			}
		}
		balanceUpdateTransaction.setProducts(products);
		BalanceUpdateTransactionDAO.getInstance().save(balanceUpdateTransaction, session);
		info(String.format("Balance update transaction created for %s transaction %s", paymentGateway, transaction.getId())); //$NON-NLS-1$
	}

	private void loadCommonProperties(BalanceUpdateTransaction transaction, Subscription subscription, Invoice invoice) {
		if (StringUtils.isBlank(transaction.getType())) {
			if (transaction.getTransactionType() != null && transaction.getTransactionType().equals(TransactionType.DEBIT.name())) {
				transaction.setType(BalanceUpdateTransaction.TYPE_REFUND);
			}
			else if (transaction.getTicketId().startsWith("WTIN_") || (transaction.getPaymentGateway() != null //$NON-NLS-1$
					&& (transaction.getPaymentGateway().equals(BalanceUpdateTransaction.PAYMENT_GATEWAY_WIRE_TRANSFER)
							|| transaction.getPaymentGateway().equals(BalanceUpdateTransaction.PAYMENT_GATEWAY_PAYPAL)))) {
				transaction.setType(BalanceUpdateTransaction.TYPE_NEW);
			}
			else {
				Date ticketCreatedDate = null;
				if (subscription != null) {
					ticketCreatedDate = new Date(subscription.getCreated() * 1000L);
				}
				else if (invoice != null) {
					ticketCreatedDate = new Date(invoice.getCreated() * 1000L);
				}
				if (ticketCreatedDate != null) {
					Date startOfDay = DateUtil.startOfDay(ticketCreatedDate);
					Date endOfDay = DateUtil.endOfDay(ticketCreatedDate);
					transaction.setType(DateUtil.between(startOfDay, endOfDay, transaction.getTransactionTime()) ? BalanceUpdateTransaction.TYPE_NEW
							: BalanceUpdateTransaction.TYPE_RECURRING);
				}
			}
			info("Balance update transaction type: " + transaction.getType()); //$NON-NLS-1$
		}
		if (transaction.getProducts() == null) {
			if (invoice != null) {
				String products = ""; //$NON-NLS-1$
				List<InvoiceLineItem> items = invoice.getLines().getData();
				for (Iterator<InvoiceLineItem> iterator = items.iterator(); iterator.hasNext();) {
					InvoiceLineItem item = (InvoiceLineItem) iterator.next();
					if (item == null || item.getPrice() == null || StringUtils.isBlank(item.getPrice().getProduct())) {
						continue;
					}
					products += item.getPrice().getProduct();
					if (iterator.hasNext()) {
						products += ","; //$NON-NLS-1$
					}
				}
				transaction.setProducts(products);
				info("Products: " + products); //$NON-NLS-1$
			}
		}
	}

	public BalanceUpdateTransaction findByTransactionId(String transactionId, TransactionType transactionType) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_ID, transactionId));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_TYPE, transactionType.name()));
			criteria.setMaxResults(1);
			return (BalanceUpdateTransaction) criteria.uniqueResult();
		}
	}

	private void info(String msg) {
		PosLog.info(getClass(), msg);
	}

	public List<BalanceUpdateTransaction> getTransactions(String paymentGateway, String customerId, String orderId, Date startDate, Date endDate) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, customerId));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_PAYMENT_GATEWAY, paymentGateway));
			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TICKET_ID, orderId));
			}
			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, startDate, endDate));
			}
			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
			return criteria.list();
		}
	}
}