/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.services.report;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.POSConstants;
import com.floreantpos.PosLog;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.Discount;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PayoutRecepient;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketSource;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.DiscountDAO;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.model.dao.PayoutReasonDAO;
import com.floreantpos.model.dao.PayoutRecepientDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.TerminalDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.model.ext.CardTypeEnum;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.DiscountData;
import com.floreantpos.report.DiscountReportDataModel;
import com.floreantpos.report.JournalReportModel;
import com.floreantpos.report.JournalReportModel.JournalReportData;
import com.floreantpos.report.MenuUsageReport;
import com.floreantpos.report.MenuUsageReport.MenuUsageReportData;
import com.floreantpos.report.ReportItem;
import com.floreantpos.report.SalesBalanceReport;
import com.floreantpos.report.SalesDetailedReport;
import com.floreantpos.report.SalesExceptionReport;
import com.floreantpos.report.SalesReportModel;
import com.floreantpos.report.ServerProductivityReport;
import com.floreantpos.report.ServerProductivityReport.ServerProductivityReportData;
import com.floreantpos.report.TaxExemptReport;
import com.floreantpos.report.payout.PayoutReportData;
import com.floreantpos.report.payout.PayoutReportDataModel;

public class ReportService {

	public static String formatFullDate(Date date) {
		return DateUtil.formatDateWithTimeAndSec(date);
	}

	public static String formatShortDate(Date date) {
		return DateUtil.formatAsShortDate(date);
	}

	public MenuUsageReport getMenuUsageReport(Date fromDate, Date toDate) {
		GenericDAO dao = new GenericDAO();
		MenuUsageReport report = new MenuUsageReport();
		Session session = null;

		try {

			session = dao.getSession();

			Criteria criteria = session.createCriteria(MenuCategory.class);
			List<MenuCategory> categories = criteria.list();
			MenuCategory miscCategory = new MenuCategory();
			miscCategory.setName(com.floreantpos.POSConstants.MISC_BUTTON_TEXT);
			categories.add(miscCategory);

			for (MenuCategory category : categories) {
				criteria = session.createCriteria(TicketItem.class, "item"); //$NON-NLS-1$
				criteria.createCriteria("ticket", "t"); //$NON-NLS-1$ //$NON-NLS-2$
				ProjectionList projectionList = Projections.projectionList();
				projectionList.add(Projections.sum(TicketItem.PROP_QUANTITY));
				projectionList.add(Projections.sum(TicketItem.PROP_SUBTOTAL_AMOUNT));
				projectionList.add(Projections.sum(TicketItem.PROP_DISCOUNT_AMOUNT));
				criteria.setProjection(projectionList);
				criteria.add(Restrictions.eq("item." + TicketItem.PROP_CATEGORY_ID, category.getId())); //$NON-NLS-1$
				criteria.add(Restrictions.ge("t." + Ticket.PROP_CREATE_DATE, fromDate)); //$NON-NLS-1$
				criteria.add(Restrictions.le("t." + Ticket.PROP_CREATE_DATE, toDate)); //$NON-NLS-1$
				criteria.add(Restrictions.eq("t." + Ticket.PROP_PAID, Boolean.TRUE)); //$NON-NLS-1$
				List datas = criteria.list();
				if (datas.size() > 0) {
					Object[] objects = (Object[]) datas.get(0);

					MenuUsageReportData data = new MenuUsageReportData();
					data.setCategoryName(category.getName());

					if (objects.length > 0 && objects[0] != null)
						data.setCount(((Number) objects[0]).intValue());

					if (objects.length > 1 && objects[1] != null)
						data.setGrossSales(((Number) objects[1]).doubleValue());

					if (objects.length > 2 && objects[2] != null)
						data.setDiscount(((Number) objects[2]).doubleValue());

					data.calculate();
					report.addReportData(data);
				}
			}

			return report;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public ServerProductivityReport getServerProductivityReport(Date fromDate, Date toDate, List<User> servers) {
		GenericDAO dao = new GenericDAO();
		ServerProductivityReport report = new ServerProductivityReport();
		Session session = null;

		try {

			session = dao.getSession();

			Criteria criteria = session.createCriteria(MenuCategory.class);
			List<MenuCategory> categories = criteria.list();
			MenuCategory miscCategory = new MenuCategory();
			miscCategory.setName(com.floreantpos.POSConstants.MISC_BUTTON_TEXT);
			categories.add(miscCategory);

			for (User server : servers) {
				ServerProductivityReportData data = new ServerProductivityReportData();
				data.setServerName(server.toString());
				criteria = session.createCriteria(Ticket.class);
				criteria.add(Restrictions.in(Ticket.PROP_OWNER_ID, server.getRoleIds()));
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.TRUE));
				criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
				criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
				criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, toDate));

				ProjectionList projectionList = Projections.projectionList();
				projectionList.add(Projections.rowCount());
				projectionList.add(Projections.sum(Ticket.PROP_NUMBER_OF_GUESTS));
				projectionList.add(Projections.sum(Ticket.PROP_SUBTOTAL_AMOUNT));
				projectionList.add(Projections.sum(Ticket.PROP_DISCOUNT_AMOUNT));

				criteria.setProjection(projectionList);

				Object[] o = (Object[]) criteria.uniqueResult();
				int totalCheckCount = 0;
				double totalServerSale = 0;
				if (o != null) {
					if (o.length > 0 && o[0] != null) {
						int i = ((Number) o[0]).intValue();
						data.setTotalCheckCount(totalCheckCount = i);
					}
					if (o.length > 1 && o[1] != null) {
						int i = ((Number) o[1]).intValue();
						data.setTotalGuestCount(i);
					}

					if (o.length > 3) {
						totalServerSale = (o[2] != null) ? ((Number) o[2]).doubleValue() : 0;
						double discountAmount = (o[3] != null) ? ((Number) o[3]).doubleValue() : 0;
						data.setTotalSales(totalServerSale - discountAmount);
					}
				}

				data.calculate();
				report.addReportData(data);

				criteria = session.createCriteria(TicketItem.class, "item"); //$NON-NLS-1$
				criteria.createCriteria(TicketItem.PROP_TICKET, "t"); //$NON-NLS-1$

				projectionList = Projections.projectionList();
				criteria.setProjection(projectionList);
				projectionList.add(Projections.sum(TicketItem.PROP_QUANTITY), "itemCount"); //$NON-NLS-1$
				projectionList.add(Projections.sum(TicketItem.PROP_SUBTOTAL_AMOUNT), "grossSales"); //$NON-NLS-1$
				projectionList.add(Projections.sum("t." + Ticket.PROP_DISCOUNT_AMOUNT), "salesDiscount"); //$NON-NLS-1$ //$NON-NLS-2$
				projectionList.add(Projections.property(TicketItem.PROP_CATEGORY_NAME), "categoryName"); //$NON-NLS-1$

				criteria.add(Restrictions.eq("item." + TicketItem.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
				criteria.add(Restrictions.ge("t." + Ticket.PROP_CREATE_DATE, fromDate)); //$NON-NLS-1$
				criteria.add(Restrictions.lt("t." + Ticket.PROP_CREATE_DATE, toDate)); //$NON-NLS-1$
				criteria.add(Restrictions.in("t." + Ticket.PROP_OWNER_ID, server.getRoleIds())); //$NON-NLS-1$
				criteria.add(Restrictions.eq("t." + Ticket.PROP_PAID, Boolean.TRUE)); //$NON-NLS-1$

				criteria.addOrder(Order.asc(TicketItem.PROP_CATEGORY_NAME));
				projectionList.add(Projections.groupProperty("item." + TicketItem.PROP_CATEGORY_NAME)); //$NON-NLS-1$

				criteria.setResultTransformer(Transformers.aliasToBean(ServerProductivityReportData.class));
				List<?> dataObjects = criteria.list();

				for (Iterator<?> iterator = dataObjects.iterator(); iterator.hasNext();) {
					ServerProductivityReportData data4 = (ServerProductivityReportData) iterator.next();
					data4.setServerName(server.toString());
					data4.setTotalCheckCount(totalCheckCount);

					if (data4.getCategoryName() == null) {
						data4.setCategoryName(POSConstants.OTHERS);
					}

					data4.calculate();
					report.addReportData(data4);

				}
			}
			return report;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public JournalReportModel getJournalReport(Date fromDate, Date toDate) {
		GenericDAO dao = new GenericDAO();
		JournalReportModel report = new JournalReportModel();
		Session session = null;

		report.setFromDate(fromDate);
		report.setToDate(toDate);
		report.setReportTime(new Date());
		try {

			session = dao.getSession();
			Criteria criteria = session.createCriteria(ActionHistory.class);
			criteria.add(Restrictions.ge(ActionHistory.PROP_ACTION_TIME, fromDate));
			criteria.add(Restrictions.le(ActionHistory.PROP_ACTION_TIME, toDate));
			criteria.addOrder(Order.desc(ActionHistory.PROP_ACTION_TIME));
			List<ActionHistory> list = criteria.list();

			for (ActionHistory history : list) {
				User user = history.getPerformer();
				String userInfo = ""; //$NON-NLS-1$
				if (user != null) {
					userInfo = user.getId();
				}
				JournalReportData data = new JournalReportData();
				data.setAction(history.getActionName());
				data.setUserInfo(userInfo);
				data.setTime(history.getActionTime());
				data.setComments(history.getDescription());
				report.addReportData(data);
			}

			return report;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public SalesBalanceReport getSalesBalanceReport(Date fromDate, Date toDate, User user) {
		GenericDAO dao = new GenericDAO();
		SalesBalanceReport report = new SalesBalanceReport();
		Session session = null;

		report.setFromDate(fromDate);
		report.setToDate(toDate);
		report.setReportTime(new Date());
		try {

			session = dao.getSession();

			//net sales
			report.setNetSalesAmount(getNetSales(session, fromDate, toDate, user));
			//gross taxable sales
			report.setGrossTaxableSalesAmount(calculateGrossSales(session, fromDate, toDate, user, true));
			//gross non-taxable sales
			report.setGrossNonTaxableSalesAmount(calculateGrossSales(session, fromDate, toDate, user, false));
			//discount
			report.setDiscountAmount(calculateDiscount(session, fromDate, toDate, user));
			//tax
			report.setSalesTaxAmount(calculateTax(session, fromDate, toDate, user));
			report.setCashTipsAmount(calculateCashTips(session, fromDate, toDate, user));
			report.setChargedTipsAmount(calculateChargedTips(session, fromDate, toDate, user));

			report.setCashReceiptsAmount(calculateCashReceipt(session, fromDate, toDate, user));
			report.setCreditCardReceiptsAmount(calculateCreditReceipt(session, CreditCardTransaction.class, fromDate, toDate, user));
			report.setDebitCardReceiptsAmount(calculateDebitReceipt(session, DebitCardTransaction.class, fromDate, toDate, user));
			report.setMemberPaymentAmount(calculateMemberPayment(session, CustomerAccountTransaction.class, fromDate, toDate, user));
			report.setCustomPaymentAmount(calculateCustomPayment(session, CustomPaymentTransaction.class, fromDate, toDate, user));
			report.setGiftCertReceipts(calculateGiftCertReceipts(session, fromDate, toDate, user));
			report.setCashBackAmount(calculateRefundAmount(session, RefundTransaction.class, fromDate, toDate, user));
			report.setCashRefundAmount(calculateCashRefundAmount(session, RefundTransaction.class, fromDate, toDate, user));
			report.setToleranceAmount(calculateToleranceAmount(session, fromDate, toDate, user));
			report.setServiceChargeAmnt(calculateServiceChargeAmount(session, fromDate, toDate, user));

			//report.setGiftCertReturnAmount(calculateCreditReceipt(session, GiftCertificateTransaction.class, fromDate, toDate));
			//		calculateToleranceAmount	
			////			gift cert
			//			criteria = session.createCriteria(GiftCertificateTransaction.class);
			//			criteria.createAlias(GiftCertificateTransaction.PROP_TICKET, "t");
			//			criteria.add(Restrictions.ge("t." + Ticket.PROP_CREATE_DATE, fromDate));
			//			criteria.add(Restrictions.le("t." + Ticket.PROP_CREATE_DATE, toDate));
			//			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE));
			//			criteria.add(Restrictions.eq("t." + Ticket.PROP_REFUNDED, Boolean.FALSE));
			//			projectionList = Projections.projectionList();
			//			projectionList.add(Projections.sum(PosTransaction.PROP_GIFT_CERT_FACE_VALUE));
			//			projectionList.add(Projections.sum(PosTransaction.PROP_GIFT_CERT_CASH_BACK_AMOUNT));
			//			criteria.setProjection(projectionList);
			//			Object[] o = (Object[]) criteria.uniqueResult();
			//			if(o.length > 0 && o[0] instanceof Number) {
			//				double amount = ((Number) o[0]).doubleValue();
			//				report.setGiftCertReturnAmount(amount);
			//			}
			//			if(o.length > 1 && o[1] instanceof Number) {
			//				double amount = ((Number) o[1]).doubleValue();
			//				report.setGiftCertChangeAmount(amount);
			//			}
			//			
			//			tips paid
			report.setGrossTipsPaidAmount(calculateTipsPaid(session, fromDate, toDate, user));
			//			
			//cash payout
			report.setCashPayoutAmount(calculateCashPayout(session, fromDate, toDate, user));
			//			
			//drawer pulls
			calculateDrawerPullAmount(session, report, fromDate, toDate, user);

			report.setVisaCreditCardAmount(calculateVisaCreditCardSummery(session, CreditCardTransaction.class, fromDate, toDate, user));
			report.setMasterCardAmount(calculateMasterCardSummery(session, CreditCardTransaction.class, fromDate, toDate, user));
			report.setAmexAmount(calculateAmexSummery(session, CreditCardTransaction.class, fromDate, toDate, user));
			report.setDiscoveryAmount(calculateDiscoverySummery(session, CreditCardTransaction.class, fromDate, toDate, user));

			report.setVisaDebitCardAmount(calculateVisaDebitCardSummery(session, DebitCardTransaction.class, fromDate, toDate, user));
			report.setMasterDebitCardAmount(calculateMasterDebitCardSummery(session, DebitCardTransaction.class, fromDate, toDate, user));

			report.calculate();
			return report;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private void calculateDrawerPullAmount(Session session, SalesBalanceReport report, Date fromDate, Date toDate, User user) {
		Criteria criteria = session.createCriteria(CashDrawer.class);
		criteria.add(Restrictions.ge(CashDrawer.PROP_REPORT_TIME, fromDate));
		criteria.add(Restrictions.le(CashDrawer.PROP_REPORT_TIME, toDate));

		addMultiUserFilter(user, criteria, CashDrawer.PROP_ASSIGNED_USER_ID);

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sum(CashDrawer.PROP_DRAWER_ACCOUNTABLE));
		projectionList.add(Projections.sum(CashDrawer.PROP_BEGIN_CASH));
		criteria.setProjection(projectionList);

		Object[] o = (Object[]) criteria.uniqueResult();
		if (o.length > 0 && o[0] instanceof Number) {
			double amount = ((Number) o[0]).doubleValue();
			report.setDrawerPullsAmount(amount);
		}
		if (o.length > 1 && o[1] instanceof Number) {
			double amount = ((Number) o[1]).doubleValue();
			report.setDrawerPullsAmount(report.getDrawerPullsAmount() - amount);
		}
	}

	private double calculateCashPayout(Session session, Date fromDate, Date toDate, User user) {
		Criteria criteria = session.createCriteria(PayOutTransaction.class);
		criteria.add(Restrictions.ge(PayOutTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PayOutTransaction.PROP_TRANSACTION_TIME, toDate));

		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateTipsPaid(Session session, Date fromDate, Date toDate, User user) {
		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.createAlias(Ticket.PROP_GRATUITY, "gratuity"); //$NON-NLS-1$
		criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));

		criteria.add(Restrictions.eq("gratuity." + Gratuity.PROP_PAID, Boolean.TRUE)); //$NON-NLS-1$

		addMultiUserFilter(user, criteria, Ticket.PROP_OWNER_ID);

		criteria.setProjection(Projections.sum("gratuity." + Gratuity.PROP_AMOUNT)); //$NON-NLS-1$

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateCreditReceipt(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CreditCardTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateRefundAmount(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//refund receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));

		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateCashRefundAmount(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//refund receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));

		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateServiceChargeAmount(Session session, Date fromDate, Date toDate, User user) {
		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
		addMultiUserFilter(user, criteria, Ticket.PROP_OWNER_ID);
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(Ticket.PROP_SERVICE_CHARGE));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateDebitReceipt(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//debit receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateCustomPayment(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//custom receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateCashReceipt(Session session, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(CashTransaction.class);
		criteria.add(Restrictions.ge(CashTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(CashTransaction.PROP_TRANSACTION_TIME, toDate));
		addMultiUserFilter(user, criteria, CashTransaction.PROP_USER_ID);
		criteria.add(Restrictions.eq(CashTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		return getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateGiftCertReceipts(Session session, Date fromDate, Date toDate, User user) {
		//gift receipt
		Criteria criteria = session.createCriteria(GiftCertificateTransaction.class);
		criteria.add(Restrictions.ge(GiftCertificateTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(GiftCertificateTransaction.PROP_TRANSACTION_TIME, toDate));
		addMultiUserFilter(user, criteria, GiftCertificateTransaction.PROP_USER_ID);
		criteria.add(Restrictions.eq(GiftCertificateTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(GiftCertificateTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateCashTips(Session session, Date fromDate, Date toDate, User user) {
		//tips
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateMemberPayment(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//Member Payment
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.MEMBER_ACCOUNT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateChargedTips(Session session, Date fromDate, Date toDate, User user) {
		//tips
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateDiscount(Session session, Date fromDate, Date toDate, User user) {
		//discounts
		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

		addMultiUserFilter(user, criteria, Ticket.PROP_OWNER_ID);

		criteria.setProjection(Projections.sum(Ticket.PROP_DISCOUNT_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double getDoubleAmount(Object result) {
		if (result != null && result instanceof Number) {
			return ((Number) result).doubleValue();
		}
		return 0;
	}

	private double calculateTax(Session session, Date fromDate, Date toDate, User user) {
		//discounts
		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		//criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

		addMultiUserFilter(user, criteria, Ticket.PROP_OWNER_ID);

		criteria.setProjection(Projections.sum(Ticket.PROP_TAX_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double getNetSales(Session session, Date fromDate, Date toDate, User user) {
		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		//criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

		addMultiUserFilter(user, criteria, Ticket.PROP_OWNER_ID);
		criteria.setProjection(Projections.sum(Ticket.PROP_SUBTOTAL_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateToleranceAmount(Session session, Date fromDate, Date toDate, User user) {
		Criteria criteria = getCriteriaForTransaction(session, CashTransaction.class);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TOLERANCE_AMOUNT));
		return getDoubleAmount(criteria.uniqueResult());
	}

	private Criteria getCriteriaForTransaction(Session session, Class transactionClass) {
		Criteria criteria = session.createCriteria(transactionClass);
		if (transactionClass.equals(RefundTransaction.class) || transactionClass.equals(PayOutTransaction.class))
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
		else
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		return criteria;
	}

	private double calculateGrossSales(Session session, Date fromDate, Date toDate, User user, boolean taxableSales) {
		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

		addMultiUserFilter(user, criteria, Ticket.PROP_OWNER_ID);

		criteria.add(Restrictions.eq(Ticket.PROP_TAX_EXEMPT, Boolean.valueOf(!taxableSales)));

		criteria.setProjection(Projections.sum(Ticket.PROP_SUBTOTAL_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	public SalesExceptionReport getSalesExceptionReport(Date fromDate, Date toDate) {
		GenericDAO dao = new GenericDAO();
		SalesExceptionReport report = new SalesExceptionReport();
		Session session = null;

		report.setFromDate(fromDate);
		report.setToDate(toDate);
		report.setReportTime(new Date());
		try {

			session = dao.getSession();

			//gross taxable sales

			//Refund Transaction
			Criteria criteriaRefund = session.createCriteria(RefundTransaction.class);
			criteriaRefund.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));

			List list = criteriaRefund.list();
			for (Iterator iter = list.iterator(); iter.hasNext();) {
				RefundTransaction refundTransaction = (RefundTransaction) iter.next();
				report.addRefundToRefundData(refundTransaction);
			}

			//void tickets
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CLOSING_DATE, toDate));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, true));

			list = criteria.list();
			for (Iterator iter = list.iterator(); iter.hasNext();) {
				Ticket ticket = (Ticket) iter.next();
				report.addVoidToVoidData(ticket);
			}

			//discounts
			criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));

			list = criteria.list();
			for (Iterator iter = list.iterator(); iter.hasNext();) {

				Ticket ticket = (Ticket) iter.next();
				report.addDiscountOnTicket(ticket);

				List<TicketItem> ticketItems = ticket.getTicketItems();
				for (TicketItem ticketItem : ticketItems) {
					report.addDiscountOnTicketItem(ticketItem);
				}
			}

			//find all valid discounts
			DiscountDAO discountDAO = new DiscountDAO();
			List<Discount> availableCoupons = discountDAO.getValidCoupons();
			report.addEmptyDiscounts(availableCoupons);

			return report;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public SalesDetailedReport getSalesDetailedReport(Date fromDate, Date toDate, Outlet outlet, boolean isShowOnlineTicket) {
		GenericDAO dao = new GenericDAO();
		SalesDetailedReport report = new SalesDetailedReport();

		report.setFromDate(fromDate);
		report.setToDate(toDate);
		report.setReportTime(new Date());
		try (Session session = dao.createNewSession()) {

			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.createAlias("ticket", "t"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.ge(CustomPaymentTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.lt(CustomPaymentTransaction.PROP_TRANSACTION_TIME, toDate));

			if (isShowOnlineTicket) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_SOURCE, TicketSource.Online.name())); //$NON-NLS-1$ //$NON-NLS-2$
			}
			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}
			//Remove Non ticket transaction 
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));

			List list = criteria.list();
			for (Iterator iter = list.iterator(); iter.hasNext();) {
				PosTransaction t = (PosTransaction) iter.next();
				report.addPaymentData(t);
			}
			return report;
		}
	}

	public SalesDetailedReport getCloudSalesDetailedReport(Date fromDate, Date toDate) {
		GenericDAO dao = new GenericDAO();
		SalesDetailedReport report = new SalesDetailedReport();
		Session session = null;

		report.setFromDate(fromDate);
		report.setToDate(toDate);
		report.setReportTime(new Date());
		try {

			session = dao.getSession();

			//			Criteria criteria = session.createCriteria(CashDrawer.class);
			//			criteria.add(Restrictions.or(Restrictions.isNull(CashDrawer.PROP_REPORT_TIME), Restrictions.ge(CashDrawer.PROP_REPORT_TIME, fromDate)));
			//			criteria.add(Restrictions.or(Restrictions.isNull(CashDrawer.PROP_REPORT_TIME), Restrictions.le(CashDrawer.PROP_REPORT_TIME, toDate)));
			//			List list = criteria.list();
			//			for (Iterator iter = list.iterator(); iter.hasNext();) {
			//				CashDrawer cashDrawer = (CashDrawer) iter.next();
			//				DrawerPullData data = new DrawerPullData();
			//				data.setDrawerPullId(cashDrawer.getId());
			//				data.setTicketCount(cashDrawer.getTicketCount());
			//				data.setIdealAmount(cashDrawer.getDrawerAccountable());
			//				data.setActualAmount(cashDrawer.getCashToDeposit());
			//				data.setVarinceAmount(cashDrawer.getDrawerAccountable() - cashDrawer.getCashToDeposit());
			//				report.addDrawerPullData(data);
			//			}

			Criteria criteria = session.createCriteria(CreditCardTransaction.class);
			criteria.add(Restrictions.ge(CreditCardTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.le(CreditCardTransaction.PROP_TRANSACTION_TIME, toDate));
			List list = criteria.list();

			for (Iterator iter = list.iterator(); iter.hasNext();) {
				CreditCardTransaction t = (CreditCardTransaction) iter.next();
				report.addCreditCardData(t);
			}

			criteria = session.createCriteria(DebitCardTransaction.class);
			criteria.add(Restrictions.ge(DebitCardTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.le(DebitCardTransaction.PROP_TRANSACTION_TIME, toDate));
			list = criteria.list();

			for (Iterator iter = list.iterator(); iter.hasNext();) {
				DebitCardTransaction t = (DebitCardTransaction) iter.next();
				report.addCreditCardData(t);
			}

			criteria = session.createCriteria(GiftCertificateTransaction.class);
			criteria.add(Restrictions.ge(GiftCertificateTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.le(GiftCertificateTransaction.PROP_TRANSACTION_TIME, toDate));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.rowCount());
			projectionList.add(Projections.sum(GiftCertificateTransaction.PROP_AMOUNT));
			criteria.setProjection(projectionList);
			Object[] object = (Object[]) criteria.uniqueResult();
			if (object != null && object.length > 0 && object[0] instanceof Number) {
				report.setGiftCertReturnCount(((Number) object[0]).intValue());
			}
			if (object != null && object.length > 1 && object[1] instanceof Number) {
				report.setGiftCertReturnAmount(((Number) object[1]).doubleValue());
			}

			criteria = session.createCriteria(GiftCertificateTransaction.class);
			criteria.add(Restrictions.ge(GiftCertificateTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.le(GiftCertificateTransaction.PROP_TRANSACTION_TIME, toDate));
			criteria.add(Restrictions.gt(GiftCertificateTransaction.PROP_GIFT_CERT_CASH_BACK_AMOUNT, Double.valueOf(0)));
			projectionList = Projections.projectionList();
			projectionList.add(Projections.rowCount());
			projectionList.add(Projections.sum(GiftCertificateTransaction.PROP_GIFT_CERT_CASH_BACK_AMOUNT));
			criteria.setProjection(projectionList);
			object = (Object[]) criteria.uniqueResult();
			if (object != null && object.length > 0 && object[0] instanceof Number) {
				report.setGiftCertChangeCount(((Number) object[0]).intValue());
			}
			if (object != null && object.length > 1 && object[1] instanceof Number) {
				report.setGiftCertChangeAmount(((Number) object[1]).doubleValue());
			}

			/*criteria = session.createCriteria(Ticket.class);
			criteria.createAlias(Ticket.PROP_GRATUITY, "g"); //$NON-NLS-1$
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
			criteria.add(Restrictions.gt("g." + Gratuity.PROP_AMOUNT, Double.valueOf(0))); //$NON-NLS-1$
			projectionList = Projections.projectionList();
			projectionList.add(Projections.rowCount());
			projectionList.add(Projections.sum("g." + Gratuity.PROP_AMOUNT)); //$NON-NLS-1$
			criteria.setProjection(projectionList);
			object = (Object[]) criteria.uniqueResult();
			if (object != null && object.length > 0 && object[0] instanceof Number) {
				report.setTipsCount(((Number) object[0]).intValue());
			}
			if (object != null && object.length > 1 && object[1] instanceof Number) {
				report.setChargedTips(((Number) object[1]).doubleValue());
			}
			
			criteria = session.createCriteria(Ticket.class);
			criteria.createAlias(Ticket.PROP_GRATUITY, "g"); //$NON-NLS-1$
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
			criteria.add(Restrictions.gt("g." + Gratuity.PROP_AMOUNT, Double.valueOf(0))); //$NON-NLS-1$
			criteria.add(Restrictions.gt("g." + Gratuity.PROP_PAID, Boolean.TRUE)); //$NON-NLS-1$
			projectionList = Projections.projectionList();
			projectionList.add(Projections.sum("g." + Gratuity.PROP_AMOUNT)); //$NON-NLS-1$
			criteria.setProjection(projectionList);
			object = (Object[]) criteria.uniqueResult();
			if (object != null && object.length > 0 && object[0] instanceof Number) {
				report.setTipsPaid(((Number) object[0]).doubleValue());
			}*/

			return report;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private double calculateVisaCreditCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.VISA.name()).ignoreCase());

		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateMasterCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(PosTransactionDAO.createMasterCardSearchCriteria());

		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateAmexSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(PosTransactionDAO.createAmexOrAmericanExpCardSearchCriteria());

		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateDiscoverySummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.DISCOVER.name()).ignoreCase());

		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateVisaDebitCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.VISA.name()).ignoreCase());

		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private double calculateMasterDebitCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = session.createCriteria(transactionClass);
		criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(PosTransactionDAO.createMasterCardSearchCriteria());

		addMultiUserFilter(user, criteria, PosTransaction.PROP_USER_ID);

		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));

		return getDoubleAmount(criteria.uniqueResult());
	}

	private void addMultiUserFilter(User user, Criteria criteria, String fieldName) {
		if (user != null) {
			PosLog.info(getClass(), "setting multi user filter for root user '" + user.getFullName() + "', id: " + user.getId()); //$NON-NLS-1$ //$NON-NLS-2$
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.eq(fieldName, user.getId()));
			List<User> linkedUsers = user.getLinkedUser();
			if (linkedUsers != null) {
				for (User linkedUser : linkedUsers) {
					if (linkedUser.getId().equals(user.getId())) {
						continue;
					}
					PosLog.info(getClass(), "linked user '" + linkedUser.getFullName() + "', id: " + linkedUser.getId()); //$NON-NLS-1$ //$NON-NLS-2$
					disjunction.add(Restrictions.eq(fieldName, linkedUser.getId()));
				}
			}
			criteria.add(disjunction);
		}
	}

	public static SalesReportModel prepareItemSalesReportModel(List<TicketItem> ticketItems) {
		SalesReportModel itemReportModel = new SalesReportModel();
		if (ticketItems != null && ticketItems.size() > 0) {
			HashMap<String, ReportItem> itemMap = new LinkedHashMap<String, ReportItem>();
			String key = null;
			for (TicketItem ticketItem : ticketItems) {
				String menuItemId = ticketItem.getMenuItemId();
				key = (menuItemId == null ? ticketItem.getName() : menuItemId) + "-" + ticketItem.getUnitPrice() + "-" + ticketItem.getUnitName(); //$NON-NLS-1$ //$NON-NLS-2$
				ReportItem reportItem = itemMap.get(key);
				if (reportItem == null) {
					reportItem = new ReportItem();
					reportItem.setId(key);
					reportItem.setPrice(ticketItem.getUnitPrice());
					reportItem.setUnit(ticketItem.getUnitName());
					reportItem.setCost(ticketItem.getUnitCost());
					reportItem.setName(ticketItem.getName());
					reportItem.setTaxRate(ticketItem.getTotalTaxRate());
					reportItem.setGroupName(ticketItem.getGroupName());
					reportItem.setBarcode(menuItemId);
					reportItem.setAdjustedPrice(ticketItem.getAdjustedUnitPrice());
					itemMap.put(key, reportItem);
				}

				reportItem.setQuantity(reportItem.getQuantity() + ticketItem.getQuantity());
				reportItem.setGrossTotal(reportItem.getGrossTotal() + ticketItem.getAdjustedTotalWithoutModifiers());
				reportItem.setDiscount(reportItem.getDiscount() + ticketItem.getAdjustedDiscountWithoutModifiers());
				reportItem.setTaxTotal(reportItem.getTaxTotal() + ticketItem.getAdjustedTaxWithoutModifiers());
				reportItem.setServiceCharge(reportItem.getServiceCharge() + ticketItem.getServiceCharge());
				double adjustedAmount = (ticketItem.isTaxIncluded())
						? (ticketItem.getAdjustedTotalWithoutModifiers() - ticketItem.getAdjustedTaxWithoutModifiers())
						: ticketItem.getAdjustedSubtotalWithoutModifiers();
				reportItem.setNetTotal(reportItem.getNetTotal() + adjustedAmount);
			}
			itemReportModel.setItems(new ArrayList<>(itemMap.values()));
		}
		return itemReportModel;
	}

	public static Map<String, Integer> prepareTicketForDashBoard(List<Date> listOfDate) {
		Map<String, Integer> map = new HashMap<>();
		if (listOfDate != null && listOfDate.size() > 0) {
			int countNumberOfDate = 0;
			for (Date date : listOfDate) {
				String formatDate = DateUtil.formatAsDefaultMonthDate(date);
				if (map.get(formatDate) == null)
					countNumberOfDate = 1;
				if (map.containsKey(formatDate))
					countNumberOfDate++;
				map.put(formatDate, countNumberOfDate);
			}
		}
		return map;
	}

	public TaxExemptReport getTaxExemptReport(Date fromDate, Date toDate) throws Exception {
		TaxExemptReport report = new TaxExemptReport();
		try (Session session = TicketDAO.getInstance().createNewSession()) {

			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CLOSING_DATE, toDate));
			criteria.add(Restrictions.eq(Ticket.PROP_TAX_EXEMPT, true));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, false));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(Ticket.PROP_ID), Ticket.PROP_ID);
			projectionList.add(Projections.property(Ticket.PROP_CREATE_DATE), Ticket.PROP_CREATE_DATE);
			projectionList.add(Projections.property(Ticket.PROP_TOTAL_AMOUNT), Ticket.PROP_TOTAL_AMOUNT);
			projectionList.add(Projections.property(Ticket.PROP_EXTRA_PROPERTIES), Ticket.PROP_EXTRA_PROPERTIES);

			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(Ticket.class));

			List<Ticket> tickets = criteria.list();
			for (Ticket ticket : tickets) {
				report.addTaxExemptTicket(ticket);
			}

			return report;
		}
	}

	public DiscountReportDataModel createTicketDiscountDataList(Date fromDate, Date toDate) {
		DiscountReportDataModel discountReport = new DiscountReportDataModel();
		try (Session session = TicketItemDAO.getInstance().createNewSession()) {

			Criteria criteria = session.createCriteria(Ticket.class);

			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.gt(Ticket.PROP_DISCOUNT_AMOUNT, 0.0));
			criteria.addOrder(Order.asc(Ticket.PROP_CREATE_DATE));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(Ticket.PROP_ID), "ticketId"); //$NON-NLS-1$ 
			projectionList.add(Projections.property(Ticket.PROP_CREATE_DATE), "date"); //$NON-NLS-1$
			projectionList.add(Projections.property(Ticket.PROP_TOTAL_AMOUNT), "totalWithDiscount"); //$NON-NLS-1$
			projectionList.add(Projections.property(Ticket.PROP_DISCOUNTS_PROPERTY), "properties"); //$NON-NLS-1$

			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(DiscountData.class));
			List<DiscountData> list = criteria.list();

			for (Iterator<DiscountData> iterator = list.iterator(); iterator.hasNext();) {
				DiscountData discountData = (DiscountData) iterator.next();
				String properties = discountData.getProperties();
				if (properties.equals("[]")) { //$NON-NLS-1$
					iterator.remove();
				}
				discountData.setTicketDiscount(Boolean.TRUE);
			}
			discountReport.addDiscountDataList(list);
			return discountReport;
		}
	}

	public DiscountReportDataModel createItemDiscountDataList(Date fromDate, Date toDate) {
		DiscountReportDataModel discountReport = new DiscountReportDataModel();
		try (Session session = TicketItemDAO.getInstance().createNewSession()) {

			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.createAlias("ticket", "t"); //$NON-NLS-1$//$NON-NLS-2$
			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$

			criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(TicketItem.PROP_CREATE_DATE, toDate));
			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.gt(TicketItem.PROP_DISCOUNT_AMOUNT, 0.0));
			criteria.addOrder(Order.asc(TicketItem.PROP_CREATE_DATE));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(TicketItem.PROP_NAME), "itemName"); //$NON-NLS-1$
			projectionList.add(Projections.property("t." + Ticket.PROP_ID), "ticketId"); //$NON-NLS-1$ //$NON-NLS-2$
			projectionList.add(Projections.property(TicketItem.PROP_CREATE_DATE), "date"); //$NON-NLS-1$
			projectionList.add(Projections.property(TicketItem.PROP_TOTAL_AMOUNT), "totalWithDiscount"); //$NON-NLS-1$
			projectionList.add(Projections.property(TicketItem.PROP_DISCOUNTS_PROPERTY), "properties"); //$NON-NLS-1$

			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(DiscountData.class));
			List<DiscountData> list = criteria.list();
			discountReport.addDiscountDataList(list);
			return discountReport;
		}
	}

	public PayoutReportDataModel findPayoutTransactions(String storeSessionId, Date fromDate, Date toDate) {
		PayoutReportDataModel reportDataModel = new PayoutReportDataModel();
		try (Session session = PosTransactionDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(PayOutTransaction.class);
			if (StringUtils.isNotEmpty(storeSessionId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSessionId));
			}
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TIME), "date"); //$NON-NLS-1$
			projectionList.add(Projections.property(PosTransaction.PROP_USER_ID), PosTransaction.PROP_USER_ID);
			projectionList.add(Projections.property(PosTransaction.PROP_RECEPIENT_ID), "recipientId"); //$NON-NLS-1$
			projectionList.add(Projections.property(PosTransaction.PROP_REASON_ID), PosTransaction.PROP_REASON_ID);
			projectionList.add(Projections.property(PosTransaction.PROP_NOTE), PosTransaction.PROP_NOTE);
			projectionList.add(Projections.property(PosTransaction.PROP_CASH_DRAWER_ID), PosTransaction.PROP_CASH_DRAWER_ID);
			projectionList.add(Projections.property(PosTransaction.PROP_AMOUNT), PosTransaction.PROP_AMOUNT);

			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(PayoutReportData.class));
			List<PayoutReportData> list = criteria.list();

			Map<String, PayoutReason> payoutResonsIds = new HashMap<String, PayoutReason>();
			Map<String, User> userIds = new HashMap<String, User>();
			Map<String, PayoutRecepient> payoutRecepientIds = new HashMap<String, PayoutRecepient>();
			Map<Integer, Terminal> terminalIds = new HashMap<Integer, Terminal>();

			if (list.size() > 0) {
				for (Iterator<?> iterator = list.iterator(); iterator.hasNext();) {
					PayoutReportData payoutData = (PayoutReportData) iterator.next();
					//Create reason
					if (payoutData.getReasonId() != null) {
						PayoutReason payoutReason = payoutResonsIds.get(payoutData.getReasonId());
						if (payoutReason == null) {
							payoutReason = PayoutReasonDAO.getInstance().get(payoutData.getReasonId());
							payoutResonsIds.put(payoutReason.getId(), payoutReason);
						}
						payoutData.setReason(payoutReason.getReason());
					}

					//create user Name
					if (payoutData.getUserId() != null) {
						User user = userIds.get(payoutData.getUserId());
						if (user == null) {
							//TODO:
							user = UserDAO.getInstance().get(payoutData.getUserId(), null);
							userIds.put(user.getId(), user);
						}
						payoutData.setUserName(user.getFullName());
					}

					//create Recipient Name
					if (payoutData.getRecipientId() != null) {
						PayoutRecepient payoutRecepient = payoutRecepientIds.get(payoutData.getRecipientId());
						if (payoutRecepient == null) {
							payoutRecepient = PayoutRecepientDAO.getInstance().get(payoutData.getRecipientId());
							payoutRecepientIds.put(payoutRecepient.getId(), payoutRecepient);
						}
						payoutData.setRecipientName(payoutRecepient.getName());
					}

					//create terminal Name
					if (payoutData.getTerminalId() != null) {
						Terminal terminal = terminalIds.get(payoutData.getTerminalId());
						if (terminal == null) {
							//TODO:
							terminal = TerminalDAO.getInstance().get(payoutData.getTerminalId(), null);
							terminalIds.put(terminal.getId(), terminal);
						}
						payoutData.setTerminalName(terminal.getName());
					}
				}
			}

			reportDataModel.addPayoutDataList(list);
			return reportDataModel;
		}
	}

}
