/**
* ************************************************************************
* * The contents of this file are subject to the MRPL 1.2
* * (the  "License"),  being   the  Mozilla   Public  License
* * Version 1.1  with a permitted attribution clause; you may not  use this
* * file except in compliance with the License. You  may  obtain  a copy of
* * the License at http://www.floreantpos.org/license.html
* * Software distributed under the License  is  distributed  on  an "AS IS"
* * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
* * License for the specific  language  governing  rights  and  limitations
* * under the License.
* * The Original Code is FLOREANT POS.
* * The Initial Developer of the Original Code is OROCUBE LLC
* * All portions are Copyright (C) 2015 OROCUBE LLC
* * All Rights Reserved.
* ************************************************************************
*/
package com.floreantpos.util;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosLog;
import com.floreantpos.model.NumberBound;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Store;
import com.floreantpos.model.util.DataProvider;

public class NumberUtil {
	private final static NumberFormat numberFormat = NumberFormat.getNumberInstance();
	private final static NumberFormat numberFormat2 = NumberFormat.getNumberInstance();
	private final static DecimalFormat decimalFormat = new DecimalFormat("#.##"); //$NON-NLS-1$
	private final static DecimalFormat twoDigitFormat = new DecimalFormat("0.00"); //$NON-NLS-1$

	private final static NumberFormat integerFormat = new DecimalFormat("0"); //$NON-NLS-1$
	private final static NumberFormat currencyFormat = new DecimalFormat("0.00##"); //$NON-NLS-1$
	private final static NumberFormat numberFormatWithSixDigit = new DecimalFormat("0.######"); //$NON-NLS-1$
	private static Map<NumberBound, Pattern> patterns = new HashMap<>();

	private static final char decimalSeparator;
	private static final char groupingSeparator;
	private static int decimalPlace = 2;

	static {
		twoDigitFormat.setMinimumFractionDigits(2);
		twoDigitFormat.setMaximumFractionDigits(2);
		twoDigitFormat.setRoundingMode(RoundingMode.HALF_UP);

		numberFormat.setMinimumFractionDigits(decimalPlace);
		numberFormat.setMaximumFractionDigits(decimalPlace);
		numberFormat.setRoundingMode(RoundingMode.HALF_UP);

		numberFormat2.setMinimumFractionDigits(3);
		numberFormat2.setMaximumFractionDigits(3);
		numberFormat2.setRoundingMode(RoundingMode.HALF_UP);

		numberFormat.setGroupingUsed(false);
		numberFormat2.setGroupingUsed(false);
		currencyFormat.setGroupingUsed(true);

		DecimalFormatSymbols symbols = new DecimalFormatSymbols(Locale.getDefault(Locale.Category.FORMAT));
		decimalSeparator = symbols.getDecimalSeparator();
		groupingSeparator = symbols.getGroupingSeparator();
	}

	/**
	 * @param value
	 * @return
	 */
	public static double round(double value) {
		if (Double.isNaN(value)) {
			return 0;
		}
		BigDecimal bd = new BigDecimal(String.valueOf(value));
		bd = bd.setScale(decimalPlace, BigDecimal.ROUND_HALF_UP);
		return bd.doubleValue();
	}

	public static BigDecimal round(BigDecimal value) {
		return value.setScale(decimalPlace, BigDecimal.ROUND_HALF_UP);
	}

	public static double round(double value, int degit) {
		if (Double.isNaN(value)) {
			return 0;
		}
		BigDecimal bd = new BigDecimal(String.valueOf(value));
		bd = bd.setScale(degit, BigDecimal.ROUND_HALF_UP);
		return bd.doubleValue();
	}

	public static double roundToOneDigit(double value) {
		if (Double.isNaN(value)) {
			return 0;
		}
		BigDecimal bd = new BigDecimal(String.valueOf(value));
		bd = bd.setScale(1, BigDecimal.ROUND_HALF_UP);
		return bd.doubleValue();
	}

	public static double roundToTwoDigit(double value) {
		return round(value);
	}

	public static double roundToThreeDigit(double value) {
		if (Double.isNaN(value)) {
			return 0;
		}
		BigDecimal bd = new BigDecimal(String.valueOf(value));
		bd = bd.setScale(3, BigDecimal.ROUND_HALF_UP);
		return bd.doubleValue();
	}

	public static String format3DigitNumber(Double number) {
		return format3DigitNumber((Number) number);
	}

	public static String format3DigitNumber(Number number) {
		if (number == null) {
			return numberFormat2.format(0);
		}
		String value = numberFormat2.format(number);

		/*if (value.startsWith("-")) { //$NON-NLS-1$
			return numberFormat2.format(0);
		}*/
		return value;
	}

	public static String format6DigitNumber(Double number) {
		if (number == null) {
			return numberFormatWithSixDigit.format(0);
		}
		String value = numberFormatWithSixDigit.format(number);
		return value;
	}

	public static String format6DigitNumber(Number number) {
		if (number == null) {
			return numberFormatWithSixDigit.format(0);
		}
		String value = numberFormatWithSixDigit.format(number);
		return value;
	}

	public static String formatNumber(Double doubleValue, boolean isAllowedNegative) {
		return formatNumberAcceptNegative(doubleValue);
	}

	public static String format(Number number) {
		return formatNumber(number, true);
	}

	public static String formatNumber(Number number, boolean isAllowedNegative) {
		if (number == null) {
			return numberFormat.format(0);
		}

		String value = numberFormat.format(number);

		if (!isAllowedNegative) {
			if (value.startsWith("-")) { //$NON-NLS-1$
				return numberFormat.format(0);
			}
		}
		return value;
	}

	public static String formatNumber(Double number) {
		return formatNumber(number, false);
	}

	public static String formatNumber(Number number) {
		return formatNumber(number, false);
	}

	public static String trimDecilamIfNotNeeded(Double doubleValue) {
		return trimDecilamIfNotNeeded((Number) doubleValue, false);
	}

	public static String trimDecilamIfNotNeeded(Number number) {
		return trimDecilamIfNotNeeded(number, false);
	}

	public static String trimDecilamIfNotNeeded(Number number, boolean allowNegative) {
		if (number == null) {
			return decimalFormat.format(0);
		}

		String value = decimalFormat.format(number);
		if (allowNegative)
			return value;
		if (value.startsWith("-")) { //$NON-NLS-1$
			return decimalFormat.format(0);
		}

		return value;
	}

	public static String formatNumberAcceptNegative(Double number) {
		return formatNumberAcceptNegative((Number) number);
	}

	public static String formatNumberAcceptNegative(Number number) {
		if (number == null) {
			return numberFormat.format(0);
		}
		return numberFormat.format(number);
	}

	public static Number parse(String number) throws ParseException {
		if (StringUtils.isBlank(number)) {
			return 0;
		}

		return numberFormat.parse(number);
	}

	public static double parseDouble(String number) {
		try {
			if (StringUtils.isBlank(number)) {
				return 0;
			}

			return numberFormat.parse(number).doubleValue();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public static Number parseOrGetZero(String number) {
		try {
			if (StringUtils.isBlank(number)) {
				return 0;
			}

			return numberFormat.parse(number);
		} catch (Exception e) {
			return 0;
		}
	}

	public static NumberFormat getNumberFormat() {
		return numberFormat;
	}

	public static String getCurrencyFormat(Object d) {
		String format = currencyFormat.format(d);
		if (format.startsWith("-")) { //$NON-NLS-1$
			return "-" + CurrencyUtil.getCurrencySymbol() + format.replaceFirst("-", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		return CurrencyUtil.getCurrencySymbol() + format;
	}

	public static String getCurrencyFormatWithoutCurrencySymbol(Object d) {
		return currencyFormat.format(d);
	}

	public static double getCurrencyFormatWithoutSymbol(Object d) {
		try {
			return currencyFormat.parse(currencyFormat.format(d)).doubleValue();
		} catch (ParseException e) {
			return 0d;
		}
	}

	public static int getIntegerFormat(Object d) {
		try {
			return integerFormat.parse(integerFormat.format(d)).intValue();
		} catch (ParseException e) {
			return 0;
		}
	}

	public static BigDecimal convertToBigDecimal(double value) {
		return new BigDecimal(Double.toString(value));
	}

	public static BigDecimal convertToBigDecimal(String value) {
		return new BigDecimal(value);
	}

	public static boolean isZero(Double number) {
		return isZero((Number) number);
	}

	public static boolean isZero(Number number) {
		String value = twoDigitFormat.format(number);
		try {
			return Math.abs(twoDigitFormat.parse(value).doubleValue()) == 0.0;
		} catch (ParseException e) {
			return false;
		}
	}

	public static String getCurrencyTwoDigitFormat(double value) {
		return twoDigitFormat.format(value);
	}

	public static Boolean isFractional(String number) {
		try {
			return isFractional(NumberUtil.parseDouble(number));
		} catch (Exception e0) {
			PosLog.error(NumberUtil.class, e0);
		}
		return null;
	}

	public static Boolean isFractional(Double number) {
		if (number == null || number.isNaN()) {
			return null;
		}
		double doubleValue = number;
		int intValue = (int) doubleValue;
		number = new Double(intValue);
		return doubleValue > number;
	}

	public static String formatNumberIfNeeded(Double number) {
		Boolean isFractional = isFractional(number);
		String value = isFractional ? formatNumber(number) : String.valueOf((int) ((double) number));
		return value;
	}

	public static String formatAmount(Double amount) {
		return formatAmount((Number) amount);
	}

	public static String formatAmount(Number amount) {
		if (amount == null) {
			return currencyFormat.format(0.0);
		}
		return currencyFormat.format(amount);
	}

	/**
	 * This method is used to check DoubleTextField validation with default length.
	 * @param value 
	 * 
	 */
	public static boolean isValidDouble(String value) {
		return isValidDouble(value, new NumberBound(8, 6));
	}

	/**
	 * This method is used to check DoubleTextField validation.
	 * @param value  
	 * @param beforeDecimal 
	 * @param afterDecimal 
	 */
	public static boolean isValidDouble(String value, int beforeDecimal, int afterDecimal) {
		return isValidDouble(value, new NumberBound(beforeDecimal, afterDecimal));
	}

	private static boolean isValidDouble(String value, NumberBound bound) {
		if (value == null) {
			return false;
		}
		Pattern pattern = getPattern(bound);
		DecimalFormatSymbols symbols = new DecimalFormatSymbols(Locale.getDefault());
		String groupingSeparator = new StringBuilder().append(symbols.getGroupingSeparator()).toString();
		value = value.replace(groupingSeparator, ""); //$NON-NLS-1$
		return pattern.matcher(value).matches();
	}

	/**
	 * This method is used to get Pattern.
	 * 
	 * @param bound NumberBound
	 * 
	 *  @see java.util.regex.Pattern#pattern()
	 */
	public static Pattern getPattern(NumberBound bound) {
		Pattern pattern;
		if (!patterns.containsKey(bound)) {
			initPattern(bound);
		}
		pattern = patterns.get(bound);
		return pattern;
	}

	private static void initPattern(NumberBound bound) {
		patterns.put(bound, Pattern.compile(createRegex(bound)));
	}

	public static String createRegex() {
		Store store = DataProvider.get().getStore();
		NumberBound bound = new NumberBound(8, store.getDecimalPlace());
		return createRegex(bound);
	}

	public static String createRegex(NumberBound bound) {
		StringBuilder regexBuilder = new StringBuilder();
		regexBuilder.append("[-]?[0-9]{0,"); //$NON-NLS-1$
		regexBuilder.append(bound.getBeforeDecimal());
		regexBuilder.append("}(\\"); //$NON-NLS-1$
		regexBuilder.append(decimalSeparator);
		regexBuilder.append("[0-9]{0,"); //$NON-NLS-1$
		regexBuilder.append(bound.getAfterDecimal());
		regexBuilder.append("})?"); //$NON-NLS-1$
		String regex = regexBuilder.toString();
		return regex;
	}

	public static char getDecimalSeparator() {
		return decimalSeparator;
	}

	public static char getGroupingSeparator() {
		return groupingSeparator;
	}

	//	@Deprecated
	//	public static void initializeNumberFormats() {
	//		initializeNumberFormats(DataProvider.get().getStore());
	//	}

	public static void initializeNumberFormats(Store store) {
		decimalPlace = store.getDecimalPlace();
		numberFormat.setMinimumFractionDigits(decimalPlace);
		numberFormat.setMaximumFractionDigits(decimalPlace);

		currencyFormat.setMinimumFractionDigits(decimalPlace);
	}

	public static void initializeNumberFormats(Outlet outlet) {
		decimalPlace = outlet.getDecimalPlace();
		numberFormat.setMinimumFractionDigits(decimalPlace);
		numberFormat.setMaximumFractionDigits(decimalPlace);

		currencyFormat.setMinimumFractionDigits(decimalPlace);
	}

	/**
	 * This method does not round. It just convert the number to string.
	 * 
	 * @param number
	 * @return
	 */
	public static String toString(double number) {
		return numberFormatWithSixDigit.format(number);
	}
}
