package com.floreantpos.extension.paypal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;

import com.floreantpos.PosLog;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.paypal.api.payments.Address;
import com.paypal.api.payments.Amount;
import com.paypal.api.payments.Details;
import com.paypal.api.payments.Item;
import com.paypal.api.payments.ItemList;
import com.paypal.api.payments.Links;
import com.paypal.api.payments.Payer;
import com.paypal.api.payments.PayerInfo;
import com.paypal.api.payments.Payment;
import com.paypal.api.payments.RedirectUrls;
import com.paypal.api.payments.Transaction;
import com.paypal.base.rest.APIContext;
import com.paypal.base.rest.PayPalRESTException;

public class PayPalProcessor {
	private String clientID;
	private String clientSecret;
	private String mode;
	private String returnUrl;
	private String cancelUrl;
	private String countryCode;

	public PayPalProcessor(String apiKey, String secretKey, String mode) {
		this.clientID = apiKey;
		this.clientSecret = secretKey;
		this.mode = mode;
	}

	public String authorize(Ticket ticket, Customer customer) {

		Payer payer = getPayer(customer, ticket);
		RedirectUrls redirectUrls = getRedirectUrls();
		List<Transaction> transactions = createTransaction(ticket);

		Payment payPalPayment = new Payment();
		payPalPayment.setPayer(payer);
		payPalPayment.setTransactions(transactions);
		payPalPayment.setRedirectUrls(redirectUrls);
		payPalPayment.setNoteToPayer(customer.getId());
		payPalPayment.setIntent("sale"); //$NON-NLS-1$
		try {
			Map<String, String> configurationMap = new HashMap<>();
			configurationMap.put("disable-funding", "paylater"); //$NON-NLS-1$ //$NON-NLS-2$
			configurationMap.put("log.LogEnabled", "true"); //$NON-NLS-1$ //$NON-NLS-2$
			configurationMap.put("log.FileName", "PayPal.log"); //$NON-NLS-1$ //$NON-NLS-2$
			configurationMap.put("log.LogLevel", "DEBUG"); //$NON-NLS-1$ //$NON-NLS-2$
			APIContext payPalapiContext = new APIContext(clientID, clientSecret, mode, configurationMap);
			Payment myPayment = payPalPayment.create(payPalapiContext);
			PosLog.debug(getClass(), "createdPayment Obejct Details ==> " + myPayment.toString()); //$NON-NLS-1$
			return getApprovalLink(myPayment);
		} catch (PayPalRESTException e) {
			String message = e.getMessage();
			JSONObject errorJSON = null;
			try {
				errorJSON = new JSONObject("{" + StringUtils.substringBetween(message, "{", "}") + "}"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			} catch (Exception e2) {
				throw new RuntimeException(message);
			}
			if (errorJSON != null && errorJSON.has("error_description")) {
				throw new RuntimeException(errorJSON.getString("error_description")); //$NON-NLS-1$
			}
			if (errorJSON != null && errorJSON.has("issue")) {
				throw new RuntimeException(errorJSON.getString("issue")); //$NON-NLS-1$
			}
			throw new RuntimeException(message);
		}
	}

	private String getApprovalLink(Payment myPayment) {
		List<Links> links = myPayment.getLinks();
		String approvalLink = null;
		for (Links link : links) {
			if (link.getRel().equalsIgnoreCase("approval_url")) { //$NON-NLS-1$
				approvalLink = link.getHref();
			}
		}
		return approvalLink;
	}

	public List<Transaction> createTransaction(Ticket ticket) {
		//		for (TicketItem ticketItem : ticket.getTicketItems()) {
		//			ticketItem.setUnitPrice(1D);
		//		}
		//		ticket.calculatePrice();

		JSONObject supplementaryData = new JSONObject();

		Details orderDetails = new Details();
		String amountString = String.format("%.2f", ticket.getDueAmount()); //$NON-NLS-1$
		orderDetails.setSubtotal(amountString);
		orderDetails.setTax(String.valueOf(0L));
		orderDetails.setShipping(String.valueOf(0L));

		Amount amount = new Amount();
		amount.setCurrency("USD"); //$NON-NLS-1$
		amount.setTotal(amountString);
		amount.setDetails(orderDetails);

		Transaction transaction = new Transaction();
		transaction.setAmount(amount);
		transaction.setDescription(ticket.getTicketItems().toString());

		ItemList itemList = new ItemList();
		List<Item> items = new ArrayList<Item>();

		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			Item item = new Item();
			item.setName(ticketItem.getNameDisplay());
			item.setQuantity(String.valueOf(ticketItem.getQuantity().longValue()));
			item.setCurrency("USD"); //$NON-NLS-1$
			String itemPriceString = String.format("%.2f", ticketItem.getAdjustedUnitPrice());//$NON-NLS-1$
			item.setPrice(itemPriceString);
			item.setTax(String.valueOf(0L));
			item.setSku(ticketItem.getMenuItemId());
			if (!ticketItem.isService() && !ticketItem.getMenuItem().isEditablePrice()) {
				String priceInterval = ticketItem.getProperty("one_time_price_interval"); //$NON-NLS-1$
				if (StringUtils.isNotBlank(priceInterval)) {
					supplementaryData.put("one_time_price_interval", priceInterval); //$NON-NLS-1$
				}
				String intervalCount = ticketItem.getProperty("one_time_price_interval_count"); //$NON-NLS-1$
				if (StringUtils.isNotBlank(intervalCount)) {
					supplementaryData.put("one_time_price_interval_count", intervalCount); //$NON-NLS-1$
				}
			}
			items.add(item);
		}
		itemList.setItems(items);
		transaction.setCustom(supplementaryData.toString());
		transaction.setItemList(itemList);
		return Arrays.asList(transaction);
	}

	public Payer getPayer(Customer customer, Ticket ticket) {
		Payer payer = new Payer();
		payer.setPaymentMethod("paypal"); //$NON-NLS-1$

		PayerInfo payerInfo = new PayerInfo();
		payerInfo.setPayerId(customer.getId());
		payerInfo.setCountryCode(countryCode);

		String addressLine = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_ADDRESS);
		String city = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_CITY);
		String state = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_STATE);
		String zip = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_STATE);

		Address address = new Address();
		address.setLine1(StringUtils.isNotBlank(addressLine) ? addressLine : customer.getAddress());
		address.setCountryCode(countryCode);
		address.setCity(city == null ? "" : city); //$NON-NLS-1$
		address.setState(state == null ? "" : state); //$NON-NLS-1$
		address.setPostalCode(zip == null ? "" : zip); //$NON-NLS-1$
		payerInfo.setFirstName(customer.getName()).setEmail(customer.getEmail()).setBillingAddress(address);
		payer.setPayerInfo(payerInfo);
		return payer;
	}

	public void setCancelUrl(String cancelUrl) {
		this.cancelUrl = cancelUrl;
	}

	public void setReturnUrl(String returnUrl) {
		this.returnUrl = returnUrl;
	}

	public RedirectUrls getRedirectUrls() {
		RedirectUrls payPalRedirectUrls = new RedirectUrls();
		payPalRedirectUrls.setCancelUrl(cancelUrl);
		payPalRedirectUrls.setReturnUrl(returnUrl);
		return payPalRedirectUrls;
	}

	public void setCountryCode(String countryCode) {
		this.countryCode = countryCode;
	}

	public Payment getPayment(String paymentId) {
		APIContext payPalapiContext = new APIContext(clientID, clientSecret, mode);
		try {
			return Payment.get(payPalapiContext, paymentId);
		} catch (PayPalRESTException e) {
			throw new RuntimeException(e);
		}
	}
}
