package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.PizzaCrust;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PizzaCrustDAO extends BasePizzaCrustDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PizzaCrustDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public List<PizzaCrust> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		if (obj instanceof PizzaCrust) {
			PizzaCrust pizzaCrust = (PizzaCrust) obj;
			session.refresh(pizzaCrust);

			List<String> listOfPriceId = PizzaPriceDAO.getInstance().findByPizzaCrust(pizzaCrust, session);
			if (listOfPriceId != null && !listOfPriceId.isEmpty()) {
				MenuItemDAO.getInstance().removePizzaPrices(listOfPriceId, session);
			}
			pizzaCrust.setDeleted(Boolean.TRUE);
			super.update(pizzaCrust, session);
		}
		else {
			throw new PosException(Messages.getString("PizzaCrustDAO.0")); //$NON-NLS-1$
		}
	}

	@Override
	public Order getDefaultOrder() {
		return Order.asc(PizzaCrust.PROP_SORT_ORDER);
	}

	public void saveOrUpdateList(List<PizzaCrust> items) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			saveOrUpdateCrustList(items, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			session.close();
		}
	}

	public void saveOrUpdateCrustList(List<PizzaCrust> items, Session session) {
		for (Iterator iterator = items.iterator(); iterator.hasNext();) {
			PizzaCrust pizzaCrust = (PizzaCrust) iterator.next();
			session.saveOrUpdate(pizzaCrust);
		}
	}

	public void saveOrUpdatePizzaCrustList(List<PizzaCrust> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<PizzaCrust> iterator = dataList.iterator(); iterator.hasNext();) {
				PizzaCrust item = (PizzaCrust) iterator.next();
				PizzaCrust existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

	public boolean nameExists(String name, String id) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(id)) {
				criteria.add(Restrictions.ne(PizzaCrust.PROP_ID, id));
			}
			criteria.add(Restrictions.eq(PizzaCrust.PROP_NAME, name).ignoreCase());
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		} finally {
			closeSession(session);
		}
	}
}