/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.text.SimpleDateFormat;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseInventoryTransaction;
import com.floreantpos.model.dao.InventoryVendorDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.InventoryUnitConvertionUtil;
import com.floreantpos.util.NumberUtil;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "unitCostDisplay", "vendor", "toInventoryLocation", "fromInventoryLocation", "transactionDateAsString" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class InventoryTransaction extends BaseInventoryTransaction implements TimedModel {
	private static final long serialVersionUID = 1L;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public InventoryTransaction() {
	}

	/**
	 * Constructor for primary key
	 */
	public InventoryTransaction(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public final static String PROPERTY_VENDOR = "vendor"; //$NON-NLS-1$
	//public final static String PROPERTY_MENU_ITEM = "menuItem";
	public final static String PROPERTY_FROM_LOC = "fromInventoryLocation"; //$NON-NLS-1$
	public final static String PROPERTY_TO_LOC = "toInventoryLocation"; //$NON-NLS-1$

	public static final String REASON_NEW_STOCK = "NEW STOCK"; //$NON-NLS-1$
	public static final String REASON_RETURN = "RETURN"; //$NON-NLS-1$
	public static final String REASON_PURCHASE = "PURCHASE"; //$NON-NLS-1$
	public static final String REASON_DAMAGED = "DAMAGED"; //$NON-NLS-1$
	public static final String REASON_TICKET_SALES = "TICKET SALES"; //$NON-NLS-1$
	public static final String REASON_TRANSFER = "TRANSFER"; //$NON-NLS-1$
	public static final String REASON_UNIT_CONVERSION = "UNIT. CONV."; //$NON-NLS-1$
	public static final String REASON_ADJUST = "ADJUST"; //$NON-NLS-1$
	public static final String REASON_ADJUST_IN = "ADJUST_IN"; //$NON-NLS-1$
	public static final String REASON_ADJUST_OUT = "ADJUST_OUT"; //$NON-NLS-1$
	public static final String REASON_PREPARE_IN = "PREPARE IN"; //$NON-NLS-1$
	public static final String REASON_PREPARE_OUT = "PREPARE OUT"; //$NON-NLS-1$
	public static final String REASON_VOID = "VOID"; //$NON-NLS-1$
	public static final String VENDOR_RETURN = "VENDOR RETURN"; //$NON-NLS-1$

	public static final String[] REASON_IN = { REASON_NEW_STOCK, REASON_RETURN, REASON_PURCHASE, REASON_ADJUST_IN, REASON_PREPARE_IN };
	public static final String[] REASON_OUT = { REASON_TICKET_SALES, REASON_DAMAGED, REASON_ADJUST_OUT, REASON_PREPARE_OUT, VENDOR_RETURN };
	public static final String[] REASON_TRANS = { REASON_TRANSFER };

	public static InventoryTransactionType transactionType;
	private static final SimpleDateFormat dateFormat = new SimpleDateFormat("MMM dd, hh:mm a"); //$NON-NLS-1$
	private static final SimpleDateFormat dateFormatWithoutTime = new SimpleDateFormat("MMM dd"); //$NON-NLS-1$

	private double openingQty;
	private double openingCost;
	private double openingTotalCost;

	private transient InventoryLocation inventoryToLocation;
	private transient InventoryLocation inventoryFromLocation;

	public InventoryTransactionType getTransactionType() {
		return transactionType = InventoryTransactionType.fromInt(super.getType());
	}

	public void setTransactionType(InventoryTransactionType type) {
		super.setType(type.getType());
	}

	public String getTransactionDateAsString() {
		String dateAsString = ""; //$NON-NLS-1$
		if (super.getTransactionDate() == null) {
			return dateAsString;
		}
		String reason = this.getReason();
		if (reason.equalsIgnoreCase(InventoryTransaction.REASON_PURCHASE)) {
			dateAsString = dateFormatWithoutTime.format(super.getTransactionDate());
		}
		else {
			dateAsString = dateFormat.format(super.getTransactionDate());
		}
		return dateAsString;
	}

	public void setTransactionDateAsString(String transactionDateAsString) {
	}

	public double getOpeningQty() {
		return openingQty;
	}

	public void setOpeningQty(double openingQty) {
		this.openingQty = openingQty;
	}

	public double getOpeningCost() {
		return openingCost;
	}

	public void setOpeningCost(double openingCost) {
		this.openingCost = openingCost;
	}

	public double getOpeningTotalCost() {
		return openingTotalCost;
	}

	public void setOpeningTotalCost(double openingTotalCost) {
		this.openingTotalCost = openingTotalCost;
	}

	public void setUnitCostDisplay(String s) {
		//do nothing
		//unitCostDisplay
	}

	@XmlTransient
	public String getUnitCostDisplay() {
		return NumberUtil.format3DigitNumber(getUnitCost());
	}

	public void setUnitPriceDisplay(String s) {
		//do nothing
		//unitCostDisplay
	}

	public String getUnitPriceDisplay() {
		return NumberUtil.format3DigitNumber(getUnitPrice());
	}

	public String getSku() {
		return getMenuItem().getSku();
	}

	public void setSku(String sku) {
	}

	public String getItemName() {
		return getMenuItem().getName();
	}

	public void setItemName(String itemName) {
	}

	public void setToInventoryLocation(InventoryLocation location) {
		this.inventoryToLocation = location;
		setToLocationId(location == null ? null : location.getId());
	}

	@XmlTransient
	public InventoryLocation getToInventoryLocation() {
		if (this.inventoryToLocation != null) {
			return this.inventoryToLocation;
		}
		return DataProvider.get().getInventoryLocationById(getToLocationId(), getOutletId());
	}

	public void setFromInventoryLocation(InventoryLocation location) {
		this.inventoryFromLocation = location;
		setFromLocationId(location == null ? null : location.getId());
	}

	@XmlTransient
	public InventoryLocation getFromInventoryLocation() {
		if (this.inventoryFromLocation != null) {
			return this.inventoryFromLocation;
		}
		return DataProvider.get().getInventoryLocationById(getFromLocationId(), getOutletId());
	}

	public void setUser(User user) {
		setUserId(user == null ? null : user.getId());
	}

	public void setVendor(InventoryVendor vendor) {
		setVendorId(vendor == null ? null : vendor.getId());
	}

	@XmlTransient
	public InventoryVendor getVendor() {
		if (getVendorId() == null) {
			return null;
		}

		return InventoryVendorDAO.getInstance().get(getVendorId());
	}

	public void calculateTotal() {
		Double unitCost = super.getUnitCost();
		IUnit transactionUnit = (IUnit) DataProvider.get().getUnitByCode(getUnit());
		if (transactionUnit == null) {
			setTotal(getQuantity() * unitCost);
			return;
		}
		MenuItem menuItem = getMenuItem();
		if (StringUtils.isEmpty(menuItem.getUnitId())) {
			setTotal(getQuantity() * unitCost);
			return;
		}
		double outUnitCost = InventoryUnitConvertionUtil.calculateCost(unitCost, transactionUnit, menuItem);
		setUnitCost(outUnitCost);
		setTotal(getQuantity() * outUnitCost);
	}

	//	public MenuItem getMenuItem() {
	//		if (menuItem != null) {
	//			return menuItem;
	//		}
	//		if (getMenuItemId() == null) {
	//			return null;
	//		}
	//		menuItem = MenuItemDAO.getInstance().get(getMenuItemId());
	//		return menuItem;
	//	}
	//	
	//	public void setMenuItem(MenuItem menuItem) {
	//		this.menuItem = menuItem;
	//		setMenuItemId(menuItem == null ? null : menuItem.getId());
	//	}
}