/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseRecepie;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.RecipeTableDAO;
import com.floreantpos.model.ext.InvMapKey;
import com.floreantpos.model.util.DataProvider;

@XmlRootElement
@JsonIgnoreProperties(ignoreUnknown = true, value = { "menuItemName" })
public class Recepie extends BaseRecepie implements IdContainer, TimedModel {
	private static final long serialVersionUID = 1L;
	private Double cost;
	private Boolean visible;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public Recepie() {
	}

	/**
	 * Constructor for primary key
	 */
	public Recepie(java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public Recepie(java.lang.String id, java.lang.String name) {

		super(id, name);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	public void addRecepieItem(RecepieItem recepieItem) {
		List<RecepieItem> recepieItems = getRecepieItems();
		if (recepieItems == null) {
			recepieItems = new ArrayList<RecepieItem>(3);
			setRecepieItems(recepieItems);
		}

		//recepieItem.setRecepie(this);
		recepieItems.add(recepieItem);
	}

	@Override
	public String toString() {
		return super.getName();
	}

	public Double getCost() {
		return cost == null ? 0 : cost;
	}

	public void setCost(Double cost) {
		this.cost = cost;
	}

	public void calculateCost() {
		double portionCost = 0;
		double recipeCost = 0;
		List<RecepieItem> items = getRecepieItems();
		if (items != null && items.size() > 0) {
			for (RecepieItem recepieItem : items) {
				recepieItem.calculatePercentage();
				recipeCost += recepieItem.getCost();
			}
		}
		portionCost = (recipeCost * getPortion()) / getYield();
		setCost(portionCost);
	}

	@Override
	public Double getPortion() {
		return super.getPortion() == 0 ? 1 : super.getPortion();
	}

	@Override
	public Double getYield() {
		return super.getYield() == 0 ? 1 : super.getYield();
	}

	public void setVisible(Boolean visible) {
		this.visible = visible;
	}

	public Boolean isVisible() {
		return visible == null ? false : visible;
	}

	private Double cookingYield;

	public void populateRecipeItemQuantity(Double mainItemQty, HashMap<InvMapKey, Double> recepieItemMap, double itemQuantity) {
		List<RecepieItem> recepieItems = getRecepieItems();
		if (recepieItems == null || recepieItems.size() == 0)
			return;

		for (RecepieItem recepieItem : recepieItems) {
			MenuItem inventoryItem = recepieItem.getInventoryItem();
			String unitCode = recepieItem.getUnitCode();
			if (inventoryItem == null || !inventoryItem.isInventoryItem()) {
				continue;
			}
			IUnit unit = DataProvider.get().getUnitByCode(unitCode);
			Double recipeItemQuantity = recepieItem.getQuantity();

			if (unit != null && unit instanceof PackagingUnit) {
				PackagingUnit packagingUnit = (PackagingUnit) unit;
				if (packagingUnit.isRecipeUnit()) {
					MenuItemDAO.getInstance().initializeUnits(inventoryItem);
					if (inventoryItem.getStockUnits() != null && inventoryItem.getStockUnits().size() > 0) {
						for (InventoryStockUnit itemStockUnit : inventoryItem.getStockUnits()) {
							if (itemStockUnit.getUnit() != null && packagingUnit.getCode().equals(itemStockUnit.getPackagingUnit().getCode())) {
								unitCode = itemStockUnit.getUnit().getCode();
								recipeItemQuantity = itemStockUnit.getUnit().getConversionRate() * itemStockUnit.getBaseUnitValue() * recipeItemQuantity;
								break;
							}
						}
					}
				}
			}
			InvMapKey menuItemUnitKey = new InvMapKey(inventoryItem.getId(), unitCode);
			Double previousValue = recepieItemMap.get(menuItemUnitKey);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			Double toBeAdjustQty = previousValue + itemQuantity * recipeItemQuantity * mainItemQty;
			recepieItemMap.put(menuItemUnitKey, toBeAdjustQty);
		}
	}

	public Double getCookingYield() {
		return cookingYield == null ? 1 : cookingYield;
	}

	public void setCookingYield(Double cookingYield) {
		this.cookingYield = cookingYield;
	}

	@XmlTransient
	public String getMenuItemName() {
		return DataProvider.get().getRecipeMenuItemName(this);
	}

	public void setMenuItemName(String menuItemName) {
	}

	/*
	 * recipe costing report methods
	 */

	public void populateRecipeItems(HashMap<String, RecepieItem> recepieItemMap, double itemQuantity, Recepie recipe, boolean group) {
		List<RecepieItem> recepieItems = getRecepieItems();
		if (recepieItems == null || recepieItems.size() == 0)
			return;

		for (RecepieItem recepieItem : recepieItems) {
			MenuItem inventoryItem = recepieItem.getInventoryItem();
			if (inventoryItem == null) {
				continue;
			}
			String key = inventoryItem.getId();
			if (!group) {
				key = recipe.getId() + "-" + key; //$NON-NLS-1$
			}
			key += "-" + recepieItem.getUnitCode(); //$NON-NLS-1$
			RecepieItem item = recepieItemMap.get(key);
			RecipeTable recipeTable = RecipeTableDAO.getInstance().findBy(inventoryItem.getId());
			if (recipeTable == null) {
				if (item == null) {
					item = new RecepieItem();
					item.setRecepie(recipe);
					item.setInventoryItem(recepieItem.getInventoryItem());
					item.setUnit(recepieItem.getUnit());
					item.setUnitCode(recepieItem.getUnitCode());
					if (!group) {
						recepieItemMap.put(recipe.getId() + "-" + key, item); //$NON-NLS-1$
					}
					else {
						recepieItemMap.put(key, item);
					}
				}
				item.setCost(inventoryItem.getCost(item.getUnitCode()));
				item.setGroupName(recipe.getName());
				item.setGroupId(recipe.getId());
				Double toBeAdjustQty = item.getQuantity() + (recepieItem.getQuantity() * itemQuantity);
				//percentage = (recepieItem.getPercentage() / 100.0);
				//Double toBeAdjustQty = item.getQuantity() + (itemQuantity * percentage);
				item.setQuantity(toBeAdjustQty);
			}
		}
	}

	public void populateRecipeSummaryItems(HashMap<String, RecepieItem> recepieItemMap, double itemQuantity, Recepie recipe, boolean group) {
		List<RecepieItem> recepieItems = getRecepieItems();
		if (recepieItems == null || recepieItems.size() == 0)
			return;

		for (RecepieItem recepieItem : recepieItems) {
			MenuItem inventoryItem = recepieItem.getInventoryItem();
			if (inventoryItem == null) {
				continue;
			}
			String key = inventoryItem.getId();
			RecepieItem item = recepieItemMap.get(key);
			if (item == null) {
				item = new RecepieItem();
				item.setRecepie(recipe);
				item.setInventoryItem(recepieItem.getInventoryItem());
				item.setUnit(inventoryItem.getUnit());
				item.setUnitCode(inventoryItem.getUnit() == null ? null : inventoryItem.getUnit().getUniqueCode());
				recepieItemMap.put(key, item);
			}
			item.setCost(inventoryItem.getCost(item.getUnitCode()));
			item.setGroupName(recipe.getName());
			item.setGroupId(recipe.getId());
			Double toBeAdjustQty = item.getQuantity()
					+ (inventoryItem.getBaseUnitQuantity(recepieItem.getUnitCode()) * recepieItem.getQuantity()) * itemQuantity;
			item.setQuantity(toBeAdjustQty);
		}
	}

	public int getCookingMin() {
		int timeInSeconds = getCookingTime();
		int hours = timeInSeconds / 3600;
		int secondsLeft = timeInSeconds - hours * 3600;
		int minutes = secondsLeft / 60;
		return minutes;
	}
}