/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.util;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.sql.Blob;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.TicketItem;
import com.floreantpos.swing.PosUIManager;

public class POSUtil {

	public static BufferedImage convertBlackAndWhiteImage(Image orginalImage) {
		int width = orginalImage.getWidth(null);
		int height = orginalImage.getHeight(null);
		BufferedImage resizedImg = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
		Graphics2D g2 = resizedImg.createGraphics();
		g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
		g2.drawImage(orginalImage, 0, 0, Color.WHITE, null);
		g2.dispose();
		for (int i = 0; i < height; i++) {
			for (int j = 0; j < width; j++) {
				Color c = new Color(resizedImg.getRGB(j, i));
				int red = (int) (c.getRed() * 0.299);
				int green = (int) (c.getGreen() * 0.587);
				int blue = (int) (c.getBlue() * 0.114);
				Color newColor = new Color(red + green + blue, red + green + blue, red + green + blue);
				resizedImg.setRGB(j, i, newColor.getRGB());
			}
		}
		return resizedImg;
	}

	public static ImageIcon convertToButtonImage(ImageIcon imageIcon) {
		return scaleImage(imageIcon, 80);
	}

	public static ImageIcon convertToBigButtonImage(ImageIcon image) {
		return scaleImage(image, 200);
	}

	public static ImageIcon scaleImage(ImageIcon image, int maxSize) {
		if (image == null) {
			return null;
		}
		int iconHeight = image.getIconHeight();
		int iconWidth = image.getIconWidth();

		if (iconWidth > iconHeight) {
			int width = maxSize;
			int height = (iconHeight * width) / iconWidth;
			Image scaledImage = image.getImage().getScaledInstance(PosUIManager.getSize(width), PosUIManager.getSize(height), Image.SCALE_SMOOTH);
			return new ImageIcon(scaledImage);
		}
		else {
			int height = maxSize;
			int width = (iconWidth * height) / iconHeight;
			Image scaledImage = image.getImage().getScaledInstance(PosUIManager.getSize(width), PosUIManager.getSize(height), Image.SCALE_SMOOTH);
			return new ImageIcon(scaledImage);
		}
	}

	public static boolean isBlankOrNull(String str) {
		if (str == null) {
			return true;
		}
		if (str.trim().equals("")) { //$NON-NLS-1$
			return true;
		}
		return false;
	}

	public static String escapePropertyKey(String propertyKey) {
		return propertyKey.replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static boolean getBoolean(String b) {
		if (b == null) {
			return false;
		}

		return Boolean.valueOf(b);
	}

	public static boolean getBoolean(String b, boolean defaultValue) {
		if (StringUtils.isBlank(b)) {
			return defaultValue;
		}

		return Boolean.valueOf(b);
	}

	public static boolean getBoolean(Boolean b) {
		if (b == null) {
			return false;
		}

		return b;
	}

	public static boolean getBoolean(Boolean b, boolean defaultValue) {
		if (b == null) {
			return defaultValue;
		}

		return b;
	}

	public static double getDouble(Double d) {
		if (d == null) {
			return 0;
		}

		return d;
	}

	public static Integer getInteger(Object result) {
		if (result != null && result instanceof Number) {
			return ((Number) result).intValue();
		}
		return 0;
	}

	public static double getDoubleAmount(Object result) {
		if (result != null && result instanceof Number) {
			return ((Number) result).doubleValue();
		}
		return 0;
	}

	public static double getRoundedDouble(Object result) {
		if (result != null && result instanceof Number) {
			return NumberUtil.round(((Number) result).doubleValue());
		}
		return 0;
	}

	public static int getInteger(Integer d) {
		if (d == null) {
			return 0;
		}

		return d;
	}

	public static int parseInteger(String s) {
		try {
			return Integer.parseInt(s);
		} catch (Exception x) {
			return 0;
		}
	}

	public static int parseInteger(String s, String parseErrorMessage) {
		try {
			return Integer.parseInt(s);
		} catch (Exception x) {
			throw new PosException(parseErrorMessage);
		}
	}

	public static double parseDouble(String s) {
		return NumberUtil.parseOrGetZero(s).doubleValue();
	}

	public static double parseDouble(String s, String parseErrorMessage, boolean mandatory) {
		if (mandatory && StringUtils.isBlank(s)) {
			throw new PosException(Messages.getString("POSUtil.1")); //$NON-NLS-1$
		}
		try {
			return NumberUtil.parse(s).doubleValue();
		} catch (Exception x) {
			if (mandatory) {
				throw new PosException(parseErrorMessage);
			}
			else {
				return 0;
			}
		}
	}

	public static String encodeURLString(String s) {
		try {
			return URLEncoder.encode(s, "UTF-8"); //$NON-NLS-1$
		} catch (Exception x) {
			return s;
		}
	}

	public static boolean isValidPassword(char[] password) {
		for (char c : password) {
			if (!Character.isDigit(c)) {
				return false;
			}
		}

		return true;
	}

	public static Blob convertImageToBlob(File imageFile, int targetSize) throws Exception {
		if (imageFile == null) {
			return null;
		}

		BufferedImage bufferedImage = ImageIO.read(imageFile);
		int iconWidth = bufferedImage.getWidth();
		int iconHeight = bufferedImage.getHeight();
		if (iconWidth > iconHeight) {
			int width = targetSize;
			int height = (iconHeight * width) / iconWidth;
			int type = bufferedImage.getType() == 0 ? BufferedImage.TYPE_INT_ARGB : bufferedImage.getType();
			bufferedImage = resizeImage(bufferedImage, width, height, type);
		}
		else {
			int height = targetSize;
			int width = (iconWidth * height) / iconHeight;
			int type = bufferedImage.getType() == 0 ? BufferedImage.TYPE_INT_ARGB : bufferedImage.getType();
			bufferedImage = resizeImage(bufferedImage, width, height, type);
		}

		ByteArrayOutputStream outputStream = new ByteArrayOutputStream(50000);

		String imageName = imageFile.getName();
		int lastIndexOfImage = imageName.lastIndexOf("."); //$NON-NLS-1$
		String imageExtension = imageName.substring(lastIndexOfImage + 1);

		if (imageExtension.equals("jpg")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "jpg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("jpeg")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "jpeg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("png")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "png", outputStream); //$NON-NLS-1$
		}
		else {
			ImageIO.write(bufferedImage, "gif", outputStream); //$NON-NLS-1$
		}

		byte[] byteArray = outputStream.toByteArray();
		outputStream.close();
		if (byteArray.length > 52428800) {
			throw new RuntimeException(Messages.getString("POSUtil.0")); //$NON-NLS-1$
		}
		return new javax.sql.rowset.serial.SerialBlob(byteArray);

	}

	@Deprecated
	public static Blob convertImageToBlob(File imageFile) throws Exception {

		if (imageFile == null) {
			return null;
		}

		BufferedImage bufferedImage = ImageIO.read(imageFile);
		if (bufferedImage.getWidth() > 400 || bufferedImage.getHeight() > 400) {
			int type = bufferedImage.getType() == 0 ? BufferedImage.TYPE_INT_ARGB : bufferedImage.getType();
			bufferedImage = resizeImage(bufferedImage, 400, 400, type);
		}

		ByteArrayOutputStream outputStream = new ByteArrayOutputStream(50000);

		String imageName = imageFile.getName();
		int lastIndexOfImage = imageName.lastIndexOf("."); //$NON-NLS-1$
		String imageExtension = imageName.substring(lastIndexOfImage + 1);

		if (imageExtension.equals("jpg")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "jpg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("jpeg")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "jpeg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("png")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "png", outputStream); //$NON-NLS-1$
		}
		else {
			ImageIO.write(bufferedImage, "gif", outputStream); //$NON-NLS-1$
		}

		byte[] byteArray = outputStream.toByteArray();
		if (byteArray.length > 500000) {
			throw new RuntimeException(Messages.getString("POSUtil.0")); //$NON-NLS-1$
		}
		return new javax.sql.rowset.serial.SerialBlob(byteArray);

	}

	public static Blob convertImageToBlob(BufferedImage bufferedImage, String imageExtension) throws Exception {
		if (bufferedImage == null) {
			return null;
		}
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream(50000);
		if (imageExtension.equals("jpg")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "jpg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("jpeg")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "jpeg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("png")) { //$NON-NLS-1$
			ImageIO.write(bufferedImage, "png", outputStream); //$NON-NLS-1$
		}
		else {
			ImageIO.write(bufferedImage, "gif", outputStream); //$NON-NLS-1$
		}
		byte[] byteArray = outputStream.toByteArray();
		if (byteArray.length > 500000) {
			throw new RuntimeException(Messages.getString("POSUtil.0")); //$NON-NLS-1$
		}
		return new javax.sql.rowset.serial.SerialBlob(byteArray);

	}

	public static Blob convertScaledImageToBlob(File imageFile) throws Exception {

		if (imageFile == null) {
			return null;
		}
		int width = 120;
		int height = 120;

		BufferedImage originalImage = ImageIO.read(imageFile);
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream(50000);

		int type = originalImage.getType() == 0 ? BufferedImage.TYPE_INT_ARGB : originalImage.getType();

		BufferedImage resizeImage = resizeImage(originalImage, width, height, type);

		String imageName = imageFile.getName();
		int lastIndexOfImage = imageName.lastIndexOf("."); //$NON-NLS-1$
		String imageExtension = imageName.substring(lastIndexOfImage + 1);

		if (imageExtension.equals("jpg")) { //$NON-NLS-1$
			ImageIO.write(resizeImage, "jpg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("jpeg")) { //$NON-NLS-1$
			ImageIO.write(resizeImage, "jpeg", outputStream); //$NON-NLS-1$
		}
		else if (imageExtension.equals("png")) { //$NON-NLS-1$
			ImageIO.write(resizeImage, "png", outputStream); //$NON-NLS-1$
		}
		else {
			ImageIO.write(resizeImage, "gif", outputStream); //$NON-NLS-1$
		}

		return new javax.sql.rowset.serial.SerialBlob(outputStream.toByteArray());

	}

	private static BufferedImage resizeImage(BufferedImage originalImage, int IMG_WIDTH, int IMG_HEIGHT, int type) {
		BufferedImage resizedImage = new BufferedImage(IMG_WIDTH, IMG_HEIGHT, type);
		Graphics2D g = resizedImage.createGraphics();
		g.drawImage(originalImage, 0, 0, IMG_WIDTH, IMG_HEIGHT, null);
		g.dispose();

		return resizedImage;
	}

	public static void storeLongProperty(Map<String, String> propertyMap, String propertyName, String propertyValue, int length) {
		Set<String> keys = propertyMap.keySet();
		Set<String> existingKeys = new HashSet<>();
		for (String key : keys) {
			if (key.startsWith(propertyName)) {
				existingKeys.add(key);
			}
		}
		for (String string : existingKeys) {
			propertyMap.remove(string);
		}

		if (propertyValue.length() <= length) {
			propertyMap.put(propertyName + "1", propertyValue); //$NON-NLS-1$
			return;
		}

		int count = 1;
		String valueToStore = ""; //$NON-NLS-1$
		while (true) {
			valueToStore = propertyValue.substring(0, length);
			propertyValue = propertyValue.substring(length);
			String nameToStore = propertyName + (count++);
			propertyMap.put(nameToStore, valueToStore);

			if (propertyValue.length() < length) {
				if (StringUtils.isNotEmpty(propertyValue)) {
					nameToStore = propertyName + (count++);
					propertyMap.put(nameToStore, propertyValue);
				}
				break;
			}
		}
	}

	public static String readLongProperty(Map<String, String> propertyMap, String propertyName) {
		List<Integer> propertyNumbers = new ArrayList<>();
		StringBuilder stringBuilder = new StringBuilder();
		Set<String> keys = propertyMap.keySet();
		for (String key : keys) {
			if (key.startsWith(propertyName)) {
				String numberString = key.replaceAll(propertyName, ""); //$NON-NLS-1$
				try {
					propertyNumbers.add(Integer.parseInt(numberString));
				} catch (Exception e) {
				}
			}
		}
		if (propertyNumbers.size() == 0) {
			return null;
		}

		Collections.sort(propertyNumbers);
		for (Integer integer : propertyNumbers) {
			String string = propertyMap.get(propertyName + integer);
			stringBuilder.append(string);
		}

		return stringBuilder.toString();
	}

	public static void removeLongProperty(Map<String, String> propertyMap, String propertyName) {
		for (Iterator<Map.Entry<String, String>> it = propertyMap.entrySet().iterator(); it.hasNext();) {
			Map.Entry<String, String> entry = it.next();
			if (entry.getKey().startsWith(propertyName)) {
				it.remove();
			}
		}
	}

	/**
	 * Copy values from collection selectedValues to collection toCollection. It also 
	 * removes element from toCollection if that element is not in selectedValues.
	 * 
	 * @param toCollection
	 * @param selectedValues
	 * @return
	 */
	public static Set copySelectedValues(Set toCollection, Set selectedValues) {
		if (toCollection == null) {
			toCollection = new LinkedHashSet<>();
			toCollection.addAll(selectedValues);
			return toCollection;
		}

		for (Iterator iterator = toCollection.iterator(); iterator.hasNext();) {
			Object existingValue = iterator.next();
			if (!selectedValues.contains(existingValue)) {
				iterator.remove();
			}
		}
		toCollection.addAll(selectedValues);
		return toCollection;
	}

	/**
	 * Copy values from collection selectedValues to collection toCollection. It also 
	 * removes element from toCollection if that element is not in selectedValues.
	 * 
	 * @param toCollection
	 * @param selectedValues
	 * @return
	 */
	public static List copySelectedValues(List toCollection, List selectedValues) {
		if (toCollection == null) {
			toCollection = new ArrayList();
			toCollection.addAll(selectedValues);
			return toCollection;
		}

		for (Iterator iterator = toCollection.iterator(); iterator.hasNext();) {
			Object existingValue = iterator.next();
			if (!selectedValues.contains(existingValue)) {
				iterator.remove();
			}
		}
		toCollection.addAll(selectedValues);
		return toCollection;
	}

	public static byte[] compress(String data) throws IOException {
		ByteArrayOutputStream bos = null;
		try {
			bos = new ByteArrayOutputStream(data.length());
			GZIPOutputStream gzip = new GZIPOutputStream(bos);
			gzip.write(data.getBytes("UTF-8")); //$NON-NLS-1$
			gzip.close();
			return bos.toByteArray();
		} finally {
			if (bos != null) {
				bos.close();
			}
		}
	}

	public static String decompress(byte[] compressed) throws IOException {
		ByteArrayInputStream bis = null;
		GZIPInputStream gis = null;
		BufferedReader br = null;
		try {
			bis = new ByteArrayInputStream(compressed);
			gis = new GZIPInputStream(bis);
			br = new BufferedReader(new InputStreamReader(gis, "UTF-8")); //$NON-NLS-1$
			StringBuilder sb = new StringBuilder();
			String line;
			while ((line = br.readLine()) != null) {
				sb.append(line);
			}
			return sb.toString();
		} finally {
			if (br != null) {
				br.close();
			}
			if (gis != null) {
				gis.close();
			}
			if (bis != null) {
				bis.close();
			}
		}
	}

	public static boolean isWindowsOs() {
		String OS = System.getProperty("os.name").toLowerCase(); //$NON-NLS-1$
		return (OS.indexOf("win") >= 0); //$NON-NLS-1$
	}

	public static boolean isAllowLocalItem() {
		String allowLocalItemProperty = System.getProperty(AppConstants.ALLOW_LOCAL_ITEM);
		if (StringUtils.isEmpty(allowLocalItemProperty)) {
			return true;
		}
		return Boolean.valueOf(allowLocalItemProperty);

	}

	public static String toFormattedString(List<?> list) {
		if (list == null || list.size() == 0) {
			return ""; //$NON-NLS-1$
		}
		String string = list.toString();
		return string.substring(1, string.length() - 1);
	}

	public static int getIntegerOrZero(String property) {
		try {
			return Integer.parseInt(property);
		} catch (Exception e) {
			return 0;
		}
	}

	public static boolean validatePhoneNo(String mobileNumber) {
		if (StringUtils.isBlank(mobileNumber)) {
			return false;
		}
		//String regex = "^[+]?(?:[0-9]?){3,14}[0-9]$";//$NON-NLS-1$
		String regex = "^[+]?[(]?[0-9]{0,3}[)]?[-\\s.]?[0-9]{0,3}[-\\s.]?[0-9]{3,9}$"; //$NON-NLS-1$
		Pattern pattern = Pattern.compile(regex);
		Matcher matcher = pattern.matcher(mobileNumber);
		if (matcher.matches()) {
			return true;
		}
		return false;
	}
	
	public static List<String> getStringIds(List<? extends Object> list, Class<?> clazz) {
		List<String> idList = new ArrayList<>();
		if (list != null) {
			try {
				for (Object model : list) {
					Method method = clazz.getMethod("getId", (Class<?>[]) null); //$NON-NLS-1$
					if (method != null) {
						Object id = method.invoke(model, (Object[]) null);
						if (id != null) {
							idList.add((String) id);
						}
					}

				}
			} catch (Exception e) {
				PosLog.error(POSUtil.class, e);
			}
		}

		return idList;
	}
	
	public static List<Integer> getIntegerIds(List<? extends Object> list, Class<?> clazz) {
		List<Integer> idList = new ArrayList<>();
		if (list != null) {
			try {
				for (Object model : list) {
					Method method = clazz.getMethod("getId", (Class<?>[]) null); //$NON-NLS-1$
					if (method != null) {
						Object id = method.invoke(model, (Object[]) null);
						if (id != null) {
							idList.add((Integer) id);
						}
					}

				}
			} catch (Exception e) {
				PosLog.error(POSUtil.class, e);
			}
		}

		return idList;
	}

	public static Component getFocusedWindow() {
		return null;
	}

	public static Date monthOfYear(int month, int year) {
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, year);
		cal.set(Calendar.MONTH, month);
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		return new Date(cal.getTimeInMillis());
	}

	public static List<Integer> getYearList() {
		Calendar cal = Calendar.getInstance();
		int year = cal.get(Calendar.YEAR);
		List<Integer> yearList = new ArrayList<Integer>();
		for (int i = year; i >= 2000; i--) {
			yearList.add(i);
		}
		return yearList;
	}

	public static List<Date> getDaysBetweenDates(Date startdate, Date enddate) {
		List<Date> dates = new ArrayList<Date>();
		Calendar calendar = new GregorianCalendar();
		calendar.setTime(startdate);

		while (calendar.getTime().before(enddate) || calendar.getTime().equals(enddate)) {
			Date result = calendar.getTime();
			dates.add(result);
			calendar.add(Calendar.DATE, 1);
		}
		return dates;
	}

	public static List<Date> getMonthBetweenDates(Date startdate, Date enddate) {
		List<Date> dates = new ArrayList<Date>();
		Calendar calendar = new GregorianCalendar();
		calendar.setTime(startdate);
		calendar.set(Calendar.DAY_OF_MONTH, 1);

		while (calendar.getTime().before(enddate) || calendar.getTime().equals(enddate)) {
			Date result = calendar.getTime();
			dates.add(result);
			calendar.add(Calendar.MONTH, 1);
		}
		return dates;
	}
	
	public static List copyList(List source) {
		if (source != null) {
			return new ArrayList(source);
		}
		
		return new ArrayList(1);
	}
	
	public static Set copySet(Set source) {
		if (source != null) {
			return new HashSet(source);
		}
		
		return new HashSet(1);
	}

	public static void clear(Collection collection) {
		if (collection != null) {
			collection.clear();
		}
	}
}
