package com.orocube.orostore.order.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.POSUtil;
import com.stripe.Stripe;
import com.stripe.exception.ApiConnectionException;
import com.stripe.exception.CardException;
import com.stripe.exception.InvalidRequestException;
import com.stripe.exception.StripeException;
import com.stripe.model.Card;
import com.stripe.model.Charge;
import com.stripe.model.ChargeCollection;
import com.stripe.model.Coupon;
import com.stripe.model.Customer;
import com.stripe.model.Invoice;
import com.stripe.model.InvoiceCollection;
import com.stripe.model.InvoiceItem;
import com.stripe.model.PaymentSource;
import com.stripe.model.Price;
import com.stripe.model.PriceCollection;
import com.stripe.model.Product;
import com.stripe.model.PromotionCode;
import com.stripe.model.PromotionCodeCollection;
import com.stripe.model.Subscription;
import com.stripe.model.SubscriptionItem;
import com.stripe.model.Token;
import com.stripe.param.ChargeListParams;
import com.stripe.param.CustomerUpdateParams;
import com.stripe.param.InvoiceCreateParams;
import com.stripe.param.InvoiceItemCreateParams;
import com.stripe.param.InvoiceItemCreateParams.Period;
import com.stripe.param.InvoiceItemCreateParams.PriceData;
import com.stripe.param.InvoiceListParams;
import com.stripe.param.InvoicePayParams;
import com.stripe.param.SubscriptionCancelParams;
import com.stripe.param.SubscriptionCreateParams;
import com.stripe.param.SubscriptionCreateParams.AddInvoiceItem;
import com.stripe.param.SubscriptionCreateParams.CollectionMethod;
import com.stripe.param.SubscriptionCreateParams.PaymentBehavior;
import com.stripe.param.SubscriptionCreateParams.ProrationBehavior;
import com.stripe.param.SubscriptionItemUpdateParams;
import com.stripe.param.SubscriptionListParams;
import com.stripe.param.SubscriptionListParams.CurrentPeriodStart;
import com.stripe.param.SubscriptionUpdateParams;

public class StripeBillingProcessor {
	private static final String CONNECTION_FAILED = "Failed to connect.";

	public static StripeBillingProcessor get() {
		return new StripeBillingProcessor();
	}

	public Product getProductById(String apiKey, String productId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			Product product = Product.retrieve(productId, params, null);
			return product;
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	public List<Price> getPrices(String apiKey, String productId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> planParams = new HashMap<>();
			planParams.put("limit", 100);
			planParams.put("active", true);
			planParams.put("product", productId);
			PriceCollection collection = Price.list(planParams);
			List<Price> plans = collection.getData();
			while (collection.getHasMore()) {
				planParams.put("starting_after", collection.getData().get(collection.getData().size() - 1).getId());
				plans.addAll((collection = Price.list(planParams)).getData());
			}
			return plans;
		} catch (Exception e0) {
			PosLog.error(getClass(), e0.getMessage());
			throw new RuntimeException(e0);
		}
	}

	public Price loadFullPrice(String apiKey, String priceId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			params.put("expand", Arrays.asList("tiers"));
			Price price = Price.retrieve(priceId, params, null);
			return price;
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	public Coupon getCoupon(String apiKey, String couponId, List<String> productIds) {
		if (StringUtils.isBlank(couponId)) {
			return null;
		}
		Stripe.apiKey = apiKey;
		try {
			List<String> expandList = new ArrayList<>();
			expandList.add("applies_to"); //$NON-NLS-1$  

			Map<String, Object> params = new HashMap<String, Object>();
			params.put("expand", expandList); //$NON-NLS-1$  
			Coupon coupon = Coupon.retrieve(couponId, params, null);
			if (!isValidCoupon(coupon, productIds)) {
				return null;
			}
			return coupon;
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0.getMessage());
		}
		return null;
	}

	private boolean isValidCoupon(Coupon coupon, List<String> productIds) {
		if (!coupon.getValid()) {
			return false;
		}
		if (productIds != null && !productIds.isEmpty() && coupon.getAppliesTo() != null) {
			List<String> products = coupon.getAppliesTo().getProducts();
			if (products != null && products.size() > 0) {
				for (String productId : productIds) {
					if (!products.contains(productId)) {
						return false;
					}
				}
			}
		}
		return true;
	}

	private com.stripe.model.Customer getCustomerById(String apiKey, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			com.stripe.model.Customer customer = com.stripe.model.Customer.retrieve(customerId);
			return customer;
		} catch (Exception e0) {
			//PosLog.error(getClass(), e0);
		}
		return null;
	}

	public com.stripe.model.Customer getCustomer(String apiKey, String customerId, String customerEmail) {
		return getCustomerById(apiKey, customerId);
	}

	public Card getCard(Customer stripeCustomer, String cardNumber, int expMonth, int expYear) {
		if (stripeCustomer == null) {
			return null;
		}
		List<PaymentSource> accounts = stripeCustomer.getSources().getData();
		if (!accounts.isEmpty()) {
			for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
				PaymentSource account = (PaymentSource) iterator.next();
				if (account instanceof Card) {
					Card creditCard = (Card) account;
					if (cardNumber.endsWith(creditCard.getLast4()) && creditCard.getExpMonth().intValue() == expMonth
							&& creditCard.getExpYear().intValue() == expYear) {
						return creditCard;
					}
				}
			}
		}
		return null;
	}

	public Customer createCustomer(String apiKey, String customerName, String customerEmail, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = null;
			try {
				customer = this.getCustomer(apiKey, customerId, customerEmail);
			} catch (Exception e1) {
			}
			if (customer == null) {
				Map<String, Object> customerParams = new HashMap<>();
				customerParams.put("email", customerEmail);
				customerParams.put("name", customerName);
				if (StringUtils.isNotEmpty(customerId)) {
					customerParams.put("id", customerId);
				}
				customer = Customer.create(customerParams);
			}
			return customer;
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	public Token createToken(String apiKey, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc, String holderName) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> tokenParams = new HashMap<String, Object>();

			Map<String, Object> cardMetaData = new HashMap<String, Object>();

			Map<String, Object> cardParams = new HashMap<String, Object>();
			cardParams.put("number", cardNumber);
			cardParams.put("exp_month", cardExpMonth);
			cardParams.put("exp_year", cardExpYear);
			cardParams.put("cvc", cvc);
			cardParams.put("name", holderName);
			cardParams.put("metadata", cardMetaData);

			tokenParams.put("card", cardParams);
			Token token = Token.create(tokenParams);
			return token;
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public Customer setSource(String apiKey, Customer customer, String tokenId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> customerParams = new HashMap<>();
			customerParams.put("source", tokenId);
			customer = customer.update(customerParams);
			return customer;
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	public Card createOrUpdateCard(String apiKey, Customer customer, Card stripeCard, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc,
			String holderName, boolean setAsDefault) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> cardMetaData = new HashMap<String, Object>();
			if (stripeCard == null) {
				Map<String, Object> tokenParams = new HashMap<String, Object>();
				Map<String, Object> cardParams = new HashMap<String, Object>();
				cardParams.put("number", cardNumber);
				cardParams.put("exp_month", cardExpMonth);
				cardParams.put("exp_year", cardExpYear);
				cardParams.put("cvc", cvc);
				cardParams.put("name", holderName);
				cardParams.put("metadata", cardMetaData);

				tokenParams.put("card", cardParams);
				Token token = Token.create(tokenParams);

				Map<String, Object> sourceParams = new HashMap<>();
				sourceParams.put("source", token.getId());
				Card card = (Card) customer.getSources().create(sourceParams);
				customer.update(CustomerUpdateParams.builder().setDefaultSource(card.getId()).build());
				return card;
			}
			else {
				Map<String, Object> cardParams = new HashMap<String, Object>();
				cardParams.put("exp_month", cardExpMonth); //$NON-NLS-1$
				cardParams.put("exp_year", cardExpYear); //$NON-NLS-1$
				cardParams.put("metadata", cardMetaData); //$NON-NLS-1$
				cardParams.put("name", holderName);
				stripeCard = (Card) stripeCard.update(cardParams);
				customer.update(CustomerUpdateParams.builder().setDefaultSource(stripeCard.getId()).build());
				return stripeCard;
			}
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public Customer createOrUpdateCustomerInfo(String apiKey, String customerName, String customerEmail, String customerId, String cardNumber,
			Integer cardExpMonth, Integer cardExpYear, String cvc, String holderName) {
		Customer stripeCustomer = getCustomer(apiKey, customerId, customerEmail);
		if (stripeCustomer == null) {
			stripeCustomer = createCustomer(apiKey, customerName, customerEmail, customerId);
			Token token = createToken(apiKey, cardNumber, cardExpMonth, cardExpYear, cvc, holderName);
			setSource(apiKey, stripeCustomer, token.getId());
		}
		else {
			updateCustomerIfNeeded(customerName, customerEmail, stripeCustomer);
			Card card = getCard(stripeCustomer, cardNumber, cardExpMonth, cardExpYear);
			createOrUpdateCard(apiKey, stripeCustomer, card, cardNumber, cardExpMonth, cardExpYear, cvc, holderName, true);
		}
		return stripeCustomer;
	}

	public void clearOldCards(String apiKey, Customer stripeCustomer, Card addedCard) {
		try {
			Stripe.apiKey = apiKey;
			List<PaymentSource> accounts = stripeCustomer.getSources().getData();
			if (!accounts.isEmpty()) {
				for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
					PaymentSource account = (PaymentSource) iterator.next();
					if (account instanceof Card) {
						Card creditCard = (Card) account;
						if (creditCard.getId().equals(addedCard.getId())) {
							continue;
						}
						creditCard.delete();
					}
				}
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public String createSubscription(String apiKey, Ticket ticket, String customerName, String customerEmail, String customerId, PosTransaction posTransaction,
			String couponId, Card card, String interval, boolean sendInvoice) {
		String cardNumber = posTransaction == null ? null : posTransaction.getCardNumber();
		String cardHolderName = posTransaction == null ? null : posTransaction.getCardHolderName();
		Integer cardExpMonth = posTransaction == null ? null : POSUtil.parseInteger(posTransaction.getCardExpMonth());
		Integer cardExpYear = posTransaction == null ? null : POSUtil.parseInteger(posTransaction.getCardExpYear());
		String cvc = posTransaction == null ? null : posTransaction.getCardCVV();
		Customer stripeCustomer = getCustomer(apiKey, customerId, customerEmail);
		if (stripeCustomer == null) {
			stripeCustomer = createCustomer(apiKey, customerName, customerEmail, customerId);
			if (!sendInvoice) {
				Token token = createToken(apiKey, cardNumber, cardExpMonth, cardExpYear, cvc, cardHolderName);
				setSource(apiKey, stripeCustomer, token.getId());
				card = token.getCard();
			}
		}
		else {
			updateCustomerIfNeeded(customerName, customerEmail, stripeCustomer);
			if (!sendInvoice) {
				if (card == null) {
					card = getCard(stripeCustomer, cardNumber, cardExpMonth, cardExpYear);
				}
				card = createOrUpdateCard(apiKey, stripeCustomer, card, cardNumber, cardExpMonth, cardExpYear, cvc, cardHolderName, true);
			}
		}
		String createSubscription = createSubscription(apiKey, ticket, stripeCustomer, card, couponId, interval, sendInvoice);
		if (card != null) {
			cardNumber = "****" + card.getLast4();
		}
		else if (StringUtils.isNotEmpty(cardNumber)) {
			cardNumber = cardNumber.substring(cardNumber.length() - 4, cardNumber.length());
		}
		if (posTransaction != null) {
			posTransaction.setCardNumber(cardNumber);
		}
		return createSubscription;
	}

	public void updateStripeCustomerIfExists(String apiKey, String customerId, String customerEmail, String customerName) {
		Customer stripeCustomer = getCustomerById(apiKey, customerId);
		if (stripeCustomer == null) {
			return;
		}
		updateCustomerIfNeeded(customerName, customerEmail, stripeCustomer);
	}

	private void updateCustomerIfNeeded(String customerName, String customerEmail, Customer stripeCustomer) {
		boolean equalsName = stripeCustomer.getName() != null && customerName != null && customerName.equalsIgnoreCase(stripeCustomer.getName());
		boolean equalsEmail = stripeCustomer.getEmail() != null && customerEmail != null && customerEmail.equalsIgnoreCase(stripeCustomer.getEmail());
		if (equalsName && equalsEmail) {
			return;
		}
		try {
			stripeCustomer.update(CustomerUpdateParams.builder().setName(customerName).setEmail(customerEmail).build());
			stripeCustomer.setName(customerName);
			stripeCustomer.setEmail(customerEmail);
		} catch (StripeException e) {
			PosLog.error(getClass(), e);
		}
	}

	public String createSubscription(String apiKey, Ticket ticket, Customer stripeCustomer, Card card, String couponId, String interval, boolean sendInvoice) {
		Stripe.apiKey = apiKey;
		try {
			//PromotionCode promotion = getPromotionByCode(apiKey, couponId, null);
			boolean oneTimeBilling = interval == null || interval.equals("one_time");
			if (oneTimeBilling) {
				return createOneTimePayment(ticket, stripeCustomer, card, couponId, sendInvoice);
			}
			else {
				return createSubscription(apiKey, ticket, stripeCustomer, card, interval, couponId, oneTimeBilling, sendInvoice);
			}
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	private String createOneTimePayment(Ticket ticket, Customer stripeCustomer, Card card, String couponId, boolean sendInvoice) throws Exception {
		List<com.stripe.param.InvoiceItemCreateParams.Builder> invoiceItemCreateBuilders = new ArrayList<>();
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			if (ticketItem.isService()) {
				//@formatter:off
				createOneTimePaymentItem(stripeCustomer, invoiceItemCreateBuilders, ticketItem);
				continue;
			}
			boolean isCombo = ticketItem.getComboItems() != null && ticketItem.getComboItems().size() > 0;
			if (isCombo) {
				if(ticketItem.getMenuItem().isModifiablePriceForComboItem()) {
					List<TicketItem> comboItems = ticketItem.getComboItems();
					for (TicketItem comboTicketItem : comboItems) {
						createOneTimePaymentItem(stripeCustomer, invoiceItemCreateBuilders, comboTicketItem);
					}
					continue;
				}
			}
			if(ticketItem.getMenuItem().isEditablePrice()) {
				com.stripe.param.InvoiceItemCreateParams.PriceData.Builder itemPriceBuilder = PriceData.builder();
				itemPriceBuilder = itemPriceBuilder
						.setProduct(ticketItem.getMenuItemId())
						.setCurrency("usd")
						.setUnitAmount((long) (ticketItem.getUnitPrice() * 100L));
				
				com.stripe.param.InvoiceItemCreateParams.Builder comboItemBuilder = InvoiceItemCreateParams.builder();
				comboItemBuilder
				.setQuantity(ticketItem.getQuantity().longValue())
				.setDescription(ticketItem.getName())
				.setCustomer(stripeCustomer.getId())
				.setCurrency("usd")
				.setPriceData(itemPriceBuilder.build());
				invoiceItemCreateBuilders.add(comboItemBuilder);
				continue;
			}
			//@formatter:on
			String oneTimePriceInterval = ticketItem.getProperty("one_time_price_interval");
			if (StringUtils.isBlank(oneTimePriceInterval)) {
				throw new PosException("Please check one time price interval is valid.");
			}
			int oneTimePriceIntervalCount = POSUtil.parseInteger(ticketItem.getProperty("one_time_price_interval_count"));
			if (oneTimePriceIntervalCount == 0) {
				throw new PosException("Please check one time price interval count is valid.");
			}
			Date startTime = new Date();
			Calendar calendar = Calendar.getInstance();
			if (oneTimePriceInterval.equalsIgnoreCase("Month")) {
				calendar.add(Calendar.MONTH, oneTimePriceIntervalCount);
			}
			else if (oneTimePriceInterval.equalsIgnoreCase("Year")) {
				calendar.add(Calendar.YEAR, oneTimePriceIntervalCount);
			}
			else {
				throw new PosException("Interval is not valid.");
			}
			Map<String, String> itemExtraParams = new HashMap<>();
			itemExtraParams.put("interval", oneTimePriceInterval.toLowerCase());
			itemExtraParams.put("interval_count", String.valueOf(oneTimePriceIntervalCount));
			//@formatter:off
			com.stripe.param.InvoiceItemCreateParams.Builder builder = InvoiceItemCreateParams.builder();
				builder
				.setQuantity(ticketItem.getQuantity().longValue())
				.setDescription(ticketItem.getName())
				.setCustomer(stripeCustomer.getId())
				.setCurrency("usd")
				.setMetadata(itemExtraParams)
				.setPeriod(Period.builder().setStart(startTime.getTime() / 1000L).setEnd(calendar.getTimeInMillis() / 1000L).build())
				.setPrice(ticketItem.getProperty("price_id")).build();

				invoiceItemCreateBuilders.add(builder);
			if(ticketItem.getComboItems()!=null) {
				for (TicketItem comboTicketItem : ticketItem.getComboItems()) {
					com.stripe.param.InvoiceItemCreateParams.PriceData.Builder itemPriceBuilder = PriceData.builder();
					itemPriceBuilder = itemPriceBuilder.setProduct(comboTicketItem.getMenuItemId())
							.setCurrency("usd")
							.setUnitAmount(0L);
					
					com.stripe.param.InvoiceItemCreateParams.Builder comboItemBuilder = InvoiceItemCreateParams.builder();
					comboItemBuilder
					.setQuantity(comboTicketItem.getQuantity().longValue())
					.setDescription(comboTicketItem.getName())
					.setCustomer(stripeCustomer.getId())
					.setCurrency("usd")
					.setMetadata(itemExtraParams)
					.setPeriod(Period.builder().setStart(startTime.getTime() / 1000L).setEnd(calendar.getTimeInMillis() / 1000L).build())
					.setPriceData(itemPriceBuilder.build());
					invoiceItemCreateBuilders.add(comboItemBuilder);
				}
			}
		}
		if(ticket.hasShippingCharge()) {
			com.stripe.param.InvoiceItemCreateParams.Builder shippingChargeBuilder = InvoiceItemCreateParams.builder();
			shippingChargeBuilder
			.setQuantity(1L)
			.setDescription("Shipping charge")
			.setCustomer(stripeCustomer.getId())
			.setCurrency("usd")
			.setPrice(ticket.getShippingChargePriceId()).build();
			invoiceItemCreateBuilders.add(shippingChargeBuilder);
		}
		for (com.stripe.param.InvoiceItemCreateParams.Builder builder : invoiceItemCreateBuilders) {
			InvoiceItem.create(builder.build());
		}
		String sourceId=card==null?null:card.getId();
		Map<String, String> metaData = new HashMap<String, String>();
		loadOrderProperties(ticket, metaData);
		com.stripe.param.InvoiceCreateParams.Builder builder = InvoiceCreateParams.builder()
				.setCustomer(stripeCustomer.getId())
				.setDefaultPaymentMethod(sourceId)
				.setDefaultSource(sourceId)
				.setCollectionMethod(sendInvoice?com.stripe.param.InvoiceCreateParams.CollectionMethod.SEND_INVOICE: com.stripe.param.InvoiceCreateParams.CollectionMethod.CHARGE_AUTOMATICALLY)
				.putAllMetadata(metaData);
		if (StringUtils.isNotBlank(couponId)) {
			List<com.stripe.param.InvoiceCreateParams.Discount> stripeDiscounts = new ArrayList<>();
			stripeDiscounts.add(com.stripe.param.InvoiceCreateParams.Discount.builder().setCoupon(couponId /*+ "_one_time"*/).build());
			builder = builder.setDiscounts(stripeDiscounts);
		}
		if(sendInvoice) {
			builder=builder.setDaysUntilDue(7L);
		}
		//@formatter:on
		Invoice invoice = Invoice.create(builder.build());
		try {
			if (sendInvoice) {
				sendInvoice(ticket, invoice);
			}
			else {
				invoice = invoice.pay();
				if (!invoice.getPaid()) {
					throw new PosException("Payment failed. Please check your card is valid and retry."); //$NON-NLS-1$
				}
			}
		} catch (Exception e) {
			invoice.delete();
			throw e;
		}
		return invoice.getId();
	}

	private void createOneTimePaymentItem(Customer stripeCustomer, List<com.stripe.param.InvoiceItemCreateParams.Builder> invoiceItemCreateBuilders,
			TicketItem ticketItem) {
		com.stripe.param.InvoiceItemCreateParams.Builder serviceItemBuilder = InvoiceItemCreateParams.builder();
		serviceItemBuilder.setQuantity(ticketItem.getQuantity().longValue()).setDescription(ticketItem.getName()).setCustomer(stripeCustomer.getId())
				.setCurrency("usd").setPrice(ticketItem.getProperty("price_id")).build();
		invoiceItemCreateBuilders.add(serviceItemBuilder);
	}

	private void loadOrderProperties(Ticket ticket, Map<String, String> metaData) {
		metaData.put("organization", ticket.getProperty("organization")); //$NON-NLS-1$ //$NON-NLS-2$
		metaData.put("commission", ticket.getProperty("commission")); //$NON-NLS-1$ //$NON-NLS-2$
		if (StringUtils.isNotBlank(ticket.getProperty("resellerId"))) {
			metaData.put("resellerId", ticket.getProperty("resellerId")); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	private String createSubscription(String apiKey, Ticket ticket, Customer stripeCustomer, Card card, String interval, String couponId,
			boolean oneTimeBilling, boolean sendInvoice) throws StripeException {
		//@formatter:off
		CollectionMethod method =sendInvoice?CollectionMethod.SEND_INVOICE: CollectionMethod.CHARGE_AUTOMATICALLY;
		String sourceId = card==null?null: card.getId();
		com.stripe.param.SubscriptionCreateParams.Builder builder = SubscriptionCreateParams.builder()
		   .setCancelAtPeriodEnd(interval!=null && interval.equals("one_time"))
		   .setCustomer(stripeCustomer.getId())
		   .setDefaultPaymentMethod(sourceId)
		   .setDefaultSource(sourceId)
		   .setPaymentBehavior(PaymentBehavior.ERROR_IF_INCOMPLETE)
		   .setCollectionMethod(method)
		   .setProrationBehavior(ProrationBehavior.ALWAYS_INVOICE)
		   .setCoupon(couponId);
		
		if(couponId != null) {
			PosLog.info(getClass(), "Coupon Id: " + couponId); //$NON-NLS-1$
		}
		
		if(sendInvoice) {
			builder=builder.setDaysUntilDue(7L);
		}
		
		com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval itemInterval = getInterval(interval);
		
		Map<String, Double> subscriptionProductMap=new HashMap<String, Double>();
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			if(!ticketItem.getMenuItem().isCloudProduct()) {
				continue;
			}
			String priceId = ticketItem.getPriceId(); 
			Double qua = subscriptionProductMap.get(priceId);
			if(qua == null) {
				qua = ticketItem.getQuantity();
			}
			else {
				qua += ticketItem.getQuantity();
			}
			subscriptionProductMap.put(priceId, qua);
		}
		
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			if(ticketItem.getMenuItem().isCloudProduct()) {
				continue;
			}
			if (ticketItem.getMenuItem().isEditablePrice()) {
				com.stripe.param.SubscriptionCreateParams.AddInvoiceItem.PriceData.Builder priceBuilder = com.stripe.param.SubscriptionCreateParams.AddInvoiceItem.PriceData.builder();
				priceBuilder= priceBuilder.setCurrency("usd")
							.setProduct(ticketItem.getMenuItemId())
							.setUnitAmount( (long) (ticketItem.getUnitPrice() * 100L));
				
				builder = builder.addAddInvoiceItem(
						AddInvoiceItem.builder().setQuantity(ticketItem.getQuantity().longValue()).setPriceData(priceBuilder.build()).build());
				continue;
			}
			boolean isCombo = ticketItem.getComboItems() != null && ticketItem.getComboItems().size() > 0;
			if (isCombo) {
				if(ticketItem.getMenuItem().isModifiablePriceForComboItem()) {
					List<TicketItem> comboItems = ticketItem.getComboItems();
					for (TicketItem comboTicketItem : comboItems) {
						boolean subscription=comboTicketItem.getBooleanProperty("subscription",false);
						if(subscription) {
							//builder = addItemWithPriceId(builder, comboTicketItem);
							String priceId = comboTicketItem.getPriceId();
							Double qua = subscriptionProductMap.get(priceId);
							if(qua == null) {
								qua = ticketItem.getQuantity();
							}
							else {
								qua += ticketItem.getQuantity();
							}
							subscriptionProductMap.put(priceId, qua);
						}
						else {
							builder = builder.addAddInvoiceItem(
									AddInvoiceItem.builder().setQuantity(comboTicketItem.getQuantity().longValue()).setPrice(comboTicketItem.getProperty("price_id")).build());
						}
					}
				}
				else{
					builder = addItem(builder, ticketItem.getMenuItemId(), ticketItem, itemInterval, (long) (ticketItem.getUnitPrice() * 100L));
					Map<String, String> metaData = new HashMap<String, String>();
					metaData.put("isComboItem", "true");
					for (TicketItem comboTicketItem : ticketItem.getComboItems()) {
						builder = addItem(builder, ticketItem.getMenuItemId(), comboTicketItem, itemInterval, 0L);
					}
				}
			}
			else if (ticketItem.isService()) {
				builder = builder.addAddInvoiceItem(
						AddInvoiceItem.builder().setQuantity(ticketItem.getQuantity().longValue()).setPrice(ticketItem.getProperty("price_id")).build());
			}
			else {
				String priceId = ticketItem.getPriceId(); 
				Double qua = subscriptionProductMap.get(priceId);
				if(qua == null) {
					qua = ticketItem.getQuantity();
				}
				else {
					qua += ticketItem.getQuantity();
				}
				subscriptionProductMap.put(priceId, qua);
				//builder = addItemWithPriceId(builder, ticketItem);
			}
		}
		for (String priceId : subscriptionProductMap.keySet()) {
			Double quantity=subscriptionProductMap.get(priceId);
			builder= builder.addItem(
					SubscriptionCreateParams.Item.builder()
					.setQuantity(quantity.longValue())
					.setPrice(priceId)
					.build());
		}
		if(ticket.hasShippingCharge()) {
			String deliveryChargePriceId = ticket.getShippingChargePriceId();
			if(StringUtils.isNotBlank(deliveryChargePriceId)) {
				builder = builder.addAddInvoiceItem(
						AddInvoiceItem.builder().setQuantity(1L).setPrice(deliveryChargePriceId).build());
			}
		}
		//@formatter:on
		Map<String, String> metaData = new HashMap<String, String>();

		loadOrderProperties(ticket, metaData);
		SubscriptionCreateParams params = builder.setMetadata(metaData).build();
		Subscription subscription = Subscription.create(params);
		if (!subscription.getStatus().equals("active")) { //$NON-NLS-1$
			throw new PosException("Payment failed. Please check your card is valid and retry."); //$NON-NLS-1$
		}
		if (sendInvoice) {
			Invoice invoice = getInvoice(apiKey, subscription.getLatestInvoice());
			sendInvoice(ticket, invoice);
		}
		return subscription.getId();
	}

	private void sendInvoice(Ticket ticket, Invoice invoice) throws StripeException {
		invoice = invoice.sendInvoice();
		ticket.addProperty("invoiceNo", invoice.getNumber());
		ticket.addProperty("invoiceId", invoice.getId());
		//ticket.addProperty("invoiceHostedUrl", invoice.getHostedInvoiceUrl());
	}

	private com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval getInterval(String interval) {
		com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval itemInterval = null;
		if (interval != null) {
			if (interval.equals("day")) {
				itemInterval = com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval.DAY;
			}
			else if (interval.equals("week")) {
				itemInterval = com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval.WEEK;
			}
			else if (interval.equals("month")) {
				itemInterval = com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval.MONTH;
			}
			else if (interval.equals("year")) {
				itemInterval = com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval.YEAR;
			}
		}
		return itemInterval;
	}

	//	private com.stripe.param.SubscriptionCreateParams.Builder addItemWithPriceId(com.stripe.param.SubscriptionCreateParams.Builder builder,
	//			TicketItem ticketItem) {
	//		//@formatter:off
    //		return addItemWithPriceId(builder, ticketItem.getProperty("price_id"), ticketItem.getQuantity());
    //		//@formatter:on
	//	}
	//	private com.stripe.param.SubscriptionCreateParams.Builder addItemWithPriceId(com.stripe.param.SubscriptionCreateParams.Builder builder, String priceId,
	//			Double quantity) {
//		//@formatter:off
//		return builder.addItem(
//				SubscriptionCreateParams.Item.builder()
//				.setQuantity(quantity.longValue())
//				.setPrice(priceId)
//				.build());
//		//@formatter:on
	//	}

	private com.stripe.param.SubscriptionCreateParams.Builder addItem(com.stripe.param.SubscriptionCreateParams.Builder builder, String parentMenuItemId,
			TicketItem ticketItem, com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.Interval interval, long amount) {
		//@formatter:off
		com.stripe.param.SubscriptionCreateParams.Item.PriceData.Builder itemPriceBuilder = SubscriptionCreateParams.Item.PriceData.builder();
		itemPriceBuilder = itemPriceBuilder.setProduct(ticketItem.getMenuItemId())
				.setRecurring(com.stripe.param.SubscriptionCreateParams.Item.PriceData.Recurring.builder()
				.setInterval(interval).build())
				.setUnitAmount(amount)
				.setCurrency("usd");
		return builder.addItem(
				SubscriptionCreateParams.Item.builder()
				.setQuantity(ticketItem.getQuantity().longValue())
				.setPriceData(itemPriceBuilder.build())
				.putMetadata("parentMenuItemId", parentMenuItemId)
				.build());
		//@formatter:on
	}

	//	private com.stripe.param.SubscriptionCreateParams.Builder addOnTimeInvoiceItem(String apiKey, com.stripe.param.SubscriptionCreateParams.Builder builder,
	//			TicketItem ticketItem) {
//		//@formatter:off
//		com.stripe.param.SubscriptionCreateParams.AddInvoiceItem.PriceData.Builder priceBuilder = SubscriptionCreateParams.AddInvoiceItem.PriceData.builder();
//		priceBuilder = priceBuilder
//				.setProduct(ticketItem.getMenuItemId())
//				.setUnitAmount((long) (ticketItem.getUnitPrice() * 100L))
//				.setCurrency("usd");
//
//		return builder.addAddInvoiceItem(
//				SubscriptionCreateParams.AddInvoiceItem.builder()
//				.setQuantity(ticketItem.getQuantity().longValue())
//				.setPriceData(priceBuilder.build())
//				.build());
//		//@formatter:on
	//	}

	public PromotionCode getPromotionByCode(String apiKey, String promoCode, List<String> productIds) {
		if (StringUtils.isBlank(promoCode)) {
			return null;
		}
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> params = new HashMap<>();
			params.put("code", promoCode); //$NON-NLS-1$  
			params.put("limit", 1); //$NON-NLS-1$  
			PromotionCodeCollection promotionCodes = PromotionCode.list(params);
			if (promotionCodes == null || promotionCodes.getData() == null || promotionCodes.getData().isEmpty()) {
				return null;
			}
			PromotionCode promotion = promotionCodes.getData().get(0);
			if (!promotion.getActive()) {
				return null;
			}
			if (getCoupon(apiKey, promotion.getCoupon().getId(), productIds) == null) {
				return null;
			}
			return promotion;
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0.getMessage());
		}
		return null;
	}

	public Subscription getSubscription(String apiKey, String subscriptionId) {
		if (StringUtils.isBlank(subscriptionId) || subscriptionId.startsWith("in_") || subscriptionId.startsWith("WTIN_")) {
			return null;
		}
		Stripe.apiKey = apiKey;
		try {
			return Subscription.retrieve(subscriptionId);
		} catch (Exception e0) {
			PosLog.error(getClass(), e0.getMessage());
		}
		return null;
	}

	public List<Subscription> getSubscriptions(String apiKey, String customerId, Date startingAfter) {
		Stripe.apiKey = apiKey;
		try {
			return Subscription
					.list(SubscriptionListParams.builder().setCustomer(customerId)
							.setCurrentPeriodStart(CurrentPeriodStart.builder().setGte(DateUtil.startOfDay(startingAfter).getTime() / 1000L).build()).build())
					.getData();
		} catch (Exception e0) {
			PosLog.error(getClass(), e0.getMessage());
		}
		return null;
	}

	public Invoice getInvoice(String apiKey, String invoiceId) {
		Stripe.apiKey = apiKey;
		try {
			Invoice invoice = Invoice.retrieve(invoiceId);
			if (invoice != null) {
				ChargeCollection paymentCollection = Charge.list(ChargeListParams.builder().setPaymentIntent(invoice.getPaymentIntent()).build());
				List<Charge> payments = paymentCollection.getData();
				if (payments == null || payments.isEmpty()) {
					return null;
				}
				for (Charge payment : payments) {
					if (payment.getRefunded() || payment.getAmountRefunded() > 0) {
						return null;
					}
				}
			}
			return invoice;
		} catch (Exception e0) {
			PosLog.error(getClass(), e0.getMessage());
		}
		return null;
	}

	public Invoice getSubscriptionInvoice(String apiKey, String invoiceId) {
		Stripe.apiKey = apiKey;
		try {
			Invoice invoice = Invoice.retrieve(invoiceId);
			return invoice;
		} catch (Exception e0) {
			PosLog.error(getClass(), e0.getMessage());
		}
		return null;
	}

	public Charge getChargeById(String apiKey, String chargeId) {
		Stripe.apiKey = apiKey;
		try {
			return Charge.retrieve(chargeId);
		} catch (Exception e0) {
			PosLog.error(getClass(), e0.getMessage());
		}
		return null;
	}

	public Subscription doCancelSubscriptionForItem(String apiKey, String subscriptionId, String menuItemId, boolean comboItem, TicketItem ticketItem)
			throws Exception {
		Stripe.apiKey = apiKey;
		try {
			//@formatter:off
			Subscription subscription=Subscription.retrieve(subscriptionId);
			if(subscription==null || !subscription.getStatus().equalsIgnoreCase("active")) {
				throw new PosException("Subscription is not valid.");
			}
			List<SubscriptionItem> items = subscription.getItems().getData();
			if(items.size()==1) {
				return subscription.update(SubscriptionUpdateParams.builder().setCancelAtPeriodEnd(true).build());
			}
			doCancelSubscriptionItem(menuItemId, ticketItem, items);
			subscription= Subscription.retrieve(subscriptionId);
			for (SubscriptionItem subscriptionItem : subscription.getItems().getData()) {
				if(subscriptionItem.getQuantity() > 0) {
					return subscription;
				}
			}
			return subscription.update(SubscriptionUpdateParams.builder().setCancelAtPeriodEnd(true).build());
		} catch (Exception e0) {
			throw e0;
		}
	}
	public Subscription doCancelSubscription(String apiKey, String subscriptionId)
			throws Exception {
		Stripe.apiKey = apiKey;
		try {
			Subscription subscription=Subscription.retrieve(subscriptionId);
			//return subscription.update(SubscriptionUpdateParams.builder().setCancelAtPeriodEnd(true).build());
			return subscription.cancel(SubscriptionCancelParams.builder().setInvoiceNow(false).setProrate(false).build());
		} catch (Exception e0) {
			throw e0;
		}
	}
	private void doCancelSubscriptionItem(String menuItemId, TicketItem ticketItem, List<SubscriptionItem> items) throws StripeException {
		MenuItem menuItem=ticketItem.getMenuItem();
		if (menuItem.isFeatured()) {
			for (TicketItem comboTicetItem : ticketItem.getComboItems()) {
				doCancelSubscriptionItem(comboTicetItem.getMenuItemId(),ticketItem.getMenuItemId(), items,true);
			}
		}
		else {
			doCancelSubscriptionItem(menuItemId,ticketItem.getMenuItemId(), items,false);
		}
	}

	private void doCancelSubscriptionItem(String menuItemId,String parentItemId, List<SubscriptionItem> items,boolean oneTimeLicense) throws StripeException {
		SubscriptionItem toBeCancelledSubscriptionItem=null;
		for (SubscriptionItem subscriptionItem : items) {
			if(oneTimeLicense) {
				String parentMenuItemId = subscriptionItem.getMetadata().get("parentMenuItemId");
				if (StringUtils.isBlank(parentMenuItemId)|| !parentItemId.equals(parentMenuItemId)) {
					continue;
				}
			}
			String stripeProductId = subscriptionItem.getPrice().getProduct();
			if(stripeProductId.equals(menuItemId)) {
				toBeCancelledSubscriptionItem=subscriptionItem;
				break;
			}
		}
		if(toBeCancelledSubscriptionItem==null) {
			throw new PosException("Subscription already canceled.");
		}
		toBeCancelledSubscriptionItem.update(SubscriptionItemUpdateParams.builder().setQuantity(0L).setProrationBehavior(com.stripe.param.SubscriptionItemUpdateParams.ProrationBehavior.NONE).build());
	}

	public Card removeCreditCard(String stripeApiKey, Card creditCard) {
		Stripe.apiKey = stripeApiKey;
		try {
			return creditCard.delete();
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public List<Invoice> getInvoices(String stripeApiKey,String subscriptionId) {
		Stripe.apiKey = stripeApiKey;
		List<Invoice> invoices=new ArrayList<>();
		try {
			InvoiceCollection invoiceCollection = Invoice.list(InvoiceListParams.builder()
					.setSubscription(subscriptionId).build());
			if (invoiceCollection != null) {
				invoices.addAll(invoiceCollection.getData());
			}
		} catch (Exception ex) {
			PosLog.error(getClass(), ex.getMessage());
		}
		return invoices;
	}

	public Subscription doPayPendingInvoice(String stripeApiKey, Card stripeCard,Subscription subscription, Invoice invoice) throws Exception {
		Stripe.apiKey = stripeApiKey;
		subscription.update(SubscriptionUpdateParams.builder().setDefaultPaymentMethod(stripeCard.getId()).setProrationBehavior(com.stripe.param.SubscriptionUpdateParams.ProrationBehavior.ALWAYS_INVOICE).setCollectionMethod(com.stripe.param.SubscriptionUpdateParams.CollectionMethod.CHARGE_AUTOMATICALLY).build());
		if(!invoice.getPaid()) {
			invoice.pay(InvoicePayParams.builder().setPaymentMethod(stripeCard.getId()).build());
		}
		return getSubscription(stripeApiKey, subscription.getId());
	}
	
	public void validateInvoice(String apiKey, Invoice invoice, Subscription subscription) {
		if (invoice == null) {
			return;
		}
		try {
			Stripe.apiKey = apiKey;
			ChargeCollection paymentCollection = Charge.list(ChargeListParams.builder().setPaymentIntent(invoice.getPaymentIntent()).build());
			List<Charge> payments = paymentCollection.getData();
			if (payments == null || payments.isEmpty() || isRefunded(payments)) {
				if (subscription != null) {
					throw new PosException("Payment for current subscription is incomplete. Could not activate license.");
				}
				throw new PosException("Order is cancelled.");
			}
		} catch (ApiConnectionException e) {
			throw new PosException(CONNECTION_FAILED);
		} catch (StripeException e) {
			throw new PosException(e.getMessage());
		}
	}
	private boolean isRefunded(List<Charge> payments) {
		for (Charge payment : payments) {
			if (payment.getRefunded() || payment.getAmountRefunded() > 0) {
				return true;
			}
		}
		return false;
	}

	public boolean isPaid(String apiKey, Subscription subscription) {
		if(subscription==null || subscription.getLatestInvoice()==null) {
			return false;
		}
		Invoice invoice=getSubscriptionInvoice(apiKey, subscription.getLatestInvoice());
		if(invoice==null) {
			return false;
		}
		try {
			Stripe.apiKey = apiKey;
			ChargeCollection paymentCollection = Charge.list(ChargeListParams.builder().setPaymentIntent(invoice.getPaymentIntent()).build());
			List<Charge> payments = paymentCollection.getData();
			if (payments == null || payments.isEmpty() || isRefunded(payments)) {
					return false;
			}
			return true;
		} catch (ApiConnectionException e) {
			throw new PosException(CONNECTION_FAILED);
		} catch (StripeException e) {
			throw new PosException(e.getMessage());
		}
	}
	
	public Subscription updateSubscription(String apiKey, Ticket ticket) {
		Stripe.apiKey = apiKey;
		try {
			Subscription subscription = Subscription.retrieve(ticket.getId());
			if (subscription == null) {
				throw new PosException("No order found.");
			}
			Map<String, Double> ticketItemMap = new HashMap<>();
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				Double exQua=ticketItemMap.get(ticketItem.getMenuItemId());
				if(exQua==null) {
					exQua=0D;
				}
				exQua+=ticketItem.getQuantity();
				ticketItemMap.put(ticketItem.getMenuItemId(), exQua);
			}
			//@formatter:off
			List<SubscriptionItem> items = subscription.getItems().getData();
//			boolean quantityDecrease=false;
//			com.stripe.param.SubscriptionUpdateParams.Builder decreaseQuantityBuilder = SubscriptionUpdateParams.builder()
//				    .setCancelAtPeriodEnd(false)
//				    .setPaymentBehavior(com.stripe.param.SubscriptionUpdateParams.PaymentBehavior.ERROR_IF_INCOMPLETE)
//				    .setProrate(false);
//			for (SubscriptionItem subscriptionItem : items) {
//				Double ticketItemQuantity = ticketItemMap.get(subscriptionItem.getPrice().getProduct());
//				if (ticketItemQuantity != null && ticketItemQuantity<subscriptionItem.getQuantity().doubleValue()) {
//					decreaseQuantityBuilder = decreaseQuantityBuilder.addItem(
//						      SubscriptionUpdateParams.Item.builder()
//						        .setId(subscriptionItem.getId())
//						        .setQuantity(ticketItemQuantity.longValue())
//						        .setPrice(subscriptionItem.getPrice().getId())
//						        .build());
//					quantityDecrease=true;
//				}
//			}
//			if(quantityDecrease) {
//				subscription =	subscription.update(decreaseQuantityBuilder.build());
//			}
			Map<String, TicketItem> comboItemMap=new HashMap<>();
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				if(ticketItem.getComboItems()!=null && ticketItem.getComboItems().size()>0) {
					for (TicketItem comboTicketItem : ticketItem.getComboItems()) {
						comboItemMap.put(ticketItem.getMenuItemId() + "_" + comboTicketItem.getMenuItemId(), comboTicketItem); //$NON-NLS-1$
					}
				}
			}
			com.stripe.param.SubscriptionUpdateParams.Builder increaseQuaOrNewItemBuilder = SubscriptionUpdateParams.builder()
				    .setCancelAtPeriodEnd(false)
				    .setPaymentBehavior(com.stripe.param.SubscriptionUpdateParams.PaymentBehavior.ERROR_IF_INCOMPLETE)
				    .setProrationBehavior(SubscriptionUpdateParams.ProrationBehavior.ALWAYS_INVOICE);
			boolean quantityIncrease=false;
			Map<String, SubscriptionItem> existingItemMap = new HashMap<>();
			for (SubscriptionItem subscriptionItem : items) {
				String parentMenuItemId=subscriptionItem.getMetadata()==null?null:subscriptionItem.getMetadata().get("parentMenuItemId"); //$NON-NLS-1$
				if(!subscriptionItem.getPrice().getProduct().equals(parentMenuItemId) && StringUtils.isNotBlank(parentMenuItemId)) {
					TicketItem comboTicketItem=comboItemMap.get(parentMenuItemId+"_"+subscriptionItem.getPrice().getProduct()); //$NON-NLS-1$
					if(comboTicketItem!=null){
						increaseQuaOrNewItemBuilder=increaseQuaOrNewItemBuilder.addItem(
							      SubscriptionUpdateParams.Item.builder()
							        .setId(subscriptionItem.getId())
							        .setQuantity(comboTicketItem.getQuantity().longValue())
							        .build());
						quantityIncrease=true;
					}
				}
				else {
					existingItemMap.put(subscriptionItem.getPrice().getProduct(), subscriptionItem);
					Double ticketItemQuantity = ticketItemMap.get(subscriptionItem.getPrice().getProduct());
					if (ticketItemQuantity != null && ticketItemQuantity>subscriptionItem.getQuantity().doubleValue()) {
						increaseQuaOrNewItemBuilder=increaseQuaOrNewItemBuilder.addItem(
							      SubscriptionUpdateParams.Item.builder()
							        .setId(subscriptionItem.getId())
							        .setQuantity(ticketItemQuantity.longValue())
							        .build());
						quantityIncrease=true;
					}
				}
			}
			com.stripe.param.SubscriptionUpdateParams.Item.PriceData.Recurring.Interval itemInterval=getUpdateInterval(ticket.getTicketItems().get(0));
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				if(StringUtils.isNotBlank(ticketItem.getId())) {
					continue;
				}
				if(existingItemMap.get(ticketItem.getMenuItemId())!=null) {
					continue;
				}
				boolean isCombo = ticketItem.getComboItems() != null && ticketItem.getComboItems().size() > 0;
				if (isCombo) {
					increaseQuaOrNewItemBuilder = addItem(increaseQuaOrNewItemBuilder, ticketItem.getMenuItemId(), ticketItem, itemInterval, (long) (ticketItem.getUnitPrice() * 100L));
					Map<String, String> metaData = new HashMap<String, String>();
					metaData.put("isComboItem", "true");
					for (TicketItem comboTicketItem : ticketItem.getComboItems()) {
						increaseQuaOrNewItemBuilder = addItem(increaseQuaOrNewItemBuilder, ticketItem.getMenuItemId(), comboTicketItem, itemInterval, 0L);
					}
				}
				else if (ticketItem.isService()) {
					increaseQuaOrNewItemBuilder = increaseQuaOrNewItemBuilder.addAddInvoiceItem(
							com.stripe.param.SubscriptionUpdateParams.AddInvoiceItem.builder().setQuantity(ticketItem.getQuantity().longValue()).setPrice(ticketItem.getProperty("price_id")).build());
				}
				else {
					increaseQuaOrNewItemBuilder = addItemWithPriceId(increaseQuaOrNewItemBuilder, ticketItem);
				}
				quantityIncrease=true;
			}
			if(!quantityIncrease) {
				throw new PosException("Please add new item or terminal.");
			}
			return subscription.update(increaseQuaOrNewItemBuilder.build());
			//@formatter:on
		} catch (CardException ex) {
			PosLog.error(StripeBillingProcessor.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (InvalidRequestException ex) {
			PosLog.error(StripeBillingProcessor.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (PosException ex) {
			throw ex;
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	private static com.stripe.param.SubscriptionUpdateParams.Item.PriceData.Recurring.Interval getUpdateInterval(TicketItem ticketItem) {
		String itemInterval = ticketItem.getProperty("price_interval");
		for (com.stripe.param.SubscriptionUpdateParams.Item.PriceData.Recurring.Interval interval : com.stripe.param.SubscriptionUpdateParams.Item.PriceData.Recurring.Interval
				.values()) {
			if (interval.getValue().equals(itemInterval)) {
				return interval;
			}
		}
		return null;
	}

	private com.stripe.param.SubscriptionUpdateParams.Builder addItem(com.stripe.param.SubscriptionUpdateParams.Builder builder, String parentMenuItemId,
			TicketItem ticketItem, com.stripe.param.SubscriptionUpdateParams.Item.PriceData.Recurring.Interval interval, long amount) {
		//@formatter:off
		com.stripe.param.SubscriptionUpdateParams.Item.PriceData.Builder itemPriceBuilder = SubscriptionUpdateParams.Item.PriceData.builder();
		itemPriceBuilder = itemPriceBuilder.setProduct(ticketItem.getMenuItemId())
				.setRecurring(com.stripe.param.SubscriptionUpdateParams.Item.PriceData.Recurring.builder()
				.setInterval(interval).build())
				.setUnitAmount(amount)
				.setCurrency("usd");
		return builder.addItem(
				SubscriptionUpdateParams.Item.builder()
				.setQuantity(ticketItem.getQuantity().longValue())
				.setPriceData(itemPriceBuilder.build())
				.putMetadata("parentMenuItemId", parentMenuItemId)
				.build());
		//@formatter:on
	}

	private com.stripe.param.SubscriptionUpdateParams.Builder addItemWithPriceId(com.stripe.param.SubscriptionUpdateParams.Builder builder,
			TicketItem ticketItem) {
		//@formatter:off
		return builder.addItem(
				SubscriptionUpdateParams.Item.builder()
				.setQuantity(ticketItem.getQuantity().longValue())
				.setPrice(ticketItem.getProperty("price_id"))
				.build());
		//@formatter:on
	}
}
