/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.awt.Color;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.base.BaseMenuModifier;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.GlobalIdGenerator;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@XmlAccessorType(XmlAccessType.FIELD)
@JsonIgnoreProperties(ignoreUnknown = true, value = { "image", "properties", "modifierGroups", "enableOnlineOrdering" })
@XmlRootElement
public class MenuModifier extends BaseMenuModifier implements TimedModel, PropertyContainer {
	private static final long serialVersionUID = 1L;

	private transient com.google.gson.JsonObject propertiesContainer;
	private String groupName;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/* [CONSTRUCTOR MARKER BEGIN] */
	public MenuModifier() {
	}

	/**
	 * Constructor for primary key
	 */
	public MenuModifier(java.lang.String id) {
		super(id);
	}

	/* [CONSTRUCTOR MARKER END] */

	private transient MenuItemModifierSpec menuItemModifierGroup;

	private String pageItemId;

	public TaxGroup getTaxGroup() {
		return DataProvider.get().getTaxGroupById(getTaxGroupId());
	}

	public void setTaxGroup(TaxGroup taxGroup) {
		String taxGroupId = null;
		if (taxGroup != null) {
			taxGroupId = taxGroup.getId();
		}
		super.setTaxGroupId(taxGroupId);
	}

	public MenuItemModifierSpec getMenuItemModifierGroup() {
		return menuItemModifierGroup;
	}

	public void setMenuItemModifierGroup(MenuItemModifierSpec menuItemModifierGroup) {
		this.menuItemModifierGroup = menuItemModifierGroup;
	}

	@Override
	public Integer getSortOrder() {
		return super.getSortOrder() == null ? 9999 : super.getSortOrder();
	}

	@Override
	public Integer getButtonColor() {
		Integer buttonColorCode = super.getButtonColor();
		if (buttonColorCode == null || buttonColorCode == 0 || buttonColorCode == -1316371) {
			buttonColorCode = Color.WHITE.getRGB();
		}
		return buttonColorCode;
	}

	@Override
	public Integer getTextColor() {
		Integer textColor = super.getTextColor();
		return textColor == 0 ? null : textColor;
	}

	@JsonIgnore
	public String getDisplayName() {
		Terminal terminal = Application.getInstance().getTerminal();
		if (terminal != null && terminal.isShowTranslatedName() && StringUtils.isNotEmpty(getTranslatedName())) {
			return getTranslatedName();
		}

		return super.getName();
	}

	@Override
	public String toString() {
		return getName();
	}

	public String getUniqueId() {
		return ("menu_modifier_" + getName() + "_" + getId()).replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}

	public double getPriceForSize(MenuItemSize size, boolean extra) {
		double defaultPrice = this.getPrice();
		if (size == null) {
			return defaultPrice;
		}

		List<PizzaModifierPrice> priceList = getPizzaModifierPriceList();
		if (priceList == null) {
			return defaultPrice;
		}

		for (PizzaModifierPrice pizzaModifierPrice : priceList) {
			if (size.equals(pizzaModifierPrice.getSize())) {
				if (extra) {
					return pizzaModifierPrice.getExtraPrice();
				}
				return pizzaModifierPrice.getPrice();
			}
		}

		return defaultPrice;
	}

	public double getPriceForMultiplier(Multiplier multiplier) {
		double defaultPrice = this.getPrice();
		if (multiplier == null || multiplier.isMain()) {
			return defaultPrice;
		}

		List<ModifierMultiplierPrice> priceList = getMultiplierPriceList();
		if (priceList == null || priceList.isEmpty()) {
			return defaultPrice * multiplier.getRate() / 100;
		}
		for (ModifierMultiplierPrice multiplierPrice : priceList) {
			if (multiplierPrice.getMultiplier() == null) {
				continue;
			}
			if (multiplier.getId().equals(multiplierPrice.getMultiplier().getId())) {
				return multiplierPrice.getPrice() != null ? multiplierPrice.getPrice() : 0.0;
			}
		}
		return defaultPrice * multiplier.getRate() / 100;
	}

	public double getPriceForSizeAndMultiplier(MenuItemSize size, boolean extra, Multiplier multiplier) {
		List<PizzaModifierPrice> priceList = getPizzaModifierPriceList();
		double regularPrice = 0;
		if (isPizzaModifier() && priceList != null) {
			for (PizzaModifierPrice pizzaModifierPrice : priceList) {
				if (size != null && size.equals(pizzaModifierPrice.getSize())) {
					List<ModifierMultiplierPrice> multiplierPriceList = pizzaModifierPrice.getMultiplierPriceList();
					if (multiplierPriceList != null) {
						Double multiplierPrice = null;
						for (ModifierMultiplierPrice price : multiplierPriceList) {
							String priceTableMultiplierName = price.getMultiplier().getId();
							if (priceTableMultiplierName.equals(Multiplier.REGULAR)) {
								regularPrice = price.getPrice();
								if (multiplier.getId().equals(Multiplier.REGULAR)) {
									return regularPrice;
								}
							}
							else if (priceTableMultiplierName.equals(multiplier.getId())) {
								multiplierPrice = price.getPrice();
							}
						}
						if (multiplierPrice != null) {
							return multiplierPrice;
						}
					}
				}
			}
		}
		return regularPrice * multiplier.getRate() / 100;
	}

	public List<TicketItemTax> getTaxByOrderType(OrderType type) {
		return getTaxByOrderType(type, null);
	}

	public List<TicketItemTax> getTaxByOrderType(OrderType type, TicketItem parentItem) {
		if (!isTaxable()) {
			return null;
		}
		Boolean taxSameAsItem = getBooleanProperty(AppConstants.TAX_SAME_AS_ITEM, false);
		if (taxSameAsItem && parentItem != null) {
			return parentItem.getTaxes();
		}

		TaxGroup taxGroup = getTaxGroup();
		if (taxGroup == null)
			return null;

		List<TicketItemTax> ticketItemTaxes = new ArrayList<>();
		List<Tax> taxes = taxGroup.getTaxes();
		if (taxes != null) {
			for (Tax tax : taxes) {
				TicketItemTax ticketItemTax = new TicketItemTax();
				ticketItemTax.setId(GlobalIdGenerator.generateGlobalId());
				ticketItemTax.setName(tax.getName());
				ticketItemTax.setRate(tax.getRate());
				ticketItemTaxes.add(ticketItemTax);
				//TODO:
				/*if (typeName != null) {
					double orderTypeTax = 0.0;
					try {
						orderTypeTax = Double.parseDouble(taxProp);
					} catch (Exception e) {
					}
				}*/
			}
		}
		return ticketItemTaxes;
	}

	public void setPageItemId(String id) {
		this.pageItemId = id;
	}

	public String getPageItemId() {
		return pageItemId;
	}

	@XmlTransient
	public double getTotalTaxRate() {
		TaxGroup taxGroup = getTaxGroup();
		if (taxGroup != null) {
			return taxGroup.getTotalTaxRate();
		}
		return 0.0;
	}

	public void setTotalTaxRate(double totalTaxRate) {
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}
		try {
			propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		} catch (Exception e) {
		}
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		try {
			propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		} catch (Exception e) {
		}
	}

	@Override
	@XmlTransient
	public JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new com.google.gson.JsonObject();
		}
		return propertiesContainer;
	}

	@XmlTransient
	public Boolean isEnableOnlineOrdering() {
		return getBooleanProperty("enable_online_ordering", true); //$NON-NLS-1$
	}

	public void setEnableOnlineOrdering(Boolean enableOnlineOrdering) {
		addProperty("enable_online_ordering", String.valueOf(enableOnlineOrdering)); //$NON-NLS-1$
	}

	@XmlTransient
	public String getModifierGroupName() {
		return groupName; //$NON-NLS-1$
	}

	public void setModifierGroupName(String groupName) {
		this.groupName = groupName;
	}

}