package com.floreantpos.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.PosLog;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class OneTimePrice {
	public static final String REF = "one_time_pricing"; //$NON-NLS-1$
	public static final String PROP_DURATION_TYPE = "durationType"; //$NON-NLS-1$
	public static final String PROP_DURATION = "duration"; //$NON-NLS-1$
	public static final String PROP_PRICE = "price"; //$NON-NLS-1$
	public static final String PROP_SUBSCRIPTION = "subscription"; //$NON-NLS-1$
	public static final String PROP_PRICE_ID = "priceId"; //$NON-NLS-1$
	public static final String PROP_PRICE_SAVE_TEXT = "priceSaveText"; //$NON-NLS-1$
	public static final String PROP_PRICE_SAVE_TEXT_IN_FRONT_PAGE = "priceSaveTextInFrontPage"; //$NON-NLS-1$
	public static final String PROP_TITLE = "title"; //$NON-NLS-1$

	public static final String YEAR = "year"; //$NON-NLS-1$
	public static final String MONTH = "month"; //$NON-NLS-1$
	public static final String ONE_TIME = "one_time"; //$NON-NLS-1$
	public static final String FOREVER = "forever"; //$NON-NLS-1$
	public static final String PROP_DEFAULT_PRICE = "defaultPrice"; //$NON-NLS-1$
	public static final String PROP_VISIBLE = "visible"; //$NON-NLS-1$

	private String durationType;
	private String duration;
	private double price;
	private boolean subscription;
	private String priceId;
	private String priceSaveText;
	private String priceSaveTextInFrontPage;
	private String title;
	private boolean defaultPrice;
	private boolean visible = true;
	private boolean showSubscriptionType;

	public OneTimePrice(String duration, String durationType, double price) {
		this.duration = duration;
		this.durationType = durationType;
		this.price = price;
	}

	public OneTimePrice() {
	}

	public void setShowSubscriptionType(boolean showSubscriptionType) {
		this.showSubscriptionType = showSubscriptionType;
	}

	public String getDurationType() {
		return durationType;
	}

	public void setDurationType(String durationType) {
		this.durationType = durationType;
	}

	public String getDuration() {
		return duration;
	}

	public void setDuration(String duration) {
		this.duration = duration;
	}

	public int getDurationAsInt() {
		if (duration == null) {
			return 0;
		}
		return POSUtil.parseInteger(duration);
	}

	public double getPrice() {
		return price;
	}

	public void setPrice(double price) {
		this.price = price;
	}

	public boolean isSubscription() {
		return subscription;
	}

	public void setSubscription(boolean subscription) {
		this.subscription = subscription;
	}

	public String getType() {
		if (isSubscription()) {
			return "Subscription";
		}

		return "One time";
	}

	public void setType(String type) {

	}

	public String getPriceId() {
		return priceId;
	}

	public void setPriceId(String priceId) {
		this.priceId = priceId;
	}

	public String getPriceSaveText() {
		return priceSaveText;
	}

	public void setPriceSaveText(String priceSaveText) {
		if (priceSaveText == null) {
			priceSaveText = "";
		}
		this.priceSaveText = priceSaveText;
	}

	public String getPriceSaveTextInFrontPage() {
		return priceSaveTextInFrontPage;
	}

	public void setPriceSaveTextInFrontPage(String priceSaveTextInFrontPage) {
		if (priceSaveTextInFrontPage == null) {
			priceSaveTextInFrontPage = "";
		}
		this.priceSaveTextInFrontPage = priceSaveTextInFrontPage;
	}

	public Double getMonthlyPrice() {
		double price = getPrice();
		if (price == 0) {
			return 0D;
		}
		int duration = POSUtil.parseInteger(getDuration());
		if (duration == 0) {
			duration = 1;
		}
		double monthlyPrice = price;
		if (getDurationType().equalsIgnoreCase(OneTimePrice.MONTH)) {
			monthlyPrice = price / duration;
		}
		else if (getDurationType().equalsIgnoreCase(OneTimePrice.YEAR)) {
			monthlyPrice = price / (duration * 12);
		}
		return monthlyPrice;
	}

	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		if (title == null) {
			title = "";
		}
		this.title = title;
	}

	public boolean isDefaultPrice() {
		return defaultPrice;
	}

	public void setDefaultPrice(boolean defaultPrice) {
		this.defaultPrice = defaultPrice;
	}

	@Override
	public String toString() {
		if (isSubscription() && showSubscriptionType) {
			return (durationType.equalsIgnoreCase(YEAR) ? "Annual subscription" : "Monthly subscription") + ", $" + NumberUtil.format(price) + " per terminal";
		}
		return "$" + NumberUtil.format(price) + " for " + duration + " "
				+ (POSUtil.parseInteger(duration) > 1 ? durationType.toLowerCase() + "s" : durationType.toLowerCase()) + " per terminal";
	}

	public static double getLowestPrice(MenuItem item) {
		List<OneTimePrice> oneTimePrices = buildOneTimePrices(item);
		if (oneTimePrices.isEmpty()) {
			return 0D;
		}
		Collections.sort(oneTimePrices, Comparator.comparing(OneTimePrice::getPrice));
		return oneTimePrices.get(0).getPrice();
	}

	public static List<OneTimePrice> buildOneTimePrices(MenuItem item) {
		List<OneTimePrice> oneTimePrices = new ArrayList<OneTimePrice>();
		try {
			if (!item.getProperties().has("one_time_pricing")) { //$NON-NLS-1$
				return oneTimePrices;
			}
			JSONArray jsonArray = item.getProperties().getJSONArray("one_time_pricing"); //$NON-NLS-1$
			for (int i = 0; i < jsonArray.length(); i++) {
				JSONObject jsonObject = (JSONObject) jsonArray.get(i);
				OneTimePrice oneTimePrice = new OneTimePrice();
				oneTimePrice.setDuration(getString(jsonObject, "duration")); //$NON-NLS-1$
				oneTimePrice.setDurationType(getString(jsonObject, "durationType")); //$NON-NLS-1$
				oneTimePrice.setPrice(getDouble(jsonObject, "price")); //$NON-NLS-1$
				oneTimePrices.add(oneTimePrice);
			}
			if (oneTimePrices.size() > 0) {
				Collections.sort(oneTimePrices, new Comparator<OneTimePrice>() {

					@Override
					public int compare(OneTimePrice o1, OneTimePrice o2) {
						String key1 = o1.getDurationType();
						String key2 = o2.getDurationType();
						return key1.compareTo(key2);
					}
				});
			}
		} catch (Exception e) {
			PosLog.error(OneTimePrice.class, e);
		}
		return oneTimePrices;
	}

	private static double getDouble(JSONObject jsonObject, String key) {
		if (!jsonObject.has(key)) {
			return 0D;
		}
		return jsonObject.getDouble(key);
	}

	private static String getString(JSONObject jsonObject, String key) {
		if (!jsonObject.has(key)) {
			return null;
		}
		return jsonObject.getString(key);
	}

	public boolean isVisible() {
		return visible;
	}

	public void setVisible(boolean visible) {
		this.visible = visible;
	}
}
