/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Tax;
import com.floreantpos.model.TaxGroup;
import com.floreantpos.util.NameBaseIdGenerator;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class TaxDAO extends BaseTaxDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public TaxDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdate(Tax tax) throws HibernateException {
		if (StringUtils.isBlank(tax.getId())) {
			tax.setId(NameBaseIdGenerator.generateId(tax.getName()));
		}
		super.saveOrUpdate(tax);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Tax tax = (Tax) obj;
		if (tax == null) {
			throw new PosException(Messages.getString("TaxDAO.0")); //$NON-NLS-1$
		}
		tax.setDeleted(true);
		update(tax, session);
		deleteTaxFromTaxGroup(tax, session);
	}

	@Override
	public List<Tax> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(Order.asc(Tax.PROP_NAME));
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public void deleteTaxFromTaxGroup(Tax tax) {
		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			deleteTaxFromTaxGroup(tax, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void deleteTaxFromTaxGroup(Tax tax, Session session) {
		Criteria criteria = session.createCriteria(TaxGroup.class);
		addDeletedFilter(criteria);
		criteria.createAlias("taxes", "tax"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("tax." + Tax.PROP_ID, tax.getId())); //$NON-NLS-1$
		List<TaxGroup> taxGroups = criteria.list();
		if (taxGroups == null || taxGroups.isEmpty()) {
			return;
		}
		for (TaxGroup taxGroup : taxGroups) {
			taxGroup.getTaxes().remove(tax);
			update(taxGroup, session);
		}
	}

	public Tax findByTaxRate(double taxRate) {
		Session session = null;
		try {
			session = createNewSession();

			Criteria criteria = session.createCriteria(Tax.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Tax.PROP_RATE, taxRate));
			return (Tax) criteria.uniqueResult();
		} finally {
			closeSession(session);
		}
	}

	public boolean nameExists(Tax tax, String code) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Tax.PROP_NAME, code).ignoreCase());
			if (tax.getId() != null)
				criteria.add(Restrictions.ne(Tax.PROP_ID, tax.getId()));

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();

			if (rowCount == null)
				return false;

			return (rowCount.intValue() > 0) ? true : false;

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<Tax> findAllUnSyncTax() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Criterion falseCriteria = Restrictions.eq(Tax.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(Tax.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateTaxs(List<Tax> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateTaxs(dataList, updateLastUpdateTime, updateSyncTime, false);
	}

	public void saveOrUpdateTaxs(List<Tax> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean forceSave) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator<Tax> iterator = dataList.iterator(); iterator.hasNext();) {
				Tax item = (Tax) iterator.next();
				Tax existingItem = get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					if (!forceSave && !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingItem.getId();
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setId(id);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public Tax get(String id, String outletId) {
		return get(new Tax(id, outletId));
	}

	@SuppressWarnings("unchecked")
	public List<Tax> findByOutletId(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Tax.PROP_OUTLET_ID, outletId));
			}
			return criteria.list();
		}
	}

}