/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.util;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import com.floreantpos.model.DayPart;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.PriceShift;
import com.floreantpos.model.Shift;
import com.floreantpos.model.dao.DayPartDAO;
import com.floreantpos.model.util.DataProvider;

public class ShiftUtil {

	private static final String DEFAULT_SHIFT = "DEFAULT SHIFT"; //$NON-NLS-1$
	private static final Calendar calendar = Calendar.getInstance();
	private static final Calendar calendar2 = Calendar.getInstance();
	private static final NumberFormat format = new DecimalFormat("00"); //$NON-NLS-1$

	private static Shift currentShift;

	static {
		calendar.clear();
	}

	public static void reInitialize() {
		calendar.clear();
	}

	/**
	 * For shift, we only care for hour and minute, not date.
	 * 
	 * @param shiftTime
	 * @return
	 */
	public static Date formatShiftTime(Date shiftTime) {
		calendar.clear();
		calendar2.setTime(shiftTime);

		calendar.set(Calendar.HOUR, calendar2.get(Calendar.HOUR));
		calendar.set(Calendar.MINUTE, calendar2.get(Calendar.MINUTE));
		calendar.set(Calendar.AM_PM, calendar2.get(Calendar.AM_PM));

		return calendar.getTime();
	}

	public static Date buildShiftStartTime(int startHour, int startMin, int startAmPm, int endHour, int endMin, int endAmPm) {
		startHour = startHour == 12 ? 0 : startHour;

		calendar.clear();

		calendar.set(Calendar.HOUR, startHour);
		calendar.set(Calendar.MINUTE, startMin);

		calendar.set(Calendar.AM_PM, startAmPm);

		return calendar.getTime();
	}

	public static Date buildShiftEndTime(int startHour, int startMin, int startAmPm, int endHour, int endMin, int endAmPm) {
		endHour = endHour == 12 ? 0 : endHour;

		calendar.clear();

		calendar.set(Calendar.HOUR, endHour);
		calendar.set(Calendar.MINUTE, endMin);

		calendar.set(Calendar.AM_PM, endAmPm);

		if (startAmPm == Calendar.PM && endAmPm == Calendar.AM) {
			calendar.add(Calendar.DAY_OF_MONTH, 1);
		}

		return calendar.getTime();
	}

	public static Date buildShiftWithoutAmPm(int hour, int min, int sec) {
		Calendar calendar = Calendar.getInstance();
		calendar.clear();

		calendar.set(Calendar.HOUR_OF_DAY, hour);
		calendar.set(Calendar.MINUTE, min);
		calendar.set(Calendar.SECOND, sec);

		return calendar.getTime();
	}

	public static String buildShiftTimeRepresentation(Date shiftTime) {
		calendar.setTime(shiftTime);

		String s = ""; //$NON-NLS-1$
		s = format.format(calendar.get(Calendar.HOUR) == 0 ? 12 : calendar.get(Calendar.HOUR));
		s += ":" + format.format(calendar.get(Calendar.MINUTE)); //$NON-NLS-1$
		s += calendar.get(Calendar.AM_PM) == Calendar.AM ? " AM" : " PM"; //$NON-NLS-1$ //$NON-NLS-2$
		return s;
	}

	public static Shift getCurrentShift() {
		Calendar calendar = Calendar.getInstance();
		if (currentShift != null && calendar.before(currentShift.getEndTime())) {
			return currentShift;
		}
		Calendar calendar2 = Calendar.getInstance();
		calendar.clear();

		calendar.set(Calendar.HOUR, calendar2.get(Calendar.HOUR));
		calendar.set(Calendar.MINUTE, calendar2.get(Calendar.MINUTE));
		calendar.set(Calendar.AM_PM, calendar2.get(Calendar.AM_PM));

		Date currentTime = calendar.getTime();
		List<DayPart> daryPartShifts = DataProvider.get().getDaryPartShifts();
		currentShift = findCurrentShift(currentTime, daryPartShifts);
		if (currentShift != null) {
			return currentShift;
		}

		calendar.add(Calendar.DATE, 1);
		currentTime = calendar.getTime();

		currentShift = findCurrentShift(currentTime, daryPartShifts);
		if (currentShift != null) {
			return currentShift;
		}
		currentShift = getDefaultShift();
		return currentShift;
	}

	private static Shift findCurrentShift(Date currentTime, List<? extends Shift> shifts) {
		Shift smallestShift = null;
		if (shifts == null) {
			return null;
		}
		for (Shift shift : shifts) {
			Date startTime = new Date(shift.getStartTime().getTime());
			Date endTime = new Date(shift.getEndTime().getTime());

			if (currentTime.after(startTime) && currentTime.before(endTime)) {
				if (smallestShift != null && shift.getShiftLength() < smallestShift.getShiftLength()) {
					smallestShift = shift;
				}
				else if (smallestShift == null) {
					smallestShift = shift;
				}
			}
		}

		return smallestShift;
	}

	private static Shift getDefaultShift() {
		Calendar calendar = Calendar.getInstance();
		Calendar calendar2 = Calendar.getInstance();

		calendar.clear();
		calendar.set(Calendar.HOUR, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);

		calendar2.clear();
		calendar2.set(Calendar.HOUR, 23);
		calendar2.set(Calendar.MINUTE, 59);
		calendar2.set(Calendar.SECOND, 59);

		DayPart defaultShift = DayPartDAO.getInstance().findByName(DEFAULT_SHIFT);
		if (defaultShift == null) {
			defaultShift = new DayPart();
			defaultShift.setName(DEFAULT_SHIFT);
			defaultShift.setStartTime(calendar.getTime());
			defaultShift.setEndTime(calendar2.getTime());
			defaultShift.setShiftLength(calendar2.getTimeInMillis() - calendar.getTimeInMillis());

			DayPartDAO.getInstance().saveOrUpdate(defaultShift);
		}

		return defaultShift;
	}

	public static List<PriceShift> getCurrentPricShifts() {
		List<PriceShift> priceShifts = DataProvider.get().getPriceShifts();
		if (priceShifts == null || priceShifts.isEmpty())
			return null;

		Calendar calendar = Calendar.getInstance();
		Calendar calendar2 = Calendar.getInstance();
		calendar.clear();

		calendar.set(Calendar.HOUR, calendar2.get(Calendar.HOUR));
		calendar.set(Calendar.MINUTE, calendar2.get(Calendar.MINUTE));
		calendar.set(Calendar.AM_PM, calendar2.get(Calendar.AM_PM));

		Date currentTime = calendar.getTime();
		List<PriceShift> currentPriceShifts = new ArrayList<>();
		for (PriceShift shift : priceShifts) {
			Date startTime = new Date(shift.getStartTime().getTime());
			Date endTime = new Date(shift.getEndTime().getTime());

			String daysOfWeek = shift.getDaysOfWeek();
			if (daysOfWeek != null && daysOfWeek.contains(String.valueOf(calendar2.get(Calendar.DAY_OF_WEEK))) && currentTime.after(startTime)
					&& currentTime.before(endTime)) {
				currentPriceShifts.add(shift);
			}
		}
		return currentPriceShifts;
	}

	public static List<MenuShift> getCurrentMenuShifts() {
		List<MenuShift> menuShifts = DataProvider.get().getMenuShifts();
		if (menuShifts == null || menuShifts.isEmpty())
			return null;

		Calendar calendar = Calendar.getInstance();
		Calendar calendar2 = Calendar.getInstance();
		calendar.clear();

		calendar.set(Calendar.HOUR, calendar2.get(Calendar.HOUR));
		calendar.set(Calendar.MINUTE, calendar2.get(Calendar.MINUTE));
		calendar.set(Calendar.AM_PM, calendar2.get(Calendar.AM_PM));

		Date currentTime = calendar.getTime();

		List<MenuShift> currentMenuShifts = new ArrayList<>();
		for (MenuShift shift : menuShifts) {
			Date startTime = new Date(shift.getStartTime().getTime());
			Date endTime = new Date(shift.getEndTime().getTime());

			String daysOfWeek = shift.getDaysOfWeek();
			if (daysOfWeek != null && daysOfWeek.contains(String.valueOf(calendar2.get(Calendar.DAY_OF_WEEK))) && currentTime.after(startTime)
					&& currentTime.before(endTime)) {
				currentMenuShifts.add(shift);
			}
		}
		return currentMenuShifts;
	}

	//	public static void adjustUserShiftAndClockIn(User user, Shift currentShift) {
	//		Application application = Application.getInstance();
	//		Calendar currentTime = Calendar.getInstance();
	//
	//		if (user.isClockedIn() != null && user.isClockedIn().booleanValue()) {
	//			Shift userShift = user.getCurrentShift();
	//			Date userLastClockInTime = user.getLastClockInTime();
	//			long elaspedTimeSinceLastLogin = Math.abs(currentTime.getTimeInMillis() - userLastClockInTime.getTime());
	//
	//			if (userShift != null) {
	//				if (!userShift.equals(currentShift)) {
	//					reClockInUser(currentTime, user, currentShift);
	//				}
	//				else if (userShift.getShiftLength() != null && (elaspedTimeSinceLastLogin >= userShift.getShiftLength())) {
	//					reClockInUser(currentTime, user, currentShift);
	//				}
	//			}
	//			else {
	//				user.doClockIn(application.getTerminal(), currentShift, currentTime);
	//			}
	//		}
	//		else {
	//			user.doClockIn(application.getTerminal(), currentShift, currentTime);
	//		}
	//	}

	//	private static void reClockInUser(Calendar currentTime, User user, Shift currentShift) {
	//		POSMessageDialog.showMessage("You will be clocked out from previous Shift");
	//
	//		Application application = Application.getInstance();
	//		AttendenceHistoryDAO attendenceHistoryDAO = new AttendenceHistoryDAO();
	//
	//		AttendenceHistory attendenceHistory = attendenceHistoryDAO.findHistoryByClockedInTime(user);
	//		if (attendenceHistory == null) {
	//			attendenceHistory = new AttendenceHistory();
	//			Date lastClockInTime = user.getLastClockInTime();
	//			Calendar c = Calendar.getInstance();
	//			c.setTime(lastClockInTime);
	//			attendenceHistory.setClockInTime(lastClockInTime);
	//			attendenceHistory.setClockInHour(Short.valueOf((short) c.get(Calendar.HOUR)));
	//			attendenceHistory.setUser(user);
	//			attendenceHistory.setTerminal(application.getTerminal());
	//			attendenceHistory.setShift(user.getCurrentShift());
	//		}
	//		user.doClockOut(attendenceHistory, currentShift, currentTime);
	//		user.doClockIn(application.getTerminal(), currentShift, currentTime);
	//	}

	public static Date buildStartTime(int startHour, int startMin, int endHour, int endMin) {
		calendar.clear();

		calendar.set(Calendar.HOUR_OF_DAY, startHour);
		calendar.set(Calendar.MINUTE, startMin);

		return calendar.getTime();
	}

	public static Date buildEndTime(int startHour, int startMin, int endHour, int endMin) {
		calendar.clear();

		calendar.set(Calendar.HOUR_OF_DAY, endHour);
		calendar.set(Calendar.MINUTE, endMin);

		return calendar.getTime();
	}
}
